#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"
#include "templates.h"

FV_SPEC = {
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int ipmi_sel_build_table_asp(int eid, webs_t wp, int argc, char **argv);

int ipmi_sel_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("ipmiGetSelTable", ipmi_sel_build_table_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_SEL, ACL_OBJ_IPMI_STATUS);
    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

/* the formular handler (parses the buttons) */
static int pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    if (form_button_clicked(wp, "action_clear_log")) {
        if (ipmi_sel_clear_log(wp)) {
            set_response(wp, ERIC_RESPONSE_ERROR, _("Could not clear System Event Log!"));
        } else {
            set_response(wp, ERIC_RESPONSE_OK, _("Successfully cleared System Event Log."));
        }
    } else if (form_button_clicked(wp, "action_refresh")) {
            /* dont show "Operation Completed" */
            set_response(wp, ERIC_RESPONSE_OK, "");
    }

    return 0;
}

/* backend for the buttons */
int ipmi_sel_clear_log(webs_t wp) {
    pp_ipmi_return_t ipmi_ret;
    int ret;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    pp_log("Clearing IPMI Event Log.\n");
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_CLEAR, NULL, &ipmi_ret, NULL, wp->user);

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return ret;
}

/* ASP calls */
static char* get_string(char *s) {
    static char nbsp[] = "&nbsp;";
    if (s) pp_trim_string(s);
    return (s && s[0]) ? s : nbsp;
}

static void add_sel_table_entry(pp_strstream_t *sel_table, int idx, char *type, char *date,
                                char *mytime, char *source, char *description, char* direction) {
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(sel_table,
            idx % 2 ? "<tr class=\"loggingRowOdd\">\n" : "<tr class=\"loggingRowEven\">\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(type));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(date));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(mytime));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(source));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(description));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "  <td>");
    pp_strappend(sel_table, get_string(direction));
    pp_strappend(sel_table, "</td>\n");

    pp_strappend(sel_table, "</tr>\n");
#else /* PP_FEAT_RARITAN_DESIGN */
    (void)idx;
    pp_strappend(sel_table, "<script>table_entry('");
    pp_strappend(sel_table, get_string(type));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(date));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(mytime));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(source));

    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(description));
    
    pp_strappend(sel_table, "','");
    pp_strappend(sel_table, get_string(direction));
    pp_strappend(sel_table, "');</script>\n");
#endif /* PP_FEAT_RARITAN_DESIGN */
}

static void add_empty_table_entry(pp_strstream_t *sel_table, const char *str) {
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(sel_table, "<tr><td colspan=\"6\" class=\"normal\">");
    pp_strappend(sel_table, str);
    pp_strappend(sel_table, "</td></tr>\n");
#else
    pp_strappend(sel_table, "<script>table_entry('");
    pp_strappend(sel_table, str);
    pp_strappend(sel_table, "','','','','','');</script>\n");
#endif    
}

static int ipmi_sel_build_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    char empty_str[1] = { 0 };
    pp_ipmi_return_t ipmi_ret;
    pp_strstream_t sel_table = PP_STRSTREAM_INITIALIZER;

    pp_strstream_init(&sel_table);
    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    if (!pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_LIST, NULL, &ipmi_ret, NULL, wp->user)) {
        int i;
        int sel_count = vector_size(ipmi_ret.data.sel_list);

        pp_log("found %d SEL entries.\n", sel_count);

        if (sel_count > 0) {
	        char * s_preinit = strdup(_("Pre-Init"));
	        char * s_unknown = strdup(_("Unknown"));
            for (i = sel_count - 1; i >= 0; i--) {
                pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);

                if (!entry) {
                    pp_log("Error: could not read entry %d\n", i);
                    continue;
                }

                add_sel_table_entry(&sel_table, sel_count - 1 - i,
                    entry->record_type_string.buf,
                    entry->timestamp.is_valid ? (entry->timestamp.is_pre_init ? s_preinit : entry->timestamp.string_date.buf) : empty_str,
                    entry->timestamp.is_valid ? entry->timestamp.string_time.buf : empty_str,
                    strcmp(entry->sensor_name.buf, s_unknown) ? entry->sensor_name.buf : entry->sensor_type.buf,
                    entry->description.buf,
                    entry->event_direction.buf);
            }
	        if (s_preinit) free(s_preinit);
	        if (s_unknown) free(s_unknown);
        } else {
            add_empty_table_entry(&sel_table, _("Event log has no entries."));
        }
    } else {
        add_empty_table_entry(&sel_table, _("Error: Could not query event log."));
    }

    ejSetResult(eid, sel_table.buf ? sel_table.buf : "");

    pp_strstream_free(&sel_table);

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return 0;
}
