#include <pp/intl.h>
#include <pp/um.h>
#include <pp/zip.h>
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "wsIntrn.h"

#define CASESENSITIVITY			(0)
#define WRITEBUFFERSIZE			(8192)
#define MAXFILENAME			(256)
#define NO_ERR 				0
#define ERR_ARBITRARY 			1
#define FILE_NOT_FOUND 			2	
#define ERR_UNCOMPRESSED_SIZE_TO_BIG	3
#define ZIP_FILE_OEM_PATH		"/flashdisk/oem/"
#define ZIP_FILE_I18N_PATH		"/flashdisk/i18n/"
#define OEM_DATA_MAX_COMP_SIZE		(2 * 1024 * 1024)
#define OEM_DATA_MAX_UNCOMP_SIZE	(3 * 1024 * 1024)
#define I18N_DATA_MAX_COMP_SIZE		(1 * 1024 * 1024)
#define I18N_DATA_MAX_UNCOMP_SIZE	(2 * 1024 * 1024)

FV_SPEC = {
    {
	id:		FV_ID_OEM_FILE_NAME,
	cfgkey:		"oem.file"
    },
    {
	id:		FV_ID_I18N_FILE_NAME,
	cfgkey:		"oem.i18n_file"
    }
};

static int reconf_oem_devel(pp_strstream_t *strbuf);
static int reconf_oem_devel_cb(pp_cfg_chg_ctx_t * ctx);

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int oem_upload_cb(webs_t wp, char * name, char * filename, void * data, size_t data_offset, size_t data_len, int more_data);
static int i18n_upload_cb(webs_t wp, char * name, char * filename, void * data, size_t data_offset, size_t data_len, int more_data);

int
oem_upload_file_tmpl_init(void)
{
    form_handler_t * fh;

    websAddPostDataCallback("oem_upload_file", oem_upload_cb);
    websAddPostDataCallback("i18n_upload_file", i18n_upload_cb);
    
    pp_cfg_add_post_tx_change_listener(reconf_oem_devel_cb, "oem.devel");

    fh = CREATE_FH_INSTANCE(TEMPLATE_OEM_UPLOAD, ACL_OBJ_OEM);

    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    int do_delete  = form_button_clicked(wp, "action_delete");
    char command[256];
    int ret = -1;

    if (wp->upload_failed) {
	/* do nothing here, errors already reported */
	goto bail;
    }

    /* request to delete oem data */
    if (do_delete) {
	int status;
	    
       	snprintf(command, sizeof(command), "rm -rf %s* %s*\n", ZIP_FILE_OEM_PATH, ZIP_FILE_I18N_PATH);
    	if ((status = pp_system(command)) == -1 || WEXITSTATUS(status) != 0) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Could not delete OEM data."));
	} else {
	    ret = 0;
	}
	goto bail;
    }

    if (form_button_clicked(wp, "action_device_reset")) {
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
	set_response(wp, ERIC_RESPONSE_OK,_("Are you sure you want to restart the device?<br>Please confirm by pressing \"Really Reset\"."));
	websSetVar(wp, "_show_device_reset_confirm", "1");
	websSetVar(wp, "_show_reset", "1");
	return 0;
    } else if (form_button_clicked(wp, "action_really_device_reset")) {
	ret = device_reset(wp);
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
	if (ret == 0) {
	    char redirect[200];
	    int redirect_to;
	    pp_cfg_get_int(&redirect_to, "reset_redirect_time");
	    snprintf(redirect, sizeof(redirect),
	        "<meta http-equiv=\"refresh\" content=\"%d; URL=/\">",
	        60 * redirect_to);

	    set_response(wp, ERIC_RESPONSE_OK, _("The device will be reset in a few seconds."));
	    websSetVar(wp, "_show_redirect_info", "1");
	    websSetVar(wp, "__meta__", redirect);
	}
	return ret;
    } else if (form_button_clicked(wp, "action_cancel_device_reset")) {
	set_response(wp, ERIC_RESPONSE_OK, _("Reset operation cancelled."));
	return 0;
    }

    /* oem data was uploaded, 'install' */
    if (wp->sd->oem_data) {
#if !defined(PP_FEAT_JFFS2_ROOTFS)
	pp_system("mount_flashdisk.sh");
#endif
	int unzip_result = pp_zip_unzip(ZIP_FILE_OEM_PATH, wp->sd->oem_data, wp->sd->oem_data_size, OEM_DATA_MAX_UNCOMP_SIZE);

	switch (unzip_result) {
	  case ERR_ARBITRARY:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Internal error on uncompressing."));
	      break;
	  case FILE_NOT_FOUND:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Could not find zip-file."));
	      break;
	  case ERR_UNCOMPRESSED_SIZE_TO_BIG:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Uncompressed size of zip-file too big."));
	      break;
	  default:
	      /* everything is fine */
	      ret = 0;
	      break;
	}

	free(wp->sd->oem_data);
	wp->sd->oem_data = NULL;
	wp->sd->oem_data_size = 0;

	// run the post-upload 
	pp_strstream_t strbuf = PP_STRSTREAM_INITIALIZER;
	if (ret == 0 && (ret = reconf_oem_devel(&strbuf))) {
	    if (pp_strstream_pos(&strbuf)) {
	        set_response(wp, ERIC_RESPONSE_ERROR, pp_strstream_buf(&strbuf));
	    } else {
	        set_response(wp, ERIC_RESPONSE_ERROR, _("Internal error"));
	    }
	}
	if (pp_strstream_pos(&strbuf)) {
	    pp_strstream_free(&strbuf);
	}

	goto bail;
    }
    
    /* i18n data was uploaded, 'install' */
    if (wp->sd->i18n_data) {
#if !defined(PP_FEAT_JFFS2_ROOTFS)
	pp_system("mount_flashdisk.sh");
#endif
	int unzip_result = pp_zip_unzip(ZIP_FILE_I18N_PATH, wp->sd->i18n_data, wp->sd->i18n_data_size, I18N_DATA_MAX_UNCOMP_SIZE);

	switch (unzip_result) {
	  case ERR_ARBITRARY:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Internal error on uncompressing."));
	      break;
	  case FILE_NOT_FOUND:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Could not find zip-file."));
	      break;
	  case ERR_UNCOMPRESSED_SIZE_TO_BIG:
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Uncompressed size of zip-file too big."));
	      break;
	  default:
	      /* everything is fine */
	      ret = 0;
	      break;
	}

	free(wp->sd->i18n_data);
	wp->sd->i18n_data = NULL;
	wp->sd->i18n_data_size = 0;
	
	if (ret == 0) {
	    websSetVar(wp, "_show_reset", "1");
	    set_response(wp, ERIC_RESPONSE_OK, _("You need to reset the device to make the new internationalization settings work."));
	}
    }
    
    ret = 0;
        
 bail:
    return ret;
}

static int
upload_cb(webs_t wp, void * data, size_t data_len, int more_data,
          void **dst_data, size_t *dst_size, size_t max_size)
{
    eric_session_touch(wp->session);

    if (PP_ERR == pp_um_user_has_permission(wp->user, ACL_OBJ_SUPER,
                                            pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	goto fail;
    }

    if (*dst_size + data_len > max_size) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Uploaded file exceeds maximum size of %d KB."),
		     (max_size / 1024));
	goto fail;
    }
    
    if ((*dst_data = realloc(*dst_data, *dst_size + data_len)) == NULL) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Not enough free memory to upload file."));
	goto fail;
    }
    
    if (data_len > 0) {
	memcpy(*dst_data + *dst_size, data, data_len);
	*dst_size += data_len;
    }
    
    if (data == NULL) {
	free(*dst_data);
	*dst_data = NULL;
	*dst_size = 0;
    } else if (!more_data) {
    	if (*dst_size >= 2) {
	    *dst_size -= 2;
	}
    }
    
    wp->upload_failed = 0;

    return 0;

 fail:
    free(*dst_data);
    *dst_data = NULL;
    *dst_size = 0;
    wp->upload_failed = 1;
    
    return -1;
}

static int
oem_upload_cb(webs_t wp, char * name UNUSED, char * filename UNUSED,
	      void * data, size_t data_offset UNUSED, size_t data_len,
	      int more_data)
{
    return upload_cb(wp, data, data_len, more_data, &wp->sd->oem_data, &wp->sd->oem_data_size, OEM_DATA_MAX_COMP_SIZE);
}

static int
i18n_upload_cb(webs_t wp, char * name UNUSED, char * filename UNUSED,
	       void * data, size_t data_offset UNUSED, size_t data_len,
	       int more_data)
{
    return upload_cb(wp, data, data_len, more_data, &wp->sd->i18n_data, &wp->sd->i18n_data_size, I18N_DATA_MAX_COMP_SIZE);
}

static int reconf_oem_devel(pp_strstream_t *strbuf) {
    int activate, status, ret = PP_SUC;

    pp_cfg_is_enabled(&activate, "oem.devel.active");
    status = pp_system("/etc/rc.oem");
    
    switch (WEXITSTATUS(status)) {
      case 0: // okay
      case 1: // oem was turned off, but no flash dir available. we dont care here
	  break;
      case 2:
      case 3:
	  pp_strappendf(strbuf, _("Could not %s share."), activate ? _("mount") : _("unmount"));
	  ret = PP_ERR;
          break;
      default:
	  pp_strappendf(strbuf, _("Error %s OEM webpages."), activate ? _("activating") : _("deactivating"));
	  ret = PP_ERR;
          break;
    }
    return ret;
}

static int reconf_oem_devel_cb(pp_cfg_chg_ctx_t * ctx) {
    pp_strstream_t strbuf = PP_STRSTREAM_INITIALIZER;
    int ret = reconf_oem_devel(&strbuf);

    if (pp_strstream_pos(&strbuf)) {
        *(ctx->retstr) = pp_strstream_buf(&strbuf);
    } else {
        pp_strstream_free(&strbuf);
    }
    return ret;
}

