#include <pp/intl.h>
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"

FV_SPEC = {
    {
	id:		FV_ID_RC_SELECTED_ENCODING,
	cfgkey:		"user[%U].rc.encoding._c_",
    },
    {
	id:		FV_ID_RC_FIX_ENCODING,
	cfgkey:		"user[%U].rc.encoding.preconf",
    },
    {
	id:		FV_ID_RC_ADV_ENCODING_CR,
	cfgkey:		"user[%U].rc.encoding.manual.compress",
    },
    {
	id:		FV_ID_RC_ADV_ENCODING_CD,
	cfgkey:		"user[%U].rc.encoding.manual.colordepth"
    }
};

typedef struct {
    const char *fix;
    const char *compression;
    const char *colordepth;
} enc_map_t;

static const enc_map_t enc_map[] = {
    {	"videohi",	"video",	"color_16bpp"},
    {	"video",	"video",	"color_8bpp" },
    {	"lanhi",	"uncompressed",	"color_16bpp"},
    {	"lan",		"uncompressed",	"color_8bpp" },
    {	"dsl",		"compr2",	"color_8bpp" },
    {	"umts",		"compr4",	"color_8bpp" },
    {	"isdn",		"compr6",	"color_4bpp" },
    {	"modem",	"compr7",	"grey_2bpp"  },
    {	"gprs",		"compr8",	"grey_2bpp"  },
    {	"gsm",		"compr9",	"grey_1bpp"  },
    {	NULL,		NULL,		NULL }
};

#ifdef PP_FEAT_RC_ENCODING_FIX
static const enc_map_t *lookup_fix_enc(const char* fix);
#endif /* PP_FEAT_RC_ENCODING_FIX */
static int pre_validate_hook(webs_t wp, form_handler_t * fh);

int
rc_encoding_tmpl_init(void)
{
    form_handler_t * fh;

    fh = CREATE_FH_INSTANCE(TEMPLATE_RC_ENCODING, ACL_OBJ_RC_ENCODING);

    fh->pre_validate_hook = pre_validate_hook;
    fh->disable_ldap=TRUE;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh)
{
    if (!form_button_clicked(wp, "action_apply")) {
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
    }
    
    /*
     * check for pre-configured encoding and configure
     * compression and color depth according to choosen value
     */
   if (strcmp(fh->fv[FV_ID_RC_SELECTED_ENCODING].val.s, "preconf")) {
        fh->fv[FV_ID_RC_FIX_ENCODING].flags |= FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    }
    if (strcmp(fh->fv[FV_ID_RC_SELECTED_ENCODING].val.s, "manual")) {
        fh->fv[FV_ID_RC_ADV_ENCODING_CR].flags |= FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
        fh->fv[FV_ID_RC_ADV_ENCODING_CD].flags |= FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    }
    
#ifdef PP_FEAT_RC_ENCODING_FIX
    const char* compression;
    const char* colordepth;

    if (!strcmp(fh->fv[FV_ID_RC_SELECTED_ENCODING].val.s, "preconf")) {
	const enc_map_t* map = lookup_fix_enc(fh->fv[FV_ID_RC_FIX_ENCODING].val.s);
	if (map != NULL) {
	    compression = map->compression;
	    colordepth  = map->colordepth;
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Unknown predefined encoding specified!"));	    
	    return -1;    
	}

        websSetVar(wp, fh->fv[FV_ID_RC_ADV_ENCODING_CR].fvname, compression);
	fh->fv[FV_ID_RC_ADV_ENCODING_CR].val.s =
            websGetVar(wp, fh->fv[FV_ID_RC_ADV_ENCODING_CR].fvname, "");

	websSetVar(wp, fh->fv[FV_ID_RC_ADV_ENCODING_CR].fvname, colordepth);
	fh->fv[FV_ID_RC_ADV_ENCODING_CD].val.s = 
            websGetVar(wp, fh->fv[FV_ID_RC_ADV_ENCODING_CR].fvname, "");

    }
# if !defined(PP_FEAT_VSC_HW_ENCODING)
    else if (!strcmp(fh->fv[FV_ID_RC_SELECTED_ENCODING].val.s, "manual")) {
	compression = fh->fv[FV_ID_RC_ADV_ENCODING_CR].val.s;
	colordepth  = fh->fv[FV_ID_RC_ADV_ENCODING_CD].val.s;

	/* sanity checks for software encoding */
	
	if ((!pp_strcmp_safe(compression, "video") || !pp_strcmp_safe(compression, "uncompressed")) &&
	    (pp_strcmp_safe(colordepth, "color_16bpp") && pp_strcmp_safe(colordepth, "color_8bpp"))) {

	    set_response(wp, ERIC_RESPONSE_ERROR, _("Without compression only 8 bit and 16 bit color are available!"));
	    return -1;	    
	} else if ((pp_strcmp_safe(compression, "video") && pp_strcmp_safe(compression, "uncompressed")) &&
		   !pp_strcmp_safe(colordepth, "color_16bpp")) {
	    
	    set_response(wp, ERIC_RESPONSE_ERROR, _("With enabled compression true color isn't available!"));
	    return -1;    
	}
    }
# endif /* !PP_FEAT_VSC_HW_ENCODING */
#else /* !PP_FEAT_RC_ENCODING_FIX */
    /* avoid unused var warning */
    (void)wp;
#endif /* !PP_FEAT_RC_ENCODING_FIX */
    return 0;
}

#ifdef PP_FEAT_RC_ENCODING_FIX
static const enc_map_t*
lookup_fix_enc(const char* fix)
{
    const enc_map_t* map = NULL;

    if (fix == NULL) goto bail;

    map = enc_map;
    while (map != NULL) {
	if (!pp_strcmp_safe(fix, map->fix)) goto bail;

	map++;
    }
    
 bail:
    return map;
}
#endif /* PP_FEAT_RC_ENCODING_FIX */

