/* system includes */
/* firmware includes */
#include <pp/ipmi.h>
#include <pp/xdefs.h>
#include <pp/um.h>
#include <pp/hal_rpc.h>

/* local includes */
#include "ej.h"
#include "eric_forms.h"
#include "templates.h"
#include "webs.h"
#include "eric_util.h"
#include "tmpl_rpc_common.h"

static int initialized = 0;
static int sdr_cache_initialized = 0;
static int power_branch_mapping_initialized = 0;
static pp_ipmi_sdr_list_entry_t *sdr_cache[RPC_MAX_SENSOR + 1];
static int power_branch_for_outlet[24];

static int init_sdr_cache(void);

/* ASPs */
static int rpc_get_sensor_number_asp(int eid, webs_t wp, int argc, char **argv);
static int rpc_init_sensor_state_asp(int eid, webs_t wp, int argc, char **argv);
static int rpc_get_outlet_name_asp(int eid, webs_t wp, int argc, char **argv);
static int rpc_get_num_outlets_asp(int eid, webs_t wp, int argc, char **argv);
static int rpc_get_sensor_name_asp(int eid, webs_t wp, int argc, char **argv);

const char *bmc_rpc_outlet_name_str = "bmc.rpc.outlet[%u].name";

int rpc_common_tmpl_init(void)
{
    if(!initialized) {
        /* register ASPs */
        websAspDefine("rpcGetSensorNumber", rpc_get_sensor_number_asp);
        websAspDefine("rpcInitSensorState", rpc_init_sensor_state_asp);
        websAspDefine("rpcGetOutletName", rpc_get_outlet_name_asp);
        websAspDefine("rpcGetNumOutlets", rpc_get_num_outlets_asp);
        websAspDefine("rpcGetSensorName", rpc_get_sensor_name_asp);
        
        initialized = 1;
    }
    
    return PP_SUC;
}

void rpc_common_tmpl_cleanup(void)
{
    if(sdr_cache_initialized) {
        u_int u;
        
        for(u = 0; u <= RPC_MAX_SENSOR; ++u) {
            if(sdr_cache[u]) {
                pp_ipmi_delete_sdr_table_entry((void*)sdr_cache[u]);
            }
        }
        
        sdr_cache_initialized = 0;
    }
}

static int update_sdr_cache(vector_t *sdr) {
    int ret = 0;
    u_int u;
    
    RPC_DEBUG_IPMI("got %u SDR entries", vector_size(sdr));
    
    /* cache SDR info */
    for(u = 0; vector_size(sdr) > u;) {
        pp_ipmi_sdr_list_entry_t *entry = vector_get(sdr, u);
        
        if(entry->sdr_id < 0 || entry->sdr_id > RPC_MAX_SENSOR) {
            /* not one of our sensors, skip */
            ++u;
            continue;
        }
        
        /* store entry to sdr_cache and avoid deletion on cleanup */
        if(sdr_cache[entry->sdr_id]) {
            pp_ipmi_delete_sdr_table_entry((void*)sdr_cache[entry->sdr_id]);
        }
        sdr_cache[entry->sdr_id] = entry;
        vector_remove_dont_delete(sdr, u);
        ++ret;
        
        RPC_DEBUG_IPMI("received SDR entry with ID %d:\n"
                       "\t\tname = %s\n\t\tunit = %s",
                       entry->sdr_id, entry->sdr_name.buf ? : _("Unknown"), entry->data.full.unit.buf ? : _("Unknown"));
    }
    
    RPC_DEBUG_IPMI("updated %d SDR entries", ret);
    
    return ret;
}

static int init_sdr_cache(void) {
    u_int u;
    pp_ipmi_return_t ipmi_ret;
    
    if(!sdr_cache_initialized) {
        vector_t *sdr;
        
        /* init */
        memset(sdr_cache, 0, 
               (RPC_MAX_SENSOR + 1) * sizeof(pp_ipmi_sdr_list_entry_t*));
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    
        /* get full SDR list */
        if (pp_ipmi_send_command(PP_IPMI_CMD_SDR,
                                 PP_IPMI_SDR_SUBCMD_LIST,
                                 NULL, &ipmi_ret, NULL, NULL) != 0) {
            RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SDR, "
                           "PP_IPMI_SDR_SUBCMD_LIST failed");
            goto bail;
        }
        sdr = ipmi_ret.data.sdr_list;
        
        update_sdr_cache(sdr);
        
        sdr_cache_initialized = 1;
        
        pp_ipmi_cleanup_ret(&ipmi_ret);
    }
    return PP_SUC;
    
 bail:
    /* clean up */
    for(u = 0; u <= RPC_MAX_SENSOR; ++u) {
        if(sdr_cache[u]) {
            pp_ipmi_delete_sdr_table_entry((void*)sdr_cache[u]);
            sdr_cache[u] = NULL;
        }
    }
    pp_ipmi_cleanup_ret(&ipmi_ret);
    return PP_ERR;
}

static int rpc_get_sensor_number_asp(int eid, webs_t wp UNUSED,
                                     int argc, char **argv) {
    int ret = -1;
    
    if(argc == 2) {
        int outlet, sensor;
        
        if((outlet = pp_strtol_10(argv[0], PP_ERR, NULL)) != PP_ERR &&
           (sensor = pp_strtol_10(argv[1], PP_ERR, NULL)) != PP_ERR) {
            char val[4];
            
            snprintf(val, 4, "%u", RPC_SENSOR(outlet, sensor));
            ejSetResult(eid, val);
            ret = 0;
        }
    }
    
    return ret;
}

static int rpc_init_sensor_state_asp(int eid UNUSED, webs_t wp,
                                     int argc, char **argv) {
    int ret = -1;
    
    if(argc == 1) {
        int sensornum;
        
        if((sensornum = pp_strtol_10(argv[0], PP_ERR, NULL)) != PP_ERR) {
            if(sensornum < 0 || sensornum > RPC_MAX_SENSOR) {
                /* invalid sensor number */
                pp_log("sensor number %d invalid!", sensornum);
                abort();
            } else {
                char sensor_name[32];
                char sensor_value[32];
                char sensor_unit[32];
                
                form_var_vec_name("sensor_name", sensornum, sensor_name, 32);
                form_var_vec_name("sensor_value", sensornum, sensor_value, 32);
                form_var_vec_name("sensor_unit", sensornum, sensor_unit, 32);

                init_sdr_cache();
                
                if(!sdr_cache[sensornum]) {
                    /* sensor not in sdr */
                    pp_log("sensor number %d not in SDR!\n", sensornum);
                    
                    websSetVar(wp, sensor_name, _("Not initialized"));
                    websSetVar(wp, sensor_value, "");
                    websSetVar(wp, sensor_unit, "");

                    /* not a "real" error... */
                } else {
                    pp_ipmi_return_t ipmi_ret;
                    pp_ipmi_parameter_t params;
                    pp_ipmi_sdr_list_entry_t *entry = sdr_cache[sensornum];
                                            
                    websSetVar(wp, sensor_name, 
                               entry->sdr_name.buf ? : _("Unknown"));
                    websSetVar(wp, sensor_unit, 
                               entry->data.full.unit.buf ? : "");

                    /* init */
                    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
                    memset(&params, 0, sizeof(params));
        
                    params.data.sensor_get_by_id_list = 
                        vector_new2(NULL, 1, sizeof(unsigned int), NULL);
                    vector_add2(params.data.sensor_get_by_id_list, &sensornum);
    
                    /* get raw sensor reading */
                    if (pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
                                             PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID,
                                             &params, &ipmi_ret,
                                             NULL, NULL) != 0) {
                        RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SENSOR,  PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID failed");

                        websSetVar(wp, sensor_value,
                                   _("currently unavailable"));
                    } else {
                        vector_t *raw_readings =
                            ipmi_ret.data.sensor_get_raw_list;
                        
                        if(!raw_readings || !vector_size(raw_readings)) {
                            RPC_DEBUG_IPMI("sensor get raw returned no reading");
    
                            websSetVar(wp, sensor_value,
                                       _("currently unavailable"));
                        } else {                        
                            char var_val[32];
                            double value;
                            pp_ipmi_raw_reading_list_entry_t *reading;
                            
                            /* we only want first reading */
                            reading = vector_get(raw_readings, 0);
                            
                            /* paranoia */
                            assert(vector_size(raw_readings) == 1);
                            assert(reading);
                            assert(reading->id == sensornum);
                            
                            // TODO: discrete sensors?
                            value = pp_ipmi_sdr_convert_sensor_reading_core(
                                        sdr_cache[sensornum]->data.full.m,
                                        sdr_cache[sensornum]->data.full.b,
                                        sdr_cache[sensornum]->data.full.b_exp,
                                        sdr_cache[sensornum]->data.full.r_exp,
                                        sdr_cache[sensornum]->data.full.analog,
                                        sdr_cache[sensornum]->data.full.linearization,
                                        reading->raw_value);
                            
                            /* TODO: format string! */
                            snprintf(var_val, 32, "%.1f", value);
                            
                            websSetVar(wp, sensor_value, var_val);
                        }
                    }
                    
                    /* clean up */
                    vector_delete(params.data.sensor_get_by_id_list);
                    pp_ipmi_cleanup_ret(&ipmi_ret);
                }
                ret = 0;
            }
        }
    }
    
    return ret;
}

static int rpc_get_outlet_name_asp(int eid, webs_t wp UNUSED,
                                   int argc, char **argv) {
    int ret = -1;
    int outlet, num_outlets = pp_hal_rpc_get_no_outlets();
    
    /* check if argv is valid outlet number */
    if(argc == 1 &&
       (outlet = pp_strtol_10(argv[0], PP_ERR, NULL)) != PP_ERR &&
       outlet > 0 && outlet <= num_outlets) {
        char *outlet_name = NULL;
        
        --outlet; // we start counting at 0, user at 1 ;-)
        
        pp_cfg_get(&outlet_name, bmc_rpc_outlet_name_str, outlet);
        ejSetResult(eid, outlet_name ? : "");
        ret = 0;
        
        free(outlet_name);
    }
    
    return ret;
}

static int rpc_get_num_outlets_asp(int eid, webs_t wp UNUSED,
                                   int argc UNUSED, char **argv UNUSED) {
    char buf[4];
    
    snprintf(buf, sizeof(buf), "%d", pp_hal_rpc_get_no_outlets());
    ejSetResult(eid, buf);

    return 0;
}

static int rpc_get_sensor_name_asp(int eid, webs_t wp UNUSED,
                                   int argc, char **argv) {
    int sensor;
    const char *sensor_name = NULL;
    
    /* check if argv is valid outlet number */
    if(argc == 1 &&
       (sensor = pp_strtol_10(argv[0], PP_ERR, NULL)) != PP_ERR &&
       sensor >= 0 && sensor <= RPC_MAX_SENSOR) {
        pp_ipmi_sdr_list_entry_t *entry;
        
        init_sdr_cache();
        
        if(NULL != (entry = sdr_cache[sensor])) {
            sensor_name = entry->sdr_name.buf;
        }
    }
    
    ejSetResult(eid, sensor_name ? : "");
    return 0;
}

int rpc_set_outlet_state(webs_t wp, unsigned char outlet, unsigned char state) {
    int ret = PP_SUC;
    char outlet_perm[16];

    assert(outlet - 1 < pp_hal_rpc_get_no_outlets());

    snprintf(outlet_perm, sizeof(outlet_perm), "outlet_%d", (int)outlet);
    
    if(PP_ERR == pp_um_user_has_permission(wp->user, outlet_perm,
                                           pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	ret = PP_ERR;
    } else {
        pp_ipmi_parameter_t params;
        pp_ipmi_return_t ipmi_ret;
    
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        memset(&params, 0, sizeof(pp_ipmi_parameter_t));
        params.data.oem_pp_rpc_set_recep_state.receptacle = outlet;
        params.data.oem_pp_rpc_set_recep_state.status = state & 0x01;
        
        if(pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
                                 PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_STATE,
                                &params, &ipmi_ret, NULL, NULL) != 0) {
            RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_OEM_PP_RPC,  PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_STATE failed");
            ret = PP_ERR;
        }
    
        pp_ipmi_cleanup_ret(&ipmi_ret);
    }
            
    return ret;
}

pp_ipmi_sdr_list_entry_t* get_sdr_entry_for_sensor(unsigned char sensornum) {
    assert(sensornum <= RPC_MAX_SENSOR);
    
    init_sdr_cache();
    
    if(!sdr_cache[sensornum]) {
        /* sensor not in sdr */
        pp_log("sensor number %d not in SDR!\n", sensornum);
    }
    
    return(sdr_cache[sensornum]);
}

int rpc_outlet_switch_hook(webs_t wp, form_handler_t * fh UNUSED) {
    int ret = PP_SUC;
    int num_outlets = pp_hal_rpc_get_no_outlets(), i;
    
    for(i = 1; i <= num_outlets; ++i) {
        char buttonstr[32];
        
        snprintf(buttonstr, 32, "action_target_switch_on_%d", i);
        if(form_button_clicked(wp, buttonstr)) {
            ret = rpc_set_outlet_state(wp, i - 1, 1);
            break;
        }
        snprintf(buttonstr, 32, "action_target_switch_off_%d", i);
        if(form_button_clicked(wp, buttonstr)) {
            ret = rpc_set_outlet_state(wp, i - 1, 0);
            break;
        }
    }
    
    return ret;
}

int rpc_get_power_branch_for_outlet(int outlet) {
    int num_outlets = pp_hal_rpc_get_no_outlets();
    
    assert(outlet >= 0 && outlet < num_outlets);

    if(!power_branch_mapping_initialized) {
        int i, outlet_num_even, outlet_num_odd, max, power_branch;
        pp_hal_rpc_relais_board_controller_t controller_type =
            pp_hal_rpc_get_controller();
        
        if(controller_type == RELAIS_BOARD_CONTROLLER_4) {
            outlet_num_even = outlet_num_odd = 4;
        } else if(controller_type == RELAIS_BOARD_CONTROLLER_5) {
            outlet_num_even = outlet_num_odd = 5;
        } else if(controller_type == RELAIS_BOARD_CONTROLLER_5_3) {
            outlet_num_even = 5;
            outlet_num_odd = 3;
        } else {
            /* something's wrong here */
            abort();
            return(0);
        }
        
        /* now init outlet mapping */
        for(i = 0, max = outlet_num_even, power_branch = 0;
            i < num_outlets; ++i) {
            if(i >= max) {
                /* we reached maximum number of outlets per controller */
                max += ++power_branch % 2 ? outlet_num_odd : outlet_num_even;
            }
            power_branch_for_outlet[i] = power_branch;
        }
        
        power_branch_mapping_initialized = 1;
    }

    return power_branch_for_outlet[outlet];
}

int rpc_get_relais_board_for_outlet(int outlet) {
    int outlets_per_board = pp_hal_rpc_get_no_outlets_per_board();
#ifndef NDEBUG
    int num_outlets = pp_hal_rpc_get_no_outlets();
    
    assert(outlet >= 0 && outlet < num_outlets);
    assert(num_outlets % outlets_per_board == 0);
#endif /* !NDEBUG */

    if(!outlets_per_board) {
        return -1;
    }
    return outlet / outlets_per_board;
}

int rpc_get_outlet_state(int outlet) {
    int state;
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;

    assert(outlet >= 0 && outlet < pp_hal_rpc_get_no_outlets());

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&params, 0, sizeof(params));
    params.data.oem_pp_rpc_get_recep_state.receptacle = outlet;

    if(pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC, 
                            PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE,
                            &params, &ipmi_ret, NULL, NULL) != PP_SUC) {
        /* ipmi_ret invalid, assume failure */
        RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_OEM_PP_RPC,  PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE failed");
        
        state = STATE_FAILURE;
    } else {
        /* we got valid ipmi_ret */
        if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.red) {
            if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.green) {
                /* red and green is yellow! only flashing */
                assert(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.blinking);
                state = STATE_RELAIS_ON_POWER_OFF; // LED yellow flashing
            } else { /* !green */
                if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.blinking) {
                    state = STATE_RELAIS_ON_POWER_CRITICAL; // LED red flashing
                } else { /* !blinking */
                    state = STATE_RELAIS_ON_POWER_ON; // LED red
                }
            }
        } else { /* !red */
            if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.green) {
                if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.blinking) {
                    state = STATE_RELAIS_OFF_POWER_OFF; // LED green flashing
                } else { /* !blinking */
                    state = STATE_RELAIS_OFF_POWER_ON; // LED green
                }
            } else { /* !green */
                /* led is off and may not be flashing */
                assert(!ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.blinking);
                state = STATE_FAILURE; // LED off
            }
        }
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    
    return(state);
}
