/* system includes */
/* firmware includes */
#include <pp/xdefs.h>

/* local includes */
#include "webs.h"
#include "templates.h"
#include "eric_forms.h"
#include "eric_util.h"

#define DNS_PORT 53
#define SMTP_PORT 25

FV_SPEC = {
    {
	id:		FV_ID_SMTP_SERVER,
	cfgkey:		"log.smtp.server"
    },
    {
	id:		FV_ID_SMTP_FROM,
	cfgkey:		"log.smtp.from"
    }
};

/* hooks */
static int post_validate_hook(webs_t wp, form_handler_t * fh);

static int is_ip (const char *name_or_ip);
static int hostname_to_ip (const char *hostname, char *ip_buf, size_t ip_buf_len);
static int check_dns_reply(const char *dns_ip);
static int check_tcp_connect(const char *hostname, u_short port);
static int check_dns_cfg(int is_auto_conf, vector_t *resp_vect);
static int check_ip_lookup(const char *hostname, const char *service_name, vector_t *resp_vect);
static void check_gateway_cfg(vector_t *resp_vect);
static void response_add_line(vector_t *resp_vect, const char *fmt, ...) __attribute__ ((format (printf, 2, 3)));

int smtp_tmpl_init(void)
{
    form_handler_t * fh;

    fh = CREATE_FH_INSTANCE(TEMPLATE_SMTP, ACL_OBJ_LOG_S);

    /* hooks */
    fh->post_validate_hook = post_validate_hook;
    
    REGISTER_FH_INSTANCE_AND_RETURN(fh);
    
    return PP_SUC;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    char *auto_conf = NULL;
    static vector_t *resp_vect;   /* response message buffer */
    
    resp_vect = vector_new(NULL, 3, free);

    if (!pp_strcmp_safe(fh->fv[FV_ID_LOG_SMTP_ENABLED].val.s, "yes") 
	    && form_button_clicked(wp, "action_apply"))
    {
	int err;
	const char *smtp_server = NULL;
	form_var_t * fv_smtp = NULL;
	int smtp_ip_lookup_ok = 0;
	int all_ok = 0;

	/** SMTP server sanity checks **/

	/* get SMTP server name (validation already done) */
	fv_smtp = &fh->fv[FV_ID_SMTP_SERVER];
	smtp_server = fv_smtp->val.s;

	if (is_ip(smtp_server)) {
	    /* looking up the IP of an IP is always ok */
	    smtp_ip_lookup_ok = 1;
	} else {
	    /* DNS lookup test */
	    if (check_ip_lookup(smtp_server, "SMTP server", resp_vect) == 0) {
		smtp_ip_lookup_ok = 1;
	    } else {
		int is_auto_conf;

		smtp_ip_lookup_ok = 0;

		/* check for DHCP */
		pp_cfg_get(&auto_conf, "network.ip_auto_config_proto");
		if (!pp_strcmp_safe(auto_conf, PP_CD_NETWORK_IP_AUTO_CONFIG_PROTO_NONE_STR)) {
		    is_auto_conf = 0;
		} else {
		    is_auto_conf = 1;
		    response_add_line(resp_vect,
				      _("Device obtains network gateway and DNS server(s)"
					" via %s protocol."), auto_conf);
		}

		/* fixed IP -> warning if no DNS or gateway configured */
		err = check_dns_cfg(is_auto_conf, resp_vect);
		if (err) {
		    response_add_line(resp_vect, 
				      _("At least one responding DNS server is required unless"
					" you specify SMTP server as IP address rather than host name."));
		    /* check if gateway may be the reason for failed DNS connect */
		    check_gateway_cfg(resp_vect);
		}
	    }
	}
	if (smtp_ip_lookup_ok) {
	    int ok = !check_tcp_connect(smtp_server, SMTP_PORT);
	    if (!ok) {
		response_add_line(resp_vect,
				  _("Connection to SMTP service on host '%s' failed!"),
			smtp_server);
	    }
	    if (ok) {
		all_ok = 1;
	    } else {
		/* check if gateway may be the reason for failed SMTP connect */
		check_gateway_cfg(resp_vect);
	    }
	}
    }

    /* join resp_vect into the web response */
    if (vector_size(resp_vect) > 0) {
	char* msg;
	pp_strstream_t *strstrm;
	int resp_cnt = vector_size(resp_vect);
	int i;

	strstrm = pp_strstream_init(NULL);
	pp_strappend(strstrm, _("Problems with SMTP logging:"));
	pp_strappend(strstrm, "\n<table class=\"rsp_message\"><tr><td><ul>");
	for (i = 0; i < resp_cnt; i++) {
	    pp_strappendf(strstrm, "<li>%s</li>", (char *)vector_get(resp_vect, i));
	}
	pp_strappend(strstrm, "</ul></td></tr></table>");
	msg = pp_strstream_buf_and_free(strstrm);

	set_response(wp, ERIC_RESPONSE_WARNING, "%s", msg);
	free(msg);
    }

    vector_delete(resp_vect);
    free(auto_conf);
    return 0;
}



/**** helper funcs for SMTP config sanity checks ****/

static void
response_add_line(vector_t *resp_vect, const char *fmt, ...)
{
    va_list args;
    pp_strstream_t *strstrm;
    char *str;

    va_start(args, fmt);
    strstrm = pp_strstream_init(NULL);
    pp_vstrappendf(strstrm, fmt, args);
    str = pp_strstream_buf_and_free(strstrm);
    va_end(args);

    vector_add(resp_vect, str);
}

static int
is_ip (const char *name_or_ip)
{
    struct in_addr addr;
    int res;

    res = inet_pton(AF_INET, name_or_ip, &addr);
    return res > 0 ? 1 : 0;
}

/* translates `hostname' to an ip */
static int
hostname_to_ip (const char *hostname, char *ip_buf, size_t ip_buf_len)
{
    int ret = -1;
    struct hostent he;
    struct hostent *resultp;
    char aux_buf[2048];
    int res, err = 0;

    /* check if server name can be resolved */
    res = gethostbyname_r (hostname, &he, 
	    aux_buf, sizeof(aux_buf), &resultp, &err);
    if (res) goto bail; /* failed */

    if (inet_ntop (he.h_addrtype, he.h_addr_list[0], 
		ip_buf, ip_buf_len) == NULL) goto bail;

    ret = 0;
bail:
    return ret;
}

/* Sends an UDP query to the DNS port at host <dns_ip> and waits for a reply. */
static int
check_dns_reply(const char *dns_ip)
{
    int ret = -1;
    int sock_fd;
    int err;
    struct sockaddr_in serv_addr;
    /* a sniffed DNS query, asking for the IP of "localhost" */
    char sbuf[] = {
	0x69, 0xe8, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x09, 0x6c, 0x6f, 0x63, 
	0x61, 0x6c, 0x68, 0x6f, 0x73, 0x74, 0x00, 0x00, 
	0x01, 0x00, 0x01 };
    char rbuf[1024];
    int recv_n, send_n;
    struct timeval tv;
    int retries = 1;

    /* init a sockaddr_in with dns_ip and DNS_PORT */
    memset(&serv_addr, 0, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = htons (DNS_PORT);
    if (!inet_pton(AF_INET, dns_ip, &serv_addr.sin_addr)) goto bail;

    /* UDP socket */
    sock_fd = socket(PF_INET, SOCK_DGRAM, 0);
    if (sock_fd < 0) goto bail;
    err = connect(sock_fd, &serv_addr, sizeof(serv_addr));
    if (err) goto bail2;

    /* set timeout */
    tv.tv_sec = 1; tv.tv_usec = 0;
    setsockopt(sock_fd, SOL_SOCKET, SO_RCVTIMEO, &tv, sizeof(tv));

    /* send/recv data */
    while (1) {
	send_n = send(sock_fd, sbuf, sizeof(sbuf), 0 /*flags*/);
	recv_n = recv(sock_fd, rbuf, sizeof(rbuf), 0 /*flags*/);
	if (recv_n >= 0) { /* ok */
	    break;
	} else { /* failed */
	    if (retries--) {
		continue;
	    } else {
		pp_log("%s: query to DNS server failed\n", ___F);
		goto bail2;
	    }
	}
    }

    ret = 0;
bail2:
    close(sock_fd);
bail:
    return ret;
}

static int
check_tcp_connect(const char *hostname, u_short port)
{
    int fd;
    struct sockaddr_in serv_addr;
    int ret = -1;
    int err;
    char ip_buf[INET_ADDRSTRLEN];
    int res;

    res = hostname_to_ip (hostname, ip_buf, sizeof(ip_buf));
    if (res) goto bail;

    /* init serv_addr with ip_buf */
    memset(&serv_addr, 0, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_port = htons (port);
    if (!inet_pton(AF_INET, ip_buf, &serv_addr.sin_addr)) goto bail;

    /* TCP connect */
    fd = socket(PF_INET, SOCK_STREAM, 0);
    if (fd < 0) goto bail;
    err = connect(fd, &serv_addr, sizeof(serv_addr));
    if (err) goto bail2;

    ret = 0;
bail2:
    close(fd);
bail:
    return ret;
}

/* check whether DNS servers are configured and responding */
static int
check_dns_cfg(int is_auto_conf, vector_t *resp_vect)
{
    char *dns_ip[2];
    int i;
    int ret = -1;

    /* get from config fs */
    pp_cfg_get(&dns_ip[0], "network.dns_ip_1");
    pp_cfg_get(&dns_ip[1], "network.dns_ip_2");

    if (dns_ip[0][0] == '\0' && dns_ip[1][0] == '\0') {
	response_add_line(resp_vect, "%s %s",
			  _("No DNS server is known to the device!"),
		is_auto_conf 
			  ? _(" Assigning the DNS server via DHCP/BOOTP must have failed.")
			  : _(" Please configure one in the network settings.")); 
    }

    for (i=0; i<2; i++) {
	if (dns_ip[i][0] == '\0') {
	    int ok = !check_dns_reply(dns_ip[i]);
	    if (ok) {
		ret = 0;
	    } else {
		response_add_line(resp_vect,
				  _("DNS server %d is %s &ndash; test query failed!"),
				  i+1, dns_ip[i]);
	    }
	}
    }

    free(dns_ip[0]);
    free(dns_ip[1]);

    return ret;
}

/* check whether gateway is set in config fs */
static void
check_gateway_cfg(vector_t *resp_vect)
{
    char *gateway;

    /* get from config fs */
    pp_cfg_get(&gateway, "network.gateway");
    if (gateway[0] == '\0') {
	response_add_line(resp_vect,
		_("No network gateway configured!  Servers outside the subnet"
		  " of the device will not be reachable!"));
    }
    free(gateway);
}

static int
check_ip_lookup(const char *hostname, const char *service_name, vector_t *resp_vect)
{
    int ret = -1;
    int res;
    char ip[512];

    memset(ip, '\0', sizeof(ip));

    /* check if server name can be resolved */
    res = hostname_to_ip (hostname, ip, sizeof(ip));
    if (res) {
	/* IP lookup failed */
	response_add_line(resp_vect,
			  _("IP address lookup for %s '%s' failed!"),
			  service_name, hostname);
	goto fail;
    }
    ret = 0;
fail:
    return ret;
}
