#include <sys/time.h>
#include <pp/cfg.h>
#include <pp/intl.h>
#include <liberic_config.h>
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

FV_SPEC = {
    {
	id:		FV_ID_TIME_SEC,
	cfgkey:		"time.proto.local.sec"
    },
    {
	id:		FV_ID_TIME_MIN,
	cfgkey:		"time.proto.local.min"
    },
    {
	id:		FV_ID_TIME_HOUR,
	cfgkey:		"time.proto.local.hour"
    },
    {
	id:		FV_ID_TIME_DAY,
	cfgkey:		"time.proto.local.day"
    },
    {
	id:		FV_ID_TIME_MONTH,
	cfgkey:		"time.proto.local.month"
    },
    {
	id:		FV_ID_TIME_YEAR,
	cfgkey:		"time.proto.local.year"
    }
};

struct fh_private_data_s {
    time_t time;
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int post_save_hook(webs_t wp, form_handler_t * fh);
static int time_set_vars_asp(int eid, webs_t wp, int argc, char ** argv);
static int get_offset(char *str_offset);

int
time_local_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("timeSetVars", time_set_vars_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_TIME_LOCAL, ACL_OBJ_TIME);

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;
    fh->post_save_hook = post_save_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp UNUSED, form_handler_t * fh)
{
    char * time_proto = get_form_var_value_unsaved(wp, TEMPLATE_TIME, FV_ID_TIME_PROTO);
    if (strcmp(time_proto, "local")) {
	fh_disable_validate_and_save(fh, -1);
    }
    free(time_proto);
    return 0;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    char * time_proto = get_form_var_value_unsaved(wp, TEMPLATE_TIME, FV_ID_TIME_PROTO);
    if (!strcmp(time_proto, "local")) {
	struct tm *t_elem;
	char * utc_offset = get_form_var_value_unsaved(wp, TEMPLATE_TIME, FV_ID_TIME_UTC_OFFSET);
	int offset = utc_offset ? get_offset(utc_offset) : 0;
	int err;
	free(utc_offset);
        
        t_elem = calloc(1, sizeof(struct tm));
        
        t_elem->tm_sec = pp_strtoul_10(fh->fv[FV_ID_TIME_SEC].val.s, 0, &err);
        if (err) goto conv_error;
        t_elem->tm_min = pp_strtoul_10(fh->fv[FV_ID_TIME_MIN].val.s, 0, &err);
	if (err) goto conv_error;
        t_elem->tm_hour = pp_strtoul_10(fh->fv[FV_ID_TIME_HOUR].val.s, 0, &err);
	if (err) goto conv_error;
        t_elem->tm_mday = pp_strtoul_10(fh->fv[FV_ID_TIME_DAY].val.s,0, &err);
	if (err) goto conv_error;
        t_elem->tm_mon = pp_strtoul_10(fh->fv[FV_ID_TIME_MONTH].val.s, 1, &err) - 1;
	if (err) goto conv_error;
        t_elem->tm_year = pp_strtoul_10(fh->fv[FV_ID_TIME_YEAR].val.s, 1900, &err) - 1900;
	if (err) goto conv_error;

        fh->private = calloc(1, sizeof(fh_private_data_t));
        fh->private->time = timelocal(t_elem) - offset * 3600;
    }
    free(time_proto);
    return 0;

 conv_error:
    set_response(wp, ERIC_RESPONSE_ERROR, _("Internal time conversion error."));
    free(time_proto);
    return -1;
}


static int
post_save_hook(webs_t wp, form_handler_t * fh)
{
    char * time_proto;
    int ret = -1;

    pp_cfg_get(&time_proto, "time.proto._c_");
    if (!pp_strcmp_safe(time_proto, "local")) {
	struct timeval reftime;
        reftime.tv_sec = fh->private->time;
        reftime.tv_usec = 0;
        /* set the system time */
        if (settimeofday(&reftime, NULL) < 0) {
	    pp_log_err("%s(): settimeofday()", ___F);
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Setting system time failed."));
	    goto bail;
        }
        /* set the hardware clock */
        pp_set_hardware_clock_exact(fh->private->time, reftime);
    }

    ret = 0;

 bail:
    free(time_proto);
    return ret;
}

static int
time_set_vars_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char ** argv UNUSED)
{
    char var[MAX_OPT_KEY_LEN+1], *str_offset;
    time_t t;
    struct tm * br_time;
    int offset;
    form_handler_t *fh = lookup_form_handler(TEMPLATE_TIME_LOCAL);
    assert(fh);
    
    pp_cfg_get(&str_offset, "time.utc_offset");
    
    offset = str_offset ? get_offset(str_offset) : 0;
    
    t = time(NULL) + offset * 3600;
    
    br_time = localtime(&t);
    
    snprintf(var, sizeof(var), "%d", br_time->tm_sec);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_SEC].fvname, var);
    snprintf(var, sizeof(var), "%d", br_time->tm_min);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_MIN].fvname, var);
    snprintf(var, sizeof(var), "%d", br_time->tm_hour);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_HOUR].fvname, var);
    snprintf(var, sizeof(var), "%d", br_time->tm_mday);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_DAY].fvname, var);
    snprintf(var, sizeof(var), "%d", br_time->tm_mon + 1);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_MONTH].fvname, var);
    snprintf(var, sizeof(var), "%d", br_time->tm_year + 1900);
    websSetVar(wp, fh->fv_tmpl[FV_ID_TIME_YEAR].fvname, var);
 
    free (str_offset);

    return 0;
}

static int
get_offset(char *str_offset)
{
    int offset;
    
    if (strchr(str_offset, '/')) {
        offset = 0;
    } else if (strchr(str_offset, '+')) {
        if (sscanf(str_offset, "+ %d h", &offset) != 1) {
            offset = 0;
        }
    } else if (strchr(str_offset, '-')) {
        if (sscanf(str_offset, "- %d h", &offset) != 1) {
            offset = 0;
        } else {
            offset = -offset;
        }     
    }     
    return offset;
}
