#include <liberic_notify.h>
#include <pp/um.h>
#include <pp/intl.h>
#include <liberic_config.h>
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

FV_SPEC = {
    {
	id:		FV_ID_GR_GROUP,
	cfgkey:		"group_name"
    },
    {
	id:		FV_ID_GR_GROUP_ID,
	cfgkey:		"group_id"
    },
    {
	id:		FV_ID_GR_NEW_GROUP,
	cfgkey:		"group[%G].name"
    },
};

typedef enum {
    ACTION_NONE,
    ACTION_NONE_NOPERM,
    ACTION_LOOKUP_GROUP,
    ACTION_CREATE_GROUP,
    ACTION_MODIFY_GROUP,
    ACTION_COPY_GROUP,
    ACTION_DELETE_GROUP,
    ACTION_DELETE_MULTIPLE_GROUPS,
} action_t;

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static action_t get_action(webs_t wp, form_handler_t * fh);
static int init_groups(webs_t wp, const char *prefix, int filter, int cut);
static int um_init_all_groups_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_init_all_generic_groups_asp(int eid, webs_t wp, 
                                          int argc, char ** argv);
static int um_init_all_individual_groups_asp(int eid, webs_t wp, 
                                             int argc, char ** argv);
static int um_init_group_and_target_group_asp(int eid, webs_t wp, 
                                              int argc, char ** argv);
static int um_group_form_state(int eid, webs_t wp, int argc, char ** argv);
static int um_group_table_view(int eid, webs_t wp, int argc, char ** argv);

int
um_groups_tmpl_init(void)
{
    form_handler_t * fh;

    if (um_common_tmpl_init() != 0) return -1;

    websAspDefine("umInitAllGroups", um_init_all_groups_asp);
    websAspDefine("umInitAllGenericGroups", um_init_all_generic_groups_asp);
    websAspDefine("umInitAllIndividualGroups",
                  um_init_all_individual_groups_asp);
    websAspDefine("umInitGroupAndTargetGroup",
                  um_init_group_and_target_group_asp);
    websAspDefine("umGroupFormState", um_group_form_state);
    websAspDefine("umGroupTableView", um_group_table_view);

    fh = CREATE_FH_INSTANCE(TEMPLATE_UM_GROUPS, ACL_OBJ_UM);

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;
    fh->disable_ldap=TRUE;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh)
{
    action_t action = get_action(wp, fh);

    if (action == ACTION_NONE_NOPERM) {
	return -1;
    } else if (action == ACTION_CREATE_GROUP) {
	fh->fv[FV_ID_GR_GROUP].flags |= FV_FLAG_SKIP_VALIDATE |
                                        FV_FLAG_DONT_SAVE;
    } else if (action == ACTION_NONE || 
               action == ACTION_DELETE_GROUP || 
               action == ACTION_LOOKUP_GROUP) {
	fh_disable_validate_and_save(fh, -1);
    } else if (action == ACTION_DELETE_MULTIPLE_GROUPS) {
	fh->fv[FV_ID_GR_GROUP].flags |= FV_FLAG_SKIP_VALIDATE |
                                        FV_FLAG_DONT_SAVE;
	fh->fv[FV_ID_GR_NEW_GROUP].flags |= FV_FLAG_SKIP_VALIDATE |
                                            FV_FLAG_DONT_SAVE;
    }
    fh->fv[FV_ID_GR_GROUP_ID].flags |= FV_FLAG_SKIP_VALIDATE |
                                       FV_FLAG_DONT_SAVE;

    return 0;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    action_t action = get_action(wp, fh);
    int ret = -1;
    int error = 0;
    int is_builtin;
    unsigned int i;
    const char * tmpwebsget;
    unsigned int numgroups;
    char groupnameindex[9 + 10 + 1];

#if !defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
    wp->fh_flags |= FH_FLAG_SKIP_SAVE;
#endif /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
    
    if (action == ACTION_NONE_NOPERM) return -1;
    if (action == ACTION_NONE) return 0;

    switch (action) {
      case ACTION_LOOKUP_GROUP:
	  if (strcmp(fh->fv[FV_ID_GR_GROUP].val.s, "")) {
	      bfreeSafe(B_L, wp->target_group);
	      wp->target_group = bstrdup(B_L, fh->fv[FV_ID_GR_GROUP].val.s);
	      wp->target_gid = pp_um_group_get_gid(wp->target_group);
	      delete_form_vars(wp);

	      websSetVar(wp, fh->fv[FV_ID_GR_GROUP].fvname, wp->target_group);
	      websSetVar(wp, fh->fv[FV_ID_GR_NEW_GROUP].fvname,
                         wp->target_group);
   	      set_response(wp, ERIC_RESPONSE_OK, "");
	      goto finish; /* do not free wp->target_group */
	  } else {
              delete_form_vars(wp);
	  }
#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  wp->fh_flags |= FH_FLAG_SKIP_SAVE;
#endif /* PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  break;
          
      case ACTION_CREATE_GROUP:

	  /* ---- set target group ----------------------------------------- */

	  bfreeSafe(B_L, wp->target_group);
	  wp->target_group = bstrdup(B_L, fh->fv[FV_ID_GR_NEW_GROUP].val.s);

	  /* ---- check groupname ------------------------------------------ */
          
          if(*wp->target_group == um_individual_group_prefix) {
              /* generic group may not start with individual group prefix */
              error = 1;
              errno = PP_UM_ERR_INVALID_GROUPNAME;
              goto error;
          }

	  /* ---- create group --------------------------------------------- */

	  error = pp_um_group_create(wp->target_group, 0);
	  
	  if (error) goto error;

	  wp->target_gid = pp_um_group_get_gid(wp->target_group);

	  /* ---- save to config fs ---------------------------------------- */

	  error = pp_cfg_save(DO_FLUSH);
	  if (error) goto error;

	  /* ---- group created successfully ------------------------------- */

#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  fh->fv[FV_ID_GR_GROUP].flags |= FV_FLAG_DONT_SAVE;
	  fh->fv[FV_ID_GR_NEW_GROUP].flags |= FV_FLAG_DONT_SAVE;
#else /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  delete_form_vars(wp);
#endif /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  websSetVar(wp, "group_table_view", "1");
	  /* FIXME: lookup newly created group */
	  set_response(wp, ERIC_RESPONSE_OK, _("Group created successfully."));
	  break;
	  
      case ACTION_MODIFY_GROUP:

	  /* ---- set target group ----------------------------------------- */

	  bfreeSafe(B_L, wp->target_group);
	  wp->target_group = bstrdup(B_L, fh->fv[FV_ID_GR_NEW_GROUP].val.s);
          wp->target_gid = pp_um_group_get_gid(fh->fv[FV_ID_GR_GROUP].val.s);
          
          is_builtin = wp->target_gid == 0 || /* Admin */
                       wp->target_gid == 1 || /* <Unknown> */
                       wp->target_gid == 2;   /* <None> */

#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  /* ---- try to change Admin group? ----------------------------- */
          
          if(wp->target_gid == 0) { /* Admin */
              /* for Admin it is not allowed to change name and ACL */
              set_response(wp, ERIC_RESPONSE_ERROR, 
                           _("You cannot modify the 'Admin' group."));
              goto error;
          }
#else /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  /* ---- try to change builtin group? ----------------------------- */
          
          if(is_builtin) {
              /* perhaps differentiate that, but for now, builtin groups do not
                 have any modifyable parameters except ACL */
              set_response(wp, ERIC_RESPONSE_ERROR, 
                           _("You cannot modify builtin groups."));
              goto error;
          }
#endif /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
          
	  /* ---- check groupname ------------------------------------------ */
          
          if (!is_builtin && (*wp->target_group == um_individual_group_prefix)) {
              /* generic group may not start with individual group prefix */
              error = 1;
              errno = PP_UM_ERR_INVALID_GROUPNAME;
              goto error;
          }

	  /* ---- rename group if necessary -------------------------------- */

	  if (!is_builtin && (strcmp(fh->fv[FV_ID_GR_GROUP].val.s, wp->target_group))) {
	      error = pp_um_group_rename(fh->fv[FV_ID_GR_GROUP].val.s,
							 wp->target_group);
	      if (error) goto error;
	  }

	  /* ---- group modified successfully ------------------------------ */
	  
#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  fh->fv[FV_ID_GR_GROUP].flags |= FV_FLAG_DONT_SAVE;
	  fh->fv[FV_ID_GR_NEW_GROUP].flags |= FV_FLAG_DONT_SAVE;
#else /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  delete_form_vars(wp);
#endif /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  websSetVar(wp, "group_table_view", "1");
	  
	  set_response(wp, ERIC_RESPONSE_OK, _("Group modified successfully."));
	  break;

      case ACTION_COPY_GROUP:

	  /* ---- set target principal ------------------------------------- */

	  bfreeSafe(B_L, wp->target_group);
	  wp->target_group = bstrdup(B_L, fh->fv[FV_ID_GR_NEW_GROUP].val.s);

	  /* ---- check groupname ------------------------------------------ */
          
          if(*wp->target_group == um_individual_group_prefix) {
              /* generic group may not start with individual group prefix */
              error = 1;
              errno = PP_UM_ERR_INVALID_GROUPNAME;
              goto error;
          }

	  /* ---- copy the group ------------------------------------------- */

	  error = pp_um_group_copy(fh->fv[FV_ID_GR_GROUP].val.s,
                                   wp->target_group);
	  if (error) goto error;
          wp->target_gid = pp_um_group_get_gid(wp->target_group);

	  /* ---- save the new group to config fs -------------------------- */

	  error = pp_cfg_save(DO_FLUSH);
	  if (error) goto error;

	  /* ---- group copied successfully -------------------------------- */
	  
	  delete_form_vars(wp);
	  set_response(wp, ERIC_RESPONSE_OK, _("Group copied successfully."));
	  break;

      case ACTION_DELETE_GROUP:

	  /* ---- delete group --------------------------------------------- */

	  error = pp_um_group_delete(fh->fv[FV_ID_GR_GROUP].val.s);
	  if (error) goto error;

	  /* ---- group deleted successfully ------------------------------- */

	  delete_form_vars(wp);
	  websSetVar(wp, "group_table_view", "1");
	  set_response(wp, ERIC_RESPONSE_OK, _("Group deleted successfully."));
#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  wp->fh_flags |= FH_FLAG_SKIP_SAVE;
#endif /* PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  break;

      case ACTION_DELETE_MULTIPLE_GROUPS:

	  /* ---- get number of groups and delete in a loop ---------------- */

	  if ((tmpwebsget = websGetVar(wp, "numgroups", NULL)) && ((numgroups = pp_strtoul_10(tmpwebsget, 0, NULL)) > 0)) {
	      pp_strstream_t rspmsg = PP_STRSTREAM_INITIALIZER;
	      pp_strstream_init(&rspmsg);
	      for (i = 0; i < numgroups; i++) {
	          snprintf(groupnameindex, sizeof(groupnameindex), "groupname%i", i);
	          if ((tmpwebsget = websGetVar(wp, groupnameindex, NULL))) {
	              if (pp_um_group_delete(tmpwebsget) == PP_SUC) {
	                  pp_strappendf(&rspmsg, _("Group \"%s\" deleted successfully."), tmpwebsget);
	              } else {
	                  pp_strappendf(&rspmsg, _("Group \"%s\" NOT deleted successfully - Error: %s"), tmpwebsget, pp_error_string(errno));
	              }
	              pp_strappendf(&rspmsg, "<br>");
	          } else {
	              pp_strappendf(&rspmsg, _("%i. group NOT deleted successfully - Error in Transmission."), i+1);
	          }
	      }
              set_response(wp, ERIC_RESPONSE_OK, rspmsg.buf);
	      pp_strstream_free(&rspmsg);
	  } else {
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Error in Transmission."));
	      goto error;
	  }

	  /* ---- groups deleted  ------------------------------------------ */

	  delete_form_vars(wp);
	  websSetVar(wp, "group_table_view", "1");
#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
	  wp->fh_flags |= FH_FLAG_SKIP_SAVE;
#endif /* PP_FEAT_RARITAN_STYLE_UM_PAGES */
	  break;
      default:
	  break;
    }

    ret = 0;

 error:
    if (error) set_response(wp, ERIC_RESPONSE_ERROR, pp_error_string(errno));
#if defined(PP_FEAT_RARITAN_STYLE_UM_PAGES)
    if (error) wp->fh_flags |= FH_FLAG_SKIP_SAVE;
#else /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */
    bfreeSafe(B_L, wp->target_group);
    wp->target_group = NULL;
    wp->target_gid = -1;
#endif /* !PP_FEAT_RARITAN_STYLE_UM_PAGES */

 finish:
    return ret;
}

static action_t
get_action(webs_t wp, form_handler_t * fh)
{
    action_t action = ACTION_NONE;

    int change_allowed = PP_SUC == 
        pp_um_user_has_permission(wp->user, fh->acl_object,
                                  pp_acl_raasip_yes_str);

    if (form_button_clicked(wp, "action_lookup_group") || websGetVar(wp, "action_lookup_group", NULL)) {
        /* RAASIP doesn't differentiate between view and change! */
	if (change_allowed) action = ACTION_LOOKUP_GROUP; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_create_group")) {
	if (change_allowed) action = ACTION_CREATE_GROUP; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_modify_group")) {
	if (change_allowed) action = ACTION_MODIFY_GROUP; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_copy_group")) {
	if (change_allowed) action = ACTION_COPY_GROUP; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_delete_group")) {
	if (change_allowed) action = ACTION_DELETE_GROUP; else goto perm_denied;
    } else if (websGetVar(wp, "action_delete_multiple_groups", NULL)) {
        if (change_allowed) action = ACTION_DELETE_MULTIPLE_GROUPS; else goto perm_denied;
    }
    return action;

 perm_denied:
    set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
    eric_notify_security_violation(wp->session);
    return ACTION_NONE_NOPERM;
}

static int
um_init_all_groups_asp(int eid UNUSED, webs_t wp, 
                       int argc UNUSED, char ** argv UNUSED) {
    return init_groups(wp, NULL, 0, 0);
}

static int
um_init_all_individual_groups_asp(int eid UNUSED, webs_t wp, 
                                  int argc UNUSED, char ** argv UNUSED) {
    char prefix[2] = { um_individual_group_prefix, '\0' };
    return init_groups(wp, prefix, 1, 1);
}

static int
um_init_all_generic_groups_asp(int eid UNUSED, webs_t wp, 
                               int argc UNUSED, char ** argv UNUSED) {
    char prefix[2] = { um_individual_group_prefix, '\0' };
    return init_groups(wp, prefix, 0, 0);
}

/**
 * if prefix != NULL
 * -> filter == 0 -> only groups without prefix
 * -> filter == 1 -> only groups with prefix
 *                -> cut == 1 -> prefix is cut from groupname
 */
static int
init_groups(webs_t wp, const char *prefix, int filter, int cut)
{
    vector_t *groups;
    char *var_name, *var_id_name;
    char opt_name[MAX_OPT_KEY_LEN + 1];
    char opt_val[11];
    u_int i, j = 0, groups_sz;
    form_handler_t *fh;
    
    fh = lookup_form_handler(TEMPLATE_UM_GROUPS);
    assert(fh);
    var_name = (char*)fh->fv_tmpl[FV_ID_GR_GROUP].fvname;
    var_id_name = (char*)fh->fv_tmpl[FV_ID_GR_GROUP_ID].fvname;

    groups = pp_um_get_all_groups();
// TODO: RAASIP:
    // principals is NULL when logged in with a ldap user
    // FIXME: find a better way to disable user management in ldap case
    if(groups) {
        int prefix_len = 0;
        
        if(prefix) {
            prefix_len = strlen(prefix);
        }
	groups_sz = vector_size(groups);
	for (i = 0; i < groups_sz; i++) {
            char *group = (char*)vector_get(groups, i);
            if(!prefix ||
               filter == !strncmp(group, prefix, prefix_len)) {
                form_var_vec_name(var_name, j, opt_name, sizeof(opt_name));
                if(!prefix || !cut) {
                    websSetVar(wp, opt_name, group);
                } else {
                    websSetVar(wp, opt_name, group + prefix_len);
                }
                form_var_vec_name(var_id_name, j++, opt_name, sizeof(opt_name));
                snprintf(opt_val, sizeof(opt_val), "%d", 
                         pp_um_group_get_gid(group));
                websSetVar(wp, opt_name, opt_val);
            }
	}
	vector_delete(groups);
        form_var_sz_name(var_name, opt_name, sizeof(opt_name));
	snprintf(opt_val, sizeof(opt_val), "%u", j);
	websSetVar(wp, opt_name, opt_val);
    }
    return 0;
}

static int
um_init_group_and_target_group_asp(int eid UNUSED, webs_t wp, 
                                   int argc UNUSED, char ** argv UNUSED)
{
    if (!form_was_submitted(wp) || 
        get_action(wp, lookup_form_handler(TEMPLATE_UM_GROUPS)) == ACTION_NONE){
	bfreeSafe(B_L, wp->target_group);
	wp->target_group = NULL;
        wp->target_gid = -1;
    }
    return 0;
}

static int
um_group_form_state(int eid, webs_t wp, 
              int argc UNUSED, char ** argv UNUSED)
{
    action_t action = get_action(wp, lookup_form_handler(TEMPLATE_UM_GROUPS));
    const char * r = "create";

    if ((action == ACTION_LOOKUP_GROUP) || (action == ACTION_MODIFY_GROUP)) {
	r = "modify";
    }

    ejSetResult(eid, r);

    return 0;
}

static int
um_group_table_view(int eid, webs_t wp, 
              int argc UNUSED, char ** argv UNUSED)
{
    const char * r = "no";

    if (websGetVar(wp, "group_table_view", NULL)) {
	r = "yes";
    }

    ejSetResult(eid, r);

    return 0;
}

