#include <openssl/md5.h>
#include <pp/um.h>
#include <pp/intl.h>
#include <pp/ldap_prof.h>
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

FV_SPEC = {
    {
	id:		FV_ID_UM_PWCHANGE_OLD_PASSWORD,
	cfgkey:		"password"
    },
    {
	id:		FV_ID_UM_PWCHANGE_PASSWORD,
	cfgkey:		"password"
    },
    {
	id:		FV_ID_UM_PWCHANGE_PASSWORD2,
	cfgkey:		"password"
    },
};

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);

int
um_pwchange_tmpl_init(void)
{
    form_handler_t * fh;

    if (um_common_tmpl_init() != 0) return -1;

    fh = CREATE_FH_INSTANCE(TEMPLATE_UM_PWCHANGE, ACL_OBJ_UM_PW);

    fh->post_validate_hook = post_validate_hook;
    fh->disable_ldap=TRUE;
    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp UNUSED, form_handler_t * fh)
{
#ifdef PP_FEAT_STRONG_PASSWORDS
    /* Skip basic validation in favor of more sophisticated check later */
    
    int strong_pw_check;
    
    pp_cfg_is_enabled(&strong_pw_check, "security.strong_pw.enabled");
    
    if(strong_pw_check) {
        fh->fv[FV_ID_UM_PWCHANGE_PASSWORD].flags |= FV_FLAG_SKIP_VALIDATE;
        fh->fv[FV_ID_UM_PWCHANGE_PASSWORD2].flags |= FV_FLAG_SKIP_VALIDATE;
    }
#endif /* PP_FEAT_STRONG_PASSWORDS */

    /* no need to validate old password, check it instead! */
    fh->fv[FV_ID_UM_PWCHANGE_OLD_PASSWORD].flags |= FV_FLAG_SKIP_VALIDATE;

    return 0;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    int ret = PP_SUC;
    int flags = PP_UM_AUTH_NO_FLAGS;
    const char *old_password = fh->fv[FV_ID_UM_PWCHANGE_OLD_PASSWORD].val.s;
    const char *password = fh->fv[FV_ID_UM_PWCHANGE_PASSWORD].val.s;
    
    
    /* check old password */
    if(pp_um_user_authenticate_with_ip_str(wp->target_user, old_password,
                                           PP_UM_AUTH_IGNORE_BLOCKING,
                                           NULL, NULL, NULL) == PP_ERR) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Authentication failed. %s"),
		     pp_ldap_has_errno(errno) ? _(pp_error_string(errno)) : "");
	return PP_ERR;
    }

    /* check new passwords */
    if (strcmp(fh->fv[FV_ID_UM_PWCHANGE_PASSWORD].val.s,
	       fh->fv[FV_ID_UM_PWCHANGE_PASSWORD2].val.s)) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("The two passwords do not match."));
	return -1;
    }

#ifdef PP_FEAT_STRONG_PASSWORDS
    {
        /* check conformity with common sequrity requirements (if enabled) */
        char * response;
        
        if(*password != '\0' && 
           PP_FAILED(pp_um_strong_pw_check_for_user(wp->target_user,
                                                    password, old_password,
                                                    &response)) ) {
            set_response(wp, ERIC_RESPONSE_ERROR, response);
            free(response);
            return -1;
        }
        
        flags |= PP_UM_AUTH_STRONG_PW_CHECKED;
    }
#endif /* PP_FEAT_STRONG_PASSWORDS */

    if(websGetVar(wp, "pw_change_forced", NULL)) {
        /* password change is forced, check if password changed */
        flags |= PP_UM_AUTH_PASSWD_FORCE_CHANGE;
    }
    
    if((ret = pp_um_user_set_password(wp->target_user, password, 
                                      flags)) == PP_ERR &&
       errno == PP_UM_ERR_PASSWORD_DIDNT_CHANGE) {
        /* show errorstring in that case! */
        set_response(wp, ERIC_RESPONSE_ERROR, _(pp_error_string(errno)));
    } else {
        /* password changed successfully */
#ifdef PP_FEAT_STRONG_PASSWORDS
        pp_um_strong_pw_add_to_history(wp->target_user, old_password,
                                       PP_UM_AUTH_NO_FLAGS);
#endif /* PP_FEAT_STRONG_PASSWORDS */
        
        if(flags & PP_UM_AUTH_PASSWD_FORCE_CHANGE) {
            /* set flag to show hint for forced password change */
            websSetVar(wp, "pw_change_forced_suc", "yes");
        }
        
        pp_cfg_save(DO_FLUSH);
    }

    return ret;
}
