#include <openssl/md5.h>
#include <pp/um.h>
#include <pp/base.h>
#include <pp/intl.h>
#include <pp/xdefs.h>
#include <liberic_config.h>
#include <liberic_net.h>
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

FV_SPEC = {
    {
	id:		FV_ID_UM_USER,
	cfgkey:		"user_name"
    },
    {
	id:		FV_ID_UM_NEW_USER,
	cfgkey:		"user[%U].login"
    },
    {
	id:		FV_ID_UM_FULLNAME,
	cfgkey:		"user[%U].name"
    },
    {
	id:		FV_ID_UM_PASSWORD,
	cfgkey:		"password"
    },
    {
	id:		FV_ID_UM_PASSWORD2,
	cfgkey:		"password"
    },
#ifdef PP_FEAT_DISABLE_WEBACCESS
    {
        id:             FV_ID_UM_DISABLE_WEB_ACCESS,
        cfgkey:         "user[%U].disable_web_access"
    },
#endif
#ifdef PP_FEAT_AUTH_SSL
    {
	id:		FV_ID_UM_AUTH_PASSWD,
	cfgkey:		"auth_passwd"
    },
    {
	id:		FV_ID_UM_AUTH_NO_PASSWD,
	cfgkey:		"user[%U].auth_no_passwd"
    },
    {
	id:		FV_ID_UM_AUTH_CERT,
	cfgkey:		"user[%U].auth_cert"
    },
    {
	id:		FV_ID_UM_CERT_HASH,
	cfgkey:		"user[%U].cert_hash"
    },
    {
	id:		FV_ID_UM_CERT_SUBJECT,
	cfgkey:		"user[%U].cert_subject"
    },
#endif /* PP_FEAT_AUTH_SSL */
    {
	id:		FV_ID_UM_EMAIL,
	cfgkey:		"user[%U].email"
    },
    {
	id:		FV_ID_UM_MOBILE,
	cfgkey:		"user[%U].mobile"
    },
#if defined(PP_FEAT_HAS_DIALBACK)
    {
	id:		FV_ID_UM_DIALBACK,
	cfgkey:		"user[%U].dial_back_number"
    },
#endif /* PP_FEAT_HAS_DIALBACK */
    {
	id:		FV_ID_UM_GROUP_ID,
	cfgkey:		"user[%U].group_id"
    },
    {
	id:		FV_ID_UM_IPMI_UID,
	cfgkey:		"user[%U].ipmi_uid"
    },
    {
	id:		FV_ID_UM_NEED_PW_CHANGE,
	cfgkey:		"user[%U].need_pw_change"
    },
#if 0 // really needed?
    {
	/* Privilege level for IPMI LAN channel */
	id:		FV_ID_UM_IPMI_PRIV_LEVEL,
	cfgkey:		"user[%U].ipmi.channel[1].priv_level"
    },
#endif
};

typedef enum {
    ACTION_NONE,
    ACTION_NONE_NOPERM,
    ACTION_LOOKUP_USER,
    ACTION_UNBLOCK_USER,
    ACTION_CREATE_USER,
    ACTION_MODIFY_USER,
    ACTION_COPY_USER,
    ACTION_DELETE_USER,
    ACTION_DELETE_MULTIPLE_USERS,
} action_t;

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);
static action_t get_action(webs_t wp, form_handler_t * fh);
#ifdef PP_FEAT_AUTH_SSL
static int cert_upload_cb(webs_t wp, char * name, char * filename,
			void * data, size_t data_offset, size_t data_len, int more_data);
#endif
static int um_init_all_users_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_init_user_list_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_init_user_and_target_user_asp(int eid, webs_t wp, int argc, char ** argv);
static int um_user_form_state(int eid, webs_t wp, int argc, char ** argv);
static int um_user_table_view(int eid, webs_t wp, int argc, char ** argv);

int
um_users_tmpl_init(void)
{
    form_handler_t * fh;

    if (um_common_tmpl_init() != 0) return -1;

    /* register ASPs */
    websAspDefine("umInitAllUsers", um_init_all_users_asp);
    websAspDefine("umInitUserList", um_init_user_list_asp);
    websAspDefine("umInitUserAndTargetUser", um_init_user_and_target_user_asp);
    websAspDefine("umUserFormState", um_user_form_state);
    websAspDefine("umUserTableView", um_user_table_view);
    
    /* register POST callbacks */
#ifdef PP_FEAT_AUTH_SSL
    websAddPostDataCallback("user_cert", cert_upload_cb);
#endif /* PP_FEAT_AUTH_SSL */

#ifdef PP_FEAT_PERM_MANAGEMENT
    fh = CREATE_FH_INSTANCE(TEMPLATE_UM_USERS, ACL_OBJ_UM);
#else
    fh = CREATE_FH_INSTANCE(TEMPLATE_UM_USERS, ACL_OBJ_SUPER);
#endif

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;
    fh->disable_ldap=TRUE;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp, form_handler_t * fh)
{
    action_t action = get_action(wp, fh);
    size_t i;
#if !defined(PP_FEAT_PERM_MANAGEMENT)
    int is_root;
#endif /* !PP_FEAT_PERM_MANAGEMENT */

    if (action == ACTION_NONE_NOPERM) {
	return -1;
    } else if (action == ACTION_NONE) {
	fh_disable_validate_and_save(fh, -1);
    } else {
#ifdef PP_FEAT_AUTH_SSL
	/* set the FV_ID_UM_AUTH_NO_PASSWD form_var for backward compatibility */
	if (*fh->fv[FV_ID_UM_AUTH_PASSWD].val.s == '\0') {
	    websSetVar(wp, fh->fv[FV_ID_UM_AUTH_NO_PASSWD].name, "yes");
	    fh->fv[FV_ID_UM_AUTH_NO_PASSWD].val.s =
		websGetVar(wp, fh->fv[FV_ID_UM_AUTH_NO_PASSWD].name, "");
	} else {
	    websSetVar(wp, fh->fv[FV_ID_UM_AUTH_NO_PASSWD].name, "");
	    fh->fv[FV_ID_UM_AUTH_NO_PASSWD].val.s =
		websGetVar(wp, fh->fv[FV_ID_UM_AUTH_NO_PASSWD].name, "");
	}
#endif /* PP_FEAT_AUTH_SSL */

#if !defined(PP_FEAT_PERM_MANAGEMENT)
        pp_um_user_is_root(&is_root, wp->target_user);
	if (action == ACTION_MODIFY_USER && is_root) {
	    fh->fv[FV_ID_UM_GROUP_ID].flags |= 
                FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
	}
#endif /* !PP_FEAT_PERM_MANAGEMENT */

	/* for MODIFY_USER, check if a user is selected */
	if (action == ACTION_MODIFY_USER &&
	    (fh->fv[FV_ID_UM_USER].val.s == '\0' ||
	     !pp_strcmp_safe(fh->fv[FV_ID_UM_USER].val.s, ""))) {

	    set_response(wp, ERIC_RESPONSE_ERROR, _("No user selected."));
	    return -1;
	}
	
	for (i = 0; i < fh->fv_cnt; i++) {
	    form_var_t * fv = &fh->fv[i];

	    if (action == ACTION_MODIFY_USER
		&& (fv->id == FV_ID_UM_PASSWORD || fv->id == FV_ID_UM_PASSWORD2)) {
		fv->flags |= FV_FLAG_ALLOW_EMPTY;
	    }

	    if ((action == ACTION_COPY_USER && fv->id != FV_ID_UM_NEW_USER)
                || (action == ACTION_CREATE_USER && fv->id == FV_ID_UM_USER)
		|| action == ACTION_DELETE_USER
		|| action == ACTION_DELETE_MULTIPLE_USERS
		|| action == ACTION_LOOKUP_USER
		|| action == ACTION_UNBLOCK_USER
                || fv->id == FV_ID_UM_GROUP_ID) {
		fv->flags |= FV_FLAG_SKIP_VALIDATE;
	    }
	}
    }

#ifdef PP_FEAT_STRONG_PASSWORDS
    {
        /* Skip basic validation in favor of more sophisticated check later */
        
        int strong_pw_check;
        
        pp_cfg_is_enabled(&strong_pw_check, "security.strong_pw.enabled");
        
        if(strong_pw_check) {
            fh->fv[FV_ID_UM_PASSWORD].flags |= FV_FLAG_SKIP_VALIDATE;
            fh->fv[FV_ID_UM_PASSWORD2].flags |= FV_FLAG_SKIP_VALIDATE;
        }
    }
#endif /* PP_FEAT_STRONG_PASSWORDS */

#if defined(PP_FEAT_CASE_INSENSITIVE_LOGINS)
    {
        /* convert login to lower case */
        
        const char *login = fh->fv[FV_ID_UM_NEW_USER].val.s;
        
        if(login && *login) {
            /* no need to convert empty string */
            
            char *login_dc = strdup(login), *c;
            int dirty = 0;
            
            for(c = login_dc; *c; ++c) {
                if(isupper(*c)) {
                    *c = tolower(*c);
                    ++dirty;
                }
            }
            
            if(dirty) {
                /* no need to set new login if nothing changed */
                websSetVar(wp, fh->fv[FV_ID_UM_NEW_USER].fvname, login_dc);
                fh->fv[FV_ID_UM_NEW_USER].val.s = 
                    websGetVar(wp, fh->fv[FV_ID_UM_NEW_USER].fvname, "");
            }
            
            free(login_dc);
        }
    }
#endif /* PP_FEAT_CASE_INSENSITIVE_LOGINS */

    return 0;
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    action_t action = get_action(wp, fh);
    char * group = NULL;
    unsigned int i;
    int ret = -1;
    int err = 0;
    int is_root;
    int renamed = 0;
    const char * tmpwebsget;
    unsigned int numusers;
    char usernameindex[8 + 10 + 1];

    wp->fh_flags |= FH_FLAG_SKIP_SAVE;

    if (action == ACTION_NONE_NOPERM) return -1;
    if (action == ACTION_NONE) return 0;

    if (action == ACTION_CREATE_USER || action == ACTION_MODIFY_USER) {
#ifdef PP_FEAT_AUTH_SSL
	if (wp->cert_data_size) {
	    char hash[9], subject_buf[256];
	    X509 *cert = NULL;
	    unsigned char *data = wp->cert_data;
	    if (d2i_X509(&cert, &data, wp->cert_data_size)) {
		snprintf(hash, sizeof(hash), "%lX", X509_issuer_and_serial_hash(cert));
		X509_NAME_oneline(X509_get_subject_name(cert), subject_buf, sizeof(subject_buf));
		X509_free(cert);
		websSetVar(wp, fh->fv[FV_ID_UM_CERT_HASH].fvname, hash);
		fh->fv[FV_ID_UM_CERT_HASH].val.s = 
                    websGetVar(wp, fh->fv[FV_ID_UM_CERT_HASH].fvname, "");
		websSetVar(wp, fh->fv[FV_ID_UM_CERT_SUBJECT].fvname, subject_buf);
		fh->fv[FV_ID_UM_CERT_SUBJECT].val.s = 
                    websGetVar(wp, fh->fv[FV_ID_UM_CERT_SUBJECT].fvname, "");
	    } else {
		ERR_print_errors_fp(stdout);
		set_response(wp, ERIC_RESPONSE_ERROR, _("Error in the certificate."));
		return -1;
	    }
	}
#endif /* PP_FEAT_AUTH_SSL */

	if (strcmp(fh->fv[FV_ID_UM_PASSWORD].val.s,
		   fh->fv[FV_ID_UM_PASSWORD2].val.s)) {
	    set_response(wp, ERIC_RESPONSE_ERROR,
			 _("The two passwords do not match."));
	    return -1;
	}

        /* new password entered */
	if (fh->fv[FV_ID_UM_PASSWORD].val.s[0] != '\0') {
#ifdef PP_FEAT_STRONG_PASSWORDS
	    /* check conformity with common sequrity requirements (if enabled)*/
	    char *response;
	    if ( PP_FAILED(pp_um_strong_pw_check_for_user(wp->target_user,
			                        fh->fv[FV_ID_UM_PASSWORD].val.s, 
                                                          NULL, &response)) ) {
		set_response(wp, ERIC_RESPONSE_ERROR, response);
                free(response);
		return -1;
	    }
#endif /* PP_FEAT_STRONG_PASSWORDS */
	}
    }

    switch (action) {
      case ACTION_LOOKUP_USER:
	  set_response(wp, ERIC_RESPONSE_OK, "");
	  if (strcmp(fh->fv[FV_ID_UM_USER].val.s, "")) {
	      bfreeSafe(B_L, wp->target_user);
	      wp->target_user = bstrdup(B_L, fh->fv[FV_ID_UM_USER].val.s);
	      wp->target_uid = pp_um_user_get_uid(wp->target_user);
	      delete_form_vars(wp);
              for (i = 0; fh->fv[i].cfgkey != NULL; i++) {
                  if (fh->fv[i].id == FV_ID_UM_FULLNAME
		      || fh->fv[i].id == FV_ID_UM_EMAIL
		      || fh->fv[i].id == FV_ID_UM_MOBILE) {
                      char *value;
                      int nodflt = (fh->fv[i].flags & FV_FLAG_DONT_SAVE
				    || fh->fv[i].flags & FV_FLAG_ALLOW_EMPTY
				    || fh->fv[i].flags & FV_FLAG_ALLOW_COND_EMPTY);
		      if (nodflt) {
			  ret = pp_cfg_get_nodflt(&value, fh->fv[i].cfgkey, wp->target_uid);
		      } else {
			  ret = pp_cfg_get(&value, fh->fv[i].cfgkey, wp->target_uid);
		      }
                      if (ret == PP_SUC) {
	                  websSetVar(wp, fh->fv[i].fvname, value);
                          free(value);
                      }
                  }
	      }
	      websSetVar(wp, fh->fv[FV_ID_UM_USER].fvname, wp->target_user);
	      websSetVar(wp, fh->fv[FV_ID_UM_NEW_USER].fvname, wp->target_user);
	      goto finish; /* do not free wp->target_user */
	  } else {
              delete_form_vars(wp);
          }
	  break;
    
#ifdef PP_FEAT_USER_BLOCKING
      case ACTION_UNBLOCK_USER:
	  pp_um_user_unblock(fh->fv[FV_ID_UM_USER].val.s);
	  ret = 0;
	  goto finish; /* do not free wp->target_user */
#endif /* PP_FEAT_USER_BLOCKING */
      case ACTION_CREATE_USER:

	  /* ---- set target principal ------------------------------------- */

	  bfreeSafe(B_L, wp->target_user);
	  wp->target_user = bstrdup(B_L, fh->fv[FV_ID_UM_NEW_USER].val.s);

	  /* ---- figure out group ----------------------------------------- */

          if(!strcmp(fh->fv[FV_ID_UM_GROUP_ID].val.s, "fourtytwo")) {
              /* so we now know the answer... ;-) */
              /* NOTE! 'group' stays NULL in that case, um will create new
                       individual group for user on it self! */
              fh->fv[FV_ID_UM_GROUP_ID].flags |= FV_FLAG_DONT_SAVE;
          } else {
              /* get real group name */
              pp_cfg_get_nodflt(&group, "group[%s].name",
                                fh->fv[FV_ID_UM_GROUP_ID].val.s);
          }

	  /* ---- create user ---------------------------------------------- */

          if(PP_ERR == (err = pp_um_user_create(wp->target_user,
                                                fh->fv[FV_ID_UM_PASSWORD].val.s, 
                                                PP_UM_AUTH_NO_FLAGS, group, 
#if defined(PP_FEAT_HAS_DIALBACK)
                                                fh->fv[FV_ID_UM_DIALBACK].val.s,
#else /* !PP_FEAT_HAS_DIALBACK */
                                                NULL,
#endif /* !PP_FEAT_HAS_DIALBACK */
                                                0
                                                ))) {
              goto error;
          }
	  wp->target_uid = pp_um_user_get_uid(wp->target_user);
          free(group);

	  /* ---- save new user to config fs ------------------------------- */

	  if (save_form_vars(wp) == PP_ERR) {
              goto error;
          }

	  /* ---- user created successfully -------------------------------- */

	  delete_form_vars(wp);
	  websSetVar(wp, "user_table_view", "1");
	  /* FIXME: lookup newly created user */
	  set_response(wp, ERIC_RESPONSE_OK, _("User created successfully."));
#ifdef PP_FEAT_AUTH_SSL
	  if (wp->cert_data_size) {
	      eric_net_reconf_listener();
	  }
#endif 
	break;

      case ACTION_MODIFY_USER:
          is_root = (wp->target_uid =
                         pp_um_user_get_uid(fh->fv[FV_ID_UM_USER].val.s)) == 0;

          if (is_root) {
              /* disable saving fixed values */
              fh->fv[FV_ID_UM_FULLNAME].flags |= FV_FLAG_DONT_SAVE;
              fh->fv[FV_ID_UM_GROUP_ID].flags |= FV_FLAG_DONT_SAVE;
          }
       
	  /* ---- set target principal ------------------------------------- */

	  bfreeSafe(B_L, wp->target_user);
	  wp->target_user = bstrdup(B_L, fh->fv[FV_ID_UM_NEW_USER].val.s);

	  /* ---- rename user if necessary --------------------------------- */

	  if (strcmp(fh->fv[FV_ID_UM_USER].val.s, wp->target_user)) {
              renamed = 1;
              if(PP_ERR == (err = pp_um_user_rename(fh->fv[FV_ID_UM_USER].val.s,
				                    wp->target_user))) {
	          goto error;
              }
	      if (!strcmp(wp->user, fh->fv[FV_ID_UM_USER].val.s)) {
		  eric_session_set_user(wp->session, wp->target_user);
	      }
              err = pp_cfg_save(DONT_FLUSH);
	  }

	  /* superusers group may not change! */
	  if (wp->target_uid > 0 ) {
              if(!strcmp(fh->fv[FV_ID_UM_GROUP_ID].val.s, "fourtytwo")) {
                  /* so we now know the answer... ;-) */
                  /* NOTE! 'group' stays NULL in that case, um will create new
                           individual group for user on it self! */
                  fh->fv[FV_ID_UM_GROUP_ID].flags |= FV_FLAG_DONT_SAVE;
              } else {
                  /* get real group name */
                  pp_cfg_get_nodflt(&group, "group[%s].name",
                                    fh->fv[FV_ID_UM_GROUP_ID].val.s);
              }
              /* NOTE! um will check if group change is really necessary.
                       if changing from individual to generic group, individual
                       group will be deleted. */
              if(PP_ERR == (err = pp_um_user_set_group(wp->target_user, 
                                                       group))) {
                  free(group);
                  goto error;
              }
              free(group);
	  }

	  /* ---- adapt username if changed -------------------------------- */
          
          if (renamed) {
              fh->fv[FV_ID_UM_USER].val.s =
                  websGetVar(wp, fh->fv[FV_ID_UM_NEW_USER].fvname, "");
          }

          /* ---- set new password if required ----------------------------- */
          
          if(fh->fv[FV_ID_UM_PASSWORD].val.s[0] != '\0') {
              char *old_password = NULL;
              
              pp_cfg_get_nodflt(&old_password, "user[%u].pw_hash",
                                wp->target_uid);
              
              if(pp_um_user_set_password(wp->target_user,
                                         fh->fv[FV_ID_UM_PASSWORD].val.s,
                                         PP_UM_AUTH_NO_FLAGS) != PP_ERR &&
                 old_password) {
#ifdef PP_FEAT_STRONG_PASSWORDS
                  pp_um_strong_pw_add_to_history(wp->target_user, old_password,
                                                 PP_UM_AUTH_PASSWD_AS_MD5_HASH);
#endif /* PP_FEAT_STRONG_PASSWORDS */
              }
              
              free(old_password);
          }

          /* ---- save modified user to config fs -------------------------- */

	  if (save_form_vars(wp) == PP_ERR) {
              goto error;
          }

	  /* ---- user modified successfully ------------------------------- */

	  delete_form_vars(wp);
	  websSetVar(wp, "user_table_view", "1");
	  
	  /* FIXME: lookup modified user */
	  set_response(wp, ERIC_RESPONSE_OK, _("User modified successfully."));

#ifdef PP_FEAT_AUTH_SSL
	  if (wp->cert_data_size) {
	      eric_net_reconf_listener();
	  }
#endif 
	  break;

#ifdef PP_FEAT_PERM_MANAGEMENT
      case ACTION_COPY_USER:
	
	  /* ---- set target principal ------------------------------------- */

	  bfreeSafe(B_L, wp->target_user);
	  wp->target_user = bstrdup(B_L, fh->fv[FV_ID_UM_NEW_USER].val.s);

	  /* ---- copy the user -------------------------------------------- */
	  if(PP_ERR == (err = pp_um_user_copy(fh->fv[FV_ID_UM_USER].val.s,
                                              wp->target_user))) {
	      goto error;
          }

	  /* ---- save the new user to config fs --------------------------- */

	  pp_cfg_save(DONT_FLUSH);
	  
	  /* ---- user copied successfully --------------------------------- */
	  
	  delete_form_vars(wp);
	  break;
#endif /* PP_FEAT_PERM_MANAGEMENT */

      case ACTION_DELETE_USER:

	  /* ---- delete user ---------------------------------------------- */

	  bfreeSafe(B_L, wp->target_user);
	  wp->target_user = bstrdup(B_L, fh->fv[FV_ID_UM_USER].val.s);

	  /* ---- delete user ---------------------------------------------- */

	  if(PP_ERR == (err = pp_um_user_delete(wp->target_user))) {
              goto error; 
          }

	  /* ---- group deleted successfully ------------------------------- */

	  delete_form_vars(wp);
	  websSetVar(wp, "user_table_view", "1");
	  set_response(wp, ERIC_RESPONSE_OK, _("User deleted successfully."));
/* FIXME: TODO, log off if you deleted yourself! */
	  break;

      case ACTION_DELETE_MULTIPLE_USERS:

	  /* ---- get number of users and delete in a loop ----------------- */

	  if ((tmpwebsget = websGetVar(wp, "numusers", NULL)) && ((numusers = pp_strtoul_10(tmpwebsget, 0, NULL)) > 0)) {
	      pp_strstream_t rspmsg = PP_STRSTREAM_INITIALIZER;
	      pp_strstream_init(&rspmsg);
	      for (i = 0; i < numusers; i++) {
	          snprintf(usernameindex, sizeof(usernameindex), "username%i", i);
	          if ((tmpwebsget = websGetVar(wp, usernameindex, NULL))) {
	              if (pp_um_user_delete(tmpwebsget) == PP_SUC) {
	                  pp_strappendf(&rspmsg, _("User \"%s\" deleted successfully."), tmpwebsget);
	              } else {
	                  pp_strappendf(&rspmsg, _("User \"%s\" NOT deleted successfully - Error: %s"), tmpwebsget, pp_error_string(errno));
	              }
	              pp_strappendf(&rspmsg, "<br>");
	          } else {
	              pp_strappendf(&rspmsg, _("%i. user NOT deleted successfully - Error in Transmission."), i+1);
	          }
	      }
              set_response(wp, ERIC_RESPONSE_OK, rspmsg.buf);
	      pp_strstream_free(&rspmsg);
	  } else {
	      set_response(wp, ERIC_RESPONSE_ERROR, _("Error in Transmission."));
	      goto error;
	  }

	  /* ---- users deleted -------------------------------------------- */

	  delete_form_vars(wp);
	  websSetVar(wp, "user_table_view", "1");
/* FIXME: TODO, log off if you deleted yourself! */
	  break;

      default:
	  break;
    }

    ret = 0;
    
 error:
    if (err) set_response(wp, ERIC_RESPONSE_ERROR, pp_error_string(errno));
    bfreeSafe(B_L, wp->target_user);
    wp->target_user = NULL;
    wp->target_uid = -1;

 finish:
    return ret;
}

static action_t
get_action(webs_t wp, form_handler_t * fh)
{
    action_t action = ACTION_NONE;
    
    int change_allowed = PP_SUC == pp_um_user_has_permission(wp->user,
                                         fh->acl_object, pp_acl_raasip_yes_str);

    if (form_button_clicked(wp, "action_lookup_user") || websGetVar(wp, "action_lookup_user", NULL)) {
        /* RAASIP doesn't differentiate between view and change! */
	if (change_allowed) action = ACTION_LOOKUP_USER; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_modify_user")) {
	if (change_allowed) action = ACTION_MODIFY_USER; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_create_user")) {
	if (change_allowed) action = ACTION_CREATE_USER; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_delete_user")) {
	if (change_allowed) action = ACTION_DELETE_USER; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_copy_user")) {
	if (change_allowed) action = ACTION_COPY_USER; else goto perm_denied;
    } else if (form_button_clicked(wp, "action_unblock_user")) {
	if (change_allowed) action = ACTION_UNBLOCK_USER; else goto perm_denied;
    } else if (websGetVar(wp, "action_delete_multiple_users", NULL)) {
        if (change_allowed) action = ACTION_DELETE_MULTIPLE_USERS; else goto perm_denied;
    }
    
    return action;

 perm_denied:
    set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
    eric_notify_security_violation(wp->session);
    return ACTION_NONE_NOPERM;
}

#ifdef PP_FEAT_AUTH_SSL
static int
cert_upload_cb(webs_t wp, char * name UNUSED, char * filename UNUSED,
	       void * data, size_t data_offset UNUSED, size_t data_len,
	       int more_data)
{
    void * cd;

    eric_session_touch(wp->session);
    
    if(PP_ERR == pp_um_user_has_permission(wp->user, ACL_OBJ_UM,
                                           pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	goto fail;
    }

    if (data_len > 0) {
	if ((cd = brealloc(B_L, wp->cert_data,
			   wp->cert_data_size + data_len)) == NULL) {
	    set_response(wp, ERIC_RESPONSE_ERROR,
			 _("Not enough free memory for certificate."));
	    goto fail;
	}
	wp->cert_data = cd;
	memcpy(wp->cert_data + wp->cert_data_size, data, data_len);
	wp->cert_data_size += data_len;
    }

    if (data == NULL) {
	goto fail;
    } else if (!more_data && wp->cert_data_size >= 2) {
	wp->cert_data_size -= 2;
    }

    wp->upload_failed = 0;

    return 0;

 fail:
    free(wp->cert_data);
    wp->cert_data = NULL;
    wp->cert_data_size = 0;
    wp->upload_failed = 1;

    return -1;

}
#endif /* PP_FEAT_AUTH_SSL */

static int
um_init_all_users_asp(int eid UNUSED, webs_t wp, 
                      int argc UNUSED, char ** argv UNUSED)
{
    vector_t *users;
    char * var_name;
    char opt_name[MAX_OPT_KEY_LEN + 1];
    char cnt_opt_val[11];
    u_int i, users_sz;
    form_handler_t *fh;
    
    fh = lookup_form_handler(TEMPLATE_UM_USERS);
    assert(fh);
    var_name = fh->fv_tmpl[FV_ID_UM_USER].fvname;

    users = pp_um_get_all_users();
// todo RAASIP:
    // principals is NULL when logged in with a ldap user
    // FIXME: find a better way to disable user management in ldap case
    if(users) {
	users_sz = vector_size(users);
	for (i = 0; i < users_sz; i++) {
            form_var_vec_name(var_name, i, opt_name, sizeof(opt_name));
	    websSetVar(wp, opt_name, vector_get(users, i));
	}
	vector_delete(users);
        form_var_sz_name(var_name, opt_name, sizeof(opt_name));
	snprintf(cnt_opt_val, sizeof(cnt_opt_val), "%u", i);
	websSetVar(wp, opt_name, cnt_opt_val);
    }

    return 0;
}

static int
um_init_user_list_asp(int eid UNUSED, webs_t wp, 
                      int argc UNUSED, char ** argv UNUSED)
{
    vector_t *users;
    char * var_login;
    char * var_fullname;
    char * var_email;
    char * var_mobile;
    char * var_group;
    char * user_name;
    char * value = NULL;
    char opt_name[MAX_OPT_KEY_LEN + 1];
    char cnt_opt_val[11];
    u_int i, users_sz, uid;
    form_handler_t *fh_um;
    form_handler_t *fh_gr;
    
    fh_um = lookup_form_handler(TEMPLATE_UM_USERS);
    assert(fh_um);
    fh_gr = lookup_form_handler(TEMPLATE_UM_GROUPS);
    assert(fh_gr);
    var_login = fh_um->fv_tmpl[FV_ID_UM_USER].fvname;
    var_fullname = fh_um->fv_tmpl[FV_ID_UM_FULLNAME].fvname;
    var_email = fh_um->fv_tmpl[FV_ID_UM_EMAIL].fvname;
    var_mobile = fh_um->fv_tmpl[FV_ID_UM_MOBILE].fvname;
    var_group = fh_gr->fv_tmpl[FV_ID_GR_GROUP].fvname;

    users = pp_um_get_all_users();
// todo RAASIP:
    // principals is NULL when logged in with a ldap user
    // FIXME: find a better way to disable user management in ldap case
    if(users) {
	users_sz = vector_size(users);
	for (i = 0; i < users_sz; i++) {
            user_name = vector_get(users, i);
            uid = pp_um_user_get_uid(user_name);
            form_var_vec_name(var_login, i, opt_name, sizeof(opt_name));
	    websSetVar(wp, opt_name, user_name);
            if (pp_cfg_get_nodflt(&value, "user[%u].name", uid) == PP_SUC) {
                form_var_vec_name(var_fullname, i, opt_name, sizeof(opt_name));
	        websSetVar(wp, opt_name, value);
	        free(value);
	    }
            if (pp_cfg_get_nodflt(&value, "user[%u].email", uid) == PP_SUC) {
                form_var_vec_name(var_email, i, opt_name, sizeof(opt_name));
	        websSetVar(wp, opt_name, value);
	        free(value);
	    }
            if (pp_cfg_get_nodflt(&value, "user[%u].mobile", uid) == PP_SUC) {
                form_var_vec_name(var_mobile, i, opt_name, sizeof(opt_name));
	        websSetVar(wp, opt_name, value);
	        free(value);
	    }
            if (pp_um_user_get_group(user_name, &value) == PP_SUC) {
                form_var_vec_name(var_group, i, opt_name, sizeof(opt_name));
	        websSetVar(wp, opt_name, value);
	        free(value);
	    }
	}
	vector_delete(users);
	snprintf(cnt_opt_val, sizeof(cnt_opt_val), "%u", i);
        form_var_sz_name(var_login, opt_name, sizeof(opt_name));
	websSetVar(wp, opt_name, cnt_opt_val);
        form_var_sz_name(var_fullname, opt_name, sizeof(opt_name));
	websSetVar(wp, opt_name, cnt_opt_val);
        form_var_sz_name(var_email, opt_name, sizeof(opt_name));
	websSetVar(wp, opt_name, cnt_opt_val);
        form_var_sz_name(var_mobile, opt_name, sizeof(opt_name));
	websSetVar(wp, opt_name, cnt_opt_val);
        form_var_sz_name(var_group, opt_name, sizeof(opt_name));
	websSetVar(wp, opt_name, cnt_opt_val);
    }

    return 0;
}

static int
um_init_user_and_target_user_asp(int eid UNUSED, webs_t wp, 
                                 int argc UNUSED, char ** argv UNUSED)
{
    if (!form_was_submitted(wp) || 
        get_action(wp, lookup_form_handler(TEMPLATE_UM_USERS)) == ACTION_NONE) {
	bfreeSafe(B_L, wp->target_user);
	wp->target_user = NULL;
        wp->target_uid = -1;
    }
    return 0;
}

static int
um_user_form_state(int eid, webs_t wp, 
              int argc UNUSED, char ** argv UNUSED)
{
    action_t action = get_action(wp, lookup_form_handler(TEMPLATE_UM_USERS));
    const char * r = "create";

    if ((action == ACTION_LOOKUP_USER) || (action == ACTION_MODIFY_USER)) {
	r = "modify";
    }

    ejSetResult(eid, r);

    return 0;
}

static int
um_user_table_view(int eid, webs_t wp, 
              int argc UNUSED, char ** argv UNUSED)
{
    const char * r = "no";

    if (websGetVar(wp, "user_table_view", NULL)) {
	r = "yes";
    }

    ejSetResult(eid, r);

    return 0;
}


