#include <stdio.h>
#include <string.h>
#include <pp/intl.h>
#include "eric_forms.h"
#include "eric_util.h"
#include "wsIntrn.h"

static int no_languages_asp(int eid, webs_t wp, int argc, char ** argv);
static int language_short_asp(int eid, webs_t wp, int argc, char ** argv);
static int language_full_asp(int eid, webs_t wp, int argc, char ** argv);
static int translate_get_language_full(int eid, webs_t wp, int argc, char ** argv);
static int translate_get_language_short(int eid, webs_t wp, int argc, char ** argv);

static int get_charset_asp(int eid, webs_t wp, int argc, char ** argv);
static int encoding_line_asp(int eid, webs_t wp, int argc, char ** argv);
static int translate_write_asp(int eid, webs_t wp, int argc, char ** argv);
static int translate_write_quot_asp(int eid, webs_t wp, int argc, char ** argv);
static int translate_asp(int eid, webs_t wp, int argc, char ** argv);
static int translate_noop_asp(int eid, webs_t wp, int argc, char ** argv);
static int writef_asp(int eid, webs_t wp, int argc, char ** argv);
static int writef_esc_asp(int eid, webs_t wp, int argc, char ** argv);

static int mysnprintf(char *dst, size_t size, const char * format, int argc, char ** argv);

int
translator_init(void) {
    /* register ASPs */

    // return the available languages
    websAspDefine("noLanguages", no_languages_asp);
    websAspDefine("languageShort", language_short_asp);
    websAspDefine("languageFull", language_full_asp);
    websAspDefine("getLanguageShort", translate_get_language_short);
    websAspDefine("getLanguageFull", translate_get_language_full);
    
    // translate sentences
    websAspDefine("encodingLine", encoding_line_asp);
    websAspDefine("getCharset", get_charset_asp);
    websAspDefine("_", translate_write_asp);
    websAspDefine("Q_", translate_write_quot_asp);
    websAspDefine("T_", translate_asp);
    websAspDefine("N_", translate_noop_asp);
    websAspDefine("writef", writef_asp);
    websAspDefine("writefEsc", writef_esc_asp);
    
    return 0;
}

// return the number of available languages
static int no_languages_asp(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    if (argc == 1) {
        char buffer[128];
        snprintf(buffer, sizeof(buffer), "%d",
            atoi(argv[0]) ? pp_intl_get_no_translations() :
                            pp_intl_get_no_possible_languages());
        ejSetResult(eid, buffer);
    }
    return 0;
}

// returns the short name of the language number in the parameter
static int language_short_asp(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    if(argc == 2) {
    	ejSetResult(eid,
    	    atoi(argv[0]) ? pp_intl_get_translation_shortname(atoi(argv[1])) :
    	                    pp_intl_get_possible_language_shortname(atoi(argv[1])));
    }
    return 0;
}

// returns the full name of the language number in the parameter
static int language_full_asp(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    if(argc == 2) {
        const char *full = 
    	    atoi(argv[0]) ? pp_intl_get_translation_fullname(atoi(argv[1])) :
    	                    pp_intl_get_possible_language_fullname(atoi(argv[1]));
    	if (full == NULL) return 0;

        const char *encoding, *utf8name;
        if ((encoding = pp_intl_get_charset()) == NULL ||
            strcasecmp(encoding, "utf-8") != 0 ||
            (utf8name = atoi(argv[0]) ? pp_intl_get_translation_utf8name(atoi(argv[1])) :
                                        pp_intl_get_possible_language_utf8name(atoi(argv[1]))) == NULL ||
            utf8name[0] == '\0') {
            
            ejSetResult(eid, full);
        } else {
            char buf[strlen(full) + strlen(utf8name) + 5];
            snprintf(buf, sizeof(buf), "%s / %s", full, utf8name);
            ejSetResult(eid, buf);
        }
    }
    return 0;
}

// returns the short name of the currently selected language
static int translate_get_language_short(int eid, webs_t wp UNUSED, int argc UNUSED, char ** argv UNUSED) {
    ejSetResult(eid, pp_intl_get_language_shortname());
    return 0;
}

// returns the short name of the currently selected language
static int translate_get_language_full(int eid, webs_t wp UNUSED, int argc UNUSED, char ** argv UNUSED) {
    ejSetResult(eid, pp_intl_get_language_fullname());
    return 0;
}

// get the character set of the currently selected language
static int get_charset_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char ** argv UNUSED) {
    ejSetResult(eid, pp_intl_get_charset());
    return 0;
}

// write the html header line with the character encoding information
static int encoding_line_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char ** argv UNUSED) {
    char line[513] = "";
    const char * encoding = pp_intl_get_charset();

    if(encoding) {
    	snprintf(line, sizeof(line) - 1,
    	    "<meta http-equiv=\"content-type\" content=\"text/html; charset=%s\">", encoding);
    }
    else {
    	snprintf(line, sizeof(line) - 1,
    	    "<meta http-equiv=\"content-type\" content=\"text/html\">");
    }

    if (websWriteBlock(wp, line, strlen(line)) < 0) return -1;
    
    return 0;
}

// translate the given string
static int translate_asp(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    if(argc >= 1) {
    	ejSetResult(eid, _(argv[0]));
    }
    return 0;
}

// translates the given string and prints it
static int translate_write_asp(int eid UNUSED, webs_t wp, int argc, char ** argv)
{
    int i;
    for (i = 0; i < argc; ) {
	const char * s = _(argv[i]);
	if (websWriteBlock(wp, s, strlen(s)) < 0) return -1;
	if (++i < argc && websWriteBlock(wp, " ", 1) < 0) return -1;
    }
    return 0;
}

// translates the given string and prints it in quotation marks
static int
translate_write_quot_asp(int eid UNUSED, webs_t wp, int argc, char ** argv)
{
    char * s_alloced = NULL;
    int i, r = 0;

    for (i = 0; i < argc; ) {
	const char * s = _(argv[i]);
	if (argc == 1) {
	    asprintf(&s_alloced, "%c%s%c", '"', s, '"');
	    s = s_alloced;
	} else if (i == 1) {
	    asprintf(&s_alloced, "%c%s", '"', s);
	    s = s_alloced;
	} else if (i == (argc - 1)) {
	    asprintf(&s_alloced, "%s%c", s, '"');
	    s = s_alloced;
	}
	if ((r = websWriteBlock(wp, s, strlen(s))) < 0) goto bail;
	if (++i < argc && (r = websWriteBlock(wp, " ", 1)) < 0) goto bail;
	free(s_alloced);
	s_alloced = NULL;
    }

 bail:
    free(s_alloced);
    return r < 0 ? -1 : 0;
}

// doesn't translate the given string
static int translate_noop_asp(int eid, webs_t wp UNUSED, int argc, char ** argv) {
    if(argc >= 1) {
    	ejSetResult(eid, argv[0]);
    }
    return 0;
}

// writes a formatted string
static int
writef_internal(int eid UNUSED, webs_t wp, int argc, char ** argv, int do_esc)
{
    int len = 0;
    int i, r = 0;
    char * str = NULL;
    
    if (argc <= 0) return 0;
    
    for (i = 0; i < argc; i++) {
    	len += strlen(argv[i]);
    }
    
    str = malloc(len + 1);
    if (!str) return -1;
    
    mysnprintf(str, len, argv[0], argc - 1, &argv[1]);
    
    if (do_esc) {
	char * esc_html = escape_html(str);
	free(str);
	str = esc_html;
    }

    r = websWriteBlock(wp, str, strlen(str));
    
    free(str);

    return r < 0 ? -1 : 0;
}

static int writef_asp(int eid, webs_t wp, int argc, char ** argv) {
    return writef_internal(eid, wp, argc, argv, 0);
}

static int writef_esc_asp(int eid, webs_t wp, int argc, char ** argv) {
    return writef_internal(eid, wp, argc, argv, 1);
}



// a "special" version of snprintf
// it takes an char*-Array instead of an argument list
// can only handle strings (may be marked with
// %s, %d, %f or whatever), but this is enough
// for the webs
static int
mysnprintf(char *dst, size_t size, const char * format, int argc, char ** argv) {
	size_t count = 0;
	int r = 0;
	const char *s = format;

	do {
		while (*format && (*format != '%')) {
			++format;
		}

		if (format-s) {		/* output any literal text in format string */
			int no = min((size_t)(format - s), size - count);
			strncpy(&dst[count], s, no);
			count += no;
			s += no;
		}

		if(count >= size) {
			break;
		}

		if (!*format) {			/* we're done */
			break;
		}

		if (format[1] == '%') {
			format += 2;
		}
		else {
			/* increase pointers */
			format += 2;
			s = format;

			/* copy the argument */
			if(r < argc && argv[r]) {
				int no = min(strlen(argv[r]), size - count);
				strncpy(&dst[count], argv[r], no);
				count += no;
				r++;
			}
		}

	} while(1);

	if(count < size) {
		dst[count] = '\0';
	}

	return count;
}
