/*
 * uemf.h -- GoAhead Micro Embedded Management Framework Header
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 */

#ifndef _UEMF_H
#define _UEMF_H 1

/******************************* Description *********************************/

/* 
 * GoAhead Web Server header. This defines the Web public APIs
 */

/****************************** Per O/S Includes *****************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>

/********************************* Includes **********************************/

#include <ctype.h>
#include <stdarg.h>
#include <string.h>
#include <pp/base.h>

/********************************* Unicode ***********************************/
/* 
 * Constants and limits. Also FNAMESIZE and PATHSIZE are currently defined 
 * in param.h to be 128 and 512
 */
#define VALUE_MAX_STRING	(4096 - 48)
#define SYM_MAX			(512)
#define XML_MAX			4096	/* Maximum size for tags/tokens */
#define BUF_MAX			4096	/* General sanity check for bufs */

/*
 * "Boolean" constants
 */

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

/*
 * GoAhead Copyright.
 */
#define GOAHEAD_COPYRIGHT "Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved."

#ifndef gopen
#define gmkdir(s)	mkdir(s,0755)
#endif /* ! gopen */

#define gfindfirst	_findfirst
#define gfinddata_t	_finddata_t
#define gfindnext	_findnext
#define gfindclose	_findclose

/********************************* Defines ***********************************/

#ifndef FNAMESIZE
#define FNAMESIZE			254	/* Max length of file names */
#endif /* FNAMESIZE */

#define E_MAX_ERROR			4096
#define URL_MAX				4096

#define a_assert(C)			assert(C)

/*****************************************************************************/
/*                                VALUE                                      */
/*****************************************************************************/
/*
 * These values are not prefixed so as to aid code readability
 */

typedef enum {
    undefined		= 0,
    integer		= 1,
    string 		= 2,
} vtype_t;

#ifndef __NO_PACK
#pragma pack(2)
#endif /* _NO_PACK */

typedef struct {

    union {
	long	integer;
	char	*string;
	const char *string_const;
    } value;

    vtype_t			type		: 16;
    unsigned int	valid		: 8;
    unsigned int	allocated	: 8;		/* String was balloced */
} value_t;

#ifndef __NO_PACK
#pragma pack()
#endif /* __NO_PACK */

/*
 * Allocation flags 
 */
#define VALUE_ALLOCATE		0x1

/*****************************************************************************/
/*
 * A ring queue allows maximum utilization of memory for data storage and is
 * ideal for input/output buffering. This module provides a highly effecient
 * implementation and a vehicle for dynamic strings.
 *
 * WARNING:  This is a public implementation and callers have full access to
 * the queue structure and pointers.  Change this module very carefully.
 *
 * This module follows the open/close model.
 *
 * Operation of a ringq where rq is a pointer to a ringq :
 *
 *   rq->buflen contains the size of the buffer.
 *   rq->buf will point to the start of the buffer.
 *   rq->servp will point to the first (un-consumed) data byte.
 *   rq->endp will point to the next free location to which new data is added
 *   rq->endbuf will point to one past the end of the buffer.
 *
 * Eg. If the ringq contains the data "abcdef", it might look like :
 *
 * +-------------------------------------------------------------------+
 * |   |   |   |   |   |   |   | a | b | c | d | e | f |   |   |   |   |
 * +-------------------------------------------------------------------+
 *   ^                           ^                       ^               ^
 *   |                           |                       |               |
 * rq->buf                    rq->servp               rq->endp      rq->enduf
 *     
 * The queue is empty when servp == endp.  This means that the queue will hold
 * at most rq->buflen -1 bytes.  It is the fillers responsibility to ensure
 * the ringq is never filled such that servp == endp.
 *
 * It is the fillers responsibility to "wrap" the endp back to point to
 * rq->buf when the pointer steps past the end. Correspondingly it is the
 * consumers responsibility to "wrap" the servp when it steps to rq->endbuf.
 * The ringqPutc and ringqGetc routines will do this automatically.
 */

/*
 * Ring queue buffer structure
 */
typedef struct {
    unsigned char	*buf;			/* Holding buffer for data */
    unsigned char	*servp;			/* Pointer to start of data */
    unsigned char	*endp;			/* Pointer to end of data */
    unsigned char	*endbuf;		/* Pointer to end of buffer */
    int				buflen;		/* Length of ring queue */
    int				maxsize;	/* Maximum size */
    int				increment;	/* Growth increment */
} ringq_t;


#define B_SHIFT		4			/* Convert size to class */

/*
 * The symbol table record for each symbol entry
 */

typedef struct sym_t {
    struct sym_t	*forw;			/* Pointer to next hash list */
    value_t		name;			/* Name of symbol */
    value_t		content;		/* Value of symbol */
    int			arg;			/* Parameter value */
} sym_t;

typedef int sym_fd_t;				/* Returned by symOpen */

/*
 *	Script engines
 */
#define EMF_SCRIPT_JSCRIPT	0		/* javascript */
#define EMF_SCRIPT_TCL	 	1		/* tcl */
#define EMF_SCRIPT_EJSCRIPT 	2		/* Ejscript */
#define EMF_SCRIPT_MAX	 	3

#define	MAXINT		INT_MAX
#ifndef BITSPERBYTE
# define BITSPERBYTE	8
#endif
#define BITS(type)	(BITSPERBYTE * (int) sizeof(type))
#define	STRSPACE	"\t \n\r\t"

#ifndef max
#define max(a,b)  (((a) > (b)) ? (a) : (b))
#endif /* max */

#ifndef min
#define min(a,b)  (((a) < (b)) ? (a) : (b))
#endif /* min */

/******************************** Prototypes *********************************/
/*
 * Balloc module
 *
 */

void 	 bclose(void);
int	 bopen(void *buf, int bufsize, int flags);

/*
 *Define NO_BALLOC to turn off our balloc module altogether
 *	#define NO_BALLOC 1
 */

#ifdef NO_BALLOC
#define balloc(B_ARGS, num) malloc(num)
#define bfree(B_ARGS, p) free(p)
#define bfreeSafe(B_ARGS, p) \
	if (p) { free(p); } else
#define brealloc(B_ARGS, p, num) realloc(p, num)
char *bstrdupNoBalloc(char *s);
char *bstrdupANoBalloc(char *s);
#define bstrdup(B_ARGS, s) bstrdupNoBalloc(s)
#define bstrdupA(B_ARGS, s) bstrdupANoBalloc(s)
#define gstrdup(B_ARGS, s) bstrdupNoBalloc(s)

#else /* BALLOC */

#ifndef B_STATS
#define balloc(B_ARGS, num) balloc(num)
#define bfree(B_ARGS, p) bfree(p)
#define bfreeSafe(B_ARGS, p) bfreeSafe(p)
#define brealloc(B_ARGS, p, size) brealloc(p, size)
#define bstrdup(B_ARGS, p) bstrdup(p)

#define bstrdupA bstrdup

#endif /* B_STATS */

void	*balloc(B_ARGS_DEC, int size);
void	bfree(B_ARGS_DEC, void *mp);
void	bfreeSafe(B_ARGS_DEC, void *mp);
void	*brealloc(B_ARGS_DEC, void *buf, int newsize);
char	*bstrdup(B_ARGS_DEC, const char *s);

#define bstrdupA bstrdup

#endif /* BALLOC */

void bstats(int handle, void (*writefn)(int handle, char *fmt, ...));

/*
 * Flags. The integrity value is used as an arbitrary value to fill the flags.
 */
#define B_USE_MALLOC		0x1	/* Okay to use malloc if required */
#define B_USER_BUF		0x2	/* User supplied buffer for mem */


#ifdef B_STATS
#define 	hAlloc(x) 				HALLOC(B_L, x)
#define		hAllocEntry(x, y, z)	HALLOCENTRY(B_L, x, y, z)
int	HALLOC(B_ARGS_DEC, void ***map);
int 	HALLOCENTRY(B_ARGS_DEC, void ***list, int *max, int size);
#else
int	hAlloc(void ***map);
int 	hAllocEntry(void ***list, int *max, int size);
#endif /* B_STATS */

int	hFree(void ***map, int handle);

int	ringqOpen(ringq_t *rq, int increment, int maxsize);
void 	ringqClose(ringq_t *rq);
int 	ringqLen(ringq_t *rq);

int 	ringqPutc(ringq_t *rq, char c);
int	ringqInsertc(ringq_t *rq, char c);
int	ringqPutStr(ringq_t *rq, const char *str);
int 	ringqGetc(ringq_t *rq);

int	fmtValloc(char **s, int n, const char *fmt, va_list arg);
int	fmtAlloc(char **s, int n, const char *fmt, ...);

#define ringqPutcA ringqPutc
#define ringqInsertcA ringqInsertc
#define ringqPutStrA ringqPutStr
#define ringqGetcA ringqGetc

int 	ringqPutBlk(ringq_t *rq, const char *buf, int len);
int 	ringqPutBlkMax(ringq_t *rq);
void 	ringqPutBlkAdj(ringq_t *rq, int size);
int 	ringqGetBlk(ringq_t *rq, char *buf, int len);
int 	ringqGetBlkMax(ringq_t *rq);
void 	ringqGetBlkAdj(ringq_t *rq, int size);
void 	ringqFlush(ringq_t *rq);
void 	ringqAddNull(ringq_t *rq);

int	scriptSetVar(int engine, char *var, char *value);
int	scriptEval(int engine, char *cmd, char **rslt, int chan);

char	*stritoa(int n, char *s, int width);

sym_fd_t	symOpen(int hash_size);
void	symClose(sym_fd_t sd);
sym_t	*symLookup(sym_fd_t sd, const char *name);
sym_t	*symEnter(sym_fd_t sd, const char *name, value_t v, int arg);
int	symDelete(sym_fd_t sd, const char *name);
void 	symWalk(sym_fd_t sd, void (*fn)(sym_t *symp));
sym_t	*symFirst(sym_fd_t sd);
sym_t	*symNext(sym_fd_t sd);
int	symSubOpen(void);
void 	symSubClose(void);

value_t 	valueInteger(long value);
value_t	valueString(const char *value, int flags);
value_t	valueErrmsg(char *value);
void 	valueFree(value_t *v);
int	vxchdir(char *dirname);

time_t	timeMsec(void);

#endif /* _UEMF_H */

/******************************************************************************/
