/*
 * url.c -- Parse URLs
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 */

/******************************* Description *********************************/

/*
 * This module parses URLs into their components.
 */

/******************************** Includes ***********************************/

#include "wsIntrn.h"

/******************************** Statics ************************************/

/*
 * htmExt is declared in this way to avoid a Linux and Solaris segmentation
 * fault when a constant string is passed to strlower which could change its
 * argument.
 */

char htmExt[] = ".htm";

/********************************** Code *************************************/

/*
 * Return the mime type for the given URL given a URL.
 * The caller supplies the buffer to hold the result.
 * charCnt is the number of characters the buffer will hold, ascii or UNICODE.
 */

char *
websUrlType(const char *url, char *buf, int charCnt)
{
    sym_t	*sp;
    char	*ext, *parsebuf;

    a_assert(url && *url);
    a_assert(buf && charCnt > 0);

    if (url == NULL || *url == '\0') {
	strcpy(buf, "text/plain");
	return buf;
    }
    if (websUrlParse(url, &parsebuf, NULL, NULL, NULL, NULL, NULL, NULL, &ext) < 0) {
	strcpy(buf, "text/plain");
	return buf;
    }
    pp_strtolower(ext);

    /*
     *	Lookup the mime type symbol table to find the relevant content type
     */
    if ((sp = symLookup(websMime, ext)) != NULL) {
	strncpy(buf, sp->content.value.string, charCnt);
    } else {
	strcpy(buf, "text/plain");
    }
    bfree(B_L, parsebuf);
    return buf;
}

/*
 * Parse the URL. A buffer is allocated to store the parsed URL in *pbuf.
 * This must be freed by the caller. NOTE: tag is not yet fully supported.
 */

int websUrlParse(const char * url, char ** pbuf, const char ** phost,
		 const char ** ppath, const char ** pport, const char ** pquery,
		 const char ** pproto, const char ** ptag, char **pext)
{
    const char *tok, *host, *path, *proto, *tag, *query;
    char *cp, *port, *ext;
    char *last_delim, *hostbuf, *portbuf, *urlbuf;
    int    c, len, ulen;

    a_assert(url);
    a_assert(pbuf);

    ulen = strlen(url);
    /*
     *	We allocate enough to store separate hostname and port number fields.
     *	As there are 3 strings in the one buffer, we need room for 3 null chars.
     *	We allocate MAX_PORT_LEN char's for the port number.
     */
    len = ulen * 2 + MAX_PORT_LEN + 3;
    if ((urlbuf = balloc(B_L, len * sizeof(char))) == NULL) {
	return -1;
    }
    portbuf = &urlbuf[len - MAX_PORT_LEN - 1];
    hostbuf = &urlbuf[ulen+1];
    strcpy(urlbuf, url);

    /*
     * Convert the current listen port to a string. We use this if the URL
     * has no explicit port setting
     */
    /* FIXME */
    stritoa(80, portbuf, MAX_PORT_LEN);
    port = portbuf;
    path = "/";
    proto = "http";
    host = "localhost";
    query = "";
    ext = htmExt;
    tag = "";

    if (strncmp(urlbuf, "http://", 7) == 0) {
	tok = cp = &urlbuf[7];
	urlbuf[4] = '\0';
	proto = urlbuf;
	host = tok;
	for (; *cp; cp++) {
	    if (*cp == '/') {
		break;
	    }
	    if (*cp == ':') {
		*cp++ = '\0';
		port = cp;
		tok = cp;
	    }
	}
	if ((cp = strchr(tok, '/')) != NULL) {
	    /*
	     * If a full URL is supplied, we need to copy the host and port 
	     * portions into static buffers.
	     */
	    c = *cp;
	    *cp = '\0';
	    strncpy(hostbuf, host, ulen);
	    strncpy(portbuf, port, MAX_PORT_LEN);
	    *cp = c;
	    host = hostbuf;
	    port = portbuf;
	    path = cp;
	    tok = cp;
	}

    } else {
	path = urlbuf;
	tok = urlbuf;
    }

    /*
     *	Parse the query string
     */
    if ((cp = strchr(tok, '?')) != NULL) {
	*cp++ = '\0';
	query = cp;
	path = tok;
	tok = query;
    } 

    /*
     *	Parse the fragment identifier
     */
    if ((cp = strchr(tok, '#')) != NULL) {
	*cp++ = '\0';
	if (*query == 0) {
	    path = tok;
	}
    }

    /*
     *	Only do the following if asked for the extension
     */
    if (pext) {
	if ((cp = strrchr(path, '.')) != NULL) {
	    if ((last_delim = strrchr(path, '/')) != NULL) {
		if (last_delim > cp) {
		    ext = htmExt;
		} else {
		    ext = cp;
		}
	    } else {
		ext = cp;
	    }
	} else {
	    if (path[strlen(path) - 1] == '/') {
		ext = htmExt;
	    }
	}
    }

    /*
     *	Pass back the fields requested (if not NULL)
     */
    if (phost)
	*phost = host;
    if (ppath)
	*ppath = path;
    if (pport)
	*pport = port;
    if (pproto)
	*pproto = proto;
    if (pquery)
	*pquery = query;
    if (ptag)
	*ptag = tag;
    if (pext)
	*pext = ext;
    *pbuf = urlbuf;
    return 0;
}
