/*
 * webs.c -- GoAhead Embedded HTTP webs server
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 */

/******************************* Description *********************************/

/*
 * This module implements an embedded HTTP/1.1 web server. It supports
 * loadable URL handlers that define the nature of URL processing performed.
 */

/******************************** Includes ***********************************/

#if defined(PP_FEAT_POWERSWITCH)
#include <pp/powerswitch.h>
#include <pp/intl.h>
#endif /* PP_FEAT_POWERSWITCH */
#include "wsIntrn.h"
#include "eric_forms.h"
#include "eric_util.h"

/******************************* Global Data *********************************/

sym_fd_t websMime; /* Set of mime types */
pp_hash_t * post_data_callbacks = NULL;

/********************************** Locals ***********************************/

/*
 *	Standard HTTP error codes
 */

websErrorType websErrors[] = {
    { 200, "Data follows" },
    { 204, "No Content" },
    { 301, "Redirect" },
    { 302, "Redirect" },
    { 304, "User local copy" },
    { 400, "Page not found" },
    { 401, "Unauthorized" },
    { 403, "Forbidden" },
    { 404, "Site or Page not Found" },
    { 405, "Access Denied" },
    { 500, "Web Error" },
    { 501, "Not Implemented" },
    { 503, "Site Temporarily Unavailable. Try again." },
    { 0, NULL }
};

/*************************** Forward Declarations ****************************/

static int	websDefaultPostDataCB(webs_t wp, char * name, char * filename,
				      void * data, size_t data_offset, size_t data_len,
				      int more_data);
static const char *websErrorMsg(int code);
static u_short	get_port_http(void);
static u_short	get_port_https(void);
static void	websReadEvent(webs_t wp);
static int 	websGetInput(webs_t wp, char **ptext, int *nbytes);
static int 	websParseFirst(webs_t wp, char *text);
static void 	websParseRequest(webs_t wp);

#define D(fmt, args...)
#define noD(fmt, args...) { \
   char __xfmt[255]; \
    snprintf(__xfmt, 255, "%s: %s\n", ___F, fmt); \
    printf(__xfmt, ##args); \
    }

/*
 * Open the GoAhead WebServer
 */

int
websOpen(void)
{
    websMimeType * mt;

    /*
     * Create a mime type lookup table for quickly determining the
     * content type
     */
    websMime = symOpen(WEBS_SYM_INIT * 4);
    a_assert(websMime >= 0);
    for (mt = websMimeList; mt->type; mt++) {
	symEnter(websMime, mt->ext, valueString(mt->type, 0), 0);
    }

    if ((post_data_callbacks = pp_hash_create(100)) == NULL) {
	pp_log("%s(): pp_hash_create() failed\n", ___F);
	goto error;
    }

    /*
     * Open the URL handler module. The caller should create the required
     * URL handlers after calling this function.
     */
    if (websUrlHandlerOpen() < 0) goto error;

    if (PP_FAILED(eric_net_register_proto(PP_NET_PROTOCOL_TYPE_HTTP, "HTTP", 1, get_port_http,
					  eric_net_init_bio_standard, eric_webs_handle_connection))) {
	websUrlHandlerClose();
	goto error;
    }

    if (PP_FAILED(eric_net_register_proto(PP_NET_PROTOCOL_TYPE_HTTPS, "HTTPS", 1, get_port_https,
					  eric_net_init_bio_https, eric_net_demuxer_handle_connection))) {
	websUrlHandlerClose();
	goto error;
    }

    return PP_SUC;

 error:
    eric_net_unregister_proto(PP_NET_PROTOCOL_TYPE_HTTP);
    eric_net_unregister_proto(PP_NET_PROTOCOL_TYPE_HTTPS);
    if (post_data_callbacks) pp_hash_delete(post_data_callbacks);
    symClose(websMime);
    return PP_ERR;
}

/*
 * Close the GoAhead WebServer
 */

void
websClose(void)
{
    /* FIXME: close all BIO's and free webs structures */

    symClose(websMime);
    if (post_data_callbacks) pp_hash_delete(post_data_callbacks);
    websUrlHandlerClose();
}

int
websAddPostDataCallback(const char * name, webs_post_data_cb_t cb)
{
    return pp_hash_set_entry(post_data_callbacks, name, cb, NULL);
}

static u_short
get_port_http(void)
{
    int force;
    u_short port = 0;
    /* check if HTTPS is forced */
    pp_cfg_is_enabled(&force, "security.force_https");
    if (!force) pp_cfg_get_ushort(&port, "network.http_port");
    return port;
}

static u_short
get_port_https(void)
{
    u_short port;
    pp_cfg_get_ushort(&port, "network.https_port");
    return port;
}

void
eric_webs_handle_connection(pp_net_conn_data_t * conn_data)
{
    webs_t wp;

    /* allocate a new webs_t structure for this connection */
    if ((wp = websAlloc()) != NULL) {
	SSL * ssl = NULL;

	wp->bio = conn_data->bio;

	/* determine if we use SSL */
	if (BIO_get_ssl(conn_data->bio, &ssl) && ssl != NULL) {
	    wp->flags |= WEBS_SECURE;
	    websSetVar(wp, "__ssl__", "1");
	} else {
	    websSetVar(wp, "__ssl__", "0");
	}

	/* read / write loop */
	websReadEvent(wp);
    } else {
	pp_log("%s(): websAlloc() failed.", ___F);
	eric_net_close(conn_data->bio, NOWAIT);
    }

    free(conn_data);
}

/*
 * The webs read handler. This is the primary read event loop. It uses a
 * state machine to track progress while parsing the HTTP request. 
 * Note: we never block as the socket is always in non-blocking mode.
 */

static void
websReadEvent(webs_t wp)
{
    char * text;
    int      rc, nbytes, len, done;
    char * buf = NULL;
    void * boundary, * data_start;
    size_t old_buf_len = 0;
    size_t buf_len = 0;
    size_t buf_off = 0;
    size_t data_off = 0;
    char name[64], filename[256];
    size_t data_len;
    char delim;
    webs_post_data_cb_t data_cb = NULL;

    a_assert(wp);

    /*
     * Read as many lines as possible. webs_bio_gets is called to read the
     * header and webs_bio_read_exact is called to read posted data.
     */
    text = NULL;
    for (done = 0; !done; ) {
	if (text) {
	    bfree(B_L, text);
	    text = NULL;
	}

	/*
	 * Get more input into "text". Returns 0, if more data is
	 * needed to continue, -1 if finished with the request, or 1
	 * if all required data is available for current state.
	 */
	while ((rc = websGetInput(wp, &text, &nbytes)) == 0);

	/*
	 * websGetInput returns -1 if it finishes with the request
	 */
	if (rc < 0) {
	    /*
	     * if there is still a data callback handler waiting for data
	     * we call him with data set to NULL.
	     */
	    if (data_cb) {
		data_cb(wp, name, filename, NULL, data_off, 0, 0);
	    }
	    break;
	}

	/*
	 * This is the state machine for the web server. 
	 */
	switch(wp->state) {
	  case WEBS_BEGIN:
	      /*
	       * Parse the first line of the Http header
	       */
	      if (websParseFirst(wp, text) < 0) {
		  done++;
		  break;
	      }
	      wp->state = WEBS_HEADER;
	      break;
		
	  case WEBS_HEADER:
	      /*
	       * Store more of the HTTP header. As we are doing line
	       * reads, we need to separate the lines with '\n'
	       */
	      if (ringqLen(&wp->header) > 0) {
		  ringqPutStr(&wp->header, "\n");
	      }
	      ringqPutStr(&wp->header, text);
	      break;

	  case WEBS_POST_CLEN:
	      /*
	       * POST request with content specified by a content length.
	       */
	      if (wp->boundary_len > 0) { /* multipart data */
		  /*
		   * Append data to 'buf' and add a '\0' character to its end.
		   * This prevents string operations running amok on binary
		   * data.
		   */
		  buf_len += nbytes;
		  if (buf == NULL || buf_len > old_buf_len) {
		      buf = brealloc(B_L, buf, buf_len + 1);
		      old_buf_len = buf_len;
		  }
		  buf[buf_len] = '\0';
		  memcpy(buf + (buf_len - nbytes), text, nbytes);
		  /*
		   * Look for the boundary string in the not yet scanned
		   * part of 'buf'.
		   */
		  while ((boundary = pp_mem_search(buf + buf_off,
						   buf_len - buf_off,
						   wp->boundary,
						   wp->boundary_len))) {
		      /*
		       * Boundary string found. Data before the boundary
		       * string is delivered to the current data handler.
		       */
		      data_len = (size_t)boundary - (size_t)buf - buf_off;
		      if (data_cb) {
			  data_cb(wp, name, filename, buf + buf_off, data_off,
				  data_len, 0);
			  data_cb = NULL;
		      }
		      data_off += data_len;
		      buf_off += data_len + wp->boundary_len;
		      /*
		       * Try to get more data if this is the final
		       * boundary string or if we dont see the end
		       * of the part header.
		       */
		      if (buf_len - buf_off < 4 ||
			  !memcmp(buf + buf_off, "--\r\n", 4) ||
			  !(data_start = pp_mem_search(buf + buf_off,
						       buf_len - buf_off,
						       "\r\n\r\n", 4))) {
			  /* safe the boundary */
			  buf_off -= wp->boundary_len;
			  break;
		      }
		      /*
		       * Parse the part header and determine the
		       * name of the form variable. Additionally we
		       * determine the data callback routine.
		       */
		      delim = ';';
		      if ((rc = sscanf(buf + buf_off + 2,
				       "Content-Disposition: form-data; "
				       "name=\"%63[^\"]\"; "
				       "filename=\"%255[^\"]\"\r\n",
				       name, filename)) == 2 || (*filename = '\0') == 1 ||
			  (rc = sscanf(buf + buf_off + 2,
				       "Content-Disposition: form-data; "
				       "name=\"%63[^\"]\"%c",
				       name, &delim)) == 2)
			   {
			  if (!wp->auth_ok) {
			      data_cb = NULL;
			  } else if (rc == 2 && delim == ';') {
			      if (*filename) {
				  data_cb = (webs_post_data_cb_t)pp_hash_get_entry(post_data_callbacks, name);
			      } else {
				  /* this happens if a file with an empty name has been submitted */
				  wp->invalid_file = 1;
			      }
			  } else {
			      filename[0] = '\0';
			      data_cb = websDefaultPostDataCB;
			  }
			  data_off = 0;
		      } else { /* Parse error */
			  /* FIXME: avoid 'buf' getting to big! */
			  name[0] = filename[0] = '\0';
			  buf_off -= wp->boundary_len;
			  if ( rc == 1 ) {
			      wp->upload_failed = 1;
			  }
			  break;
		      }
		      buf_off = (size_t)data_start + 4 - (size_t)buf;
		  }
		  /*
		   * If boundary string is not found deliver already scanned
		   * data to the current data handler.
		   */
		  if (!boundary &&
		      (buf_len - buf_off) >= wp->boundary_len) {
		      size_t new_buf_off = buf_len - wp->boundary_len + 1;
		      if (data_cb) {
			  if (data_cb(wp, name, filename, buf + buf_off, data_off,
				      new_buf_off - buf_off, 1) == -1) {
			      data_cb = NULL;
			  }
		      }
		      data_off += new_buf_off - buf_off;
		      buf_off = new_buf_off;
		  }
		  /*
		   * shift data from end of begin of 'buf'
		   */
		  buf_len -= buf_off;
		  memmove(buf, buf + buf_off, buf_len);
		  buf_off = 0;
	      } else if (wp->query) {
		   wp->query_len = strlen(wp->query);
		   if (!(wp->flags & WEBS_POST_DATA)) {
		      /*
		       * Special case where the POST request also had query
		       * data specified in the URL, ie. url?query_data. In
		       * this case the URL query data is separated by a '&'
		       * from the posted query data. This is only called the
		       * first time, because the second time WEBS_POSTDATA is
		       * set
		       */

		      wp->query = brealloc(B_L, wp->query, strlen(wp->query) + nbytes + 2);
		      wp->query[wp->query_len++] = '&';
		      wp->post_start = wp->query_len;
		      wp->query_len += nbytes;
		      memcpy(&wp->query[wp->query_len - nbytes], text, nbytes);
		      wp->query[wp->query_len] = '\0';
		  } else {
		      /*
		       * The existing query data came from the POST request
		       * so just
		       * append it.
		       */
		      wp->query_len += nbytes;
		      wp->query = brealloc(B_L, wp->query, wp->query_len + 1);
		      memcpy(&wp->query[wp->query_len - nbytes], text, nbytes);
		      wp->query[wp->query_len] = '\0';
		  }
	      } else {
		  wp->post_start = 0;
		  wp->query_len = nbytes;
		  wp->query = balloc(B_L, wp->query_len + 1);
		  memcpy(wp->query, text, wp->query_len);
		  wp->query[wp->query_len] = '\0';
	      }

	      /*
	       * Calculate how much more post data is to be read.
	       */
	      wp->flags |= WEBS_POST_DATA;
	      wp->clen -= nbytes;
	      if (wp->clen > 0) {
		  if (nbytes > 0) {
		      break;
		  }
		  done++;
		  break;
	      }

	      /*
	       * No more data so process the request
	       */
	      websUrlHandlerRequest(wp);
	      done++;
	      break;

	  case WEBS_POST:
	      /*
	       * POST without content-length specification
	       */

	      if (wp->query && *wp->query && !(wp->flags & WEBS_POST_DATA)) {
		  len = strlen(wp->query);
		  wp->query = brealloc(B_L, wp->query, (len + strlen(text) +
							2) * sizeof(char));
		  if (wp->query) {
		      wp->query[len++] = '&';
		      wp->post_start = len;
		      strcpy(&wp->query[len], text);
		  } else {
		      wp->post_start = 0;
		  }

	      } else {
		  wp->query = bstrdup(B_L, text);
		  wp->post_start = 0;
	      }
	      wp->flags |= WEBS_POST_DATA;
	      done++;
	      break;

	  default:
	      websError(wp, 404, "Bad state");
	      done++;
	      break;
	}
    }

    bfreeSafe(B_L, buf);
    bfreeSafe(B_L, text);
}

/*
 * Get input from the browser. Return TRUE (!0) if the request has been 
 * handled. Return -1 on errors or if the request has been processed, 
 * 1 if input read, and 0 to instruct the caller to call again for more input.
 *
 * Note: socketRead will Return the number of bytes read if successful. This
 * may be less than the requested "bufsize" and may be zero. It returns -1 for
 * errors. It returns 0 for EOF. Otherwise it returns the number of bytes 
 * read. Since this may be zero, callers should use socketEof() to 
 * distinguish between this and EOF.
 */

static int
websGetInput(webs_t wp, char **ptext, int *pnbytes) 
{
    char	*text;
    char	buf[WEBS_SOCKET_BUFSIZ+1];
    int		nbytes, len, clen;

    a_assert(wp);
    a_assert(ptext);
    a_assert(pnbytes);

    *ptext = text = NULL;
    *pnbytes = 0;

    /*
     * If this request is a POST with a content length, we know the number
     * of bytes to read so we use socketRead().
     */
    if (wp->state == WEBS_POST_CLEN) {
	len = (wp->clen > WEBS_SOCKET_BUFSIZ) ? WEBS_SOCKET_BUFSIZ : wp->clen;
    } else {
	len = 0;
    }

    if (len > 0) {
	if (webs_bio_read_exact(wp, buf, len) == -1) {
	    websDone(wp, 0);
	    return -1;
	} else { /* Valid data */
	    buf[len] = '\0';
	    if ((text = balloc(B_L, len + 1)) != NULL) {
		memcpy(text, buf, len);
		text[len] = '\0';
	    } else {
		websError(wp, 503, _("Insufficient memory"));
		return -1;
	    }
	}
	nbytes = len;
    } else {
	nbytes = webs_bio_gets(wp, &text);

	if (nbytes < 0) {
	    int eof;
	    /*
	     * Error, EOF or incomplete
	     */
	    eof = webs_bio_eof(wp);

	    if (eof) {
		/*
		 * If this is a post request without content
		 * length, process the request as we now have
		 * all the data. Otherwise just close the
		 * connection.
		 */
		if (wp->state == WEBS_POST) {
		    websUrlHandlerRequest(wp);
		} else {
		    websDone(wp, 0);
		}	
		/*
		 * If state is WEBS_HEADER and the ringq is empty, then this is a
		 * simple request with no additional header fields to process and
		 * no empty line terminator.
		 */
	    } else if (wp->state == WEBS_HEADER && ringqLen(&wp->header) <= 0) {
		websParseRequest(wp);
		websUrlHandlerRequest(wp);
	    }
	    return -1;

	} else if (nbytes == 0) {
	    if (wp->state == WEBS_HEADER) {
		/*
		 * Valid empty line, now finished with header
		 */
		websParseRequest(wp);
		if (wp->flags & WEBS_POST_REQUEST) {
		    if (wp->flags & WEBS_CLEN) {
			wp->state = WEBS_POST_CLEN;
			clen = wp->clen;
		    } else {
			wp->state = WEBS_POST;
			clen = 1;
		    }
		    if (clen > 0) {
			/*
			 * Return 0 to get more data.
			 */
			return 0;
		    }
		    return 1;
		}
		/*
		 * We've read the header so go and handle the request
		 */
		websUrlHandlerRequest(wp);
	    }
	    return -1;
	}
    }
    a_assert(text);
    a_assert(nbytes > 0);
    *ptext = text;
    *pnbytes = nbytes;
    return 1;
}

/*
 * Parse the first line of a HTTP request
 */

static int
websParseFirst(webs_t wp, char *text)
{
    const char *proto, *url, *host, *query, *path, *port;
    char *op, *protoVer, *ext;
    char *buf;
    char * tok_ptr;

    a_assert(wp);
    a_assert(text && *text);

    /*
     *	Determine the request type: GET, HEAD or POST
     */
    op = strtok_r(text, " \t", &tok_ptr);
    if (op == NULL || *op == '\0') {
	websError(wp, 400, _("Bad HTTP request"));
	return -1;
    }
    if (strcmp(op, "GET") != 0) {

	if (strcmp(op, "POST") == 0) {
	    wp->flags |= WEBS_POST_REQUEST;
	} else if (strcmp(op, "HEAD") == 0) {
	    wp->flags |= WEBS_HEAD_REQUEST;
	} else {
	    websError(wp, 400, _("Bad request type"));
	    return -1;
	}
    }

    url = strtok_r(NULL, " \t\n", &tok_ptr);
    if (url == NULL || *url == '\0') {
	websError(wp, 400, _("Bad HTTP request"));
	return -1;
    }
    protoVer = strtok_r(NULL, " \t\n", &tok_ptr);

    /*
     * Parse the URL and store all the various URL components. websUrlParse
     * returns an allocated buffer in buf which we must free. We support
     * both proxied and non-proxied requests. Proxied requests will have
     * http://host/ at the start of the URL. Non-proxied will just be
     * local path names.
     */
    host = path = port = proto = query = ext = NULL;
    if (websUrlParse(url, &buf, &host, &path, &port, &query, &proto, 
		     NULL, &ext) < 0) {
	websError(wp, 400, _("Bad URL format"));
	return -1;
    }

    wp->url = bstrdup(B_L, url);

    wp->query = bstrdup(B_L, query);
    wp->host = bstrdup(B_L, host);
    wp->path = bstrdup(B_L, path);
    wp->protocol = bstrdup(B_L, proto);
    wp->protoVersion = bstrdup(B_L, protoVer);

    /* look for last path component and remember that as page
    if ((cp = strrchr(wp->path, '/')) != NULL) {
	wp->page = cp;
    } else {
	wp->page = wp->path;
    }
    */
    
    if (strcmp(ext, ".asp") == 0) {
	wp->flags |= WEBS_ASP;
    }
    bfree(B_L, buf);

    websUrlType(url, wp->type, sizeof(wp->type));

    ringqFlush(&wp->header);
    return 0;
}

/*
 * Parse a full request
 */

#define isgoodchar(s) (gisalnum((s)) || ((s) == '/') || ((s) == '_') || \
						((s) == '.')  || ((s) == '-') )

static void
websParseRequest(webs_t wp)
{
    char	*upperKey, *cp, *browser, *lp, *key, *value;
    char * tok_ptr;
    char emptyValue[1] = "\0";

    a_assert(wp);

    /*
     * set a default value for the user agent size
     * as it is likely that this is not contained
     * -1 means unknown
     */
    wp->uadim.width = -1;
    wp->uadim.height = -1;

    /* 
     *	Parse the header and create the Http header keyword variables
     *	We rewrite the header as we go for non-local requests.  NOTE: this
     * 	modifies the header string directly and tokenizes each line with '\0'.
     */
    browser = NULL;
    for (lp = (char*) wp->header.servp; lp && *lp; ) {
	cp = lp;
	if ((lp = strchr(lp, '\n')) != NULL) {
	    lp++;
	}

	if ((key = strtok_r(cp, ": \t\n", &tok_ptr)) == NULL) {
	    continue;
	}

	if ((value = strtok_r(NULL, "\n", &tok_ptr)) == NULL) {
	    value = emptyValue;
	} else {
	    while (isspace(*value)) value++;
	}
	pp_strtolower(key);

	/*
	 * Create a variable (CGI) for each line in the header
	 */
	fmtAlloc(&upperKey, (strlen(key) + 6), "HTTP_%s", key);
	for (cp = upperKey; *cp; cp++) {
	    if (*cp == '-')
		*cp = '_';
	}
	pp_strtoupper(upperKey);
	websSetVar(wp, upperKey, value);
	bfree(B_L, upperKey);

	/*
	 * Track the requesting agent (browser) type
	 */
	if (strcmp(key, "user-agent") == 0) {
	    wp->userAgent = bstrdup(B_L, value);
#if defined(PP_FEAT_WS_MANAGEMENT)
	    /*
	     * parse user authorization if given
	     */
	} else if (strcmp(key, "authorization") == 0) {
	    wp->wsman_auth = bstrdup(B_L, value);
	    if (value && strncasecmp(value, "basic", strlen("basic")) == 0) {
	    	wp->flags |= WEBS_AUTH_BASIC;
	    } else if (value && strncasecmp(value, "digest", strlen("digest")) == 0) {
	    	wp->flags |= WEBS_AUTH_DIGEST;
	    }
#endif /* PP_FEAT_WS_MANAGEMENT */

	    /*
	     * parse user Agent size if given
	     */
	} else if (strcmp(key, "ua-pixels") == 0) {
	    char* ep;
	    wp->uadim.width = strtol(value, &ep, 10);
	    if (*ep != '\0') wp->uadim.height = pp_strtol_10(++ep, 0, NULL);
	    
	    /*
	     *	Parse the content length
	     */
	} else if (strcmp(key, "content-length") == 0) {
	    wp->flags |= WEBS_CLEN;
	    wp->clen = atoi(value);
	    websSetVar(wp, "CONTENT_LENGTH", value);

	    /*
	     * Parse the content type
	     */
	} else if (strcmp(key, "content-type") == 0) {
	    websSetVar(wp, "CONTENT_TYPE", value);
	    if (sscanf(value, "multipart/form-data; boundary=%253[^\r]\r\n",
		       &wp->boundary[2]) == 1) {
		wp->boundary[0] = wp->boundary[1] = '-';
		wp->boundary_len = strlen(wp->boundary);
	    }

#ifdef WEBS_KEEP_ALIVE_SUPPORT
	} else if (strcmp(key, "connection") == 0) {
	    strlower(value);
	    if (strcmp(value, "keep-alive") == 0) {
		wp->flags |= WEBS_KEEP_ALIVE;
	    }
#endif

	    /*
	     * Store the cookie
	     */
	} else if (strcmp(key, "cookie") == 0) {
	    wp->flags |= WEBS_COOKIE;
	    wp->cookie = bstrdup(B_L, value);
	}
    }

    wp->auth_ok = check_auth(wp) ? 1 : 0;
}

static int
websDefaultPostDataCB(webs_t wp, char * name, char * filename UNUSED,
		      void * data, size_t data_offset UNUSED, size_t data_len, int more_data)
{
    if (data != NULL) {
	if (data_len > 0) {
	    wp->mp_form_data_len += data_len;
	    wp->mp_form_data = brealloc(B_L, wp->mp_form_data, wp->mp_form_data_len + 1);
	    memcpy(wp->mp_form_data + wp->mp_form_data_len - data_len, data, data_len);
	    wp->mp_form_data[wp->mp_form_data_len] = '\0';
	}

	if (more_data) return 0;

	wp->mp_form_data_len -= 2;
	wp->mp_form_data[wp->mp_form_data_len] = '\0';
	{
	    const char * old_val = websGetVar(wp, name, "");
	    size_t old_len = strlen(old_val);
	    size_t new_len = old_len + wp->mp_form_data_len;
	    char * new_val = balloc(B_L, new_len + 1);
	    const char * delim = old_len ? " " : "";
	    snprintf(new_val, new_len + 1, "%s%s%s", old_val, delim, wp->mp_form_data);
	    websSetVar(wp, name, new_val);
	    bfreeSafe(B_L, new_val);
	}
    }

    bfreeSafe(B_L, wp->mp_form_data);
    wp->mp_form_data = NULL;
    wp->mp_form_data_len = 0;
    return 0;
}

/*
 * Set the variable (CGI) environment for this request. Create variables
 * for all standard CGI variables. Also decode the query string and create
 * a variable for each name=value pair.
 */

void
websSetEnv(webs_t wp)
{
    char	*keyword, *value;
    char	emptyValue[1] = "\0";
#if 0
    char	*valCheck, *valNew;
#endif
    char * tok_ptr;

    a_assert(wp);
    /*
     *	Decode and create an environment query variable for each query keyword.
     *	We split into pairs at each '&', then split pairs at the '='.
     *	Note: we rely on wp->decodedQuery preserving the decoded values in the
     *	symbol table.
     */
    wp->decodedQuery = bstrdup(B_L, wp->query);
    keyword = strtok_r(wp->decodedQuery, "&", &tok_ptr);
    while (keyword != NULL) {
	if ((value = strchr(keyword, '=')) != NULL) {
	    *value++ = '\0';
	    websDecodeUrl(keyword, keyword, strlen(keyword));
	    websDecodeUrl(value, value, strlen(value));
	} else {
	    value = emptyValue;
	}

	if (*keyword) {
	    /*
	     * If keyword has already been set, append the new value to what
	     * has been stored.
	     */
	    /*
	     * ATTENTION: changed by thomas:
	     * the last one will win, i.e. POST params will
	     * get a higher prio then GET params
	     */
#if 0
	    if ((valCheck = websGetVar(wp, keyword, NULL)) != 0) {
		fmtAlloc(&valNew, 256, "%s %s", valCheck, value);
		websSetVar(wp, keyword, valNew);
		bfreeSafe(B_L, valNew);
	    } else
#endif	    
		websSetVar(wp, keyword, value);
	    wp->flags |= WEBS_GET_DATA;
	}
	keyword = strtok_r(NULL, "&", &tok_ptr);
    }
}

/*
 * Define a webs (CGI) variable for this connection. Also create in relevant
 * scripting engines. Note: the incoming value may be volatile. 
 */

void
websSetVar(webs_t wp, const char *var, const char *value)
{
    value_t v;

    a_assert(wp);

    /*
     *	value_instring will allocate the string if required.
     */
    if (value) {
	v = valueString(value, VALUE_ALLOCATE);
    } else {
	v = valueString("", VALUE_ALLOCATE);
    }

    D("wp: %p, added var '%s'='%s'", wp, var, value ? value : "");
    symEnter(wp->cgiVars, var, v, 0);
}

/*
 * Get a webs variable but return a default value if string not found.
 * Note, defaultGetValue can be NULL to permit testing existence.
 */

const char *
websGetVar(webs_t wp, const char *var, const char *defaultGetValue)
{
    sym_t *sp;

    a_assert(wp);
    a_assert(var && *var);
 
    if ((sp = symLookup(wp->cgiVars, var)) != NULL) {
	a_assert(sp->content.type == string);
	if (sp->content.value.string) {
            D("wp: %p, found var '%s'='%s'", wp, var, sp->content.value.string);
	    return sp->content.value.string;
	} else {
	    return "";
	}
    }
    D("wp: %p, var '%s' not found", wp, var);
    return defaultGetValue;
}

void
websTrimVar(webs_t wp, const char *var)
{
    sym_t *sp;

    a_assert(wp);
    a_assert(var && *var);

    if ((sp = symLookup(wp->cgiVars, var)) != NULL) {
        a_assert(sp->content.type == string);
        if (sp->content.value.string) {
	    pp_trim_string(sp->content.value.string);
            D("wp: %p, trimmed var '%s'='%s'", wp, var, sp->content.value.string);
        }
    } else {
	D("wp: %p, var '%s' not found", wp, var);
    }
}

void
websTransCaseVar(webs_t wp, const char *var, webs_trans_case_type_t type)
{
    sym_t *sp;

    a_assert(wp);
    a_assert(var && *var);

    if ((sp = symLookup(wp->cgiVars, var)) != NULL) {
        a_assert(sp->content.type == string);
        if (sp->content.value.string) {
	    if (type == CASE_UP)	pp_strtoupper(sp->content.value.string);
	    else if (type == CASE_DOWN)	pp_strtolower(sp->content.value.string);
	    else { assert(0); }
            D("wp: %p, trans case var '%s'='%s'", wp, var, sp->content.value.string);
        }
    } else {
	D("wp: %p, var '%s' not found", wp, var);
    }
}



char **
websSearchVars(webs_t wp, const char *var)
{
    sym_t * pea;
    char **ret = NULL;
    int ret_cnt = 0;
    int i = 0;
    char * search;
    int error = 1;
    
    a_assert(wp);
    a_assert(var && *var);
    
    for (pea = symFirst(wp->cgiVars); pea != NULL; pea = symNext(wp->cgiVars)){
	search = strstr(pea->name.value.string, var);
	if ((search == NULL) || (search != pea->name.value.string)) {
	    // no match
	    continue;
	}
	// found an matching entry
	if (pp_check_and_alloc_mem((void*)&ret, sizeof(*ret), &ret_cnt, i+1) < 0) {
	    goto bail;
	}
	if ((ret[i++] = strdup(pea->name.value.string)) == NULL) {
	    goto bail;
	}
    }
    if (pp_check_and_alloc_mem((void*)&ret, sizeof(*ret), &ret_cnt, i+1) < 0) {
	goto bail;
    }
    ret[i] = NULL;
    // all things ok, so error = 0
    error = 0;
 bail:
    if (error) {
	pp_free_dynamic_array(ret, i, free);
	ret = NULL;
    }
    return ret;
}



/*
 * Output a HTTP response back to the browser. If redirect is set to a 
 * URL, the browser will be sent to this location.
 */

void
websResponse(webs_t wp, int code, const char * cookie,
	     const char * additional_header,
	     const char * message, const char * redirect)
{
    char * date;

    a_assert(wp);

    /*
     * IE3.0 needs no Keep Alive for some return codes.
     */
    wp->flags &= ~WEBS_KEEP_ALIVE;

    /*
     * Only output the header if a header has not already been output.
     */
    if ( !(wp->flags & WEBS_HEADER_DONE)) {
	wp->flags |= WEBS_HEADER_DONE;
	websWrite(wp, "HTTP/1.1 %d %s\r\n", code, websErrorMsg(code));
	/*		
	 * By license terms the following line of code must not be modified.
	 */
	websWrite(wp, "Server: %s\r\n", WEBS_NAME);

	/*		
	 * Timestamp/Date is usually the next to go
	 */
	if ((date = websGetDateString(NULL)) != NULL) {
	    websWrite(wp, "Date: %s\r\n", date);
	    bfree(B_L, date);
	}

	if (wp->flags & WEBS_KEEP_ALIVE) {
	    websWrite(wp, "Connection: keep-alive\r\n");
	}

	websWrite(wp,
		  "Pragma: no-cache\r\n"
		  "Cache-Control: no-cache\r\n");

        const char * encoding = pp_intl_get_charset();
        if (encoding) {
            websWrite(wp, "Content-Type: text/html; charset=%s\r\n", encoding);
        } else {
            websWrite(wp, "Content-Type: text/html\r\n");
        }

	if (cookie) {
	    websWrite(wp, "Set-Cookie: %s\r\n", cookie);
	}

	if (additional_header) {
	    websWrite(wp, "%s\r\n", additional_header);
	}

	/* P3P stuff needed for IE6 */
	websWrite(wp, "P3P: CP=\"NON DSP CURa OUR NOR UNI\"\r\n");

	/*
	 * We don't do a string length here as the message may be multi-line. 
	 * Ie. <CR><LF> will count as only one and we will have a
	 * content-length that is too short.
	 *
	 * websWrite(wp, "Content-Length: %s\r\n", message);
	 */

	if (redirect) {
	    websWrite(wp, "Location: %s\r\n", redirect);
	}
	websWrite(wp, "\r\n");
    }

    /*
     *	If the browser didn't do a HEAD only request, send the message as well.
     */
    if ((wp->flags & WEBS_HEAD_REQUEST) == 0 && message && *message) {
	websWrite(wp, "%s\r\n", message);
    }
    websDone(wp, code);
}

/*
 * Redirect the user to another webs page
 */

void
websRedirect(webs_t wp, const char * url, const char * cookie)
{
    const char *redirectFmt;
    char *msgbuf, *urlbuf;
    char hostname[256];
    char * bulk_msg;
    
    a_assert(wp);
    a_assert(url);

    msgbuf = urlbuf = NULL;
    
    /*
     *	Some browsers require a http://host qualified URL for redirection
     */
    if (strstr(url, "http://") == NULL &&
	strstr(url, "https://") == NULL) {
	if (*url == '/') {
	    url++;
	}
	if (wp->flags & WEBS_SECURE) {
	    redirectFmt = "https://%s/%s";
	} else {
	    redirectFmt = "http://%s/%s";
	}

	if (gethostname(hostname, sizeof(hostname) - 1) == -1) {
	    pp_log_err("%s(): gethostname()", ___F);
	    /* Note: this is only a last resort, which should hardly occur.  */
	    strcpy(hostname, "localhost");
	}

	fmtAlloc(&urlbuf, WEBS_MAX_URL + 80, redirectFmt,
		 websGetVar(wp, "HTTP_HOST", hostname), url);
	url = urlbuf;
    }
    
    /*
     *	Add human readable message for completeness. Should not be required.
     */
    bulk_msg = gen_response_msg_for_bulk(wp);
    fmtAlloc(&msgbuf, WEBS_MAX_URL + 80 + (bulk_msg ? strlen(bulk_msg) : 0),
	     "<html><head></head><body>\r\n"
	     "Moved to this <a href=\"%s\">location</a>.\r\n"
	     "%s</body></html>\r\n", url, bulk_msg ? bulk_msg : "");

    websResponse(wp, 302, cookie, NULL, msgbuf, url);
    free(bulk_msg);
    bfreeSafe(B_L, msgbuf);
    bfreeSafe(B_L, urlbuf);
}

/*	
 * Output an error message and cleanup
 */

void
websError(webs_t wp, int code, const char *fmt, ...)
{
    va_list	args;
    char	*userMsg, *buf;
    const char	*msg;

    a_assert(wp);
    a_assert(fmt);

    va_start(args, fmt);
    userMsg = NULL;
    fmtValloc(&userMsg, WEBS_BUFSIZE, fmt, args);
    va_end(args);

    msg = "<html><head><title>Document Error: %s</title></head>\r\n<body><h2>Access Error: %s</h2>\r\n when trying to obtain <b>%s</b><br><p>%s</p></body></html>\r\n";
    /*
     *	Ensure we have plenty of room
     */
    buf = NULL;
    fmtAlloc(&buf, WEBS_BUFSIZE, msg, websErrorMsg(code), 
	     websErrorMsg(code), wp->url, userMsg);

    websResponse(wp, code, NULL, NULL, buf, NULL);
    bfreeSafe(B_L, buf);
    bfreeSafe(B_L, userMsg);
}

/*
 * Return the error message for a given code
 */

static const char *
websErrorMsg(int code)
{
    websErrorType *ep;

    for (ep = websErrors; ep->code; ep++) {
	if (code == ep->code) {
	    return ep->msg;
	}
    }
    a_assert(0);
    return "";
}

/*
 * Do formatted output to the browser. This is the public ASP and form
 * write procedure.
 */

int
websWrite(webs_t wp, const char *fmt, ...)
{
    va_list	 vargs;
    char	*buf = NULL;
    int		 rc = 0;
	
    va_start(vargs, fmt);

    if (fmtValloc(&buf, WEBS_BUFSIZE, fmt, vargs) >= WEBS_BUFSIZE) {
	pp_log("%s(): lost data, buffer overflow\n", ___F);
    }
    va_end(vargs);
    a_assert(buf);
    if (buf) {
	rc = websWriteBlock(wp, buf, strlen(buf));
	bfree(B_L, buf);
    }
    return rc;
}

/*
 * Write a block of data of length "nChars" to the user's browser. Public
 * write block procedure.
 */

int
websWriteBlock(webs_t wp, const char * buf, int nChars)
{
    int r = 0;

    if (!wp->skipOutput) {
	r = webs_bio_write_exact(wp, buf, nChars);
    }

    return r == -1 ? r : nChars;
}

/*
 * Decode a URL (or part thereof). Allows insitu decoding.
 */

void
websDecodeUrl(char *decoded, char *token, int len)
{
    char	*ip,  *op;
    int		num, i, c;
	
    a_assert(decoded);
    a_assert(token);

    op = decoded;
    for (ip = token; *ip && len > 0; ip++, op++) {
	if (*ip == '+') {
	    *op = ' ';
	} else if (*ip == '%' && isxdigit(ip[1]) && isxdigit(ip[2])) {

	    /*
	     * Convert %nn to a single character
	     */
	    ip++;
	    for (i = 0, num = 0; i < 2; i++, ip++) {
		c = tolower(*ip);
		if (c >= 'a' && c <= 'f') {
		    num = (num * 16) + 10 + c - 'a';
		} else {
		    num = (num * 16) + c - '0';
		}
	    }
	    *op = (char) num;
	    ip--;

	} else {
	    *op = *ip;
	}
	len--;
    }
    *op = '\0';
}

/*
 * Called when the request is done.
 */

void
websDone(webs_t wp, int code)
{
    a_assert(wp);

    if (code != 200) {
	wp->flags &= ~WEBS_KEEP_ALIVE;
    }

    /*
     * Close any opened document by a handler
     */
    websPageClose(wp);

    /*
     * Exit if secure.
     */
    if (wp->flags & WEBS_SECURE) {
	websFree(wp);
	return;
    }

    /*
     * If using Keep Alive (HTTP/1.1) we keep the socket open for a period
     * while waiting for another request on the socket. 
     */
    if (wp->flags & WEBS_KEEP_ALIVE) {
	wp->state = WEBS_BEGIN;
	wp->flags |= WEBS_REQUEST_DONE;
	if (wp->header.buf) {
	    ringqFlush(&wp->header);
	}
	return;
    }

    websFree(wp);
}

/*
 * allocate a webs structure
 */
webs_t
websAlloc(void)
{
    webs_t wp;

    /* ---- allocate a new webs structure ------------------------------ */

#ifdef B_STATS
    if ((wp = (webs_t)balloc(B_ARGS, sizeof(struct websRec))) == NULL)
#else
    if ((wp = (webs_t)balloc(B_L, sizeof(struct websRec))) == NULL)
#endif
    {
	pp_log("%s(): balloc() failed.\n", ___F);
	return NULL;
    }

    memset(wp, 0, sizeof(struct websRec));

    /* ---- initialize this webs structure ----------------------------- */

    wp->state = WEBS_BEGIN;
    wp->docfd = -1;
    wp->download_fd = -1;

    if (ringqOpen(&wp->header, WEBS_HEADER_BUFINC, WEBS_MAX_HEADER) == -1) {
	pp_log("%s(): ringqOpen() failed\n", ___F);
	bfree(B_L, wp);
	return NULL;
    }

    /*
     * Create storage for the CGI variables. We supply the symbol tables
     * for both the CGI variables and for the global functions. The
     * function table is common to all webs instances (ie. all browsers)
     */
    wp->cgiVars = symOpen(WEBS_SYM_INIT);

    return wp;
}

/*
 * free a webs structure
 */
void
websFree(webs_t wp)
{
#if defined(PP_FEAT_POWERSWITCH)
    /* close power switch connections if requested */
    if (wp->close_power_switches) {
	pp_power_logout(PP_POWER_PORT_ID_SERIAL_1);
	pp_power_logout(PP_POWER_PORT_ID_SERIAL_2);
    }
#endif /* PP_FEAT_POWERSWITCH */

    bfreeSafe(B_L, wp->target_user);
    bfreeSafe(B_L, wp->path);
    bfreeSafe(B_L, wp->url);
    bfreeSafe(B_L, wp->host);
    bfreeSafe(B_L, wp->lpath);
    bfreeSafe(B_L, wp->query);
    bfreeSafe(B_L, wp->decodedQuery);
    bfreeSafe(B_L, wp->cookie);
    bfreeSafe(B_L, wp->userAgent);
    bfreeSafe(B_L, wp->dir);
    bfreeSafe(B_L, wp->protocol);
    bfreeSafe(B_L, wp->protoVersion);
    bfreeSafe(B_L, wp->support_data);
    bfreeSafe(B_L, wp->cert_data);
    bfreeSafe(B_L, wp->mp_form_data);
#if defined(PP_FEAT_WS_MANAGEMENT)
    bfreeSafe(B_L, wp->wsman_auth);
#endif
    ringqClose(&wp->header);
    if (wp->session) {
	eric_session_release(wp->session);
    }

    if (wp->form_vars != NULL) {
        delete_form_vars(wp);
	free(wp->form_vars);
    }

    if (wp->fh_vec) vector_delete(wp->fh_vec);

    symClose(wp->cgiVars);

    webs_bio_close(wp);

    bfree(B_L, wp);
}

/*
 * Set the local path for the request
 */

void
websSetRequestLpath(webs_t wp, const char *lpath)
{
    a_assert(wp);
    a_assert(lpath && *lpath);

    if (wp->lpath) {
	bfree(B_L, wp->lpath);
    }
    wp->lpath = bstrdup(B_L, lpath);
}

/*
 * Update the URL path and the directory containing the web page
 */

void
websSetRequestPath(webs_t wp, const char *dir, const char *path)
{
    char	*tmp;

    a_assert(wp);

    if (dir) { 
	tmp = wp->dir;
	wp->dir = bstrdup(B_L, dir);
	if (tmp) {
	    bfree(B_L, tmp);
	}
    }
    if (path) {
	tmp = wp->path;
	wp->path = bstrdup(B_L, path);
	if (tmp) {
	    bfree(B_L, tmp);
	}
    }
}

/*
 * Build an ASCII time string.  If sbuf is NULL we use the current time,
 * else we use the last modified time of sbuf;
 */

char *
websGetDateString(websStatType *sbuf)
{
    char	cbuf[32];
    char*	cp, *r;
    time_t	now;

    if (sbuf == NULL) {
	time(&now);
    } else {
	now = sbuf->mtime;
    }
    if ((cp = ctime_r(&now, cbuf)) != NULL) {
	cp[strlen(cp) - 1] = '\0';
	r = bstrdup(B_L, cp);
	return r;
    }
    return NULL;
}
