/**	
 *	@file	RDM_DDA_Port.cpp
 *	@brief	Implementation of CDDA_Port
 *
 *	Object that represents  .
 */

#include <assert.h>
#include <pp/syms.h>
#include <pp/RDM_DDA_Device.h>
#include <pp/RDM_DDA_Port.h>
#include <pp/RDM_DDA_Utility.h>
#include <pp/SXDB_Parse_Table.h>

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Static Data
 *--------------------------------------*/

//----------------------------------------
//			Port Parse Table
//----------------------------------------

typedef	struct
{
	const char	*pID;					// id of the device
	const char	*pClass;				// Class attribute
	const char	*pType;					// Type atribute
	int		status;					// Status attribute
	const char	*pName;					// Name element
	int		deviceID;				// Device ID for switching
	int		index;
	const char	*pSecID;				// SecurityID
	int		internal;				// 1 = This is an internal port
	int		capacity;				// 
} PORT_DATA;

#define	PT_STRUCT	PORT_DATA
PT_BEGIN	( "Port",       portTable,	PT_NO_UNKNOWN )
PT_ATT		( "id",         pID,            0,	PT_STRING_PTR )
PT_ATT		( "Class",      pClass,         0,	PT_STRING_PTR )
PT_ATT		( "Type",       pType,          0,	PT_STRING_PTR )
PT_ATT		( "Status",     status,         0,	PT_INT )
PT_ELEM		( "Name",       pName,          0,	PT_STRING_PTR )
PT_ELEM		( "DeviceID",   deviceID,       0,	PT_INT | PT_CONDITION_1 )
PT_ATT		( "index",      index,          0,	PT_INT )
PT_ATT		( "SecurityID", pSecID,         0,	PT_STRING_PTR | PT_CONDITION_2 )
PT_ATT		( "Internal",   internal,       0,	PT_INT | PT_CONDITION_3 )
PT_ATT		( "Capacity",   capacity,       0,	PT_INT )
PT_END
#undef	PT_STRUCT

/*----------------------------------------
 *	Port Class
 *--------------------------------------*/

CDDA_Port::CDDA_Port (CRDM* pNewRDM) {
    pRDM = pNewRDM;
}

CDDA_Port::CDDA_Port (
        CRDM        * pNewRDM, 
        CDDA_Device * pParentDevice,     // Device object this port belongs to
        const char  * pID,               // id of the device
        const char  * pClass,            // Class ("KVM","Serial","Outlet")
        const char  * pType,             // Type Attribute
        const char  * _pName,            // Port name
        int           index,             // >= 0 - port index < 0  = no index attribute
        int           access,            // !0 = Add SecurityID="Access"
        int           internal,          // !0 = this is an internal port
        int           capacity           // Capacity attribute
    )
{
    pRDM = pNewRDM;
    initialize(pRDM, pParentDevice, pID, pClass, pType, _pName, index, access, internal, capacity);
}

void CDDA_Port::initialize
    (
        CRDM        * _pRDM, 
        CDDA_Device * pParentDevice,    // Device object this port belongs to
        const char  * pID,             // id of the device
        const char  * pClass,          // Class ("KVM","Serial","Outlet")
        const char  * pType,           // Type Attribute
        const char  * _pName,          // Port name
        int           index,           // >= 0 - port index < 0  = no index attribute
        int           access,          // !0 = Add SecurityID="Access"
        int           internal,        // !0 = this is an internal port
        int           capacity         // Capacity attribute
    )
{
	PORT_DATA   port;
	char        xPath[255];
	int         result;

	int deviceID = index; // hack - need to specify ???

	// Init data
	strncpy(id,pID,RDM_MAX_ID-1);

	// Add Device node

	BeginUpdate();

	if (pType == NULL)
		pType = "";

	sprintf(xPath,"//*[@id=\"%s\"]",pParentDevice->GetID());
	memset (&port, 0, sizeof (port));

	port.pID         = pID;
	port.pClass      = pClass;
	port.pType       = pType;
	port.status      = deviceID;
	port.pName       = _pName;
	port.deviceID    = index;		// HACK
	port.index       = index;
	port.pSecID      = "Access";
	port.internal    = internal;
	port.capacity    = capacity;

	result = SXDB_PT_Put(	
	                     _pRDM->db, xPath, 
	                     PT_CHILD_APPEND, 
	                     portTable, 
	                     &port, 
	                     (deviceID >= 0 ? PT_CONDITION_1 : 0) | (access ? PT_CONDITION_2 : 0) | (internal ? PT_CONDITION_3 : 0)
	                    );


	assert(result >= 0);

	EndUpdate();
}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/

CDDA_Port::~CDDA_Port( )
{
    // Delete our device node

    CSXDB_Node *p = pRDM->db->GetNodeFromID( id );
    if (p != NULL)
	delete p;
}


/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/

void CDDA_Port::BeginUpdate()
{
    pRDM->db->BeginAccess(1);
}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/

void CDDA_Port::EndUpdate()
{
    pRDM->db->EndAccess();

}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/

void CDDA_Port::SetStatus( int status )
{
    DDA_UpdatePathStatus( pRDM, id, status );    
}
/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/

const char * CDDA_Port::GetID( )
{
    return id;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h 
 *
 * @todo In the future we might need an extra parameter to determine the key
 *       type like (e.g SXDB_TYPE_ELEMENT or SXDB_TYPE_ATTRIBUTE )
**/
int  CDDA_Port::WriteData( const char * pID, const char * pKey, char * pData )
{
	char xPath[ 256 ];

	sprintf( xPath, "//*Port[@id=\"%s\"]/%s", pID, pKey );

	BeginUpdate();

	if ( pData ) {
		pRDM->db->PutData( xPath, NULL, SXDB_TYPE_ELEMENT, pData ); 
	}
	else {
		pRDM->db->Delete( xPath, NULL );
	}
	EndUpdate();

	pRDM->nm.Notify( GetID(), RDM_EC_Port_Changed, NULL );

	return 0;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/
void CDDA_Port::CommitData()
{
}

/*  --------------------------------------------------------------------*/
/** @see RDM_DDA_Device.h **/
int  CDDA_Port::ReadData( const char * NOTUSED(pID), const char * NOTUSED(pKey), const char * NOTUSED(pData) )
{
	return -1;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_BranchManager.h **/
int CDDA_Port::AssociateOutlet(const char * NOTUSED(port_id), const char * NOTUSED(outlet_id))
{
	return -1;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_BranchManager.h **/
int CDDA_Port::SetPowerState(const char * NOTUSED(outlet_id), int NOTUSED(state))
{
	return -1;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_BranchManager.h **/
char * CDDA_Port::GetPowerStripStatus (const char * NOTUSED(strip_id))
{
	return NULL;
}

/*  --------------------------------------------------------------------*/
/** @see RDM_BranchManager.h **/
int CDDA_Port::Update( CSession * NOTUSED(pUserSession), CSXDB_Node * NOTUSED(pNode), CSXDB_Node * NOTUSED(pData) )
{
	return -1;
}
