/*--------------------------------------------------------------------------

	RDM_Database.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager Protocol Database service handler class (CRDM_Database)

---------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <pp/syms.h>
#include <pp/Hash32.h>
#include <pp/RDM_Database.h>
#include <pp/SXDB_Parse_Table.h>
#include <pp/CmdLineParse.h>

//----------------------------------------
//				Equates
//----------------------------------------

	// Function IDs

enum
{
	CMD_Version,
	CMD_Get,
	CMD_Update,
	CMD_Append,
	CMD_Delete,
	CMD_GetUpdatedID,
	CMD_Count,
	CMD_Order
};

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

	// Function List

static TOKEN	functions[] = 
{
	{ "Version",	CMD_Version },
	{ "Get",		CMD_Get },
	{ "Update",		CMD_Update },
	{ "Append",		CMD_Append },
	{ "Delete",		CMD_Delete },
	{ "GetUpdatedID",		CMD_GetUpdatedID },
	{ "Count",		CMD_Count },
	{ "Order",		CMD_Order },
	{ NULL,			0 } // End of list
};

//----------------------------------------
//		Get Function Parse Table
//----------------------------------------

	// Get Function Structure

typedef	struct
{
	const char	*pXPath;
	const char	*pNodes;		
	const char	*pSubNodes;
	int		depth;
} GET_FUNC_DATA;

	// Get Function Parse Table

#define	PT_STRUCT	GET_FUNC_DATA
PT_BEGIN	( "Get",		getTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_ELEM		( "Nodes",		pNodes,		0,	PT_STRING_PTR )
PT_ELEM		( "SubNodes",	pSubNodes,	0,	PT_STRING_PTR )
PT_ELEM		( "Depth",		depth,		0,	PT_INT )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Count Function Parse Table
//----------------------------------------

	// Count Function Structure

typedef struct
{
	const char	*pXPath;
} COUNT_FUNC_DATA;

	// Count Function Parse Table

#define	PT_STRUCT	COUNT_FUNC_DATA
PT_BEGIN	( "Count",		countTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Order Function Parse Table
//----------------------------------------

	// Order Function Structure

typedef struct
{
	const char	*pXPath;
} ORDER_FUNC_DATA;

	// Count Function Parse Table

#define	PT_STRUCT	ORDER_FUNC_DATA
PT_BEGIN	( "Order",		orderTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Append Function Parse Table
//----------------------------------------

	// Append Function Structure

typedef	struct
{
	const char	*pXPath;
	CSXDB_Node	*pDataNode;
} APPEND_FUNC_DATA;

// Append Function Parse Table

#define	PT_STRUCT	APPEND_FUNC_DATA
PT_BEGIN	( "Append",		appendTable,	PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_ELEM		( "Data",		pDataNode,	0,	PT_NODE | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Update Function Parse Table
//----------------------------------------

	// Update Function Structure

typedef	struct
{
	const char		*pXPath;
	CSXDB_Node	*pDataNode;
} UPDATE_FUNC_DATA;

	// Update Function Parse Table

#define	PT_STRUCT	UPDATE_FUNC_DATA
PT_BEGIN	( "Update",		updateTable,	PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_ELEM		( "Data",		pDataNode,	0,	PT_NODE | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Delete Function Parse Table
//----------------------------------------

	// Delete Function Structure

typedef struct
{
	const char	*pXPath;
} DELETE_FUNC_DATA;

	// Count Function Parse Table

#define	PT_STRUCT	DELETE_FUNC_DATA
PT_BEGIN	( "Delete",		deleteTable,	PT_NO_UNKNOWN )
PT_ELEM		( "Select",		pXPath,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT


//----------------------------------------
//		GetUpdatedID Function Parse Table
//----------------------------------------

	// GetUpdatedID Function Structure

typedef	struct
{
	const char		*OldID;
} GETUPDATEDID_FUNC_DATA;

	// GetUpdatedID Function Parse Table

#define	PT_STRUCT	GETUPDATEDID_FUNC_DATA
PT_BEGIN	( "GetUpdatedID",		getUpdatedIDTable,	PT_NO_UNKNOWN )
PT_ELEM		( "OldID",  OldID,		0,  PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Code
//----------------------------------------

//---------------------------------------------------------------------------
//									CRDM_Database
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CRDM_Database::CRDM_Database
	(
		class CRDM	*	pNewRDM			// Ptr to the RDM object
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	// Initialize everthing

	pName		= "Database";
	hashCode	= Hash32( 0, pName );
	pBMList		= NULL;
	pRDM		= pNewRDM;
	cs = OS_CreateCriticalSection( OS_CRITICAL_SECTION_NORMAL );

	// Install our service

	pRDM->rdmp.AddService( this );
}

//---------------------------------------------------------------------------
//
	CRDM_Database::~CRDM_Database
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
	OS_DeleteCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	int									// 0 or error code if request failed
	CRDM_Database::Command
	(
		CSession	*	pUserSession,	// User session
		CSXDB_Node	*	pRootNode,		// Ptr to root node of our functions
		CSIO		*	pResponse		// The response -
										// Contains the return value on exit
	)
//
//	Processes an RDMP request for the database service & returns the response
//
//---------------------------------------------------------------------------
{
	const char		*pFunctionName;
	int			cmdID;
	int			result;
	CSXDB_Node	*pNode;

	// Get ptr to first function node

	pNode = this->GetFirstFunction( pRootNode );

	while (pNode != NULL)
	{
		// Function Name

		pFunctionName = pNode->GetName();

		// Find function

		cmdID = GetTokenValue( pFunctionName, functions );

		// Exexute the function

		pResponse->Printf("<%s>",pFunctionName);
		pRDM->db->BeginAccess( 1 );
		switch (cmdID)
		{
			case CMD_Version:
				pResponse->Puts("1");
				break;

			case CMD_Get:
				this->Get( pUserSession, pNode, pResponse );
				break;

			case CMD_Count:
				this->Count( pUserSession, pNode, pResponse );
				break;

			case CMD_Order:
				this->Order( pUserSession, pNode, pResponse );
				break;

			case CMD_Append:
				this->Append( pUserSession, pNode, pResponse );
				break;

			case CMD_Update:
				this->Update( pUserSession, pNode, pResponse );
				break;

			case CMD_Delete:
				this->Delete( pUserSession, pNode, pResponse );
				break;
				
			case CMD_GetUpdatedID:
			    this->GetUpdatedID( pUserSession, pNode, pResponse );
			    break;

			default:
				this->PrintError( pResponse, RDM_ERROR_FUNCTION_NOT_FOUND );
				break;

		}
		pRDM->db->EndAccess( );
		result = pResponse->Printf("</%s>",pFunctionName);

		pNode = this->GetNextFunction(pNode);
	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Get
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Get function
//
//---------------------------------------------------------------------------
{
	GET_FUNC_DATA	get;
	int				result;
	CSIO_DMEM		tempSIO;

	// Default values

	get.pXPath		= "/";
	get.pNodes		= "*";
	get.pSubNodes	= "*";
	get.depth		= 0;

	// Parse the function

	result = SXDB_PT_Get(pNode, getTable, &get, 0);

	if (result != 0 || get.pNodes == NULL)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Do it

	result = pRDM->db->Get( get.pXPath, get.pNodes, get.pSubNodes, get.depth, &tempSIO, pUserSession );

	// Show results

	if (result < 0)
		return PrintError( pResponse, result );
		
	pResponse->Printf("<Data>\n");
	tempSIO.CopyTo( pResponse );
	return pResponse->Printf("</Data>\n");
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Count
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Count function
//
//--------------------------------------------------------------------------
{
	COUNT_FUNC_DATA	count;
	int				result;

	// Default values

	count.pXPath		= "/";

	// Parse the function

	result = SXDB_PT_Get(pNode, countTable, &count, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Do it

	result = pRDM->db->Count( count.pXPath, pUserSession );

	// Show results

	if (result < 0)
		return PrintError( pResponse, result );
		
	return pResponse->Printf("%d", result);
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Order
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Count function
//
//--------------------------------------------------------------------------
{
	ORDER_FUNC_DATA	order;
	int				result;

	// Default values

	order.pXPath		= "/";

	// Parse the function

	result = SXDB_PT_Get(pNode, orderTable, &order, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Do it

	result = pRDM->db->Order( order.pXPath, pUserSession );

	// Show results

	if (result < 0)
		return PrintError( pResponse, result );
		
	return pResponse->Printf("%d", result);
}

//--------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Append
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The node of the function element
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Append function
//
//--------------------------------------------------------------------------
{
	CRDM_BranchManager	*pBM;
	APPEND_FUNC_DATA	append;
	CSXPath				xpath;
	int					result;
	int					count = 0;
	CSIO_DMEM			tempSIO;
	int					x;

	// Default values

	append.pXPath		= "/";

	// Parse the function

	result = SXDB_PT_Get(pNode, appendTable, &append, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Find the nodes

	result = xpath.Parse( append.pXPath, this->pRDM->db, NULL, 0, pUserSession );

	// If no nodes match, return 0

	if (result == 0)
		return pResponse->Puts("0");

	// Attempt to update each element found

	for (x=0; (pNode = xpath.Enum(x)); x++ )
	{
		// Find the branch manager for this node

		pBM = FindBranchManager( pUserSession, pNode );

		if (pBM == NULL)
			continue;

		// Call the BM to append the data

		result = pBM->Append( pUserSession, pNode, append.pDataNode );

		if (result >= 0)
			count ++;
	}

	// Show the results

	if (count > 0)
		return pResponse->Printf("%d",count);
	else
		return PrintError( pResponse, result );
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Update
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Update function
//
//---------------------------------------------------------------------------
{
	CRDM_BranchManager	*pBM;
	APPEND_FUNC_DATA	update;
	CSXPath				xpath;
	int					result;
	int					count = 0;
	CSIO_DMEM			tempSIO;
	int					x;

	// Default values
	update.pXPath		= "/";

	// Parse the function
	result = SXDB_PT_Get(pNode, updateTable, &update, 0);
	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Find the target node(s) for this update
	result = xpath.Parse( update.pXPath, this->pRDM->db, NULL, 0, pUserSession );
	// If no nodes match, return 0
	if (result == 0) {
		return pResponse->Puts("0");
	}

	// Attempt to update each element found
	for (x=0; (pNode = xpath.Enum(x)); x++ )
	{
		// Find the branch manager for this node
		pBM = FindBranchManager( pUserSession, pNode );
		if (pBM == NULL) {
			continue;
		}

		// Call the BM to update the data
		result = pBM->Update( pUserSession, pNode, update.pDataNode );
		if (result >= 0)
			count++;
	}

	// Show the results
   
	if (count > 0)
		return pResponse->Printf("%d",count);
	else {
		return PrintError( pResponse, result );
 	}
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::Delete
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Update function
//
//---------------------------------------------------------------------------
{
	CRDM_BranchManager	*pBM;
	DELETE_FUNC_DATA	del;
	CSXPath				xpath;
	int					result;
	int					count = 0;
	int					x;

	// Default values

	del.pXPath		= "/";

	// Parse the function

	result = SXDB_PT_Get(pNode, deleteTable, &del, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Check for BMs that want to do pre processing

	pBM = NULL;
	do
	{
		pBM = EnumBranchManager( NULL, pBM );

		if (pBM == NULL)
			break;
		
		result = pBM->PreProcessDelete( pUserSession, del.pXPath, pResponse );

		if (result != 0)
			return 0;
	} while (1);

	// Find the nodes

	result = xpath.Parse( del.pXPath, this->pRDM->db, NULL, 0, pUserSession );

	if (result < 0)
		return PrintError( pResponse, result );

	// If no nodes match, return 0

	if (result == 0)
		return pResponse->Puts("0");

	// Attempt to delete each element found

	for (x=0; (pNode = xpath.Enum(x)); x++ )
	{
		// Find the branch manager for this node

		pBM = FindBranchManager( pUserSession, pNode );

		if (pBM == NULL)
			continue;

		// Call the BM to delete it

		result = pBM->Delete( pUserSession, pNode );

		if (result >= 0)
			count ++;
	}

	// Show the results

	if (count > 0)
		return pResponse->Printf("%d",count);
	else
		return PrintError( pResponse, result );

	return 0;	
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Database::GetUpdatedID
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the GetUpdatedID function
//  Since Port id may be changed based on CIM serial number
//  So, old Port ID will be replaced by new Port ID.
//  RDMP Client can use this command to get new Port ID
//
//  Old Port covention: P_<Device Serial Number>_<Index>
//
//---------------------------------------------------------------------------
{
    int                     result;
    CSXPath				    xpath;
    char                    str_xpath[512];
    char                    id[RDM_MAX_ID];
    GETUPDATEDID_FUNC_DATA  getUpdatedID;
    
    // Parse the function

	result = SXDB_PT_Get(pNode, getUpdatedIDTable, &getUpdatedID, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

    sprintf(str_xpath, "//*[@id=\"%s\"]", getUpdatedID.OldID);	
	result = xpath.Parse(str_xpath, this->pRDM->db, NULL, 0, pUserSession);
	
    if(result == 1)
    {
        // Found this id in RDM database
        return pResponse->Printf("<OldID>%s</OldID><NewID>%s</NewID>", getUpdatedID.OldID, getUpdatedID.OldID);
    }
    
    // find new ID
    
    if(getUpdatedID.OldID[0] == 'P')
    {
        char device_id[RDM_MAX_ID];
        char index[8];
        strncpy(id, getUpdatedID.OldID, RDM_MAX_ID);
        
        char *ptr = &(id[0]);
        ptr = strtok(ptr, "_");   // point to "P"
        
        ptr = strtok(NULL, "_");  // point to device id
        if(ptr) sprintf(device_id, "P_%s", ptr);
        else device_id[0] = 0;
        
        ptr = strtok(NULL, "_");  // point to index
        if(ptr) strcpy(index, ptr);
        else index[0] = 0;
        
        sprintf(str_xpath, "/System/Device[@id=\"%s\"]/Port[@index=%s]", device_id, index);	
    	result = xpath.Parse(str_xpath, this->pRDM->db, NULL, 0, pUserSession);
    	
    	if(result < 0)
    	{
    	    // Parsing error
    	    printf("%s: Error - XML[%s] parsing fialed\n", __PRETTY_FUNCTION__, str_xpath);
    	    return PrintError( pResponse,  result );
    	}
    	else if(result==1)
    	{

        	CSXDB_Node	*pPortNode = xpath.Enum(0);
        	CSXDB_Node	*pPortIDNode = pPortNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "id", NULL );
        	if(pPortIDNode && pPortIDNode->GetData())
        	{
        	    return pResponse->Printf("<OldID>%s</OldID><NewID>%s</NewID>", getUpdatedID.OldID, pPortIDNode->GetData());    
        	}
        }
    }
    
    return pResponse->Printf("<OldID>%s</OldID>", getUpdatedID.OldID);
}	

//---------------------------------------------------------------------------
//
	CRDM_BranchManager	*			// BM ptr or NULL if none found
	CRDM_Database::FindBranchManager
	(
		CSession	* NOTUSED(pUserSession),// User session or NULL for super user
		CSXDB_Node	*pNode			// The node in question
	)
//
//	Processes the Update function
//
//---------------------------------------------------------------------------
{
	CSXDB_Node			*pBMNode;
	CRDM_BranchManager	*pBM;

	do
	{
		pBMNode = pNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "BM", NULL );

		if (pBMNode != NULL)
		{
			pBM = EnumBranchManager( pBMNode->GetData(), NULL );

			if (pBM != NULL)
				return pBM;
		}
	
		pNode = pNode->Parent();
	
	} while (pNode != NULL);

	return NULL;
}

//---------------------------------------------------------------------------
//
	CRDM_BranchManager	*			// BM ptr or NULL if none found
	CRDM_Database::EnumBranchManager
	(
		const char		*_pName,// The BM name (NULL = Find all BMs)
		CRDM_BranchManager	*pLast	// NULL to find first,
									// otherwise last found to find next
	)
//
//	Processes the Update function
//
//---------------------------------------------------------------------------
{
	CRDM_BranchManager	*pBM;
	int					hash;

	if (pLast == NULL)
		pBM = pBMList;
	else
		pBM = pLast->pNext;

	if (_pName == NULL)
		return pBM;

	hash = Hash32(0,_pName);

	while (pBM != NULL)
	{
		if (hash == pBM->GetHashCode())
		{
			if (strcmp( _pName, pBM->GetName() ) == 0)
			{
				// Found
				
				break;
			}
		}

		pBM = pBM->pNext;
	}

	return pBM;
}

//---------------------------------------------------------------------------
//
	void
	CRDM_Database::AddBranchManager
	(
		CRDM_BranchManager	*pBM		// Ptr to the Branch Manager to add
	)
//
//	Adds a Branch Manager to the Database manager
//
//--------------------------------------------------------------------------
{
	OS_EnterCriticalSection( cs );

	pBM->pNext = pBMList;
	pBMList = pBM;

	OS_LeaveCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	void
	CRDM_Database::RemoveBranchManager
	(
		CRDM_BranchManager	*pBM		// Ptr to the subscription to remove
	)
//
//	Removes a Branch Manager from the database manager
//
//---------------------------------------------------------------------------
{
	CRDM_BranchManager *pTemp;

	OS_EnterCriticalSection( cs );

	if (pBMList == pBM)
		pBMList = pBM->pNext;
	else
	{
		for (pTemp = pBMList; pTemp != NULL && pTemp->pNext != pBM; pTemp = pTemp->pNext)
			;

		if (pTemp != NULL)
			pTemp->pNext = pBM->pNext;
	}

	pBM->pNext = NULL;

	OS_LeaveCriticalSection( cs );
}

