/*-------------------------------------------------------------------------------

	RDM_Power.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager Protocol Power service handler class (CRDM_Power)


--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<pp/syms.h>
#include	<pp/RDM.h>
#include	<pp/Hash32.h>
#include	<pp/CmdLineParse.h>
#include	<pp/RDM_Power.h>
#include	<pp/RDM_Database.h>
#include	<pp/SXDB_Parse_Table.h>
#include	<pp/RDM_DDA_Utility.h>

//----------------------------------------
//				Equates
//----------------------------------------
#define     MAX_PORTS       6

#define	    NOT_SUPPORT_ASSOCIATION_ON_P2_ZCIM 

//#define	_DEBUG_POWER	1				// Define this only for debug

	// Function IDs

enum
{
	CMD_Version,
	CMD_On,
	CMD_Off,
	CMD_Cycle,
	CMD_Associate,
	CMD_PowerStripStatus
};

	// Power States for SetPower()

enum
{
	POWER_CYCLE	= 0,
	POWER_ON = 1,
	POWER_OFF = 2
};

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

#ifdef _DEBUG_POWER
	void
	CreateTestPowerStrip
	(
	);
#endif

//----------------------------------------
//				Static Data
//----------------------------------------

	// Function List

static TOKEN	functions[] = 
{
	{ "Version",			CMD_Version },
	{ "On",					CMD_On },
	{ "Off",				CMD_Off },
	{ "Cycle",				CMD_Cycle },
	{ "Associate",			CMD_Associate },
	{ "PowerStripStatus",	CMD_PowerStripStatus },
	{ NULL,			0 } // End of list
};

//----------------------------------------
//		Associated Outlet table
//----------------------------------------

	// Outlet Structure

typedef struct
{
	const char	*ConnID;					// The connection ID
	int		matched;					// Used to figure out what associations are new
} ASSOCIATED_OUTLET;

	// Outlet Parse Table

#define	PT_STRUCT	ASSOCIATED_OUTLET
PT_BEGIN	( "AssociatedOutlet",	outletTable,	PT_UNKNOWN_OK )
PT_ATT		( "ConnID",			ConnID,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		On Function Parse Table
//----------------------------------------

	// On Function Structure

typedef	struct
{
	const char		*pID;		
} ON_FUNC_DATA;

	// On Function Parse Table

#define	PT_STRUCT	ON_FUNC_DATA
PT_BEGIN	( "On",				onTable,			PT_NO_UNKNOWN )
PT_ROOT_DATA(					pID,			0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Off Function Parse Table
//----------------------------------------

	// Off Function Structure

typedef	struct
{
	const char		*pID;		
} OFF_FUNC_DATA;

	// Off Function Parse Table

#define	PT_STRUCT	OFF_FUNC_DATA
PT_BEGIN	( "Off",			offTable,			PT_NO_UNKNOWN )
PT_ROOT_DATA(					pID,			0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Cycle Function Parse Table
//----------------------------------------

	// Cycle Function Structure

typedef	struct
{
	const char		*pID;		
} CYCLE_FUNC_DATA;

	// Off Function Parse Table

#define	PT_STRUCT	CYCLE_FUNC_DATA
PT_BEGIN	( "Cycle",			cycleTable,			PT_NO_UNKNOWN )
PT_ROOT_DATA(					pID,			0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Associate Function Parse Table
//----------------------------------------

typedef	struct
{
	const char		*pPortID;				// ID of the Port to associate
	int			outletCount;			// # of associated outlets found
	ASSOCIATED_OUTLET AssociatedOutlet[4];	// The associated outlet data
} ASSOCIATE_FUNC_DATA;

	// Associate Function Parse Table

#define	PT_STRUCT	ASSOCIATE_FUNC_DATA
PT_BEGIN	( "Associate",		associateTable,		PT_UNKNOWN_OK )
PT_MOVE_DOWN( "AssociatedPort",						PT_REQUIRED )
PT_ATT		( "ConnID",			pPortID,		0,	PT_STRING_PTR )
PT_MOVE_UP	( )
PT_ARRAY	( outletTable,		AssociatedOutlet, 4, 0, outletCount )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Port/Associated Outlet Parse Table
//----------------------------------------

typedef	struct
{
	int			outletCount;			// # of associated outlets found
	ASSOCIATED_OUTLET AssociatedOutlet[4];	// The associated outlet data
	const char	*	Name;					// Name of the port
	const char	*	Status;					// Status of the port
	const char	*	Class;					// The port's class
	const char	*	AssociatedPort;			// Associated Port for outlets
} PORT_OUTLET_DATA;

	// Associate Function Parse Table

#define	PT_STRUCT	PORT_OUTLET_DATA
PT_BEGIN	( "Port",			portOutletTable,	PT_UNKNOWN_OK )
PT_ARRAY	( outletTable,		AssociatedOutlet, 4, 0, outletCount )
PT_ATT		( "Status",			Status,		0,		PT_STRING_PTR )
PT_ELEM		( "Name",			Name,		0,		PT_STRING_PTR )
PT_ATT		( "Class",			Class,		0,		PT_STRING_PTR )
PT_ELEM		( "AssociatedPort",	AssociatedPort, 0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		PowerStripStatus Function Parse Table
//----------------------------------------

	// On Function Structure

typedef	struct
{
	const char		*pID;		
} POWER_STRIP_FUNC_DATA;

	// On Function Parse Table

#define	PT_STRUCT	ON_FUNC_DATA
PT_BEGIN	( "PowerStripStatus",	powerStripStatusTable,	PT_NO_UNKNOWN )
PT_ROOT_DATA(						pID,			0,		PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CRDM_Power
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRDM_Power::CRDM_Power
	(
		class CRDM	*	pNewRDM		// Ptr to the new RDM object
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	pName				= "Power";
	hashCode			= Hash32( 0, pName );
	pRDM				= pNewRDM;

	// Install our service

	pRDM->rdmp.AddService( this );
}

//--------------------------------------------------------------------------------
//
	CRDM_Power::~CRDM_Power
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
   pRDM->rdmp.RemoveService( this );
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code if request failed
	CRDM_Power::Command
	(
		CSession	*	pUserSession,	// The user session or NULL for super user
		CSXDB_Node	*	pRootNode,		// Ptr to the root node of our functions
		CSIO		*	pResponse		// The response -
										// Contains the return value on exit
	)
//
//	Processes an RDMP request for the notification service and returns the response.
//
//------------------------------------------------------------------------------//
{
	const char		*pFunctionName;
	int			cmdID;
	int			result;
	CSXDB_Node	*pNode;

	// Get ptr to first function node

	pNode = this->GetFirstFunction( pRootNode );

	while (pNode != NULL)
	{
		// Function Name

		pFunctionName = pNode->GetName();

		// Find function

		cmdID = GetTokenValue( pFunctionName, functions );

		// Exexute the function

		pResponse->Printf("<%s>",pFunctionName);

		switch (cmdID)
		{
			case CMD_Version:
				pResponse->Puts("1");

				#ifdef _DEBUG_POWER
					CreateTestPowerStrip();
				#endif

				break;

			case CMD_On:
				this->On( pUserSession, pNode, pResponse );
				break;

			case CMD_Off:
				this->Off( pUserSession, pNode, pResponse );
				break;

			case CMD_Cycle:
				this->Cycle( pUserSession, pNode, pResponse );
				break;

			case CMD_Associate:
				this->Associate( pUserSession, pNode, pResponse );
				break;

			case CMD_PowerStripStatus:
				this->PowerStripStatus( pUserSession, pNode, pResponse );
				break;

			default:
				this->PrintError( pResponse, RDM_ERROR_FUNCTION_NOT_FOUND );
				break;

		}

		result = pResponse->Printf("</%s>",pFunctionName);

		pNode = this->GetNextFunction(pNode);
	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::On
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the On function
//
//---------------------------------------------------------------------------
{
	ON_FUNC_DATA		on;
	int					result = 0;

	// Default values

	memset(&on,0, sizeof(on));

	// Parse the function

	result = SXDB_PT_Get(pNode, onTable, &on, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	return SetPowerState( pUserSession, on.pID, pResponse, POWER_ON );
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::Off
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the On function
//
//---------------------------------------------------------------------------
{
	OFF_FUNC_DATA		off;
	int					result = 0;;

	// Default values

	memset(&off,0, sizeof(off));

	// Parse the function

	result = SXDB_PT_Get(pNode, offTable, &off, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	return SetPowerState( pUserSession, off.pID, pResponse, POWER_OFF );
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::Cycle
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Cycle function
//
//---------------------------------------------------------------------------
{
	CYCLE_FUNC_DATA		cycle;
	int					result = 0;;

	// Default values

	memset(&cycle,0, sizeof(cycle));

	// Parse the function

	result = SXDB_PT_Get(pNode, cycleTable, &cycle, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	return SetPowerState( pUserSession, cycle.pID, pResponse, POWER_CYCLE );
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::SetPowerState
	(
		CSession	*pUserSession,	// User session
		const char		*pID,		// ID of the port to control
		CSIO		*pResponse,		// SIO Response
		int			state			// power state
	)
//
//	Sets the power state of a port
//
//---------------------------------------------------------------------------
{
	int					result = 0;
	int					tempResult;
	PORT_OUTLET_DATA	port;
	char				xpath[100];
	int					x;

	// Get the port info

	sprintf(xpath,"//*[@id=\"%s\"]",pID);

	result = SXDB_PT_Get(pRDM->db, xpath, portOutletTable, &port );

	if (result < 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// See if this is a normal port or an outlet

	if (port.Class != NULL && strcmp(port.Class,"Outlet") == 0)
	{
		// Direct Outlet Control

		// Check permissions

		if (pUserSession != NULL && pUserSession->GetUserObject()->GetPermission("Access",pID) != TR_PERM_READWRITE)
			return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

		// Set state

		result = DEP_SetPowerStateOfOutlet( pID, state );

		if (result < 0)
			return PrintError( pResponse, RDM_ERROR_FAILED );
	}
	else
	{
		// In direct outlet control via AssociatedOutlets

		// If there are no associated outlets, then it's an error

		if (port.outletCount == 0)
			return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

		// Check permissions

		for (x=0;x<port.outletCount;x++)
		{
			if (pUserSession != NULL && pUserSession->GetUserObject()->GetPermission("Access",port.AssociatedOutlet[x].ConnID) != TR_PERM_READWRITE)
				return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );
		}

		// Set state

		result = 0;

		if (state != POWER_CYCLE || port.outletCount == 1)
		{
			// Normal case

			for (x=0;x<port.outletCount;x++)
			{
				tempResult = DEP_SetPowerStateOfOutlet( port.AssociatedOutlet[x].ConnID, state );
				if (tempResult < 0)
					result = RDM_ERROR_FAILED;
			}
		}
		else
		{
			// Special case ofr CYCLE of more than one outlet

			// All OFF

			for (x=0;x<port.outletCount;x++)
			{
				tempResult = DEP_SetPowerStateOfOutlet( port.AssociatedOutlet[x].ConnID, POWER_OFF );

				if (tempResult < 0)
					result = RDM_ERROR_FAILED;
			}

			// Delay for Cycle of more that one outlet

			// NOTE: Advised by RT. Should wait approx. 30 sec to power-on after power-off
			OS_Sleep(30*1000); // HACK - FIXME - Need cycle time from peter

			// All On

			for (x=0;x<port.outletCount;x++)
			{
				tempResult = DEP_SetPowerStateOfOutlet( port.AssociatedOutlet[x].ConnID, POWER_ON );

				if (tempResult < 0)
					result = RDM_ERROR_FAILED;
			}
		}

		if (result != 0)
			return PrintError( pResponse, result );
	}

	// Done


	return result;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::DEP_SetPowerStateOfOutlet
	(
		const char	*	NOTUSED(pID),
		int			NOTUSED(state)
	)
//
//	Utility function for SetPowerState
//
//---------------------------------------------------------------------------
{
	return -1; // This method must be implemented in a derived class
}

//---------------------------------------------------------------------------
//
    char *                              // I/O error on output
    CRDM_Power::DEP_GetPowerStripStatus
    (
        const char        *   NOTUSED(pID)
    )
//
//  Utility function for GetPowerStripStatus
//
//---------------------------------------------------------------------------
{
    return NULL; // This method must be implemented in a derived class
}


//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Power::Associate
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Associate function
//
//---------------------------------------------------------------------------
{
	ASSOCIATE_FUNC_DATA	associate;
	PORT_OUTLET_DATA	port;
	PORT_OUTLET_DATA	outlet[4];
	int					result = 0;
	int					x,y;
	CSXDB_Node		*	pOutletNode;
	CRDM_BranchManager *pBM;
	CRDM_Database	*	pDatabaseService;
	char				xpath[100];
	// char			*	pString;

	// Check Permissions

	if (pUserSession->GetUserObject()->GetNodePermission( "DeviceManagement" ) != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	// We will need the database service

	pDatabaseService = (CRDM_Database *) pRDM->rdmp.FindService( "Database" );

	if (pDatabaseService == NULL)
		return PrintError( pResponse, RDM_ERROR_MEMORY_ERROR );

	// Default values

	memset(&associate,0, sizeof(associate));
	memset(&port,0,sizeof(port));
	memset(&outlet,0,sizeof(outlet));

	// Parse the function

	result = SXDB_PT_Get(pNode, associateTable, &associate, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Make sure all the outlets are really outlets

	for (x=0;x<associate.outletCount;x++)
	{
		sprintf(xpath,"//*[@id=\"%s\"]",associate.AssociatedOutlet[x].ConnID);
		result = SXDB_PT_Get( pRDM->db, xpath, portOutletTable, &outlet[x] );

		if ( outlet[x].Class == NULL || strcmp(outlet[x].Class,"Outlet") != 0)
			return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
	}

	// Find the port and parse existing data

	sprintf(xpath,"//*[@id=\"%s\"]",associate.pPortID);
	
#ifdef NOT_SUPPORT_ASSOCIATION_ON_P2_ZCIM

	// Becasue power association on P2-ZCIM isn't supported by UMT and will cause serious 
	// problem on UMT. So we temprarily block it. 
	// We will remove this block once UMT can support it.
	 
	CSXPath 		parser;
	CSXDB_Node		*	pPortNode = NULL;
	CSXDB_Node		*	pDeviceNode = NULL;
	CSXDB_Node		*	pDeviceModelNode = NULL;

	// check if the port is P2-ZCIM, return error (RDM_NOT_SUPPORT)
	result = parser.Parse(xpath, pRDM->db, NULL, 1, NULL);
	
	if(result != 1)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	pPortNode = parser.Enum(0);
	if(pPortNode) pDeviceNode = pPortNode->Parent();
	if(pDeviceNode) pDeviceModelNode = pDeviceNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE,"Model", NULL);
	
	if(pDeviceModelNode && pDeviceModelNode->GetData())
	{
		if(strcmp(pDeviceModelNode->GetData(), "P2-ZCIM") == 0)
			return PrintError( pResponse, RDM_ERROR_NOT_SUPPORTED );
	}
	
#endif 	

	result = SXDB_PT_Get(pRDM->db, xpath, portOutletTable, &port );

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Find Matches

	for (x=0;x<port.outletCount;x++)
	{
		for (y=0;y<associate.outletCount;y++)
		{
			if (strcmp(port.AssociatedOutlet[x].ConnID, associate.AssociatedOutlet[y].ConnID) == 0 )
			{
				port.AssociatedOutlet[x].matched = 1;
				associate.AssociatedOutlet[y].matched = 1;
			}
		}
	}

	// Remove associations that are no longer in use

	for (x=port.outletCount-1;x>=0;x--)
	{
		if (!port.AssociatedOutlet[x].matched)
		{
			sprintf(xpath,"//*[@id=\"%s\"]",port.AssociatedOutlet[x].ConnID);
			pOutletNode = pRDM->db->GetNodeFromXPath( xpath );
			if (pOutletNode != NULL)	// was already checked, should never be NULL
			{
				// Tell Low Level to dis-associate

				pBM = pDatabaseService->FindBranchManager( NULL, pOutletNode );

				while (pBM != NULL)
				{
					int q;

					//printf("POWER: Calling %s\b",pBM->GetName());
					q = pBM->AssociateOutlet( NULL, port.AssociatedOutlet[x].ConnID );
					//printf("POWER: AssociateOutlet NULL, %d ) returned %d\n",port.AssociatedOutlet[x].ConnID,q);

					if (q >= 0)
						pBM = NULL;
					else
						pBM = pDatabaseService->EnumBranchManager( pBM->GetName(), pBM );
				}

				// Update the Outlet database

			//	sprintf(xpath,"//*[@id=\"%s\"]/AssociatedPort",port.AssociatedOutlet[x].ConnID);
			//	pRDM->db->Delete( xpath );

				// Update the Port database

			//	sprintf(xpath,"//*[@id=\"%s\"]/AssociatedOutlet[@ConnID=\"%s\"]",associate.pPortID, port.AssociatedOutlet[x].ConnID);
			//	pRDM->db->Delete( xpath );

			}
		}
	}
					
	// Add new associations

	for (x=0;x<associate.outletCount;x++)
	{
		if (!associate.AssociatedOutlet[x].matched)
		{
			sprintf(xpath,"//*[@id=\"%s\"]",associate.AssociatedOutlet[x].ConnID);
			pOutletNode = pRDM->db->GetNodeFromXPath( xpath );
			if (pOutletNode != NULL)	// was already checked, should never be NULL
			{
				// Tell Low Level to associate

				pBM = pDatabaseService->FindBranchManager( NULL, pOutletNode );

				while (pBM != NULL)
				{
					int	q;

					//printf("POWER: Calling %s\b",pBM->GetName());
					q = pBM->AssociateOutlet( associate.pPortID, associate.AssociatedOutlet[x].ConnID );
					//printf("POWER: AssociateOutlet( %s, %s ) returned %d\n",associate.pPortID, associate.AssociatedOutlet[x].ConnID,q);

					if (q >= 0)
						pBM = NULL;
					else
						pBM = pDatabaseService->EnumBranchManager( pBM->GetName(), pBM );
				}

				// Update the Outlet database

				//sprintf(xpath,"//*[@id=\"%s\"]",associate.AssociatedOutlet[x].ConnID);
				//pRDM->db->PutData( xpath, "AssociatedPort", SXDB_TYPE_ELEMENT, NULL );
				//strcat( xpath,"/AssociatedPort" );
				//pRDM->db->PutData( xpath, "ConnID", SXDB_TYPE_ATTRIBUTE, associate.pPortID );

				// Update the Port database

				//sprintf(xpath,"//*[@id=\"%s\"]",associate.pPortID);
				//result = SXDB_PT_Put( pRDM->db, xpath, PT_CHILD_APPEND, outletTable, &associate.AssociatedOutlet[x] );

			}
		}
	}
	
	// Send the notification

//	sprintf(xpath,"//*[@id=\"%s\"]/../Name",associate.pPortID); // Device Name

//	pString = pRDM->db->GetData( xpath );

//	DDA_Notify_Port_Change( pRDM, associate.pPortID, pString, port.Name, port.Status );

	// Done

	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_Power::PowerStripStatus
	(
		CSession	* NOTUSED(pUserSession),// User session
		CSXDB_Node	* pNode,		// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Associate function
//
//---------------------------------------------------------------------------
{
	POWER_STRIP_FUNC_DATA	status;
	int						result = 0;
	char				*	pStatus = NULL;

	// Default values

	memset(&status,0, sizeof(status));

	// Parse the function

	result = SXDB_PT_Get(pNode, powerStripStatusTable, &status, 0);

	if (result != 0){
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );
	}

	pStatus = DEP_GetPowerStripStatus(status.pID);

	if (pStatus == NULL)
		return PrintError( pResponse, RDM_ERROR_FAILED );
	
	pResponse->Puts(pStatus);

	delete pStatus;

	return 0;

}




#ifdef _DEBUG_POWER

#include "DeviceDBMgr.h"

//---------------------------------------------------------------------------
//
	void
	CreateTestPowerStrip
	(
	)
//
//	This is a function used for testing onlt. It create a dummy power strip.
//	This code is IP-Reach specific.
//
//---------------------------------------------------------------------------
{
	int		x;
	char	id[16];
	static	created = 0;

	if (created)
		return;

	AddRDMDevice(	"PowerStrip_ID",
					"PowerStrip",
					"PCR8",
					NULL,
					"Test Power Strip",
					0
				);

	for (x=0;x<8;x++)
	{
		sprintf(id,"Outlet_%d",x);
		AddRDMPort(
					"//*[@id=\"PowerStrip_ID\"]",
					id,
					"Outlet",
					NULL,
					1,
					id
				  );
	}

	created = 1;
}

#endif
