/*-------------------------------------------------------------------------------

	RDM_Secure_SXDB.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	A Secure version of SXDB. Secure means users are authorized and it is thread
	safe.

	Two methods have been added to the base SXDB class:
		BeginUpdate( int write );
		EndUpdate();

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<pp/syms.h>
#include	<pp/RDM_Secure_SXDB.h>
#include	<pp/Session.h>

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CRDM
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRDM_Secure_SXDB::CRDM_Secure_SXDB
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	cs = OS_CreateCriticalSection( OS_CRITICAL_SECTION_NORMAL );
}

//--------------------------------------------------------------------------------
//
	CRDM_Secure_SXDB::~CRDM_Secure_SXDB
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	if (cs != NULL)
		OS_DeleteCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Secure_SXDB::BeginAccess
	(
		int	NOTUSED(write)						// 0 = read only, !0 = read/write access
	)
//
//	Locks the database for access.
//
//	If <write> = 0, then all writes are locked out. Others threads can read, but cannot write.
//	If <write> is !0, then no other thread can read or write.
//
//	BeginAccess() calls can be nested as long as EndAccess is called the same number
//	of times.
//
//------------------------------------------------------------------------------//
{
	// NOTE: This implementation locks reads and writes the same way.
	//		 Only one thread can access the data at once.

	OS_EnterCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Secure_SXDB::EndAccess
	(
	)
//
//	Unlocks the database
//
//------------------------------------------------------------------------------//
{
	OS_LeaveCriticalSection( cs );
}

//--------------------------------------------------------------------------------
//
	int									// !0 if access is allowed
										// 0 = Not allowed
										// -1 = unknown for this node
	CRDM_Secure_SXDB::CheckAccess
	(
		CSXDB_Node	*pNode,				// The node to authorize
		void		*pUser,				// User account
		int			accessType,			// SXDB_ACCESS_xxx
		int			oneLevel			// !0 = only check pNode, not it's ansestors
	)
//
//	Checks to see if the user is allowed to perform the specifide access to the 
//	node.
//
//------------------------------------------------------------------------------//
{
	CSession	*pUserSession = (CSession *) pUser;
	const char	*pSecurityID;
	const char	*pID;
	DWORD		perm;

	if (pUser == NULL)
		return 1;

	if (pNode == NULL)
		return 0;

	// Special case for private nodes starting with "_"

	if (pNode->GetName()[0] == '_')
		return 0;

	// Check this node or this and it's ansestors

	if (oneLevel)
	{
		GetSecurityIDForNode( pNode, &pSecurityID, &pID );
		
		if (pSecurityID == NULL)
			return -1; // Unknown security
	}
	else
	{
		GetSecurityID( pNode, &pSecurityID, &pID );
		
		if (pSecurityID == NULL)
			return 0; // Default to no access
	}

	// Ask the security module for access permissions

	if (pID != NULL)
		perm = pUserSession->GetUserObject()->GetPermission( pSecurityID, pID );
	else
		perm = pUserSession->GetUserObject()->GetNodePermission( pSecurityID );

	// Determine if the accessType is allowed for the found permissions

	switch (accessType)
	{
		case SXDB_ACCESS_READ:
			if (perm == TR_PERM_NONE)
				return 0;
			break;

		case SXDB_ACCESS_WRITE:
			if (perm <= TR_PERM_READONLY)
				return 0;
			break;

		default:
		case SXDB_ACCESS_DELETE:
			if (perm <= TR_PERM_READONLY)
				return 0;
			break;
	}

	return 1;
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Secure_SXDB::GetSecurityIDForNode
	(
		CSXDB_Node	*pNode,				// The node to search
		const char	**pSecurityID,		// Returns the securityID, or NULL if none found
		const char	**pID				// Returns the id attribute found at the same
										// level as the securityID or NULL if none found
	)
//
//	Looks for the SecurityID="" attribute for the given node.
//	Not all nodes have a security ID. Null is returned if none is found
//	This function only searches the specified node, not any ansestors
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node	*pChild;

	*pSecurityID	= NULL;
	*pID		= NULL;

	if (pNode == NULL)
		return;

	pChild = pNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "SecurityID", NULL );

	if (pChild)
	{
		*pSecurityID = pChild->GetData();

		pChild = pNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, "id", NULL );

		if (pChild)
		{
			*pID = pChild->GetData();
		}
	}
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_Secure_SXDB::GetSecurityID
	(
		CSXDB_Node	*pNode,				// The node to search
		const char	**pSecurityID,		// Returns the securityID, or NULL if none found
		const char	**pID				// Returns the id attribute found at the same
										// level as the securityID or NULL if none found
	)
//
//	Looks for the SecurityID="" attribute for the given node.
//	If the specified node does not have a SecurityID, then all of it's
//	ansestors are searched until one is found. IF none are found all the way
//	to the root of the document, then NULL is returned.
//
//------------------------------------------------------------------------------//
{
	*pSecurityID	= NULL;
	*pID			= NULL;

	if (pNode == NULL)
		return;

	do
	{
		GetSecurityIDForNode( pNode, pSecurityID, pID );

		if (*pSecurityID != NULL)
			return;

		pNode = pNode->Parent();
	} while (pNode != NULL);

	return;
}

