/**	
 *	@file	RDM_DDA_Device.h
 *	@brief	Public Device Data API (DDA)
 *
 *	Object that represents  .
 */

#ifndef _RDM_DDA_Device_h_
#define _RDM_DDA_Device_h_

#include "pp/RDM_Definitions.h"
#include "pp/RDM_BranchManager.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Device Class
 *--------------------------------------*/

class CDDA_Device : public CRDM_BranchManager
{
protected:
    /**
     * Private minimal constructor, used internally only by derived objects
     * that call initialize() themselves.
     * @param  pRDM            Ptr to RDM object
     */
     CDDA_Device(CRDM* pRDM);
     
public:

	/*  --------------------------------------------------------------------*/
	/** 
	 * @brief	Public constructor for a device
	 *
	 * Create a new device node in the database.
	 *
	 *  @param  pRDM	Ptr to RDM object
	 *  @param  pID         ID of this object
	 *  @param  pType	The device type ("IP-Reach",...)
	 *  @param  pModel	The model ("DKX","KX101",...)
	 *  @param  pName	The name of the device
	 *  @param  baseDevice  set to 0 if this is a subdevice
	 */
	CDDA_Device(CRDM * pRDM, const char* pID, const char *pType, const char *pModel, const char *pName, int basedevice = 1);
	~CDDA_Device();

protected:
    /**
     * Internal code of public constructor, used for decoupled initialization
     * in subclasses. Must be called in public constructors.
     * @see CDDA_Device
     */
    void initialize(CRDM * pRDM, const char* pID, const char *pType, const char *pModel, const char *pName, int basedevice);

public:
	/*  --------------------------------------------------------------------*/
	/** 
	 *	Functions to change device attributes.
	 *  Must call BeginUpdate() before changing attributes and EndUpdate()
	 *  when finished changing attributes.
	 *  BeginUpdate() locks the database and EndUpdate() unlocks the database
	 *  and then send the RDM events
	 * 
	 */

	void BeginUpdate();
	void SetName( const char *pName );
	void SetCalibrationSpeed( int speed );
	void EndUpdate();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Utility functions
	 * 
	 */

	const char *GetID();		///< Returns the port ID

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Called when data needs to be saved for the device
	 *
	 *	Called when there is an update to the writable device elements.
	 *  This function can be overridden and so the data can be stored in an
	 *  external config database.
	 *  The default implementation saves the data under the ID key int he 
	 *  RDM DeviceSettings branch.
	 *  CommitData() will be called after all data has been written.
	 * 
	 *  NOTE: Device only supports "Name" for now.
	 *
	 *	@param	pID				The ID of this port
	 *	@param	pKey			The data key name, such as "Name"
	 *	@param	pData			The data written
	 */

	virtual int  WriteData( const char *pID, const char *pKey, const char *pData );
	virtual void CommitData();


protected:

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Private function that implement the RDM branch manager interface.
	 * 
	 */

	virtual
	int									// 0 or Error code
	Update
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The Node to be appended too
		CSXDB_Node	*pData			// Ptr to data to append
	);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Private Data
	 * 
	 */

private:
	CRDM * pRDM;                 ///< Ptr to the RDM object
	int    sendEvents;           ///< !0 = Send events for changes
	char   id[RDM_MAX_ID];       ///< Our ID
	int    sendChgEvent;         ///< !0 = we need to send a device change event
	int    sendStatusEvent;      ///< !0 = we need to send a status event
	int    okToSendEvents;       ///< !0 = it's ok to send events
};

#endif // _RDM_DDA_Device_h_

