/**	
 *  @file	RFPPack.cpp
 *  @brief	Implementation of class CRFPPack class
 *
 *	
 */

#include	<stdio.h>
#include	<string.h>
#include	<assert.h>

#include <openssl/rsa.h>       /* SSLeay stuff */
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rc4.h>
#include <openssl/evp.h>

#include	<pp/syms.h>
#include	<pp/RFPPack.h>
#include	<pp/Scramble.h>

namespace pp
{

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

#define	RFP_BUF_SIZE	(RFP_MAX_HEADER_SIZE + RFP_MAX_SIGNATURE_SIZE)

#define	RFP_SERVER_PUBLIC_KEY_FILE	"RFPRSAKEY.PEM"
#define	RFP_SERVER_PRIVATE_KEY_FILE	"RFPRSAKEY_PRIVATE.PEM"
#define	RFP_SERVER_RC4_FILE			"RFPRC4KEY.TXT"

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Code
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
CRFPPack::CRFPPack(  )
{
	pHeaderText = NULL;
	SetKeyFiles(NULL,NULL);
}

/*  --------------------------------------------------------------------*/
CRFPPack::~CRFPPack()
{
	if (pHeaderText != NULL)
		delete pHeaderText;
}

/*  --------------------------------------------------------------------*/
void CRFPPack::SetKeyFiles( char * pRSAKeyFile, char * pRC4KeyFile  )
{
	if (pRSAKeyFile == NULL)
		strcpy(this->rsaKeyFile,RFP_SERVER_PRIVATE_KEY_FILE);
	else
		strcpy(this->rsaKeyFile,pRSAKeyFile);

	if (pRC4KeyFile == NULL)
		strcpy(this->rc4KeyFile,RFP_SERVER_RC4_FILE);
	else
		strcpy(this->rc4KeyFile,pRC4KeyFile);
}

/*  --------------------------------------------------------------------*/
void CRFPPack::AddHeaderText( char * pText )
{
	int len = strlen(pText);
	char *p;

	if (pHeaderText != NULL)
	{
		len += strlen(pHeaderText);
		p = new char[len+1];
		if (p == NULL)
			return;
		strcpy(p,pHeaderText);
		strcat(p,pText);
		delete pHeaderText;
		pHeaderText = p;
	}
	else
	{
		pHeaderText = new char[len+1];
		if (pHeaderText == NULL)
			return;
		strcpy(pHeaderText,pText);
	}
}

/*  --------------------------------------------------------------------*/
int CRFPPack::CreateSchema( CSIO * pSIO )
{
	int error = ParseFile( pSIO );
	errorFile = GetFileCount()-1;
	return error;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::ValidateSchema( )
{
	RFP *	pRFP;
	RFP_FILE* pRFP_File;
	int		isSigned = 0;
	int		isEncrypted = 0;
	int		x;

	pRFP = GetHeader();

	errorFile = -1;
	if (GetFileCount() < 1)
		return RFP_ERROR_NO_FILES;

    for (x=0;x<GetFileCount();x++)
	{
		pRFP_File = EnumFile(x);
		if (pRFP_File == NULL)
			return RFP_ERROR_INTERNAL_ERROR;

		errorFile = x;

		if (pRFP_File->maxLength && x < GetFileCount()-1)
			return RFP_ERROR_MAXLENGTH_NOT_LAST;

		if (pRFP_File->maxLength == 0 && !pRFP_File->lengthSpecified)
			return RFP_ERROR_NO_LENGTH;
		
		if (pRFP_File->maxLength != 0 && pRFP_File->lengthSpecified)
			return RFP_ERROR_MAXLENGTH_AND_LENGTH;
		
		isSigned |= pRFP_File->isSigned;
		isEncrypted |= pRFP_File->isEncrypted;
	}

	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::PackRFP( CSIO * pSIO )
{
	char		buffer[RFP_BUF_SIZE];
	int			count;
	char	*	p;
	int			x;
	RFP		*	pRFP = GetHeader();
	RFP_FILE *	pRFP_File;
	FILE	*	fp;
	int			xferCount;
	int			headerLength;
	int			totalLength = 0;
	int			sigLength;
	int			error = RFP_OK;
	unsigned char	keyText[256];
	int				keyTextLength;
	RC4_KEY			key;

 	// ---------------------------------
	// Check the schema

	error = ValidateSchema();
	if (error != RFP_OK)
		return error;

 	// ---------------------------------
	// Create the header

	errorFile = -1;

	p = buffer;

	p += sprintf(p,"<?xml version=\"1.0\"?>\n");
	p += sprintf(p,"<RFP>\n"/*  --------------------------------------------------------------------*/);
	p += sprintf(p," <Title>%s</Title>\n", pRFP->title);
	p += sprintf(p," <Description>%s</Description>\n", pRFP->description);
	p += sprintf(p," <Copyright>%s</Copyright>\n", pRFP->copyRight);
	p += sprintf(p," <Publisher>%s</Publisher>\n", pRFP->publisher);
	p += sprintf(p," <Model>%s</Model>\n", pRFP->model);
	p += sprintf(p," <Version>%s</Version>\n", pRFP->version);
	p += sprintf(p," <VersionMin>%s</VersionMin>\n", pRFP->versionMin);
	p += sprintf(p," <VersionMax>%s</VersionMax>\n", pRFP->versionMax);
	if (pRFP->isSigned)
		p += sprintf(p," <Signed/>\n");

	p += sprintf(p," <SupportedDensity>%s</SupportedDensity>\n", pRFP->supportedDensity);
	p += sprintf(p," <SupportedHardware>%s</SupportedHardware>\n", pRFP->supportedHardware);

	if (pHeaderText != NULL)
		p += sprintf(p,pHeaderText);

	if (pRFP->tags != NULL)
		p += sprintf(p,pRFP->tags);

	for (x=0;x<GetFileCount();x++)
	{
		errorFile = x;
		pRFP_File = EnumFile(x);
		p += sprintf(p," <RFP_File>\n");

		// Insert file name tag

		if (pRFP_File->isMetaFile)
		{
			if (pRFP_File->sourceFile[0] != 0)
				strcpy((char *) &pRFP_File->resolvedFileName,pRFP_File->sourceFile);
			else
			{
				error = ResolveMetaFile( (char*) pRFP_File->metaFileName, (char *) pRFP_File->resolvedFileName );
				if (error != RFP_OK)
					goto EXIT;
			}
			p += sprintf(p,"  <MetaFileName>%s</MetaFileName>\n",pRFP_File->metaFileName);
		}
		else
		{
			p += sprintf(p,"  <FileName>%s</FileName>\n",pRFP_File->fileName);
			if (pRFP_File->sourceFile[0] != 0)
				strcpy((char *) pRFP_File->resolvedFileName,pRFP_File->sourceFile);
			else
				strcpy((char *) pRFP_File->resolvedFileName,pRFP_File->fileName);
		}

		// DBLog(("  RFP Send: sending file %s\n", realFileName[x]));

		// Insert length tag

		if (pRFP_File->maxLength)
			p += sprintf(p,"  <MaxLength>%d</MaxLength>\n",pRFP_File->maxLength);
		else
		{
			pRFP_File->length = 0;
			fp = fopen( (char *) pRFP_File->resolvedFileName, "rb");
			if (fp != NULL)
			{
				fseek(fp, 0, SEEK_END);
				pRFP_File->length = ftell(fp);
				fclose( fp );
			}
			else
			{
				// DBLog(("  RFP Send: requested file %s not found\n", realFileName[x]));
				error = RFP_ERROR_FILE_NOT_FOUND;
				goto EXIT;
			}
	
			p += sprintf(p,"  <Length>%d</Length>\n",pRFP_File->length);
			totalLength += pRFP_File->length;
		}

		// Other tags

		if (pRFP_File->isScript)
			p += sprintf(p,"  <Script>%s</Script>\n",pRFP_File->script);
		if (pRFP_File->isSigned)
		{
			p += sprintf(p,"  <Signature>\n");
			error = SignFile( (char *) pRFP_File->resolvedFileName,pRFP_File->rsaKeyFile[0] != 0 ? pRFP_File->rsaKeyFile : this->rsaKeyFile,p,&sigLength);
			if (error != RFP_OK)
				goto EXIT;
			p += sigLength;
			p += sprintf(p,"  </Signature>\n");
		}
		if (pRFP_File->isEncrypted)
			p += sprintf(p,"  <Encrypted/>\n");
		if (strlen(pRFP_File->model) > 0)
			p += sprintf(p,"  <Model>%s</Model>\n",pRFP_File->model);
		if (strlen(pRFP_File->component) > 0)
			p += sprintf(p,"  <Component>%s</Component>\n",pRFP_File->component);
		if (strlen(pRFP_File->version) > 0)
			p += sprintf(p,"  <Version>%s</Version>\n",pRFP_File->version);
		if (strlen(pRFP_File->versionMin) > 0)
			p += sprintf(p,"  <VersionMin>%s</VersionMin>\n",pRFP_File->versionMin);
		if (strlen(pRFP_File->versionMin) > 0)
			p += sprintf(p,"  <VersionMax>%s</VersionMax>\n",pRFP_File->versionMax);
		if (pRFP_File->isMemoryFile)
			p += sprintf(p,"  <MemoryFile/>\n");

		if (pRFP_File->tags != NULL)
			p += sprintf(p,pRFP_File->tags);

		// End of tag

		p += sprintf(p," </RFP_File>\n");
	}

	p += sprintf(p,"</RFP>\n");
	p += sprintf(p,"\n%c",0);

 	// ---------------------------------
	// Create the header signature

	if (pRFP->isSigned)
	{
		headerLength = p - buffer;
		error = SignBuffer(buffer,headerLength,pRFP->rsaKeyFile[0] != 0 ? (char *)pRFP->rsaKeyFile : (char *)this->rsaKeyFile,p,&sigLength);
		if (error != RFP_OK)
			goto EXIT;
		p += sigLength;
		*p++ = 0; // null terminator
	}

	headerLength = p - buffer;
	totalLength += headerLength;

 	// ---------------------------------
	// Output the header

	error = RFPOutput(pSIO,buffer,headerLength,totalLength);
	if (error != RFP_OK)
		goto EXIT;

 	// ---------------------------------
	// Send the data for each fileCRFPPack::

	for (x=0;x<GetFileCount();x++)
	{
		errorFile = x;
		pRFP_File = EnumFile(x);

		if (pRFP_File->length == 0)
			continue;

		// Initialize the key

		if (pRFP_File->isEncrypted)
		{
			fp = fopen(pRFP_File->rc4KeyFile[0] != 0 ? pRFP_File->rc4KeyFile : this->rc4KeyFile,"rb");
			if (fp == NULL)
				return RFP_ERROR_RC4_KEY_MISSING;
			keyTextLength = fread(keyText,1,256,fp);
			fclose( fp );
			if (keyTextLength < 1)
				return RFP_ERROR_RC4_KEY_MISSING;
			Scramble( (char *) keyText, keyTextLength );
			RC4_set_key( &key, keyTextLength, &keyText[0] );
		}

		// Open the file

		fp = fopen((char *) pRFP_File->resolvedFileName, "rb");
		if (fp == NULL)
		{
			error = RFP_ERROR_FILE_NOT_FOUND;
			goto EXIT;
		}

		// Send data

		for (xferCount = pRFP_File->length; xferCount > 0; )
		{
			count = fread( buffer,1,RFP_BUF_SIZE,fp );

			if (count <= 0)
			{
				// DBLog(("  RFP Send: error reading file\n"));
				error = RFP_ERROR_FILE_NOT_FOUND;
				goto EXIT;
			}

			xferCount -= count;

			if (xferCount < 0)
			{
				// DBLog(("  RFP Send: file size changed while reading\n"));
				error = RFP_ERROR_FILE_NOT_FOUND;
				goto EXIT;
			}

			if (pRFP_File->isEncrypted)
				RC4( &key, count, (unsigned char *) buffer, (unsigned char *) buffer );

			error = RFPOutput(pSIO,buffer,count,totalLength);
			if (error != RFP_OK)
				goto EXIT;
		}

		// Clean up after meta files

		if (pRFP_File->isMetaFile)
			ReleaseMetaFile( pRFP_File->metaFileName, pRFP_File->resolvedFileName );

		fclose(fp);

		// hack used to send DDA_Notify_DeviceConfig_Backup_Restore()

	}

	// Tell client we are done

	error = RFPOutput(pSIO,NULL,0,totalLength);
	if (error != RFP_OK)
		goto EXIT;

	// Done !!!!

EXIT:

	return error;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::RFPOutput( CSIO * pSIO, char * pData, int dataLength, int NOTUSED(totalLength) )
{
	int	error = RFP_ERROR_CANNOT_WRITE_FILE;
	if (pSIO != NULL)
	{
		int result = pSIO->Write(pData,dataLength);
		if (result >= 0)
			error = RFP_OK;
	}

	return error;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::ResolveMetaFile( char * NOTUSED(pMetaFileName), char * NOTUSED(pOutRealFileName) )
{
	return RFP_ERROR_BAD_METAFILE;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::ReleaseMetaFile( char * NOTUSED(pMetaFileName), char * NOTUSED(pResolvedFileName) )
{
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::SignFile( char * pFileName, char *pKeyFileName, char * pOutBuffer, int *rtnLength )
{
	int err;
	unsigned int sig_len;
	unsigned char sig_buf [4096];
	unsigned char bioMem[4096];
	EVP_MD_CTX		md_ctx;
	EVP_PKEY *      pkey;
	FILE *          fpKey;
	FILE *			fpIn;
	int				result = -1;

	// Just load the crypto library error strings,
	// SSL_load_error_strings() loads the crypto AND the SSL ones 
	// SSL_load_error_strings();
	ERR_load_crypto_strings();

	// Read private key 

	fpKey = fopen (pKeyFileName, "rb");
	if (fpKey == NULL) 
		return RFP_ERROR_PUBLIC_KEY_MISSING;

	pkey = PEM_read_PrivateKey(fpKey, NULL, NULL, NULL);

	fclose (fpKey);

	if (pkey == NULL)
	{ 
		ERR_print_errors_fp (stderr);
		return RFP_ERROR_PUBLIC_KEY_MISSING;
	}

	// Do the signature

	fpIn = fopen(pFileName,"rb");
	if (fpIn == NULL)
		return RFP_ERROR_CANNOT_READ_FILE;

	EVP_SignInit   (&md_ctx, EVP_sha1());

	do
	{
		result = fread(bioMem,1,4096,fpIn);

		if (result > 0)
			EVP_SignUpdate (&md_ctx, bioMem, result);

	} while (result == 4096);

	sig_len = sizeof(sig_buf);
	err = EVP_SignFinal (&md_ctx, sig_buf, &sig_len, pkey);

	if (err != 1)
	{
		ERR_print_errors_fp(stderr);
		return RFP_ERROR_INTERNAL_ERROR;
	}

	fclose(fpIn);

	// Write the PEM signature to the buffer

	BIO *bio = BIO_new(BIO_s_mem());
	char hdr[] = "";/* use non-const buffer because of broken openssl API */
	result = PEM_write_bio( bio, "SIGNATURE", hdr, sig_buf, sig_len );
	BIO_seek(bio,0);
	result = BIO_read(bio,pOutBuffer,4096);
	BIO_free(bio);

	if (rtnLength != NULL)
		*rtnLength = result;

	EVP_PKEY_free (pkey);

	return result >=0 ? 0 : RFP_ERROR_INTERNAL_ERROR;
}

/*  --------------------------------------------------------------------*/
int CRFPPack::SignBuffer( char * pInBuffer, int length, char *pKeyFileName, char * pOutBuffer, int *rtnLength )
{
	int err;
	unsigned int sig_len;
	unsigned char sig_buf [4096];
	EVP_MD_CTX     md_ctx;
	EVP_PKEY *      pkey;
	FILE *          fpKey;
	int				result = -1;

	// Just load the crypto library error strings,
	// SSL_load_error_strings() loads the crypto AND the SSL ones 
	// SSL_load_error_strings();
	ERR_load_crypto_strings();

	// Read private key 

	fpKey = fopen (pKeyFileName, "rb");
	if (fpKey == NULL) 
		return RFP_ERROR_PUBLIC_KEY_MISSING;

	pkey = PEM_read_PrivateKey(fpKey, NULL, NULL, NULL);

	fclose (fpKey);

	if (pkey == NULL)
	{ 
		ERR_print_errors_fp (stderr);
		return RFP_ERROR_PUBLIC_KEY_MISSING;
	}

	// Do the signature

	EVP_SignInit   (&md_ctx, EVP_sha1());
	EVP_SignUpdate (&md_ctx, pInBuffer, length);
	sig_len = sizeof(sig_buf);
	err = EVP_SignFinal (&md_ctx, sig_buf, &sig_len, pkey);

	if (err != 1)
	{
		ERR_print_errors_fp(stderr);
		return RFP_ERROR_INTERNAL_ERROR;
	}

	// Write the PEM signature to the buffer

	BIO *bio = BIO_new(BIO_s_mem());
	char hdr[] = "";/* use non-const buffer because of broken openssl API */
	result = PEM_write_bio( bio, "SIGNATURE", hdr, sig_buf, sig_len );
	BIO_seek(bio,0);
	result = BIO_read(bio,pOutBuffer,4096);
	BIO_free(bio);

	if (rtnLength != NULL)
		*rtnLength = result;

	EVP_PKEY_free (pkey);

	return result >=0 ? 0 : RFP_ERROR_INTERNAL_ERROR;
}

} // namespace


