/*--------------------------------------------------------------------------------

	RFPParser.cpp

	Copyright (c) 2002, Raritan Computer, Inc.

	Parser for RFP File Headers.
	This parser is based on the SXML XML parser class.

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	"pp/RFPParser.h"

namespace pp
{

//----------------------------------------
//				Equates
//----------------------------------------

	// Parser Finite State Machine States (FSM)

enum
{
	RFP_STATE_NULL = 0,					// Starting point
	RFP_STATE_RFP,						// RFP Document
	RFP_STATE_RFP_File,					// <RFP_File> tag
	RFP_STATE_STRING_CAT				// Append a string
};

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									Methods
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRFPParser::CRFPParser
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	memset(&rfp,0,sizeof(rfp));
	filePackageName = NULL;
	copyTags	  = 0;
}

//--------------------------------------------------------------------------------
//
	CRFPParser::~CRFPParser
	(
	)
//
//	Clean up
//
//------------------------------------------------------------------------------//
{
	Free();
}

//--------------------------------------------------------------------------------
//
	int									// Error Code
	CRFPParser::ParseHeader
	(
		char		*	pHeader			// Ptr to the NULL terminated header data
	)
//
//	Parses the RFP header.
//
//------------------------------------------------------------------------------//
{
	int	result;

	Free();

	// Parse the header

	result = Parse( pHeader, strlen( pHeader ) );

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// Error Code
	CRFPParser::ParseFile
	(
		CSIO		*	pSIO,			// SIO to read the data from
		int			*	pHeaderOffset,	// Returns offset into file of the header (usually 0)
		int			*	pHeaderLength,	// Returns length of the header
		int			*	pSignatureOffset,// Returns offset to the signature
		int			*	pSignatureLength,// Returns length of the signature (0 if no signature)
		int			*	pDataOffset		// Returns the offset to the first data file
	)
//
//	Identifies the header and header signature in a file and then parses the header
//
//------------------------------------------------------------------------------//
{
	char	*	header;
	int			result;
	int			length;

	// Clear everything

	Free();

	if (pHeaderOffset != NULL)
		*pHeaderOffset		= 0;
	if (pHeaderLength != NULL)
		*pHeaderLength		= 0;
	if (pSignatureOffset != NULL)
		*pSignatureOffset	= 0;
	if (pSignatureLength != NULL)
		*pSignatureLength	= 0;
	if (pDataOffset != NULL)
		*pDataOffset		= 0;

	// Allocate Buffer space

	header = (char *) malloc( RFP_MAX_HEADER_SIZE+RFP_MAX_SIGNATURE_SIZE );

	if (header == NULL)
		return RFP_ERROR_MEMORY;

	// Read enough of the file to get the header and signature

	length = pSIO->Read( header, RFP_MAX_HEADER_SIZE+RFP_MAX_SIGNATURE_SIZE );

	if (length <= 0)
	{
		result = RFP_ERROR_CANNOT_READ_FILE;
		goto EXIT;
	}

	// Scan for header and signature

	result = ParseHeaderAndSignature( header, length, pHeaderOffset, pHeaderLength, pSignatureOffset, pSignatureLength, pDataOffset );

	if (result != 0)
		goto EXIT;

	// Return the results

EXIT:
	free(header);
	return result;
}

//--------------------------------------------------------------------------------
//
	int									// Error Code
	CRFPParser::ParseHeaderAndSignature
	(
		char		*	pData,			// Ptr to the data (NULL Termincated)
		int				length,			// length of the data
		int			*	pHeaderOffset,	// Returns offset into file of the header (usually 0)
		int			*	pHeaderLength,	// Returns length of the header
		int			*	pSignatureOffset,// Returns offset to the signature
		int			*	pSignatureLength,// Returns length of the signature (0 if no signature)
		int			*	pDataOffset		// Returns the offset to the first data file
	)
//
//	Identifies the header and header signature in a buffer and then parses the header
//
//------------------------------------------------------------------------------//
{
	char	*	p;
	int			result;
	int			dataOffset;
	int			headerLength;
	int			signatureLength;

	// Clear everything

	if (pHeaderOffset != NULL)
		*pHeaderOffset		= 0;
	if (pHeaderLength != NULL)
		*pHeaderLength		= 0;
	if (pSignatureOffset != NULL)
		*pSignatureOffset	= 0;
	if (pSignatureLength != NULL)
		*pSignatureLength	= 0;
	if (pDataOffset != NULL)
		*pDataOffset		= 0;

	// Scan for header and signature

	p = pData;
	headerLength = 0;

	while (*p && headerLength < RFP_MAX_HEADER_SIZE && headerLength < length)
	{
		p++;
		headerLength ++;
	}

	headerLength++;
	p++;

	// Parse the header

	result = ParseHeader( pData );

	if (result != 0)
		goto EXIT;

	// Find signature

	signatureLength = 0;

	if (rfp.isSigned)
	{
		while (*p && signatureLength < RFP_MAX_SIGNATURE_SIZE && headerLength + signatureLength < length)
		{
			p++;
			signatureLength++;
		}
		signatureLength++;
		p++;
	}

	dataOffset = headerLength + signatureLength;


	// Return the results

	if (pHeaderOffset != NULL)
		*pHeaderOffset		= 0;
	if (pHeaderLength != NULL)
		*pHeaderLength		= headerLength;
	if (pSignatureOffset != NULL)
		*pSignatureOffset	= headerLength;
	if (pSignatureLength != NULL)
		*pSignatureLength	= signatureLength;
	if (pDataOffset != NULL)
		*pDataOffset		= dataOffset;

EXIT:
	return result;
}

//--------------------------------------------------------------------------------
//
	void
	CRFPParser::Free
	(
	)
//
//	Frees all memory allocated by the ParseHeader function.
//
//------------------------------------------------------------------------------//
{
	int		x;

	for (x=0;x<fileCount;x++)
	{
		if (rfpFile[x]->tags != NULL)
			delete rfpFile[x]->tags;
		delete rfpFile[x];
		rfpFile[x] = NULL;
	}

	if (rfp.tags != NULL)
		delete rfp.tags;

	memset(&rfp,0,sizeof(rfp));
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::VerifyHeaderSignature
	(
		char		*	pHeader,		// Ptr to the header data, NULL terminated
		char		*	pSignature,		// Ptr to the signature data (null terminated)
		char		*	pPublicKey		// Ptr File name of the public key in PEM format
	)
//
//	Verifies the header signature.
//
//------------------------------------------------------------------------------//
{
	int		result;

	if (rfp.isSigned == 0)
		return 0;

	CSIO_SMEM sio(pHeader,strlen(pHeader)+1,0);

	result = VerifySignature( &sio, pSignature, pPublicKey );

	return result;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::VerifyFileSignatures
	(
		char		*	pPublicKey		// Ptr File name of the public key in PEM format
	)
//
//	Verifies the signature.for all files. It assumes the same public key is used
//	for all of the files.
//
//------------------------------------------------------------------------------//
{
	int	x;
	int	result = 0;

	for (x=0; x<fileCount; x++)
	{
		result = VerifyFileSignature( rfpFile[x], NULL, pPublicKey );

		if (result != 0)
			break;
	}

	return result;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::VerifyFileSignature
	(
		RFP_FILE	*	pRFPFile,		// Description of file to check
		char		*	pTempFileName,	// Ptr to the temp file name or NULL to use actual name in pRFPFile
		char		*	pPublicKey		// Ptr File name of the public key in PEM format
	)
//
//	Verifies the signature.for a single file.
//
//------------------------------------------------------------------------------//
{
	CSIO_FP	sio;
	int		result;

	if (pRFPFile->isSigned == 0)
		return 0;

	if (pTempFileName == NULL)
		pTempFileName = pRFPFile->resolvedFileName;

	result = sio.Open(pTempFileName, "rb");

	if (result < 0)
		return RFP_ERROR_CANNOT_READ_FILE;

	result = VerifySignature( &sio, pRFPFile->signature, pPublicKey );

	sio.Close();

	return result;
}

//--------------------------------------------------------------------------------
//
	RFP_FILE *							// Ptr the file data or NULL
	CRFPParser::EnumFile
	(
		int				index			// 0 based index in to files
	)
//
//	Returns a ptr to the file data or NULL if the index > the # of files.
//
//------------------------------------------------------------------------------//
{
	if (index < fileCount && index >= 0)
		return rfpFile[index];

	return NULL;
}

//--------------------------------------------------------------------------------
//
	RFP *
	CRFPParser::GetHeader
	(
	)
//
//	Returns a ptr to the RFP struct for the header
//
//------------------------------------------------------------------------------//
{
	return &rfp;
}

//--------------------------------------------------------------------------------
//							Private Methods
//							Private Methods
//							Private Methods
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::VerifySignature
	(
		CSIO		*	pSIO,			// Data source
		char		*	pSignature,		// Ptr to the signature data (null terminated)
		char		*	pPublicKey		// Ptr File name of the public key in PEM format
	)
//
//	Verifies the signature.for a single file.
//
//------------------------------------------------------------------------------//
{
	EVP_MD_CTX			md_ctx;
	EVP_PKEY	*		pkey;
	FILE		*		fp;
	X509		*		x509;
	int					result;
	int					total = 0;
	char				data[1024];
	BIO			*		bpSig;
	char		*		pName;
	char		*		pPEMHeader;
	unsigned char *		pSig;
	long				sigLen;

	// Read public key 
  
	fp = fopen (pPublicKey, "rb");
	if (fp == NULL) 
		return RFP_ERROR_PUBLIC_KEY_MISSING;

	x509 = PEM_read_X509(fp, NULL, NULL, NULL);
	fclose (fp);

	if (x509 == NULL) 
		return RFP_ERROR_RSA_KEY_BAD;

	// Get public key - eay 
	pkey=X509_get_pubkey(x509);
	if (pkey == NULL)
		return RFP_ERROR_RSA_KEY_BAD;

	// Create a BIO for the signature & read it in

	bpSig = BIO_new_mem_buf( pSignature, strlen(pSignature) );

	if (bpSig == NULL)
		return RFP_ERROR_MEMORY;

	result = PEM_read_bio( bpSig, &pName, &pPEMHeader, &pSig, &sigLen );

	BIO_free( bpSig );

	if ( result != 1)
		return RFP_ERROR_BAD_SIGNATURE;

	// Verify the signature

	EVP_VerifyInit   (&md_ctx, EVP_sha1());

	pSIO->Seek(0,SEEK_SET);

	do
	{
		result = pSIO->Read(data,1024);

		if (result > 0)
		{
			EVP_VerifyUpdate ( &md_ctx, data, result );
			total += result;
		}

	} while (result == 1024);

	result = EVP_VerifyFinal (&md_ctx, pSig, sigLen, pkey);

	EVP_PKEY_free (pkey);

	OPENSSL_free(pName);
	OPENSSL_free(pPEMHeader);
	OPENSSL_free(pSig);

	if (result != 1)
		return RFP_ERROR_BAD_SIGNATURE;

	return 0;
}



//--------------------------------------------------------------------------------
//
	int
	CRFPParser::Event
	(
		int			event
	)
//
//	Overrides the SXML Event handler.
//
//------------------------------------------------------------------------------//
{
	if (event == SXML_EVENT_START_DOCUMENT)
		state = RFP_STATE_NULL;

	switch (state)
	{
		case RFP_STATE_NULL:		return State_Null(event);			break;
		case RFP_STATE_RFP:			return State_RFP(event);			break;
		case RFP_STATE_RFP_File:	return State_RFP_File(event);		break;
		case RFP_STATE_STRING_CAT:	return State_String_Cat(event);		break;

		default:
			return HelperStates(event);
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::State_Null
	(
		int			event
	)
//
//	Starting point for RFP document.
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_START_TAG:

			if( filePackageName != NULL ){
				SXML_Tag_State( filePackageName, RFP_STATE_RFP );
			}

			SXML_Tag_State( "Raritan_File_Package", RFP_STATE_RFP );
			SXML_Tag_State( "RFP", RFP_STATE_RFP );

			// No tags match...

			SkipTag();
			
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::State_RFP
	(
		int			event
	)
//
//	Parses sub tags of <Raritan_File_Package>
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_String		( "Title",					&rfp.title,				RFP_MAX_TITLE );
			SXML_String		( "Description",			&rfp.description,		RFP_MAX_DESCRIPTION );
			SXML_String		( "Copyright",				&rfp.copyRight,			RFP_MAX_COPYRIGHT );
			SXML_String		( "Publisher",				&rfp.publisher,			RFP_MAX_PUBLISHER );
			SXML_Int		( "AutoRestart",			&rfp.autoRestart);
			SXML_Int		( "DisableOtherSessions",	&rfp.disableOtherSessions);
			SXML_Tag_Action	( "Signed",					rfp.isSigned = 1; 		SkipTag(); );
			SXML_String		( "SupportedDensity",		&rfp.supportedDensity,	RFP_MAX_SUPPORTED_DENSITY );
			SXML_String		( "SupportedHardware",		&rfp.supportedHardware,	RFP_MAX_SUPPORTED_HARDWARE );
			SXML_String		( "Model",					&rfp.model,				RFP_MAX_MODEL );
			SXML_String		( "Version",				&rfp.version,			RFP_MAX_VERSION );
			SXML_String		( "VersionMin",				&rfp.versionMin,		RFP_MAX_VERSIONMIN );
			SXML_String		( "VersionMax",				&rfp.versionMax,		RFP_MAX_VERSIONMAX );
			SXML_String		( "RSAKeyFile",				&rfp.rsaKeyFile,		RFP_MAX_FILENAME );
			SXML_Tag_State	( "RFP_File",				RFP_STATE_RFP_File		);

			// Unknown tag...

			if (!copyTags)
			{
				SkipTag();
			}
			else
			{
				// Append unknown tag and it's data to rfp.tags string
				ppStr = &rfp.tags;
				prevState = state;
				state = RFP_STATE_STRING_CAT;
				return Event( SXML_EVENT_INIT_STATE );
			}

			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::State_RFP_File
	(
		int			event
	)
//
//	Parses sub tags of <RFP_File>
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_INIT_STATE:

			// Create a new RFP_File node

			if (fileCount == RFP_MAX_FILES)
				return RFP_ERROR_TOO_MANY_FILES;

			rfpFile[fileCount] = new RFP_FILE;

			if (rfpFile[fileCount] == NULL)
				return RFP_ERROR_MEMORY;

			memset(rfpFile[fileCount], 0, sizeof(RFP_FILE));

			pCurFile = rfpFile[fileCount];

			fileCount++;

			break;

		case SXML_EVENT_START_TAG:

			SXML_String		( "FileName",		&pCurFile->fileName,	RFP_MAX_FILENAME );
			SXML_String_Action( "MetaFileName",	&pCurFile->metaFileName, RFP_MAX_FILENAME, pCurFile->isMetaFile = 1 );
			SXML_String		( "SourceFile",		&pCurFile->sourceFile,	RFP_MAX_FILENAME );
			SXML_String_Action( "Script",		&pCurFile->script,		RFP_MAX_SCRIPT, pCurFile->isScript = 1 );
			SXML_String_Action( "Signature",	&pCurFile->signature,	RFP_MAX_SIGNATURE, pCurFile->isSigned = 1 );
			SXML_Tag_Action	( "MemoryFile",		pCurFile->isMemoryFile = 1; 	SkipTag(); );
			SXML_String		( "Model",			&pCurFile->model,		RFP_MAX_MODEL );
			SXML_String		( "Component",		&pCurFile->component,	RFP_MAX_COMPONENT );
			SXML_String		( "Version",		&pCurFile->version,		RFP_MAX_VERSION );
			SXML_String		( "VersionMin",		&pCurFile->versionMin,	RFP_MAX_VERSIONMIN );
			SXML_String		( "VersionMax",		&pCurFile->versionMax,	RFP_MAX_VERSIONMAX );
			SXML_Int_Action	( "Length",			&pCurFile->length,		pCurFile->lengthSpecified = 1 );
			SXML_Int		( "MaxLength",		&pCurFile->maxLength );
			SXML_Tag_Action	( "Encrypted",		pCurFile->isEncrypted = 1; SkipTag() );

			SXML_String		( "RC4KeyFile",		&pCurFile->rc4KeyFile,	RFP_MAX_FILENAME );
			SXML_String		( "RSAKeyFile",		&pCurFile->rsaKeyFile,	RFP_MAX_FILENAME );


			// Unknown tag...

			if (!copyTags)
			{
				SkipTag();
			}
			else
			{
				// Append unknown tag and it's data to rfp.tags string
				ppStr = &pCurFile->tags;
				prevState = state;
				state = RFP_STATE_STRING_CAT;
				return Event( SXML_EVENT_INIT_STATE );
			}

			break;

		case SXML_EVENT_END_TAG:
			state = RFP_STATE_RFP;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CRFPParser::State_String_Cat
	(
		int			event
	)
//
//	Parses a string and concatinates it to a previous string.
//	Uses strPtr, strLen, and prevState
//
//------------------------------------------------------------------------------//
{
	char	empty[] = "";
	char	*pOld,*pNew;

	switch (event)
	{
		case SXML_EVENT_START_TAG:
			SkipTag();
			break;

		case SXML_EVENT_DATA:

			// Append unknown tag and it's data to rfp.tags string
			pOld = *ppStr;
			if (pOld == NULL)
				pOld = empty;
			pNew = new char[strlen(pOld) + strlen(GetName()) * 2 + GetDataLength() + 10];
			if (pNew == NULL)
				return RFP_ERROR_MEMORY;
			sprintf(pNew,"%s <%s>%s</%s>\n",pOld,GetName(),GetData(),GetName());
			if (*ppStr != NULL)
				delete *ppStr;
			*ppStr = pNew;

			SkipTag();
			// Fall through to end tag

		case SXML_EVENT_END_TAG:
			state = prevState;
			break;
	}
	
	return 0;
}

} // namespace


