/**	
 *	@file	RFP_Protocol.cpp
 *	@brief	Implementation of CRFP_Protocol_Factory & CRFP_Protocol
 *
 */

#include <assert.h>
#include <pp/syms.h>
#include <pp/OS_Port.h>
#include <pp/SIO.h>
#include <pp/Session.h>
#include <pp/NetConn.h>
#include <pp/ppRFPPack.h>
#include <pp/ppRFPUnpack.h>
#include <pp/RFP_Protocol.h>

namespace pp
{

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

#define	DBLog(...)    printf(__VA_ARGS__)
#define DBLog2(...)   printf(__VA_ARGS__)
#define DBLog3(...)   printf(__VA_ARGS__)

#define RFP_SEND_DATA_TIMEOUT   (120000)        // 120 seconds
#define RFP_RCV_DATA_TIMEOUT    (120000)        // 120 seconds

#define	RFP_MESSAGE_COMMAND	0x35		// TRSP Req Command for RFP

/** RFP Protocol State machine sates */

enum
{
	RFP_SERVER_STATE_NULL,				// Nothing is going on
	RFP_SERVER_STATE_RECEIVE,			// Recieving an RFP
	RFP_SERVER_STATE_REQUEST,			// Processing a request
};

/** RFP Transfer Phases  These are events into the RFP client/server FSM */

enum
{
	TR_RFP_REQUEST_START		= 0,	// Request an RFP transfer
	TR_RFP_REQUEST_DATA			= 1,	// Continuing request header data
	TR_RFP_REQUEST_DONE			= 2,	// Request is done.
	TR_RFP_START				= 1,	// Ready to send data
	TR_RFP_DATA					= 2,	// Transmitting RFP DATA
	TR_RFP_DONE					= 3,	// File is done,
	TR_RFP_CANCEL				= 4,	// Cancel the transfer
	TR_RFP_RESULT				= 5,	// The result of the transfer

	TRCMD_RFP_MESSAGE			= 0x35, // Send RFP data
	TRRSP_RFP_MESSAGE			= 0x23, // Send RFP data
};

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

DWORD RFP_Protocol_Thread(LPVOID param);
DWORD RFP_Unpack_Thread(LPVOID param);

/*----------------------------------------
 *	static data
 *--------------------------------------*/

/*----------------------------------------
 *	Local Derived version of ppRDMUnpack
 *	This derived class gets data from the 
 *	RFP protocol stream. It also
 *	runs asynchronously in it's own
 *  Thread.
 *--------------------------------------*/

class RFP_Protocol_Unpack : public ppRFPUnpack
{
public:
	RFP_Protocol_Unpack();
	virtual ~RFP_Protocol_Unpack();
	virtual int RFPInput( CSIO * pSIO, char * pData, int dataLength, int *pBytesRead );
	void DataReady( char * pData, int length );
	int  WaitForDataDone( );
	void Cancel( int error );
	int AsyncUnpack();
	int AsyncFinish();

	OS_THREAD	hThread;				// Handle to our async processing thread
	OS_EVENT	dataReady;				// Set when new data is available
	OS_EVENT	dataDone;				// Set when all data has been consumed
	int			bufferCount;			// # of bytes of data left
	char *		pBuffer;	 		   	// Ptr to the data;
	int			error;					// If there was an error, it lives here
};

/*  --------------------------------------------------------------------*/
RFP_Protocol_Unpack::RFP_Protocol_Unpack()
{
	bufferCount		= 0;
	error			= RFP_OK;
	dataReady		= OS_CreateEvent( OS_EVENT_NORMAL );
	dataDone		= OS_CreateEvent( OS_EVENT_NORMAL );
}

/*  --------------------------------------------------------------------*/
RFP_Protocol_Unpack::~RFP_Protocol_Unpack() 
{
	OS_DeleteEvent(dataReady);
	OS_DeleteEvent(dataDone);
	if (hThread != NULL)
	{
		OS_WaitForEvent( hThread, RFP_SEND_DATA_TIMEOUT );
		OS_DeleteThread( hThread );
	}
}

/*  --------------------------------------------------------------------*/
int RFP_Protocol_Unpack::RFPInput( CSIO * NOTUSED(pSIO), char * pData, int dataLength, int *pBytesRead )
{
	int	count;
	int	total = 0;
	int	result;

	while (total < dataLength)
	{
		if (error != RFP_OK)
			return error;
		if (bufferCount)
		{
			if (bufferCount > dataLength - total)
				count = dataLength - total;
			else
				count = bufferCount;
			memcpy(pData,pBuffer,count);
			pData		+= count;
			pBuffer		+= count;
			total		+= count;
			bufferCount -= count;
		}
		else
		{
			OS_SetEvent( dataDone );
			result = OS_WaitForEvent( dataReady, RFP_RCV_DATA_TIMEOUT );
			if ((DWORD) result == OS_ERROR_TIMEOUT)
				return (error = RFP_ERROR_TIMEOUT);
			if (bufferCount == 0)
			{
				*pBytesRead = 0;
				return error;
			}
		}
	}

	*pBytesRead = total;
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
void RFP_Protocol_Unpack::DataReady( char * pData, int length ) 
{
	this->pBuffer = pData; 
	this->bufferCount = length; 
	OS_SetEvent( dataReady ); 
}

/*  --------------------------------------------------------------------*/
int  RFP_Protocol_Unpack::WaitForDataDone( ) 
{
	if (error != RFP_OK) return error;
	DWORD result = OS_WaitForEvent( dataDone, RFP_SEND_DATA_TIMEOUT );
	if (result == OS_ERROR_TIMEOUT)
		error = RFP_ERROR_INTERNAL;
	return error;
}

/*  --------------------------------------------------------------------*/
void RFP_Protocol_Unpack::Cancel( int newError ) 
{
	error = newError;
}

/*  --------------------------------------------------------------------*/
int RFP_Protocol_Unpack::AsyncUnpack()
{
	hThread = OS_CreateThread( (void *) RFP_Unpack_Thread, 0, this );
	if (hThread == NULL)
		return RFP_ERROR_INTERNAL;
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
int RFP_Protocol_Unpack::AsyncFinish()
{
	if (hThread == NULL)
		return RFP_ERROR_INTERNAL;
	int err = OS_WaitForThread( hThread );
	OS_DeleteThread( hThread );
	hThread = NULL;
	return err;
}

/*----------------------------------------
 *	Local Derived version of ppRDMPack
 *--------------------------------------*/
class RFP_Protocol_Pack : public ppRFPPack
{
public:
	RFP_Protocol_Pack( );
	virtual ~RFP_Protocol_Pack();
	virtual int RFPInput( CSIO * pSIO, char * pData, int dataLength, int totalLength );

	int		sentSoFar;			// Keep tracks of how much data has been sent
	int		error;				// Tracks error codes
};

/*  --------------------------------------------------------------------*/
RFP_Protocol_Pack::RFP_Protocol_Pack( ) 
{
	sentSoFar		= 0;
	error			= RFP_OK;
}

/*  --------------------------------------------------------------------*/
RFP_Protocol_Pack::~RFP_Protocol_Pack() 
{
}

/*  --------------------------------------------------------------------*/
int RFP_Protocol_Pack::RFPInput( CSIO * pSIO, char * pData, int dataLength, int totalLength )
{
	int			count;
	int			total = 0;
	RFP_MESSAGE	msg;
	CRFP_Protocol *pRFP = (CRFP_Protocol *) pSIO;

	if (sentSoFar == 0)
	{
		error = pRFP->SendRFPMessage( TR_RFP_START, totalLength );
		if (error != RFP_OK)
			return error;
	}

	if (dataLength == 0)
		return (error = pRFP->SendRFPMessage( TR_RFP_DONE, 0 ));

	while (total < dataLength)
	{
		if ((int) sizeof(msg.data) < dataLength - total)
			count = sizeof(msg.data);
		else
			count = dataLength - total;

		sentSoFar 	+= count;

		msg.header.command		= TRRSP_RFP_MESSAGE;
		msg.header.cmdLength 	= sizeof(msg.header) + count;
		msg.header.pktID		= 0;
		msg.header.phase		= TR_RFP_DATA;
		msg.header.data			= sentSoFar;

		memcpy(msg.data,pData,count);
		error = pRFP->SendRFPMessage( TR_RFP_START, totalLength );
		if (error != RFP_OK)
			return error;

		pData		+= count;
		total		+= count;
	}

	return RFP_OK;
}


/*----------------------------------------
 *	CRFP_Protocol_Factory Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
void CRFP_Protocol_Factory::AcceptClientSocket(CSession *pSession, CNetConn *pNetConn, int NOTUSED(ipAddress), int NOTUSED(tcpPort))
{
    printf("CRFP_Protocol_Factory::AcceptClientSocket()\n");
	CRFP_Protocol * p = new CRFP_Protocol( pNetConn, pSession );
	assert( p != NULL );
	if (p == NULL)
	{
		pSession->Release();
		pNetConn->Shutdown();
	}
}

/*----------------------------------------
 *	CRFP_Protocol Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
CRFP_Protocol::CRFP_Protocol( CNetConn *pNewNetConn, CSession * pNewSession )
{
	assert(pNewNetConn != NULL);

	pNetConn = pNewNetConn;

	if (pNewSession != NULL) {
		pSession = pNewSession;
		pSession->AddSessionObject( this );
		pSession->Event( pSession, SESSION_EVENT_CONNECT, "RDM" );
	} else {
		this->pSession = NULL;
	}

	// Create a thread to process this RDM connection

	hThread = OS_CreateThread( (void *) RFP_Protocol_Thread, 0, this );
	assert( hThread != NULL );
	if (hThread == NULL)
	{
		if (pSession != NULL)
			pSession->Release();
		pNetConn->Shutdown();
	}
}

/*  --------------------------------------------------------------------*/
CRFP_Protocol::~CRFP_Protocol()
{
	pNetConn->Shutdown(); // Causes Process() to exit
	if (hThread != NULL)
	{
		OS_WaitForThread(hThread);
		OS_DeleteThread(hThread);
	}
}

/*  --------------------------------------------------------------------*/
void CRFP_Protocol::Event(CSession * NOTUSED(pUserSession), int event, const void * NOTUSED(pData))
{
	if (event == SESSION_EVENT_CLOSE)
		pNetConn->Shutdown(); // Causes Process() to exit
}

/*  --------------------------------------------------------------------*/
void CRFP_Protocol::Process()
{
	BOOL		bOk;
	int			error = RFP_OK;
	RFP_MESSAGE	msg;

	// Loop and process RFP messages

	while (1)
	{
		// ---------------------------------------
		// Read request

		bOk = pNetConn->Read( (BYTE *) &msg, sizeof(msg.header) );
		if (!bOk)
			goto EXIT;

		msg.header.cmdLength	= ntohs(msg.header.cmdLength);
		msg.header.data			= ntohl(msg.header.data);
		msg.header.phase		= ntohl(msg.header.phase);

		if (msg.header.cmdLength < sizeof(msg.header))
		{
			DBLog(("RFP Request length < 12\n"));
			goto EXIT;
		}

		if (msg.header.command != RFP_MESSAGE_COMMAND)
		{
			DBLog(("RFP Bad Command\n"));
			goto EXIT;
		}

		// ---------------------------------------
		// Read in the rest of the message

		if (msg.header.cmdLength > sizeof(msg.header) )
		{
			bOk = pNetConn->Read( (BYTE *) &msg.data, msg.header.cmdLength - sizeof(msg.header) );
			if (!bOk)
				goto EXIT;
		}

		// ---------------------------------------
		// Process the request

		switch (state)
		{
		case RFP_SERVER_STATE_NULL:
			switch (msg.header.phase)
			{
			case TR_RFP_REQUEST_START:	error = StartRequest(&msg);			break;
			case TR_RFP_START:			error = StartReceive(&msg);			break;
			default:					error = Cancel(RFP_ERROR_INTERNAL);	break;
			}
			break;

		case RFP_SERVER_STATE_RECEIVE:
			switch (msg.header.phase)
			{
			case TR_RFP_DATA:			error = ReceiveData(&msg);			break;
			case TR_RFP_DONE:			error = ReceiveDone(&msg);			break;
			default:					error = Cancel(RFP_ERROR_INTERNAL);	break;
			}
			break;

		case RFP_SERVER_STATE_REQUEST:
			switch (msg.header.phase)
			{
			case TR_RFP_REQUEST_DATA:	error = RequestData(&msg);			break;
			case TR_RFP_REQUEST_DONE:	error = RequestDone(&msg);			break;
			default:					error = Cancel(RFP_ERROR_INTERNAL);	break;
			}
			break;

		default:	error = Cancel(RFP_ERROR_INTERNAL);	break;
		}
	}

EXIT:
	Cancel(RFP_ERROR_INTERNAL);

	if (pSession != NULL)
	{
		pSession->Event( pSession, SESSION_EVENT_DISCONNECT, "RFP" );
		pSession->Release();
	}
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Prepares to receive an RFP request
 * 
 */

int CRFP_Protocol::StartRequest( RFP_MESSAGE * pMsg )
{
	expectedLength = pMsg->header.data;
	receivedLength = 0;
	if (expectedLength > RFP_MAX_HEADER_SIZE)
		return Cancel(RFP_ERROR_REQUEST_FORMAT);
	reqBuffer = new char[RFP_MAX_HEADER_SIZE+1];
	if (reqBuffer == NULL)
		return Cancel(RFP_ERROR_INTERNAL);
	state = RFP_SERVER_STATE_REQUEST;
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Receives request data
 * 
 */

int CRFP_Protocol::RequestData( RFP_MESSAGE * pMsg )
{
	int	dataLength = pMsg->header.cmdLength - sizeof(pMsg->header);
	if (dataLength < 1)
		return Cancel(RFP_ERROR_REQUEST_FORMAT);
	if (receivedLength + dataLength > expectedLength)
		return Cancel(RFP_ERROR_REQUEST_FORMAT);
	SendRFPMessage(TR_RFP_RESULT, RFP_PACKET_ACK);
	memcpy(&reqBuffer[receivedLength],&pMsg->data,dataLength);
	receivedLength += dataLength;
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Request is done, process it.
 * 
 */

int CRFP_Protocol::RequestDone( RFP_MESSAGE * NOTUSED(pMsg) )
{
	int	error;

	if (expectedLength != receivedLength)
		return Cancel(RFP_ERROR_REQUEST_FORMAT);

	reqBuffer[receivedLength++] = 0;

	CSIO_SMEM	sio(reqBuffer,receivedLength);

	RFP_Protocol_Pack pack;

	error = pack.CreateSchema( &sio );
	if (error != RFP_OK)
		return Cancel(error);

	error = pack.PackRFP( (CSIO *) this );
	if (error != RFP_OK)
		return Cancel(error);

	return GotoNull(error);
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Start receiving and RFP.
 * 
 */

int CRFP_Protocol::StartReceive( RFP_MESSAGE * pMsg )
{
	expectedLength = pMsg->header.data;
	receivedLength = 0;
	state = RFP_SERVER_STATE_RECEIVE;
	pUnpack = new RFP_Protocol_Unpack;
	if (pUnpack == NULL)
		return Cancel(RFP_ERROR_INTERNAL);
	int error = pUnpack->AsyncUnpack();
	if (error != RFP_OK)
		return Cancel(error);
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Receive RFP Data
 * 
 */

int CRFP_Protocol::ReceiveData( RFP_MESSAGE * pMsg )
{
	int error;
	int	dataLength = pMsg->header.cmdLength - sizeof(pMsg->header);
	if (pUnpack == NULL || dataLength < 1)
		Cancel(RFP_ERROR_INTERNAL);
	pUnpack->DataReady((char *) pMsg->data, pMsg->header.cmdLength - sizeof(pMsg->header));
	SendRFPMessage(TR_RFP_RESULT, RFP_PACKET_ACK);
	error = pUnpack->WaitForDataDone();
	if (error != RFP_OK)
		return Cancel(error);
	return RFP_OK;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Receive RFP done
 * 
 */

int CRFP_Protocol::ReceiveDone( RFP_MESSAGE * NOTUSED(pMsg) )
{
	pUnpack->DataReady(NULL, 0);
	int error = pUnpack->AsyncFinish();
	if (error != RFP_OK)
		return Cancel(error);
	return GotoNull(error);
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	We are done with the RFP, go back to the null state
 * 
 */

int CRFP_Protocol::Cancel( int error )
{
	SendRFPMessage( TR_RFP_CANCEL, error );
	return GotoNull(error);
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	We are done with the RFP, go back to the null state
 * 
 */

int CRFP_Protocol::GotoNull( int error )
{
	state = RFP_SERVER_STATE_NULL;

	if (pUnpack != NULL)
	{
		delete pUnpack;
		pUnpack = NULL;
	}

	if (reqBuffer != NULL)
	{
		delete reqBuffer;
		reqBuffer = NULL;
	}

	return error;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Utility function to construct and send an RFP message
 * 
 */
int CRFP_Protocol::SendRFPMessage( int phase, int data )
{
	RFP_MESSAGE		msg;

	msg.header.command		= TRRSP_RFP_MESSAGE;
	msg.header.cmdLength 	= sizeof(msg.header);
	msg.header.pktID   		= 0;
	msg.header.phase		= phase;
	msg.header.data			= data;

	return SendRFPMessage( &msg );
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Utility function to construct and send an RFP message
 * 
 */
int CRFP_Protocol::SendRFPMessage( RFP_MESSAGE * pRFP )
{
	int	length = pRFP->header.cmdLength;

	pRFP->header.cmdLength	= htons(pRFP->header.cmdLength);
	pRFP->header.phase		= htonl(pRFP->header.phase);
	pRFP->header.data		= htonl(pRFP->header.data);

	BOOL bOk = pNetConn->Write( pRFP, length );

	return bOk ? RFP_OK : RFP_ERROR_INTERNAL;
}


/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Glue code to get into the listen thread method.
 * 
 */

DWORD RFP_Protocol_Thread(LPVOID param)
{
	CRFP_Protocol * p = (CRFP_Protocol *) param;

	p->Process();

	return 0;
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Glue code to get into the listen thread method.
 * 
 */

DWORD RFP_Unpack_Thread(LPVOID param)
{
	RFP_Protocol_Unpack * p = (RFP_Protocol_Unpack *) param;

	return p->UnpackRFP( NULL );
}

} // namespace

