/*-------------------------------------------------------------------------------

	SIO.cpp

	Copyright (c) 2002, Raritan Computer, Inc.

	An simple IO object. Simular in concept to the BIO found in the SSL libray.

	CSIO is the base class and is almost all virtual.
	CSIO_FP uses POSIX file file functions (fopen, fwrite, etc)
	CSIO_SMEM uses a memory buffer allocated by the user. The buffer is fixed size. (STATIC ALLOCATION)
	CSIO_DMEM allocates memory as needed and will grow when written to up to a maximum (DYNAMIC ALLOCATION)

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<stdarg.h>
#include	<ctype.h>
#include	<pp/syms.h>
#include	"pp/SIO.h"

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CSIO
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSIO::CSIO
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	features = 0;
}

//--------------------------------------------------------------------------------
//
	CSIO::~CSIO
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO::Read
	(
		char	*	NOTUSED(pData),		// Were to put the data
		int		NOTUSED(count)		// # to read
	)
//
//	Reads data from the SIO object
//
//------------------------------------------------------------------------------//
{
	return 0;
}


//--------------------------------------------------------------------------------
//
	int									// returns bytes written or error code (POSIX)
	CSIO::Write
	(
		const char	*	NOTUSED(pData),		// Data to write
		int			count				// # to write
	)
//
//	Writes data to the SIO object
//
//------------------------------------------------------------------------------//
{
	return count;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO::Seek
	(
		int			NOTUSED(offset),		// Number of bytes from the origin
		int			NOTUSED(origin)		// SEEK_SET, SEEK_CUR, SEEK_END
	)
//
//	Seeks into the file
//
//------------------------------------------------------------------------------//
{
	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO::Flush
	(
	)
//
//	Flushes the output to the file.
//
//------------------------------------------------------------------------------//
{
	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO::Tell
	(
	)
//
//	Returns the current position of the file
//
//------------------------------------------------------------------------------//
{
	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO::Size
	(
	)
//
//	Returns the size of the file
//
//------------------------------------------------------------------------------//
{
	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSIO::Printf
	(
		const char	*format,				// The format string
		...								// variable arg list	
	)
//
//	printf() function to the SIO
//
//--------------------------------------------------------------------------------
{
	va_list	args;
	char	buffer[300];
	int	count;

	va_start(args, format);
	count = vsnprintf(buffer, sizeof(buffer), format, args);
	va_end(args);

	if (count < 0)
		return count;

	return Write( buffer, count );
}

//--------------------------------------------------------------------------------
//
	int
	CSIO::Putc
	(
		char		c	
	)
//
//	putc function to the SIO
//
//--------------------------------------------------------------------------------
{
	int	 result = Write( &c, 1 );

	if (result == 1)
		return c;
	else
		return EOF;
}

//--------------------------------------------------------------------------------
//
	int
	CSIO::Puts
	(
		const char		*pData	
	)
//
//	putc function to the SIO
//
//--------------------------------------------------------------------------------
{
	// HACK assert( pData != NULL );

	if (pData == NULL)
		return 0;

	int	 result = Write( pData, strlen(pData) );

	if (result >= 0)
		return 0;
	else
		return EOF;
}

//--------------------------------------------------------------------------------
//
	int									// Returns # of bytes written or error
	CSIO::CopyTo
	(
		CSIO	*pOut,					// The target SIO
		int		srcPos,					// Source Position, 0 = Start of file
										// Default value is 0
		int		length,					// # of bytes to copy, -1 = Whole File
										// Default value is -1
		int		destPos					// Destination Position, -1 = Cur Position
										// Default value is -1
	)
//
//	Appends the data from this SIO to <pOut> at the current file position.
//
//--------------------------------------------------------------------------------
{
	int		result,result2;
	int		written = 0;
	int		count;

	// Figure out starting points and length
	
	Seek(srcPos,SEEK_SET);
	
	if (length < 0)
		length = 0x7FFFFFFF;

	if (destPos >= 0)
		pOut->Seek( destPos, SEEK_SET );

	// Copy the data

/*
	if (features & SIO_READ_NO_COPY)
	{
		// This SIO support ReadNoCopy, use it

		result = ReadNoCopy( &p, length );
		if (result > 0)
		result2 = pOut->Write( p, result );
		if (result2 > 0)
			written += result2;
	}
	else*/
	{
		// Old fasioned copy

		char	buffer[256];

		do
		{
			count = 256;
			if (256 > length)
				count = length;

			result = Read(buffer,count);

			if (result < 1)
				break;

			result2 = pOut->Write(buffer,result);

			written += result2;

			if (result2 != result)
				break;

			length -= result;

		} while (result == count && length > 0);
	}

	return written;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO::ReadNoCopy
	(
		char	**	pData,				// Ptr to a Ptr to return buffer ptr
		int		NOTUSED(count)		// Number of bytes wanted
	)
//
//	Special version of read that allows the caller to "read" the data without
//	copying the data. This is done by returning a ptr to the SIO's internal
//	buffer.
//	The internal buffer may be segmented, so the length returned will often
//	be less than the length requested.
//
//	The file position is advanced.
//
//--------------------------------------------------------------------------------
{
	*pData = NULL;
	return -1;
}

//--------------------------------------------------------------------------------
//									CSIO_FP
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSIO_FP::CSIO_FP
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	fp = NULL;
	weOpened = 0;
}

//--------------------------------------------------------------------------------
//
	CSIO_FP::~CSIO_FP
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	Close();
}

//--------------------------------------------------------------------------------
//
	void
	CSIO_FP::SetFP
	(
		FILE	*	_fp
	)
//
//	Sets the file pointer for access
//
//------------------------------------------------------------------------------//
{
	Close();	// Close any files we may have opened before
	fp = _fp;
}

//--------------------------------------------------------------------------------
//
	FILE *								// The file ptr
	CSIO_FP::GetFP
	(
		FILE	*	_fp
	)
//
//	Returns the file pointer
//
//------------------------------------------------------------------------------//
{
	return _fp;
}

//--------------------------------------------------------------------------------
//
	int									// 0 = no error, !0 = POSIX error
	CSIO_FP::Open
	(
		const char	*	pName,				// File name
		const char	*	pMode				// File mode
	)
//
//	Opens the file
//
//------------------------------------------------------------------------------//
{
	Close();	// Close any files we may have opened before

	if (pName == NULL || pName[0] == 0 || pMode == NULL)
		return -1;

	fp = fopen(pName, pMode);

	if (fp != NULL)
	{
		weOpened = 1;
		return 0;
	}
	else
		return -1;
}

//--------------------------------------------------------------------------------
//
	int									// 0 = no error, !0 = POSIX error
	CSIO_FP::Close
	(
	)
//
//	Closes the file. (No real need to call this, deleting the object closes the file)
//
//------------------------------------------------------------------------------//
{
	int	result = 0;

	if (weOpened)
	{
		result = fclose(fp);
		fp = NULL;
		weOpened = 0;
	}

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO_FP::Read
	(
		char	*	pData,				// Were to put the data
		int			count				// # to read
	)
//
//	Reads data from the SIO object
//
//------------------------------------------------------------------------------//
{
	return fread( pData, 1, count, fp );
}


//--------------------------------------------------------------------------------
//
	int									// returns bytes written or error code (POSIX)
	CSIO_FP::Write
	(
		const char	*	pData,				// Data to write
		int			count				// # to write
	)
//
//	Writes data to the SIO object
//
//------------------------------------------------------------------------------//
{
	return fwrite( pData, 1, count, fp );
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO_FP::Seek
	(
		int			offset,				// Number of bytes from the origin
		int			origin				// SEEK_SET, SEEK_CUR, SEEK_END
	)
//
//	Seeks into the file
//
//------------------------------------------------------------------------------//
{
	return fseek( fp, offset, origin );
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO_FP::Flush
	(
	)
//
//	Flushes the output to the file.
//
//------------------------------------------------------------------------------//
{
	return fflush( fp );
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_FP::Tell
	(
	)
//
//	Returns the current position of the file
//
//------------------------------------------------------------------------------//
{
	return ftell( fp );
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_FP::Size
	(
	)
//
//	Returns the size of the file
//
//------------------------------------------------------------------------------//
{
	int	pos,size;

	pos = ftell( fp );
	fseek(fp, 0, SEEK_END);
	size = ftell( fp );
	fseek(fp, pos, SEEK_SET);

	return size;
}

//--------------------------------------------------------------------------------
//									CSIO_SMEM
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSIO_SMEM::CSIO_SMEM
	(
		char *	pNewBuffer,				// Ptr to the user buffer
		int		count,					// # of bytes of data in the buffer
		int		newMaxSize					// max size the data can grow to in this buffer
										// if maxSize = 0, then buffer will not grow past count.
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	pBuffer = pNewBuffer;
	size = count;
	position = 0;
	if (newMaxSize == 0)
		maxSize = count;
	else
		maxSize = newMaxSize;
	features |= SIO_READ_NO_COPY;
}

//--------------------------------------------------------------------------------
//
	CSIO_SMEM::~CSIO_SMEM
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO_SMEM::Read
	(
		char	*	pData,				// Were to put the data
		int			count				// # to read
	)
//
//	Reads data from the SIO object
//
//------------------------------------------------------------------------------//
{
	if (position >= size)
		return 0;

	if (position + count > size)
		count = size - position;

	memcpy(pData,&pBuffer[position],count);
	position += count;

	return count;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes written or error code (POSIX)
	CSIO_SMEM::Write
	(
		const char	*	pData,				// Data to write
		int			count				// # to write
	)
//
//	Writes data to the SIO object
//
//------------------------------------------------------------------------------//
{
	if (position >= maxSize)
		return 0;

	if (position + count > maxSize)
		count = maxSize - position;

	memcpy(&pBuffer[position],pData,count);
	position += count;
	if (position > size)
		size = position;

	return count;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO_SMEM::Seek
	(
		int			offset,				// Number of bytes from the origin
		int			origin				// SEEK_SET, SEEK_CUR, SEEK_END
	)
//
//	Seeks into the file
//
//------------------------------------------------------------------------------//
{
	int		pos;

	switch (origin)
	{
		case SEEK_SET:
			pos = offset;
			break;
		case SEEK_CUR:
			pos = position + offset;
			break;
		case SEEK_END:
			pos = size + offset;
			break;
		default:
			return -1;
	}

	if (pos < 0)
		pos = 0;
	else if (pos > size)
		pos = size;

	position = pos;

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_SMEM::Tell
	(
	)
//
//	Returns the current position of the file
//
//------------------------------------------------------------------------------//
{
	return position;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_SMEM::Size
	(
	)
//
//	Returns the size of the file
//
//------------------------------------------------------------------------------//
{
	return size;
}

//--------------------------------------------------------------------------------
//
	char *								// Returns the ptr to the memory buffer
	CSIO_SMEM::GetBuffer
	(
	)
//
//	Returns the ptr to the memory buffer
//
//------------------------------------------------------------------------------//
{
	return pBuffer;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO_SMEM::ReadNoCopy
	(
		char	**	pData,				// Ptr to a Ptr to return buffer ptr
		int			count				// Number of bytes wanted
	)
//
//	Special version of read that allows the caller to "read" the data without
//	copying the data. This is done by returning a ptr to the SIO's internal
//	buffer.
//	The internal buffer may be segmented, so the length returned will often
//	be less than the length requested.
//
//	The file position is advanced.
//
//--------------------------------------------------------------------------------
{
	*pData = NULL;

	if (position >= size)
		return 0;

	if (position + count > size)
		count = size - position;

	*pData = (char *) &pBuffer[position];

	position += count;

	return count;
}

//--------------------------------------------------------------------------------
//									CSIO_DMEM
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSIO_DMEM::CSIO_DMEM
	(
		int		newMaxSize					// max size the data can grow to in this buffer
										// if maxSize = 0, then there is no limit
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	if (newMaxSize == 0)
		maxSize = 0x7FFFFFFF;
	else
		maxSize = newMaxSize;
	size = position = 0;
	pFirstBlock = NULL;
	features |= SIO_READ_NO_COPY;
}

//--------------------------------------------------------------------------------
//
	CSIO_DMEM::~CSIO_DMEM
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	DMEM_BLOCK	*p,*p2;

	p = pFirstBlock;
	pFirstBlock = NULL;

	while (p != NULL)
	{
		p2 = p->pNext;
		delete p;
		p = p2;
	}
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO_DMEM::Read
	(
		char	*	pData,				// Were to put the data
		int			count				// # to read
	)
//
//	Reads data from the SIO object
//
//------------------------------------------------------------------------------//
{
	DMEM_BLOCK	* p;
	int startBlock = position / SIO_DMEM_BLOCK_SIZE;
	int	offset = position % SIO_DMEM_BLOCK_SIZE;
	int remaining;
	int	length;

	if (position >= size)
		return 0;

	if (position + count > size)
		count = size - position;

	if (count > 0)
	{
		p = pFirstBlock;

		while (startBlock--)
			p = p->pNext;

		remaining = count;

		while (remaining)
		{
			length = SIO_DMEM_BLOCK_SIZE - offset;
			if (length > remaining)
				length = remaining;

			memcpy(pData,&p->data[offset],length);

			remaining -= length;
			pData += length;
			offset += length;

			if (offset == SIO_DMEM_BLOCK_SIZE)
			{ 
				if (p->pNext == NULL)
					return -1;
				p = p->pNext;
				offset = 0;
			}
		}

		position += count;
	}

	return count;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes written or error code (POSIX)
	CSIO_DMEM::Write
	(
		const char	*	pData,				// Data to write
		int			count				// # to write
	)
//
//	Writes data to the SIO object
//
//------------------------------------------------------------------------------//
{
	DMEM_BLOCK	* p;
	int startBlock = position / SIO_DMEM_BLOCK_SIZE;
	int	offset = position % SIO_DMEM_BLOCK_SIZE;
	int	length;
	int	written = 0;

	if (position + count > maxSize)
		count = maxSize - position;

	if (count > 0)
	{
		if (pFirstBlock == NULL)
		{
			if (AddBlock() != 0)
				return -1;
		}

		p = pFirstBlock;

		while (startBlock--)
			p = p->pNext;

		while (count)
		{
			length = SIO_DMEM_BLOCK_SIZE - offset;
			if (length > count)
				length = count;

			memcpy(&p->data[offset],pData,length);

			count -= length;
			pData += length;
			written += length;
			offset += length;

			if (offset == SIO_DMEM_BLOCK_SIZE)
			{ 
				if (p->pNext == NULL)
				{
					if (AddBlock() != 0)
						return -1;
				}
				p = p->pNext;
				offset = 0;
			}
		}

		position += written;
	}

	if (position > size)
		size = position;

	return written;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code (POSIX)
	CSIO_DMEM::Seek
	(
		int			offset,				// Number of bytes from the origin
		int			origin				// SEEK_SET, SEEK_CUR, SEEK_END
	)
//
//	Seeks into the file
//
//------------------------------------------------------------------------------//
{
	int		pos;

	switch (origin)
	{
		case SEEK_SET:
			pos = offset;
			break;
		case SEEK_CUR:
			pos = position + offset;
			break;
		case SEEK_END:
			pos = size + offset;
			break;
		default:
			return -1;
	}

	if (pos < 0)
		pos = 0;
	else if (pos > size)
		pos = size;

	position = pos;

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_DMEM::Tell
	(
	)
//
//	Returns the current position of the file
//
//------------------------------------------------------------------------------//
{
	return position;
}

//--------------------------------------------------------------------------------
//
	int									// returns current position or error code (POSIX)
	CSIO_DMEM::Size
	(
	)
//
//	Returns the size of the file
//
//------------------------------------------------------------------------------//
{
	return size;
}

//--------------------------------------------------------------------------------
//
	int									// 0 = No error
	CSIO_DMEM::AddBlock
	(
	)
//
//	Allocates more memory
//
//------------------------------------------------------------------------------//
{
	DMEM_BLOCK	*pNew = new DMEM_BLOCK;
	DMEM_BLOCK	*p;

	if (pNew == NULL)
		return -1;

	pNew->pNext = NULL;

	if (pFirstBlock == NULL)
		pFirstBlock = pNew;
	else
	{
		p = pFirstBlock;

		while (p->pNext)
			p = p->pNext;

		p->pNext = pNew;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// returns bytes read or error code (POSIX)
	CSIO_DMEM::ReadNoCopy
	(
		char	**	pData,				// Ptr to a Ptr to return buffer ptr
		int			count				// Number of bytes wanted
	)
//
//	Special version of read that allows the caller to "read" the data without
//	copying the data. This is done by returning a ptr to the SIO's internal
//	buffer.
//	The internal buffer may be segmented, so the length returned will often
//	be less than the length requested.
//
//	The file position is advanced.
//
//--------------------------------------------------------------------------------
{
	DMEM_BLOCK	* p;
	int startBlock = position / SIO_DMEM_BLOCK_SIZE;
	int	offset = position % SIO_DMEM_BLOCK_SIZE;
	int	length;

	*pData = NULL;

	if (position >= size)
		return 0;

	if (position + count > size)
		count = size - position;

	p = pFirstBlock;

	while (startBlock--)
		p = p->pNext;

	length = SIO_DMEM_BLOCK_SIZE - offset;

	if (length < count)
		count = length;

	position += count;

	*pData = &p->data[offset];

	return count;
}

