/*--------------------------------------------------------------------------------

	Timer.cpp

	Copyright (c) 2001, Scott Coleman!

	I hate having to create yet another thread just so I can have a timer.
	This class will provide mutiple millisecond timers from a single thread.

--------------------------------------------------------------------------------*/

#include	"pp/OS_Port.h"
#include	"pp/timer.h"
//#include	"Debug.h"

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

	UINT 
	CTimer_Thread
	(
		LPVOID pParam
	);

//----------------------------------------
//				Static Data
//----------------------------------------

BOOL		timerStarted = FALSE;		// True if there is at least one CTimer
CTimer	*	pFirst;						// handle to the first connection
OS_THREAD	hThread = NULL;				// handle to our thread
OS_EVENT	hEvent = NULL;				// Our event
OS_CRITICAL_SECTION		timerCS = NULL;	// Our critical section
BOOL		stopThread;					// TRUE when we want to stop the thread
DWORD		lastTime;					// Last time a timer triggered

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//						Functions
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CTimer::CTimer
	(
	)
//
//	Creates a CTimer
//
//--------------------------------------------------------------------------------
{
	// Make sure the timer has been initialized

	if (!timerStarted)
	{
		pFirst			= NULL;
		stopThread		= FALSE;

		hEvent = OS_CreateEvent( OS_EVENT_NORMAL );
		
		if (hEvent == NULL)
			return;

		timerCS = OS_CreateCriticalSection( OS_CRITICAL_SECTION_NORMAL );
		
		if (timerCS == NULL)
			return;

		hThread = OS_CreateThread( (void *) CTimer_Thread, OS_THREAD_NORMAL, this );

		if (hThread == NULL)
			return;

		lastTime		= OS_GetTickCount();
		timerStarted	= TRUE;
	}

	OS_EnterCriticalSection( timerCS );

	// Init this object
	
	this->running		= FALSE;
	this->timeLeft		= 0;
	this->reload		= 0;
	this->pProc			= 0;
	this->pNext			= 0;

	// Add this timer to the list

	this->pNext = pFirst;
	pFirst = this;

	OS_LeaveCriticalSection( timerCS );
}

//--------------------------------------------------------------------------------
//
	CTimer::~CTimer
	(
	)
//
//	Destroyies a CTimer
//
//--------------------------------------------------------------------------------
{
	CTimer *pObj,*pLast;

	OS_EnterCriticalSection( timerCS );

	// Remove this form the linked list

	pLast = NULL;
	for (pObj = pFirst; pObj; pObj = pObj->pNext)
	{
		if (pObj == this)
		{
			if (pLast == NULL)
				pFirst = pObj->pNext;
			else
				pLast->pNext = pObj->pNext;
			break;
		}

		pLast = pObj;
	}

	// Stop this timer

	this->running = FALSE;

	// If this is the last timer, kill the thread

	if (pFirst == NULL)
	{
		stopThread = TRUE;
		OS_SetEvent( hEvent );

		timerStarted = FALSE;

		if (hThread != NULL)
		{
            OS_LeaveCriticalSection( timerCS );
			OS_WaitForThread( hThread );
			OS_DeleteThread( hThread );
			hThread = NULL;
            OS_EnterCriticalSection( timerCS );
		}
		
		if (hEvent != NULL)
		{
			OS_DeleteEvent( hEvent );
			hEvent = NULL;
		}
		
		if (timerCS != NULL)
		{
			OS_LeaveCriticalSection( timerCS );
			OS_DeleteCriticalSection( timerCS );
			timerCS = NULL;
		}
	}

	if (timerCS != NULL)
		OS_LeaveCriticalSection( timerCS );
}

//--------------------------------------------------------------------------------
//
	void
	CTimer::SetCTimerProc
	(
		PTIMERPROC		_pProc,			// The proc to call when timer triggers
		void		*	param1,			// User param
		void		*	param2			// User param
	)
//
//	Sets the timer proc
//
//--------------------------------------------------------------------------------
{
	OS_EnterCriticalSection( timerCS );
	p1 = param1;
	p2 = param2;
	pProc = _pProc;
	OS_LeaveCriticalSection( timerCS );
}

//--------------------------------------------------------------------------------
//
	void
	CTimer::SetCTimer
	(
		DWORD	period,					// # of ms until triggering
		BOOL	oneShot					// TRUE = trigger only once
	)
//
//	Sets the timer period. Also, the timer can be setup to be a one shot timer
//
//--------------------------------------------------------------------------------
{
	OS_EnterCriticalSection( timerCS );
	if (period == 0)
		this->running = FALSE;

	this->timeLeft = this->reload = period;

	if (oneShot)
		this->reload = 0;

	OS_LeaveCriticalSection( timerCS );
}

//--------------------------------------------------------------------------------
//
	void
	CTimer::StopCTimer
	(
	)
//
//	Stops the timer.
//
//--------------------------------------------------------------------------------
{
	this->running = FALSE;
	OS_SetEvent( hEvent );
}

//--------------------------------------------------------------------------------
//
	void
	CTimer::StartCTimer
	(
	)
//
//	Starts the timer.
//
//--------------------------------------------------------------------------------
{
	this->timeLeft += OS_GetTickCount() - lastTime;
	this->running = TRUE;

	OS_SetEvent( hEvent );
}

//--------------------------------------------------------------------------------
//
	BOOL
	CTimer::IsTimerRunning
	(
	)
//
//	Starts the timer.
//
//--------------------------------------------------------------------------------
{
	return this->running;
}

//--------------------------------------------------------------------------------
//
	DWORD
	CTimer::GetRemainingTime
	(
	)
//
//	Returns the # of ms until the timer triggers.
//
//--------------------------------------------------------------------------------
{
	DWORD	result;

	OS_EnterCriticalSection( timerCS );

	result = this->timeLeft - (OS_GetTickCount() - lastTime);

	OS_LeaveCriticalSection( timerCS );

	return result;
}

//--------------------------------------------------------------------------------
//
	void 
	CTimer::TimerProc
	(
	)
//
//	Virtual function called when timer is triggered
//
//--------------------------------------------------------------------------------
{
}

//--------------------------------------------------------------------------------
//
	void 
	CTimer::CTimerProcess
	(
	)
//
//	This thread waits form the next CTimer to expire and then calls it
//
//--------------------------------------------------------------------------------
{
	DWORD	timeout;
	CTimer	*pTimer;
	DWORD	min;

	do
	{
		// compute the next timeout value

		OS_EnterCriticalSection( timerCS );
	
		for (pTimer = pFirst, min = 0xFFFFFFFF; pTimer; pTimer = pTimer->pNext )
		{
			if (pTimer->running)
			{
				if (pTimer->timeLeft < min)
					min = pTimer->timeLeft;
			}
		}

		OS_LeaveCriticalSection( timerCS );
		
		if (min == 0xFFFFFFFF)
			min = INFINITE;

		// Wait...

		lastTime = OS_GetTickCount();

		//_DBLog2(("CTimer:Wait for Event, Last Time = %d\n",lastTime));

		OS_WaitForEvent( hEvent, min );

		if (stopThread)
			break;

		// See if any timers expired

		timeout = OS_GetTickCount() - lastTime;

		lastTime = OS_GetTickCount();

		//_DBLog2(("CTimer:Wakeup, timeout = %d, lastTime = %d\n",timeout,lastTime));

		// Make a list of the expired timers

		OS_EnterCriticalSection( timerCS );

		for (pTimer = pFirst; pTimer; pTimer = pTimer->pNext )
		{
			if (pTimer->running)
			{
				//_DBLog2(("CTimer:Timeleft = %d\n",pTimer->timeLeft));

				if (timeout >= pTimer->timeLeft)
				{
					pTimer->timeLeft = pTimer->reload;

					if (pTimer->timeLeft == 0)
						pTimer->running = FALSE;
						
					// Call the timer

					//_DBLog2(("CTimer:Calling TimerProc\n"));

					if (pTimer->pProc != NULL)
						(pTimer->pProc)( pTimer, pTimer->p1, pTimer->p2);

					pTimer->TimerProc();
				}
				else
				{
					pTimer->timeLeft -= timeout;
				}
			}
		}

		OS_LeaveCriticalSection( timerCS );

	} while (!stopThread);

}

//--------------------------------------------------------------------------------
//
	UINT 
	CTimer_Thread
	(
		LPVOID pParam
	)
//
//	Glue code to get to the thread proc
//
//--------------------------------------------------------------------------------
{
	CTimer * pThis = (CTimer *) pParam;

	pThis->CTimerProcess();

	return 0;
}
