/**
 * \file dlist.c
 *
 * Description: Double Linked List
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <stddef.h>
#include <malloc.h>
#include <assert.h>
#include <pp/dlist.h>
 
struct dlist_iter_s {
	dlist_iter_t* next;
	dlist_iter_t* prev;
	void* elem;
};

struct dlist_s {
	dlist_iter_t root;
	int count;
	void (*delelem)(void*);
};

dlist_t* dlist_new(void (*delelem)(void*))
{
	dlist_t *l = (dlist_t*)malloc(sizeof(dlist_t));
	l->delelem = delelem;
	l->root.next = &l->root;
	l->root.prev = &l->root;
	l->count = 0;
	return l;
}

void dlist_delete(dlist_t* l)
{
	assert(l);
	dlist_clear(l);
	free(l);
}


int dlist_count(const dlist_t* l)
{
	assert(l);
	return l->count;
}

void* dlist_get(const dlist_t* l, const dlist_iter_t* it)
{
        assert(l);
	(void)l;
	assert(it);
	return it->elem;
}

dlist_iter_t* dlist_find(const dlist_t* l, const void* e)
{
	dlist_iter_t *it;
	assert(l);

	for (it = dlist_first(l); it; it = dlist_next(l, it))
		if (dlist_get(l, it) == e) return it;
	
	return NULL;
}


dlist_iter_t* dlist_insert(dlist_t* l, dlist_iter_t* succ, void* e)
{
        dlist_iter_t *it;
	assert(l);
	if (succ == NULL) succ = &l->root;
	it = (dlist_iter_t*)malloc(sizeof(dlist_iter_t));
	it->prev = succ;
	it->next = succ->next;
	it->prev->next = it;
	it->next->prev = it;
	it->elem = e;
	l->count++;
	return it;
}

void* dlist_remove(dlist_t* l, dlist_iter_t* it)
{
        void *e;
        assert(l);
	assert(it);
	e = it->elem;
	it->prev->next = it->next;
	it->next->prev = it->prev;
	free(it);
	l->count--;
	return e;
}

dlist_iter_t* dlist_erase(dlist_t* l, dlist_iter_t* it)
{
	dlist_iter_t *ret = it->next == &l->root ? NULL : it->next;
	l->delelem(dlist_remove(l, it));
	return ret;
}

void dlist_clear(dlist_t* l)
{
	assert(l);
	while (l->root.next != &l->root) dlist_erase(l, l->root.next);
}


dlist_iter_t* dlist_first(const dlist_t* l)
{
	assert(l);
	return l->root.next == &l->root ? NULL : l->root.next;
}

dlist_iter_t* dlist_last(const dlist_t* l)
{
	assert(l);
	return l->root.prev == &l->root ? NULL : l->root.prev;
}

dlist_iter_t* dlist_next(const dlist_t* l, const dlist_iter_t* it)
{
	assert(l);
	if (it == NULL) it = &l->root;
	return it->next == &l->root ? NULL : it->next;
}

dlist_iter_t* dlist_prev(const dlist_t* l, const dlist_iter_t* it)
{
	assert(l);
	if (it == NULL) it = &l->root;
	return it->prev == &l->root ? NULL : it->prev;
}


dlist_iter_t* dlist_push_back(dlist_t* l, void* e)
{
	assert(l);
	return dlist_insert(l, l->root.prev, e);
}

dlist_iter_t* dlist_push_front(dlist_t* l, void* e)
{
	assert(l);
	return dlist_insert(l, NULL, e);
}

void* dlist_pop_back(dlist_t* l)
{
	assert(l);
	if (l->count == 0) return NULL;
	return dlist_remove(l, l->root.prev);
}

void* dlist_pop_front(dlist_t* l)
{
	assert(l);
	if (l->count == 0) return NULL;
	return dlist_remove(l, l->root.next);
}
