#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <pp/base.h>

#if !defined(PP_FW_TYPE_PRODUCTION)
# include <pp/intl.h>
#endif

/*
 * global errno registry
 */

#define MAX_ERRNO_LIBS 16
#define MIN_EXTENDED_ERRNO 0x00010000

typedef struct {
    int min_errno;
    const char* (*error_string)(int);
} errno_entry_t;

// use static array, so no need to init, i.e. nothing can go wrong
static errno_entry_t errno_libs[MAX_ERRNO_LIBS + 1];
static int errno_libs_no = 0; // actual number of registered errno libs

int
pp_register_errnos(int no, const char* (*error_string)(int))
{
    int errno_range_start;
    
    assert(errno_libs_no < MAX_ERRNO_LIBS);
    assert(error_string != NULL);

    // first time init
    if (errno_libs_no == 0)
	errno_libs[errno_libs_no].min_errno = MIN_EXTENDED_ERRNO;

    errno_range_start = errno_libs[errno_libs_no].min_errno;
    errno_libs[errno_libs_no].error_string = error_string;
    errno_libs[++errno_libs_no].min_errno = errno_range_start + no;

    return errno_range_start;
}

// unregister has to happen in same order as register
void pp_unregister_errnos(int range_start)
{
    assert(errno_libs_no > 0);
    PP_ASSERT(range_start, errno_libs[errno_libs_no - 1].min_errno == range_start);

    --errno_libs_no;
}

/*
 * since the normal strerror function is not thread safe
 * but we wonna avoid the trouble of passing buffers to
 * pp_error_string, we create a thread-specific strerror buffer
 * on-the-fly and return a pointer to that buffer in case of
 * normal errnos.
 *
 * moreover, it seems that strerror_t is not using the supplied
 * buffer unconditionally. So, in order to detect this, we use
 * a stack-buffer and if that one has been used we copy it over
 * to the thread buffer...
 *
 * (really crazy, never knew that using strerror is such a science :-|)
 */

#define STRERR_BUF_SIZE 128

static pthread_once_t strerr_buf_key_once = PTHREAD_ONCE_INIT;
static pthread_key_t  strerr_buf_key;

static void strerr_buf_key_create(void) {
    pthread_key_create(&strerr_buf_key, free);
}

static char* get_strerr_buf(void) {
    char* buf;
    pthread_once(&strerr_buf_key_once, strerr_buf_key_create);
    if (NULL == (buf = pthread_getspecific(strerr_buf_key))) {
	buf = malloc(STRERR_BUF_SIZE);
	pthread_setspecific(strerr_buf_key, buf);
    }
    return buf;
}

const char* pp_error_string(int error)
{
    int i;

    if (error < MIN_EXTENDED_ERRNO) {
	char* strerr;
	char  buf[STRERR_BUF_SIZE];
	
	/* NOTE: strerror_r may, but need not, use the user supplied buffer */
	if (NULL == (strerr = strerror_r(error, buf, STRERR_BUF_SIZE))) {
	    return N_("strerror_r failed");
	} else if (strerr == buf) {/* buffer has been used, so we need to    *
				   * preserve it in a thread-specific buffer */
	    strerr = get_strerr_buf();
	    strcpy(strerr, buf); /* no check - buffers are equal sized */
	    return strerr;
	}

    } else {
	
	// step trough our list of errno ranges and select the right function
	for (i = 0; i < errno_libs_no; ++i) {
	    if (error >= errno_libs[i].min_errno &&
		error < errno_libs[i+1].min_errno) {
		return errno_libs[i].error_string(error);
	    }
	}
    }

    return N_("Unknown Error");
}
