#include <pthread.h>
#include <pp/base.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <pp/i2c.h>
#include "base_intern.h"

#define noDEBUG
#ifdef DEBUG
# define D(fmt, args...)   { printf(fmt, ##args); }
#else
# define D(fmt, args...)   { }
#endif

#define I2C_DEV_INITIALIZER(bus_id, name, device, mtx)		\
    { name, device, -1, mtx, 0, 0, bus_id }
#define I2C_DEV_TERMINATOR \
    { NULL, NULL, -1, NULL, 0, 0, -1 }

typedef struct {
    const char* name;
    const char* devicename;
    int fd;
    pthread_mutex_t * mtx;
    u_long refcount;
    u_char slave;
    char bus_id; /* arbitrary bus id, used by BMC */
} i2c_device_t;
    
static pthread_mutex_t device_list_mtx = PTHREAD_MUTEX_INITIALIZER;

#if defined(PP_BOARD_KIRA)
static pthread_mutex_t faraday_1_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t faraday_2_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t faraday_3_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t faraday_4_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t faraday_5_mtx = PTHREAD_MUTEX_INITIALIZER;
#else
static pthread_mutex_t asics_0_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t asics_1_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t asics_2_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t asics_3_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t asics_4_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t ibm_0_mtx   = PTHREAD_MUTEX_INITIALIZER;
#endif

static i2c_device_t i2c_devices[] = {
#if defined(PP_BOARD_KIRA)
    I2C_DEV_INITIALIZER(1, "faraday-1", "/dev/i2c-faraday-1", &faraday_1_mtx),
    I2C_DEV_INITIALIZER(2, "faraday-2", "/dev/i2c-faraday-2", &faraday_2_mtx),
    I2C_DEV_INITIALIZER(3, "faraday-3", "/dev/i2c-faraday-3", &faraday_3_mtx),
    I2C_DEV_INITIALIZER(4, "faraday-4", "/dev/i2c-faraday-4", &faraday_4_mtx),
    I2C_DEV_INITIALIZER(5, "faraday-5", "/dev/i2c-faraday-5", &faraday_5_mtx),
#else /* !PP_BOARD_KIRA */
    I2C_DEV_INITIALIZER(0, "asics-0", "/dev/i2c-asics-0", &asics_0_mtx),
    I2C_DEV_INITIALIZER(1, "asics-1", "/dev/i2c-asics-1", &asics_1_mtx),
    I2C_DEV_INITIALIZER(2, "asics-2", "/dev/i2c-asics-2", &asics_2_mtx),
    I2C_DEV_INITIALIZER(3, "asics-3", "/dev/i2c-asics-3", &asics_3_mtx),
    I2C_DEV_INITIALIZER(4, "asics-4", "/dev/i2c-asics-4", &asics_4_mtx),
    I2C_DEV_INITIALIZER(5, "ibm-0",   "/dev/i2c-ibm_iic-0", &ibm_0_mtx),
    I2C_DEV_INITIALIZER(-1,"ipmb-0",  "/dev/i2c-ipmi", &ibm_0_mtx), 
#endif /* !PP_BOARD_KIRA */
    I2C_DEV_TERMINATOR
};

static const u_char i2c_dev_cnt = sizeof(i2c_devices) / sizeof(i2c_device_t);
    
int
pp_i2c_init(void)
{
    /* nothing to do, could open all devices here... */
    return 0;
}

void
pp_i2c_cleanup(void)
{
    i2c_device_t* dev = i2c_devices;

    MUTEX_LOCK(&device_list_mtx);

    while (dev && dev->name) {
	if (dev->refcount != 0) {
	    pp_log("%s i2c device '%s' was still in use, closing.\n", ___F, dev->name);
	    close(dev->fd);
	    dev->fd = -1;
	}
	dev++;
    }
    MUTEX_UNLOCK(&device_list_mtx);
}

static u_char
pp_i2c_open_core(u_char handle, int *error)
{
    i2c_device_t* dev = &i2c_devices[handle];
    
    assert(handle < i2c_dev_cnt);
    assert(error);
    *error = PP_I2C_ERROR;

    MUTEX_LOCK(&device_list_mtx);

    if (dev->fd == -1) {
	assert(dev->refcount == 0);
	
	if ((dev->fd = open(dev->devicename, O_RDWR)) < 0) {
	    pp_log_err("%s(): Error Opening I2C '%s' (device %s)\n", ___F,
		       dev->name, dev->devicename);
	    goto bail;
	}
    }

    dev->refcount++;
    
    *error = PP_I2C_NO_ERROR;
    
 bail:
    MUTEX_UNLOCK(&device_list_mtx);
    return handle;
}

u_char
pp_i2c_open(const char* name, int *error)
{
    int i;
    for (i = 0; i2c_devices[i].name != NULL; ++i) {
	if (!pp_strcmp_safe(name, i2c_devices[i].name)) {
	    return pp_i2c_open_core(i, error);
	}
    }
    pp_log_err("I2C device named '%s' not supported\n", name);
    *error = PP_I2C_ERROR;
    return i;
}

u_char
pp_i2c_open_device(const char* devicename, int *error)
{
    int i;
    for (i = 0; i2c_devices[i].devicename != NULL; ++i) {
	if (!pp_strcmp_safe(devicename, i2c_devices[i].devicename)) {
	    return pp_i2c_open_core(i, error);
	}
    }
    pp_log_err("I2C device named '%s' not supported\n", devicename);
    *error = PP_I2C_ERROR;
    return i;
}

u_char
pp_i2c_open_bus(unsigned int number, int *error) {
    int i;
    for (i = 0;  i2c_devices[i].devicename != NULL; ++i) {
	if (i2c_devices[i].bus_id > 0
	    && i2c_devices[i].bus_id == number) {
	    return pp_i2c_open_core(i, error);
	}
    }
    pp_log_err("I2C bus number '%d' not supported\n", number);
    *error = PP_I2C_ERROR;
    return i;
}

void
pp_i2c_close(u_char handle)
{
    i2c_device_t* dev;

    assert(handle < i2c_dev_cnt);

    MUTEX_LOCK(&device_list_mtx);
    
    dev = &i2c_devices[handle];

    if (dev->refcount == 0) {
	pp_log("%s(): Trying to close '%s' while no reference pending, ignored\n",
		   ___F, dev->name);
	goto bail;
    }    

    dev->refcount--;

    if (dev->refcount == 0) {
	close(dev->fd);
	dev->fd = -1;
    }
    
 bail:
    MUTEX_UNLOCK(&device_list_mtx);
}

const char* pp_i2c_dev_by_hndl(u_char handle) {
    assert(handle < i2c_dev_cnt);
    return i2c_devices[handle].devicename;
}
    

/**
 * Check, if this slave address is already set for the device,
 * else set it via ioctl.
 */
static int set_slave_address(i2c_device_t *dev, u_char slave) {
    if (dev->slave != slave) {
	if (ioctl(dev->fd, I2C_SLAVE, slave) < 0) {
	    pp_log_err("%s(): Setting I2C slave address failed\n", ___F);
	    return PP_ERR;
	}
	dev->slave = slave;
    }
    return PP_SUC;
}

int pp_i2c_set_slave_address(u_char handle, u_char slave) {
    i2c_device_t* dev;
    assert(handle < i2c_dev_cnt);
    dev = &i2c_devices[handle];
    return set_slave_address(dev, slave);
}
   

int pp_i2c_get_fd(u_char handle) {
    i2c_device_t* dev;
    assert(handle < i2c_dev_cnt);
    dev = &i2c_devices[handle];
    return dev->fd;
}
    
void
pp_i2c_lock(u_char handle)
{
    i2c_device_t* dev;
    
    assert(handle < i2c_dev_cnt);
    
    dev = &i2c_devices[handle];
    
    MUTEX_LOCK(dev->mtx);
}

void
pp_i2c_unlock(u_char handle)
{
    i2c_device_t* dev;
    
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    MUTEX_UNLOCK(dev->mtx);
}


u_char
pp_i2c_rx_byte(u_char handle, u_char slave, int *error)
{
    i2c_device_t* dev;
    __s32 ret = 0;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    pp_i2c_lock(handle);
    ret = pp_i2c_rx_byte_core(handle, PP_I2C_DO_LOG_ERR, slave, error);
    pp_i2c_unlock(handle);

    return ret;
}

u_char
pp_i2c_rx_byte_core(u_char handle, int do_log_err, u_char slave, int *error)
{
    i2c_device_t* dev;
    int ret = 0;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    *error = PP_I2C_ERROR;
    
    if (PP_SUCCED(set_slave_address(dev, slave))) {
	if ((ret = i2c_smbus_read_byte(dev->fd)) < 0) {   
	    if (do_log_err) {
		pp_log("%s(): @0x%02x on '%s' failed\n", ___F, slave,
		       dev->name);
	    }
	} else {
	    *error = PP_I2C_NO_ERROR;
	    D("I2C RX  %02x, %02x on '%s' (%s)\n",
	      dev->slave, ret, dev->name, dev->devicename);
	}
    }
    return (u_char) ret;
}
    
void
pp_i2c_tx_byte(u_char handle, u_char slave, u_char value, int *error)
{
    assert(error);
    assert(handle < i2c_dev_cnt);
    
    pp_i2c_lock(handle);
    pp_i2c_tx_byte_core(handle, PP_I2C_DO_LOG_ERR, slave, value, error);
    pp_i2c_unlock(handle);
}

void
pp_i2c_tx_byte_core(u_char handle, int do_log_err, u_char slave,
		    u_char value, int *error)
{
    i2c_device_t* dev;    
    int ret;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    *error = PP_I2C_ERROR;
   
    if (PP_SUCCED(set_slave_address(dev, slave))) {
	D("I2C TX  %02x, %02x on '%s' (%s)\n",
	  dev->slave, value, dev->name, dev->devicename);
	if ((ret = i2c_smbus_write_byte(dev->fd, value)) < 0) {
	    if (do_log_err) {
		pp_log("%s(): @0x%02x on '%s' failed\n",
		       ___F, slave, dev->name);
	    }
	} else {
	    *error = PP_I2C_NO_ERROR;
	}
    }
}

u_char
pp_i2c_rx_byte_data(u_char handle, u_char slave, u_char reg, int *error)
{
    __s32 ret;

    assert(error);
    assert(handle < i2c_dev_cnt);

    pp_i2c_lock(handle);
    ret = pp_i2c_rx_byte_data_core(handle, PP_I2C_DO_LOG_ERR, slave,
				   reg, error);
    pp_i2c_unlock(handle);
    return ret;
}

u_char
pp_i2c_rx_byte_data_core(u_char handle, int do_log_err, u_char slave,
			 u_char reg, int *error)
{
    i2c_device_t* dev;
    int ret = 0;    

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];

    *error = PP_I2C_ERROR;

    if (PP_SUCCED(set_slave_address(dev, slave))) {
	if ((ret = i2c_smbus_read_byte_data(dev->fd, reg)) < 0) {    
	    if (do_log_err) {			
		pp_log("%s(): @0x%02x on '%s' failed\n",
		       ___F, slave, dev->name);
	    }
	} else {
	    D("I2C RXD %02x, %02x, %02x on '%s' (%s)\n",
	      dev->slave, reg, ret, dev->name, dev->devicename);
	    *error = PP_I2C_NO_ERROR;
	}
    }
    return (u_char) ret;
}

void
pp_i2c_tx_byte_data(u_char handle, u_char slave, u_char reg, u_char value, int *error)
{
    assert(error);
    assert(handle < i2c_dev_cnt);
    
    pp_i2c_lock(handle);
    pp_i2c_tx_byte_data_core(handle, PP_I2C_DO_LOG_ERR, slave, reg,
			     value, error);
    pp_i2c_unlock(handle);
}

void
pp_i2c_tx_byte_data_core(u_char handle, int do_log_err, u_char slave UNUSED,
			 u_char reg, u_char value, int *error)
{
    i2c_device_t* dev;
    int ret = 0;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    *error = PP_I2C_ERROR;

    if (PP_SUCCED(set_slave_address(dev, slave))) {
	D("I2C TXD %02x, %02x, %02x on '%s' (%s)\n",
	  dev->slave, reg, value, dev->name, dev->devicename);
	if ((ret = i2c_smbus_write_byte_data(dev->fd, reg, value)) < 0) {
	    if (do_log_err) {
		pp_log("%s(): @0x%02x on '%s' failed\n",
		       ___F, slave, dev->name);
	    }
	} else {
	    *error = PP_I2C_NO_ERROR;
	}
    }
}

u_short
pp_i2c_rx_word_data(u_char handle, u_char slave, u_char reg, int *error)
{
    __s32 ret;

    assert(error);
    assert(handle < i2c_dev_cnt);

    pp_i2c_lock(handle);
    ret = pp_i2c_rx_word_data_core(handle, PP_I2C_DO_LOG_ERR, slave,
				   reg, error);
    pp_i2c_unlock(handle);
    return ret;
} 

u_short
pp_i2c_rx_word_data_core(u_char handle, int do_log_err, u_char slave, u_char reg, int *error)
{
    i2c_device_t* dev;
    int ret = 0;    
	
    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];

    *error = PP_I2C_ERROR;

    //pp_i2c_lock(handle);

    if (PP_SUCCED(set_slave_address(dev, slave))) {
	if ((ret = i2c_smbus_read_word_data(dev->fd, reg)) < 0) {
		if (do_log_err) {			
		pp_log("%s(): @0x%02x on '%s' failed\n",
		       ___F, slave, dev->name);
	    }
	} else {
	    D("I2C RXD %02x, %02x, %02x on '%s' (%s)\n",
	      dev->slave, reg, ret, dev->name, dev->devicename);
	    *error = PP_I2C_NO_ERROR;
	}
    }
    
    //pp_i2c_unlock(handle);
    //printf("word %x\n",ret);//george

    return (u_short) ret;
} 

void
pp_i2c_tx_word_data(u_char handle, u_char slave, u_char reg,
		    u_short value, int *error)
{
    assert(error);
    assert(handle < i2c_dev_cnt);
    
    pp_i2c_lock(handle);
    pp_i2c_tx_word_data_core(handle, PP_I2C_DO_LOG_ERR, slave, reg,
			     value, error);
    pp_i2c_unlock(handle);    
}



void
pp_i2c_tx_word_data_core(u_char handle, int do_log_err, u_char slave, u_char reg,
		    u_short value, int *error)
{
    i2c_device_t* dev;
    int ret = 0;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    
    *error = PP_I2C_ERROR;

    //pp_i2c_lock(handle);
    
    if (PP_SUCCED(set_slave_address(dev, slave))) {
	D("I2C TXWD %02x, %02x, %02x on '%s' (%s)\n",
	  dev->slave, reg, value, dev->name, dev->devicename);
	if ((ret = i2c_smbus_write_word_data(dev->fd, reg, value)) < 0) {
         if (do_log_err) {
		pp_log("%s(): @0x%02x on '%s' failed\n",
		       ___F, slave, dev->name);
	    }
	} else {
	    *error = PP_I2C_NO_ERROR;
		}
    }

    //pp_i2c_unlock(handle);    
}
void pp_i2c_tx_burst_core(u_char handle, u_char slave, const u_char* buf,
		     u_short count, int *error) {
    i2c_device_t* dev;
    int ret;

    assert(error);
    assert(handle < i2c_dev_cnt);

    dev = &i2c_devices[handle];
    *error = PP_I2C_ERROR;

    if (PP_SUCCED(set_slave_address(dev, slave))) {
	D("I2C TXBurst %02x, Length %d on '%s' (%s)\n",
	  dev->slave, count, dev->name, dev->devicename);
	ret = write(dev->fd, buf, count);
	if (ret < count) {
	    pp_log("%s(): @0x%02x on '%s' failed\n", ___F, slave, dev->name);
	} else {
	    *error = PP_I2C_NO_ERROR;
	}
    }
}

void pp_i2c_tx_burst(u_char handle, u_char slave, const u_char* buf,
		     u_short count, int *error) {
    assert(error);
    assert(handle < i2c_dev_cnt);
    pp_i2c_lock(handle);
    pp_i2c_tx_burst_core(handle, slave, buf, count, error);
    pp_i2c_unlock(handle);
}

int pp_i2c_rx_burst_core(u_char handle, u_char slave, u_char* buf,
			 u_short buf_size, int *error) {
    i2c_device_t* dev;
    int ret = 0;

    assert(error);
    assert(handle < i2c_dev_cnt);
    
    dev = &i2c_devices[handle];
    *error = PP_I2C_ERROR;
    
    if (PP_SUCCED(set_slave_address(dev, slave))) {
	D("I2C RXBurst %02x, Length %d on '%s' (%s)\n",
	  dev->slave, buf_size, dev->name, dev->devicename);
	ret = read(dev->fd, buf, buf_size);
	if (ret < 0) {
	    ret = 0;
	    pp_log("%s(): @0x%02x on '%s' failed\n", ___F, slave, dev->name);
	} else {
	    *error = PP_I2C_NO_ERROR;
	}
    }
    return ret;
}

int pp_i2c_rx_burst(u_char handle, u_char slave, u_char* buf,
		    u_short buf_size, int *error) {
    int ret;
    assert(error);
    assert(handle < i2c_dev_cnt);
    pp_i2c_lock(handle);
    ret = pp_i2c_rx_burst_core(handle, slave, buf, buf_size, error);
    pp_i2c_unlock(handle);
    return ret;
}

