/**
 * Copyright 2001 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: little vector implementation to store
 *              an arbitrary number of elements.
 *              the elements have to be of the same
 *              type in case the element delete function is
 *              going to be used
 */

#ifndef __ERIC_VECTOR_H
#define __ERIC_VECTOR_H

#include <assert.h>
#include <stdlib.h>
#include <pp/mallocator.h>

#ifdef __cplusplus
extern "C" {
#endif
    
#define VECTOR_INCREMENT  32

typedef void (*vector_elem_del_func)(pp_mallocator_t*a, void* element);
typedef void (*vector_elem_del_func_simple)(void*);

/*
 * the vector object
 */
typedef struct {
    /*
     * current size of the vector
     */
    size_t size;

    /*
     * current capacity of the vector
     */
    size_t capacity;

    /*
     * size of an element
     */
    size_t elemsize;

    /*
     * myown is a flag rememering whether these methods
     * are resonsible for deallocating the vector or not
     */
    int state;

    int has_const_elements;
    int has_non_const_elements;

    /*
     * array of elements
     */
    union {
	void** elements;
	const void ** elements_const;
    };

    /*
     * memory allocator
     */
    pp_mallocator_t* alloc;

    /*
     * method to cleanup one element
     * we define it only ones per vector as we assume a vector
     * contains only elements of the same type
     */
    vector_elem_del_func delelem;
} vector_t;

/**
 * initializes the vector for void* pointers
 * in case the overgiven vector_t sructure pointer is NULL
 * we are going to allocate a new one
 */
vector_t* vector_new(vector_t* vec,
		     size_t initcapacity,
		     vector_elem_del_func_simple delelem);

/**
 * initializes a vector with the given alloc
 */
vector_t* vector_new_with_alloc(vector_t* vec,
				size_t initcapacity,
				vector_elem_del_func delelem,
				pp_mallocator_t* a);
    
/**
 * initializes a vector for elements of the given element size
 * ATTENTION: if you provicde a delelem function make sure the function
 *            doesn't try to reclaim the memory of element itself but
 *            only of its members.
 *            The element's memory is managed by the vector
 *            (in difference to a vector initialized with vector_new).
 */
vector_t* vector_new2(vector_t* vec, 
		      size_t initcapacity,
		      size_t elemsize,
		      vector_elem_del_func_simple delelem);

vector_t* vector_new2_with_alloc(vector_t* vec,
				 size_t initcapacity,
				 size_t elemsize,
				 vector_elem_del_func delelem,
				 pp_mallocator_t* a);

/**
 * add the element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
void vector_add(vector_t* vec, void* element);
    
/**
 * add the const element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
void vector_add_const(vector_t* vec, const void* element);
    
/**
 * copy a structure into the vector.
 * the vector, most probably, must have been initialized with vector_new2
 */
void vector_add2(vector_t* vec, void* elemptr);

/**
 * add the integer element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
static inline void vector_add_int(vector_t* vec, int element)
{
    vector_add_const(vec, (const void*)element);
}

/**
 * add the unsigned integer element to the vector
 * this may lead to dynamically increasing the
 * capacity of the vector
 */
static inline void vector_add_u_int(vector_t* vec, unsigned int element)
{
    vector_add_const(vec, (const void*)element);
}

/**
 * add all elements of the given vector to this
 * vector.
 * Note that all the elements will be deleted with
 * the delete function of this vector
 */
void vector_addvec(vector_t* vec, vector_t* addvec, void* (*copyelem)(void* elm));

/*
 * In case the copy function is empty, a bit-copy will be
 * performed, elements are not deleted from the other vector
 */
void
vector_addvec2(vector_t* vec, vector_t* addvec,
	       void (*copyelem)(void* dest, void* src, size_t size));
    
/**
 * returns the element at the specified index
 */
void * vector_get(const vector_t* vec, unsigned long idx);
    
/**
 * returns the const element at the specified index
 */
const void * vector_get_const(const vector_t* vec, unsigned long idx);
    
/**
 * returns a pointer to the element at the specified index
 */
void* vector_get2(const vector_t* vec, unsigned long idx);

/**
 * returns the integer element at the specified index
 */
static inline int vector_get_int(const vector_t* vec, unsigned long idx)
{
    const void * v = vector_get_const(vec, idx);
    return (int)v;
}
    
/**
 * returns the unsigned integer element at the specified index
 */
static inline unsigned int vector_get_u_int(const vector_t* vec, unsigned long idx)
{
    const void * v = vector_get_const(vec, idx);
    return (unsigned int)v;
}
    
/**
 * sets the element at the specified index
 * ATTENTION, asserts that the index exists
 */
void vector_set(const vector_t* vec, unsigned long idx, void* v);
    
/**
 * sets the const element at the specified index
 * ATTENTION, asserts that the index exists
 */
void vector_set_const(const vector_t* vec, unsigned long idx, const void* v);
    
/**
 * copies a element pointed to by vptr into the vector
 */
void vector_set2(const vector_t* vec, unsigned long idx, void* vptr);

/**
 * sets the integer element at the specified index
 * ATTENTION, asserts that the index exists
 */
static inline void vector_set_int(const vector_t* vec, unsigned long idx, int v)
{
    vector_set_const(vec, idx, (const void*)v);
}

/**
 * sets the unsigned integer element at the specified index
 * ATTENTION, asserts that the index exists
 */
static inline void vector_set_u_int(const vector_t* vec, unsigned long idx, unsigned int v)
{
    vector_set_const(vec, idx, (const void*)v);
}

/**
 * swaps two elements in the vector
 */
void vector_swap(const vector_t* vec, unsigned long idx1, unsigned long idx2);
    
/**
 * removes the last element and returns it. The element
 * is not deleted. asserts that vector_size > 0.
 */
void* vector_pop(vector_t* vec);
    
/**
 * returns the number of elements currently
 * contained in the vector
 */
size_t vector_size(const vector_t* vec);

/**
 * removes element at the given index and moves
 * all other elements one index to the front
 */
void vector_remove(vector_t* vec, unsigned long idx);

/**
 * removes element at the given index and moves
 * all other elements one index to the front
 * ATTENTION: element memory is not deleted!
 */
void vector_remove_dont_delete(vector_t* vec, unsigned long idx);

/**
 * sortes elements in vector using the specified 
 * compare function
 */
void vector_qsort(vector_t* vec, int(*comp)(const void*, const void*));

/**
 * cleans up all the elements using the element
 * delete function, if any, and deletes the allocated
 * memory
 */
void vector_delete(vector_t* vec);

/**
 * deletes the vector but leaves the elements untouched
 * nomatter whether the element delete function is set or not
 */
void vector_delete_wo_elems(vector_t* vec);
    
#ifdef __cplusplus
}
#endif

#endif /* __PP_VECTOR_H */
