/**
 * Copyright 2002 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: Properties are like config entries,
 *              i.e name value pairs however they are not
 *              stored in a file, but in a hashtable
 *              only. Properties may be serialized into a flat buffer and
 *              deserialized from such a buffer.
 *              The serialization format is identical to that one
 *              used by the config file system
 */
#include <stdio.h>
#include <malloc.h>
#include <assert.h>
#include <pp/base.h>

/* ----------------------------------------- *
 * local stuff
 * ----------------------------------------- */
 
#define KEY_STATE       0
#define VALUE_STATE     1
#define COMMENT_STATE   2

typedef struct listener_entry_s {
    struct list_head listnode;
    char* prop;
    pp_prop_listener_t* listener;
} listener_entry_t;

typedef void (*for_all_listeners_cb)(pp_prop_t* props,
				     pp_prop_listener_t* l,
				     const char* prop);
static void fire(pp_prop_t* props, for_all_listeners_cb cb,
			      const char* prop);
static void fire_added(pp_prop_t* props, pp_prop_listener_t* l,
		       const char* prop);
static void fire_removed(pp_prop_t* props, pp_prop_listener_t* l,
			 const char* prop);
static void fire_changed(pp_prop_t* props, pp_prop_listener_t* l,
			 const char* prop);

/* -----------------------------------------
 * API Implementation
 * ----------------------------------------- */
    
int pp_prop_read(pp_prop_t* props, void * buf, size_t buf_size)
{
    return pp_hash_deserialize(props->nvs, buf, buf_size);
}

int pp_prop_write(pp_prop_t* props, void * buf, size_t buf_size)
{
    return pp_hash_serialize(props->nvs, buf, buf_size);
}

pp_prop_t * pp_prop_new(pp_prop_t* props, size_t initsize) {
    if(NULL == props) {
	if(NULL == (props = (pp_prop_t*)malloc(sizeof(pp_prop_t))))
	    return NULL;
    	props->myown = 1;
    } else {
	props->myown = 0;
    }
    props->nvs = NULL;
    props->listeners.next = props->listeners.prev = NULL;
    if(NULL == (props->nvs = pp_hash_create(initsize))) {
	pp_prop_delete(props);
	return NULL;
    }
    INIT_LIST_HEAD(&props->listeners);
    return props;
}

void pp_prop_delete(pp_prop_t* props) {
    struct list_head *ptr, *pcl = &props->listeners;
    if(pcl->next != NULL && pcl->prev != NULL) {
	listener_entry_t* le;
	for (ptr = pcl->next; ptr != pcl; ptr = ptr->next) {
	    le = list_entry(ptr, listener_entry_t, listnode);
	    list_del(&le->listnode);
	    free(le);
	}
    }
    pcl->next = pcl->prev = NULL;
    if(props->nvs != NULL) pp_hash_delete(props->nvs);
    if(props->myown) free(props);
}

char* pp_prop_get(pp_prop_t* props, const char* name) {
    return pp_hash_get_entry(props->nvs, name);
}

int pp_prop_set(pp_prop_t* props, const char* prop, char* value) {
    char* old = pp_hash_get_entry(props->nvs, prop);
    if(NULL == old) {
	if(NULL == value) return -1; // no old and no new value
	if(0 > pp_hash_set_entry(props->nvs, prop, strdup(value), free))
	    return -1;
	fire(props, fire_added, prop);
    } else {
	if(NULL == value) {
	    char* cp = strdup(prop);  // prop might point into hash and be del
	    if (NULL == cp) return -1;
	    pp_hash_delete_entry(props->nvs, prop);
	    fire(props, fire_removed, cp);
	    free(cp);
	} else {
	    if(strcmp(value, old)) { // set only if really different
		if(0 > pp_hash_set_entry(props->nvs, prop,
						strdup(value), free))
		    return -1;
		fire(props, fire_changed, prop);
	    }
	}
    }
    return 0;
}

int pp_prop_add_listener(pp_prop_t* props, pp_prop_listener_t* listener,
			 const char* prop) {
    listener_entry_t* le;
    assert(props->listeners.next != NULL && props->listeners.prev != NULL);
    if (NULL == (le = malloc(sizeof(listener_entry_t)))) return -ENOMEM;
    le->listener = listener;
    le->prop = prop == NULL ? NULL : strdup(prop);
    list_add(&le->listnode, &props->listeners);
    return 0;
}

int pp_prop_remove_listener(pp_prop_t* props, pp_prop_listener_t* listener) {
    struct list_head *ptr, *pcl = &props->listeners;
    listener_entry_t *le;
    assert(props->listeners.next != NULL && props->listeners.prev != NULL);
    for(ptr = pcl->next; ptr != pcl; ptr = ptr->next) {
	le = list_entry(ptr, listener_entry_t, listnode);
	if(le->listener == listener) {
	    list_del(&le->listnode);
	    free(le->prop);
	    free(le);
	    return 0;
	}
    }
    return -EINVAL;
}

/* -----------------------------------------
 * local Implementation 
 * ----------------------------------------- */
static void fire(pp_prop_t* props, for_all_listeners_cb cb, const char* prop) {
    struct list_head *ptr, *pcl = &props->listeners;
    listener_entry_t *le;
    assert(props->listeners.next != NULL && props->listeners.prev != NULL);
    for(ptr = pcl->next; ptr != pcl; ptr = ptr->next) {
	le = list_entry(ptr, listener_entry_t, listnode);
	if(NULL == le->prop || !strcmp(prop, le->prop))
	    cb(props, le->listener, prop);
    }
}

static void fire_added(pp_prop_t* props, pp_prop_listener_t* l,
		const char* prop) {
    if(l->added) {
	l->added(props, prop);
    }
}

static void fire_removed(pp_prop_t* props, pp_prop_listener_t* l,
		  const char* prop) {
    if(l->removed) {
	l->removed(props, prop);
    }
}

static void fire_changed(pp_prop_t* props, pp_prop_listener_t* l,
		  const char* prop) {
    if(l->changed) {
	l->changed(props, prop);
    }
}
