#include <stdlib.h>
#include <pp/base.h>

struct _pp_queue_t {
    pp_queue_entry_t * head;
    pp_queue_entry_t * tail;
    pp_queue_type_t type;
};

pp_queue_t *
pp_queue_alloc(pp_queue_type_t queue_type)
{
    pp_queue_t * q;

    if ((q = (pp_queue_t *)malloc(sizeof(pp_queue_t))) == NULL) {
	return NULL;
    }

    q->type = queue_type;
    q->head = NULL;
    q->tail = NULL;

    return q;
}

void
pp_queue_free(pp_queue_t * queue)
{
    pp_queue_entry_t * qe;

    assert(queue);

    while ((qe = queue->head) != NULL) {
	queue->head = qe->prev;
	if (qe->len > 0) {
	    free(qe->data);
	}
	free(qe);
    }

    free(queue);
}

int
pp_queue_enqueue(pp_queue_t * queue, void * data, 
		size_t data_len, unsigned int priority)
{
    pp_queue_entry_t * qe, * qlook, * qprev;
    int reverse;

    assert(queue);

    if ((qe = (pp_queue_entry_t *)malloc(sizeof(pp_queue_entry_t))) == 0) {
	return -1;
    }

    if (data_len > 0) {
	if ((qe->data = malloc(data_len)) == NULL) {
	    free(qe);
	    return -1;
	}
	memcpy(qe->data, data, data_len);
    } else {
	qe->data = data;
    }

    qe->len	 = data_len;
    qe->priority = priority;
    qe->prev     = NULL;

    reverse = queue->type == QUEUE_TYPE_PRIORITY_REVERSE;
    
    if (queue->type == QUEUE_TYPE_PRIORITY || reverse) {
	qprev = NULL;
	if (queue->head != NULL) {
	    qlook = queue->head;
	    while (qlook) {
		if ((!reverse && qlook->priority < priority) ||
                    (reverse && qlook->priority > priority)) {
		    break;
		}
		qprev = qlook;
		qlook = qlook->prev;
	    }
	    qe->prev    = qlook;
	    if (qprev) {
		qprev->prev = qe;
	    } else {
		queue->head = qe;
	    }
	} else {
	    queue->head = qe;
	}
    } else {
	if (queue->tail != NULL) {
	    queue->tail->prev = qe;
	} else {
	    queue->head = qe;
	}
	queue->tail = qe;
    }
    return 0;
}

pp_queue_entry_t *
pp_queue_dequeue(pp_queue_t * queue)
{
    pp_queue_entry_t * qe;

    assert(queue);

    if ((qe = queue->head) != NULL) {
	queue->head = qe->prev;
	qe->prev = NULL;
    }

    if (queue->head == NULL) {
        queue->tail = NULL;
    }

    return qe;
}

/**
 * returns pointer to first entry in queue without dequeuing it
 */
pp_queue_entry_t *
pp_queue_first(pp_queue_t * queue)
{
    assert(queue);

    return queue->head;
}

/**
 * dequeues first entry in queue with assigned priority
 * returns NULL if no entry with that priority can be found
 */
pp_queue_entry_t * 
pp_queue_get(pp_queue_t * queue, unsigned int priority) {
    pp_queue_entry_t * qe, *succ;

    assert(queue);
    
    for(qe = queue->head, succ = NULL;
        qe && qe->priority != priority; /* search qe with priority */ 
        succ = qe, qe = qe->prev);
    
    if(qe) { /* we found an entry... */
        if(qe == queue->head) /* entry is head, set head to prev */
            queue->head = qe->prev;
        else { /* entry is not head, so we link succ and prev */
            if(qe == queue->tail) /* entry is tail, set tail to succ */
                queue->tail = succ;
            succ->prev = qe->prev;
        }
        qe->prev = NULL;
        
        if (queue->head == NULL) {
            queue->tail = NULL;
        }
    }

    return qe;
}

