/**
 * rbuf.c
 *
 * ring buffer implementation
 * 
 * generic ring buffer that is capable of tracking 
 * overflow conditions, if initialized to do so.
 *
 * (c) 2004 Peppercon AG, 2005/5/7, tbr@peppecon.de


 * TODO: implement missing fkts
 *       run valgrind 
 */

#include <string.h>
#include <pp/rbuf.h>
#include <pp/vector.h>

/* ringbuffer */
struct pp_rbuf_s {
    char*     buf;     // data buffer
    u_int     cap;     // capacity
    u_int     ridx;    // current read index
    u_int     widx;    // current write index
    int       is_ovfl; // overflow is current condition
    vector_t* ovfls;   // overflow idxes, may be NULL, depending on init
};

static int idx_distance(pp_rbuf_t* rbuf, u_int sidx, u_int eidx);
static u_int idx_inc(pp_rbuf_t* rbuf, u_int idx, u_int inc);
static u_int idx_dec(pp_rbuf_t* rbuf, u_int idx);
static void rbuf_ovfl_add(pp_rbuf_t* rbuf, u_int idx);
static int rbuf_ridx_is_ovfl(pp_rbuf_t* rbuf);
static size_t rbuf_remove(pp_rbuf_t* rbuf, char* buf, const size_t len);

/*
 * public method implementation
 * -----------------------------
 */

/*
 * initialices a ring buffer of the given size
 * the track_ovfl bits indicates whether overflows
 * will be tracked or not
 */
pp_rbuf_t* pp_rbuf_create(size_t capacity, int track_ovfl) {
    pp_rbuf_t* rbuf = malloc (sizeof(pp_rbuf_t));
    rbuf->buf = malloc (capacity);
    rbuf->cap = capacity;
    rbuf->ridx = rbuf->widx = 0;
    rbuf->is_ovfl = 0;
    if (track_ovfl) rbuf->ovfls = vector_new(NULL, 5, NULL);
    else rbuf->ovfls = NULL;
    return rbuf;
}

/*
 * cleans up rbuf
 */
void pp_rbuf_destroy(pp_rbuf_t* rbuf) {
    if (rbuf != NULL) {
	vector_delete(rbuf->ovfls);
	free(rbuf->buf);
	free(rbuf);
    }
}

/*
 * returns the free space in the ring buffer
 * 1 position we need to keep overflow seperate from ridx, that's why -1
 */
size_t pp_rbuf_space(pp_rbuf_t* rbuf) {
    return rbuf->cap - idx_distance(rbuf, rbuf->ridx, rbuf->widx) - 1;
}

/*
 * returns the number of bytes in the ring buffer
 */
size_t pp_rbuf_size(pp_rbuf_t* rbuf) {
    return idx_distance(rbuf, rbuf->ridx, rbuf->widx);
}

/*
 * Appends the buffer of given len to the ringbuf.
 * The buffer will not overflow but discard bytes that
 * don't fit in.
 *
 * returns the number of bytes that were actually
 * appended. this number may be smaller as requested
 * because of overflow.
 */
size_t pp_rbuf_append(pp_rbuf_t* rbuf, const char* buf, const size_t len) {
    u_int widx = rbuf->widx;
    u_int ridx = idx_dec(rbuf, rbuf->ridx);
    size_t clen;

    if (ridx == widx) {                      // full
	rbuf_ovfl_add(rbuf, widx);           // remember ovfl if first time
	return 0;
    } 
                                             // space left
    if (ridx > widx) {                       // ridx infront of widx
	if ((widx + len) > ridx) {           // overflow
	    clen = ridx - widx;
	    memcpy(&rbuf->buf[widx], buf, clen);
	    rbuf->widx = idx_inc(rbuf, widx, clen);
	    rbuf_ovfl_add(rbuf, rbuf->widx); // remember ovflow 
	    return clen;
	}
    } else {                                 // end of buf infront of widx
	if ((widx + len) > rbuf->cap) {      // wrap around
	    clen = rbuf->cap - widx;
	    memcpy(&rbuf->buf[widx], buf, clen);
	    rbuf->widx = 0;
	    clen += pp_rbuf_append(rbuf, &buf[clen], len - clen);
	    return clen;
	}
    }
    memcpy(&rbuf->buf[widx], buf, len);      // simply fits
    rbuf->widx = idx_inc(rbuf, widx, len);
    return len;
}

/*
 * Discards all data in rbuf
void pp_rbuf_clear(pp_rbuf_t* rbuf) {
    // TODO
}
 */

/*
 * checks whether next remove starts with a overflow condition,
 * i.e. would set the ovfl param of pp_rbuf_remove to TRUE
 */
int pp_rbuf_next_is_ovfl(pp_rbuf_t* rbuf) {
    return rbuf_ridx_is_ovfl(rbuf);
}

/*
 * Copies the requested number of bytes or all what is available
 * from the head of the ringbuf to the target buffer. In case a
 * overflow condition is inbetween the requested bytes, the continous
 * area will be copied only
 *
 * returns the number of bytes copied, i.e. 0 if empty
 *         the ovfl out parameter indicates whether there
 *         was a overflow condition infront of the copied buffer
 */
size_t pp_rbuf_remove(pp_rbuf_t* rbuf, int* ovfl, char* buf, 
		      const size_t len) {
    if (ovfl != NULL) *ovfl = 0;
    if (len == 0 || rbuf->widx == rbuf->ridx) return 0; // empty

    if (rbuf_ridx_is_ovfl(rbuf)) {
	vector_remove(rbuf->ovfls, 0);
	if (ovfl != NULL) *ovfl = 1;
    }
    rbuf->is_ovfl = 0;
    return rbuf_remove(rbuf, buf, len);
}


/*
 * private stuff
 * ----------------
 */

static size_t rbuf_remove(pp_rbuf_t* rbuf, char* buf, const size_t len) {
    u_int widx = rbuf->widx;
    u_int ridx = rbuf->ridx;
    vector_t* ovfls = rbuf->ovfls;
    u_int eidx;
    size_t clen, length = len;

    if (ridx == widx) return 0; // empty

    if (ovfls != NULL && (vector_size(ovfls)) > 0) {
	void * _eidx = vector_get(ovfls, 0);
 	eidx = (u_int)_eidx;
    } else {
	eidx = widx;
    }

    if (eidx <= ridx && len > (clen = rbuf->cap - ridx)) {  // wrapping needed
	memcpy(buf, &rbuf->buf[ridx], clen);
	rbuf->ridx = 0;
	clen += rbuf_remove(rbuf, &buf[clen], len - clen);
	return clen;
    } else if (len > (clen = eidx - ridx)) { // no wrap, but other limit
	length = clen;
    }
    memcpy(buf, &rbuf->buf[ridx], length);
    rbuf->ridx = idx_inc(rbuf, ridx, length);
    return length;
}
    
/*
 * calculates diff between indexes, taking wraparound into account
 */
static int idx_distance(pp_rbuf_t* rbuf, u_int sidx, u_int eidx) {
    int dist;

    if (sidx <= eidx) {
	dist = eidx - sidx;
    } else {
	dist = (rbuf->cap - sidx) + eidx;
    }
    return dist;
}

static u_int idx_inc(pp_rbuf_t* rbuf, u_int idx, u_int inc) {
    return (idx + inc) % rbuf->cap;
}

static u_int idx_dec(pp_rbuf_t* rbuf, u_int idx) {
    if (idx == 0) return rbuf->cap - 1;
    return idx - 1;
}

static void rbuf_ovfl_add(pp_rbuf_t* rbuf, u_int idx) {
    vector_t* ovfls = rbuf->ovfls;
    if (ovfls && !rbuf->is_ovfl) {
	vector_add(ovfls, (void*)(long)idx);
	rbuf->is_ovfl = 1;
    }
}

static int rbuf_ridx_is_ovfl(pp_rbuf_t* rbuf) {
    vector_t* ovfls = rbuf->ovfls;
    if (ovfls != NULL && vector_size(ovfls) > 0) {
	void * _idx = vector_get(ovfls, 0);
	return rbuf->ridx == (u_int)_idx;
    }
    return 0;
}
