/**
 * Access Control Lists
 * ACL is a wrapper around the config system providing higher level and
 * faster access to the permission objects and well as to ACL browsing
 * and manipulation funktions
 *
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 * Copyright 2004 Peppercon AG
 */

/* TODO: check for thread-safety/locking */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <regex.h>
#include <assert.h>
#include <pp/base.h>
#include <pp/vector.h>
#include <pp/cfg.h>
#include <pp/cfgx.h>
#include "acl_cache.h"
#include <pp/acl.h>
#include <pp/xdefs.h>
#include "cfg_intern.h"

#define str_compare (int(*)(const void*, const void*))strcmp

static int acl_objects_register(pp_acl_objects_t* objs);
static pp_acl_object_entry_t* acl_object_entry_create(const char* name,
						      const char* longname,
                                                      const char* type,
						      const char* ro_flag);
static void acl_object_entry_free(void * _object);
static pp_acl_perm_entry_t * acl_perm_entry_create(const char * name);
static void acl_perm_entry_free(void * _permission);
static pp_acl_t* acl_load_from_cfg(int gid);
static int acl_invalidate_local_cache_cb(pp_cfg_chg_ctx_t *ctx);

#if defined(PP_FEAT_RARITAN_PERMISSIONS)
static void perm_mapping_init(void);
static pp_hash_t* perm_mapping = NULL;
#endif /* PP_FEAT_RARITAN_PERMISSIONS */

static pp_acl_objects_t* acl_objs = NULL;
static pp_acl_custom_compare_cb_t acl_custom_compare;
static pthread_mutex_t acl_custom_compare_mtx;

const char* pp_acl_raasip_yes_str       = "yes";
const char* pp_acl_raasip_no_str        = "no";
const char* pp_acl_raasip_none_str      = "none";
const char* pp_acl_raasip_view_str      = "view";
const char* pp_acl_raasip_control_str   = "control";
const char* pp_acl_raasip_deny_str      = "deny";
const char* pp_acl_raasip_readonly_str  = "readonly";
const char* pp_acl_raasip_readwrite_str = "readwrite";


static const char * acl_errors[] = {
    /*00*/ "An internal ACL error occured.",
    /*01*/ "The ACL object doesn't exist.",
    /*02*/ "The ACL object's permission doesn't exist.",
    /*03*/ "The ACL doesn't exist.",    
};

static int errno_base = 1;
inline int pp_acl_errno_base() { return errno_base; }

static const char* get_error_string(const int error) {
    return acl_errors[error - errno_base];
}

void pp_acl_register_errnos() {
    errno_base = pp_register_errnos(sizeof(acl_errors) / sizeof(*acl_errors),
				    get_error_string);
}

void pp_acl_unregister_errnos() {
    pp_unregister_errnos(errno_base);
}

/*
 * ACL objects repository operations
 * =====================================
 */
/*
 * static
 * -------------
 */
static pp_acl_objects_t* acl_objects_create(void) {
    pp_acl_objects_t* os;
    
    os = (pp_acl_objects_t *)malloc(sizeof(pp_acl_objects_t));
    os->objects = pp_hash_create(32);

    MUTEX_CREATE_ERRORCHECKING(&os->mtx);
    return os;
}
    
static void acl_objects_free(void* _objects) {
    pp_acl_objects_t* os = (pp_acl_objects_t*) _objects;
    if (os != NULL) {
	pp_hash_delete(os->objects);
	free(os);
    }
}

static pp_acl_object_entry_t* acl_object_entry_create(const char* name,
						      const char* longname,
                                                      const char* type,
						      const char* ro_flag) {
						      
    pp_acl_object_entry_t* oe;

    assert(name);
    assert(longname);
    assert(type);
    assert(ro_flag);

    oe = malloc(sizeof(pp_acl_object_entry_t));
    memset(oe, 0, sizeof(pp_acl_object_entry_t));

    oe->name = strdup(name);
    oe->longname = strdup(longname);
    oe->perms = pp_hash_create(3);
    oe->type = strdup(type);
    oe->ro_flag = strdup(ro_flag);
    
    return oe;
}

static void acl_object_entry_free(void * _object) {
    pp_acl_object_entry_t * oe = (pp_acl_object_entry_t *)_object;
    if (oe) {
        free(oe->name);
        free(oe->longname);
        pp_hash_delete(oe->perms);
        free(oe->type);
        free(oe);
    }
}

/*
 * build up acl-objects from config
 */
#define INIT_PERM(__name__) \
            myperm = acl_perm_entry_create(__name__); \
            pp_hash_set_entry(acl_obj->perms, myperm->name, \
                              myperm, acl_perm_entry_free);
static int acl_objects_register(pp_acl_objects_t* objs) {
    const char* fn = ___F;
    pp_cfg_iter_t *oiter;
    const char* idx, *type, *object, *desc, *ro_flag;
    pp_acl_object_entry_t* acl_obj;
    pp_acl_perm_entry_t* myperm;

    if (PP_ERR != pp_cfg_iter_create_at_layer(&oiter, PP_PROFILE_LOCAL_CODE,
				              "device.perms[")) {
	while (pp_cfg_iter_next(oiter, &idx, NULL)) {
	    if (PP_ERR != pp_cfg_iter_get(oiter, &type, "type") &&
		PP_ERR != pp_cfg_iter_get(oiter, &object, "object") &&
		PP_ERR != pp_cfg_iter_get(oiter, &desc, "desc") && 
		PP_ERR != pp_cfg_iter_get(oiter, &ro_flag, "ro_flag")) {
		acl_obj = acl_object_entry_create(object, desc, type, ro_flag);
/* FIXME: type should be enum! */
                if(!strcmp(type, PP_CD_PERMTYPE_ADMIN_STR) || 
                   !strcmp(type, PP_CD_PERMTYPE_USER_STR)) {
                    /* user or admin perm */
		    INIT_PERM(pp_acl_raasip_yes_str);
		    INIT_PERM(pp_acl_raasip_no_str);
#if defined(PRODUCT_XX01IP_ANY) || defined(PP_FEAT_EXTENDED_PORT_ACLS)
                } else if(!strcmp(type, PP_CD_PERMTYPE_PORT_STR)) {
                    /* port perm */
 		    INIT_PERM(pp_acl_raasip_none_str);
		    INIT_PERM(pp_acl_raasip_view_str);
		    INIT_PERM(pp_acl_raasip_control_str);
#endif /* PRODUCT_XX01IP_ANY || PP_FEAT_EXTENDED_PORT_ACLS */
#if defined(PP_FEAT_EXTENDED_PORT_ACLS)
                } else if(!strcmp(type, PP_CD_PERMTYPE_VM_STR)) {
                    /* port perm */
 		    INIT_PERM(pp_acl_raasip_deny_str);
		    INIT_PERM(pp_acl_raasip_readonly_str);
		    INIT_PERM(pp_acl_raasip_readwrite_str);
#endif /* PP_FEAT_EXTENDED_PORT_ACLS */
#ifdef PP_FEAT_IPMI_SERVER
                } else if(!strcmp(type, PP_CD_PERMTYPE_PRIVLVL_STR)) {
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_NO_ACCESS_STR);
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_CALLBACK_STR);
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_USER_STR);
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_OPERATOR_STR);
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR);
                    INIT_PERM(PP_CD_IPMIPRIVLEVEL_OEM_STR);
#endif /* PP_FEAT_IPMI_SERVER */
                } else {
                    /* should not happen */
                    pp_log("invalid permission type %s\n", type);
                    assert(0);
                }
		pp_hash_set_entry(objs->objects, acl_obj->name, acl_obj,
				  acl_object_entry_free);
	    } else {
		pp_log("%s: ERROR at perms[%s]\n", fn, idx);
	    }
	}
    } else {
	pp_log("%s: WARNING: no device perms at all!\n", fn);
    }
    pp_cfg_iter_destroy(oiter);
    return PP_SUC;
}

#if defined(PP_FEAT_RARITAN_PERMISSIONS)
static void perm_mapping_init(void) {
    int size = 50; // fallback default
    pp_cfg_iter_t *oiter;
    const char* idx, *old_perm, *new_perm;

    pp_cfg_get_int_nodflt(&size, "device.perm_mapping._s_");
    perm_mapping = pp_hash_create(size);
    if (PP_ERR != (pp_cfg_iter_create_at_layer(&oiter, PP_PROFILE_LOCAL_CODE, "device.perm_mapping["))) {
        while (pp_cfg_iter_next(oiter, &idx, NULL)) {
            if (!PP_FAILED(pp_cfg_iter_get(oiter, &old_perm, "old_perm")) &&
                !PP_FAILED(pp_cfg_iter_get(oiter, &new_perm, "new_perm"))) {
                // initialize mapping table
                pp_hash_set_entry(perm_mapping, old_perm, strdup(new_perm), free);
            }
        }
    }
    pp_cfg_iter_destroy(oiter);
}
#endif /* PP_FEAT_RARITAN_PERMISSIONS */

/*
 * public
 * -------------
 */
int pp_acl_init(void) {
    int r;

    assert(acl_objs == NULL);
    pp_acl_register_errnos();
    
    acl_objs = acl_objects_create();

    // build up acl-objects from config
    r = acl_objects_register(acl_objs);
    
    // init acl-cache for user-acls
    acl_cache_init(acl_load_from_cfg);
    
    pp_cfg_add_change_listener(acl_invalidate_local_cache_cb, "group[?].acl");
#ifdef PP_FEAT_IPMI_SERVER
    pp_cfg_add_change_listener(acl_ipmi_set_perm_by_priv_level_cb,
                               "group[?].acl[ipmi_priv]");
#endif /* PP_FEAT_IPMI_SERVER */
    
#if defined(PP_FEAT_RARITAN_PERMISSIONS)
    assert(perm_mapping == NULL);
    perm_mapping_init();
#endif /* PP_FEAT_RARITAN_PERMISSIONS */

    return r;
}

void pp_acl_cleanup() {
    assert(acl_objs != NULL);
#ifdef PP_FEAT_IPMI_SERVER
    pp_cfg_rem_change_listener(acl_ipmi_set_perm_by_priv_level_cb,
                               "group[?].acl[ipmi_priv]");
#endif /* PP_FEAT_IPMI_SERVER */
    pp_cfg_rem_change_listener(acl_invalidate_local_cache_cb, "group[?].acl");
    acl_objects_free(acl_objs);
    pp_acl_unregister_errnos();

#if defined(PP_FEAT_RARITAN_PERMISSIONS)
    pp_hash_delete(perm_mapping);
    perm_mapping = NULL;
#endif /* PP_FEAT_RARITAN_PERMISSIONS */
}

static int acl_objects_name_compare(const void * x, const void * y) {
    pp_acl_object_entry_t * object1, * object2;
    int ret = 0;

    MUTEX_LOCK(&acl_objs->mtx);

    if ((object1 = pp_hash_get_entry(acl_objs->objects, *(const char * const *)x))!= NULL &&
	(object2 = pp_hash_get_entry(acl_objs->objects, *(const char * const *)y))!= NULL &&
	object1->longname && object2->longname) {

	ret = strcmp(object1->longname, object2->longname);
    }

    MUTEX_UNLOCK(&acl_objs->mtx);
    return ret;
}

static int
acl_object_custom_compare(const void * x, const void * y) {
    pp_acl_object_entry_t * object1, * object2;
    int ret = 0;
    
    assert(acl_custom_compare);

    MUTEX_LOCK(&acl_objs->mtx);

    if ((object1 = pp_hash_get_entry(acl_objs->objects, *(const char * const *)x)) != NULL &&
	(object2 = pp_hash_get_entry(acl_objs->objects, *(const char * const *)y)) != NULL) {

	ret = acl_custom_compare(object1, object2);
    }

    MUTEX_UNLOCK(&acl_objs->mtx);
    return ret;
}  

vector_t* pp_acl_objects_get_all(char* filter_incl, char* filter_excl) {
    return pp_acl_objects_get_all_custom_compare(filter_incl, filter_excl, NULL);
}

vector_t* pp_acl_objects_get_all_custom_compare(char* filter_incl, char* filter_excl,
                                          pp_acl_custom_compare_cb_t custom_compare) {
    pp_acl_object_entry_t* object;
    vector_t* ret;
    int use_obj;
    regex_t rgexp_incl, rgexp_excl;

    MUTEX_LOCK(&acl_objs->mtx);

    if (filter_incl != NULL) {
	if (regcomp(&rgexp_incl, filter_incl,
		    REG_ICASE | REG_EXTENDED | REG_NOSUB) != 0) {
	    pp_log_err("Could not compile including regexp %s\n", filter_incl);
	    filter_incl = NULL;
	}
    }
    if (filter_excl != NULL) {
	if (regcomp(&rgexp_excl, filter_excl,
		    REG_ICASE | REG_EXTENDED | REG_NOSUB) != 0) {
	    pp_log_err("Could not compile excluding regexp %s\n", filter_excl);
	    filter_excl = NULL;
	}
    }

    ret = vector_new(NULL, 30, free);
    for (object = pp_hash_get_first_entry(acl_objs->objects);
	 object != NULL;
	 object = pp_hash_get_next_entry(acl_objs->objects)) {

	/* if we use a filter, return only objects (not)
	 * matching the filter regex */
	use_obj = 1;
	if (filter_incl != NULL) {
	    use_obj &= (regexec(&rgexp_incl, object->name, 0, NULL, 0) == 0)
		? 1 : 0;
	}
	if (filter_excl != NULL) {
	    use_obj &= (regexec(&rgexp_excl, object->name, 0, NULL, 0) == 0)
		? 0 : 1;
	}

	if(use_obj) vector_add(ret, strdup(object->name));
    }

    MUTEX_UNLOCK(&acl_objs->mtx);

    if (custom_compare != NULL) {
        MUTEX_LOCK(&acl_custom_compare_mtx);
        acl_custom_compare = custom_compare;
        vector_qsort(ret, acl_object_custom_compare);
        MUTEX_UNLOCK(&acl_custom_compare_mtx);
    } else {
        vector_qsort(ret, acl_objects_name_compare);
    }

    if (filter_incl != NULL) { regfree(&rgexp_incl); }
    if (filter_excl != NULL) { regfree(&rgexp_excl); }

    return ret;
}

char* pp_acl_objects_get_long_name(const char * name) {
    pp_acl_object_entry_t * object;
    char * longname = NULL;

    assert(name);

    MUTEX_LOCK(&acl_objs->mtx);

    if ((object = pp_hash_get_entry(acl_objs->objects, name)) == NULL) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
    } else if (object->longname) {
	longname = strdup(object->longname);
    }

    MUTEX_UNLOCK(&acl_objs->mtx);

    return longname;
}

char* pp_acl_objects_get_read_only_flag(const char * name) {
    pp_acl_object_entry_t * object;
    char * ro_flag = NULL;

    assert(name);

    MUTEX_LOCK(&acl_objs->mtx);

    if ((object = pp_hash_get_entry(acl_objs->objects, name)) == NULL) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
    } else if (object->ro_flag) {
	ro_flag = strdup(object->ro_flag);
    }

    MUTEX_UNLOCK(&acl_objs->mtx);

    return ro_flag;
}

vector_t* pp_acl_objects_get_possible_perms(const char *name,
					    const char* prefix) {
    pp_acl_object_entry_t *object;
    pp_acl_perm_entry_t *perm;
    vector_t* ret = NULL;
    int pfx_len = 0;

    assert(name);

    MUTEX_LOCK(&acl_objs->mtx);

    if ((object = pp_hash_get_entry(acl_objs->objects, name)) == NULL) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
	goto bail;
    }

    if (prefix) pfx_len = strlen(prefix);

    ret = vector_new(NULL, 3, free);
    for (perm = pp_hash_get_first_entry(object->perms);
	 perm != NULL;
	 perm = pp_hash_get_next_entry(object->perms)) {
	char* t;
	    
	if (prefix) {
	    t = malloc(strlen(perm->name) + pfx_len + 1);
	    strcpy(t, prefix);
	    strcpy(t + pfx_len, perm->name);
	} else {
	    t = strdup(perm->name);
	}
	vector_add(ret, t);
    }
    vector_qsort(ret, str_compare);
 bail:
    MUTEX_UNLOCK(&acl_objs->mtx);
    return ret;
}

int pp_acl_objects_has_perm(const char* objname, const char* permname) {
    pp_acl_object_entry_t* o;
    pp_acl_perm_entry_t *perm;
    
    if (NULL == (o = pp_hash_get_entry(acl_objs->objects, objname))) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
	return PP_ERR;
    }
    for (perm = pp_hash_get_first_entry(o->perms); perm != NULL;
	 perm = pp_hash_get_next_entry(o->perms)) {
	if (!strcmp(perm->name, permname))
	    return 1;
    }
    return 0;
}

char* pp_acl_lookup_real_permission_name(const char * code_permission_name) {
    char * real_permission_name;

    if (!code_permission_name) return NULL;
#if defined(PP_FEAT_RARITAN_PERMISSIONS)
    if (strstr(code_permission_name, "access_port_") || 
        strstr(code_permission_name, "access_vm_") || 
        strstr(code_permission_name, "outlet_")) {
        real_permission_name = strdup(code_permission_name);
    }
    else {
        if ((real_permission_name = pp_hash_get_entry(perm_mapping, code_permission_name))) {
            real_permission_name = strdup(real_permission_name);
        }
#if !defined(NDEBUG)
        assert(real_permission_name);
#endif /* !NDEBUG */
    }
#else /* !PP_FEAT_RARITAN_PERMISSIONS */
    real_permission_name = strdup(code_permission_name);
#endif /* !PP_FEAT_RARITAN_PERMISSIONS */

    return real_permission_name;
}

/* FIXME: type should be enum! */
int pp_acl_objects_get_type(char **type, const char* objname) {
    pp_acl_object_entry_t* o;
    
    if (NULL == (o = pp_hash_get_entry(acl_objs->objects, objname))) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
	return PP_ERR;
    }
    *type = strdup(o->type);
    return PP_SUC;
}

/*
 * ACL operations that rely on the acl objects
 * =============================================
 */
#if 0 // TODO: unneeded?
int pp_acl_store_to_cfg(pp_acl_t* acl, int uid) {
    pp_hash_t* nacl;
    vector_t* perms;
    unsigned int i;
    char* obj;
// tmp RAASIP    char perm[PP_ACL_MAX_PERM_LEN + 1];
    int gid;
    
    if(PP_ERR == pp_cfg_get_int(&gid, "user[%u].group_id", uid)) {
        // todo, set errno?
        return PP_ERR;
    }

    pp_cfg_remove("group[%u].acl[]", gid);

    nacl = pp_acl_sort_perms_by_obj(acl);

    for (perms = pp_hash_get_first_entry(nacl); perms != NULL;
         perms = pp_hash_get_next_entry(nacl)) {
	
        assert(vector_size(perms) > 0);
	obj = ((pp_acl_entry_t*)vector_get(perms, 0))->object->name;
	    
	for (i = 0; i < vector_size(perms); ++i) {
            pp_acl_entry_t* ae = vector_get(perms, i);
// tmp RAASIP	    perm[0] = ae->is_negative ? '-' : '+';
// tmp RAASIP	    strncpy(&perm[1], ae->perm->name, PP_ACL_MAX_PERM_LEN - 1);
// tmp RAASIP	    pp_cfg_set(perm, "user[%u].acl[%s][%u]", uid, obj, i);
	    pp_cfg_set(ae->perm->name, "group[%u].acl[%s][%u]", gid, obj, i);
	}
    }
    
    return PP_SUC;
}
#endif

static pp_acl_t* acl_load_from_cfg(int gid) {
    const char* obj;
    const char* perm;
    pp_acl_t* acl = NULL;
    pp_cfg_iter_t* oiter;

    acl = pp_acl_create();

    if (PP_ERR != pp_cfg_iter_create(&oiter, "group[%u].acl[", gid)) {
	while(pp_cfg_iter_next(oiter, &obj, &perm)) {
// TODO: dummy... change pp_acl_add_entry
            if(0 > pp_acl_add_entry(acl, obj, perm, 1)) {
                pp_log("Adding ACL entry '%s: %s' failed: %s\n",
                       obj, perm, pp_error_string(errno));
            }
	}
	pp_cfg_iter_destroy(oiter);
    }
    
    return acl;
}

static pp_acl_perm_entry_t * acl_perm_entry_create(const char * name) {
    pp_acl_perm_entry_t * perm_entry;
    assert(name);

    perm_entry = malloc(sizeof(pp_acl_perm_entry_t));
    perm_entry->name = strdup(name);
    return perm_entry;
}

static void acl_perm_entry_free(void* _perm_entry) {
    pp_acl_perm_entry_t* perm_entry = (pp_acl_perm_entry_t *)_perm_entry;
    if (perm_entry) {
	free(perm_entry->name);
	free(perm_entry);
    }
}

static int acl_invalidate_local_cache_cb(pp_cfg_chg_ctx_t *ctx UNUSED) {
    pp_profile_type_t prof_type = PP_PROFILE_LOCAL;
    pp_profile_t *prof;
    int ret;
    
    if (PP_ERR == pp_cfg_get_profile(&prof, prof_type, NULL, NULL)) {
        abort(); // should never happen!
	return PP_ERR;
    }
    
    ret = pp_profile_invalidate_acl_cache(prof);
    
    pp_profile_release(prof);
    
    return ret;
}

pp_acl_t* pp_acl_create() {
    pp_acl_t* acl;

    acl = malloc(sizeof(pp_acl_t));
    acl->acl = pp_hash_create(50);
    acl->objects = acl_objs;
    acl->parent = NULL;
    pp_mallocator_mutex_init(pp_mallocator_heap(), &acl->mtx, PP_MUTEX_NORMAL);
    return acl;
}

int pp_acl_add_entry(pp_acl_t* acl, const char* object,
		     const char* perm, const int negative) {
    pp_acl_object_entry_t *oe;
    pp_acl_perm_entry_t *perm_ce;
    pp_acl_entry_t *ae;
    char * s;
    int s_size, ret = PP_ERR;

    assert(acl);
    assert(object);
    assert(perm);

    MUTEX_LOCK(&acl->objects->mtx);
    
    if ((oe = pp_hash_get_entry(acl->objects->objects, object)) == NULL) {
	errno = PP_ACL_ERR_OBJECT_DOESNT_EXIST;
	goto bail;
    }

    if ((perm_ce = pp_hash_get_entry(oe->perms, perm)) == NULL) {
	/* perm not allowed for object */
	errno = PP_ACL_ERR_OBJECT_PERM_DOESNT_EXIST;
	goto bail;
    }

    ae = malloc(sizeof(pp_acl_entry_t));
    ae->object = oe;
    ae->perm = perm_ce;
    ae->is_negative = negative;

    /* FIXME: use static string */
    s_size = strlen(object) + strlen(perm) + 2;
    s = malloc(s_size);
    snprintf(s, s_size, "%s:%s", object, perm);
    pp_hash_set_entry(acl->acl, s, ae, free);
    free(s);
    ret = PP_SUC;

 bail:
    MUTEX_UNLOCK(&acl->objects->mtx);
    return ret;
}

/* TODO: RAASIP! */
#if 0 // tmp RAASIP

/* calculates the union of two acls, following algo:
 * - acl will contain all perms of acl and other
 * - perms will be joined as follows:
 *   x, x  = x
 *   x, o  = x
 *   x, !x = optimistic ? + : -
 */
int pp_acl_union(pp_acl_t* me, pp_acl_t* other, int optimistic) {
    const char* ae_key;
    pp_acl_entry_t *ae_me;
    pp_acl_entry_t *ae_other;
        
    MUTEX_LOCK(&me->objects->mtx);
    // loop over me objects
    for (ae_key = pp_hash_get_first_key(me->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(me->acl)) {
	// if found in other objects
	if(NULL != (ae_other = pp_hash_get_entry(other->acl, ae_key))) {
	    ae_me = pp_hash_get_entry(me->acl, ae_key);
	    // apply rule
	    if(ae_me->is_negative != ae_other->is_negative) {
		ae_me->is_negative = optimistic ? 0 : 1;
	    }
	}
    }
    // loop over other objects
    for (ae_key = pp_hash_get_first_key(other->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(other->acl)) {
	// if not found in me
	if(NULL == (ae_me = pp_hash_get_entry(me->acl, ae_key))) {
	    ae_other = pp_hash_get_entry(other->acl, ae_key);
	    // duplicate acl entry
	    ae_me = malloc(sizeof(pp_acl_entry_t));
	    *ae_me = *ae_other;
	    // add acl entry
	    pp_hash_set_entry(me->acl, ae_key, ae_me, free);
	}
    }
    
    MUTEX_UNLOCK(&me->objects->mtx);
    return PP_SUC;
}

#endif //0

int pp_acl_del_entry(pp_acl_t* acl, const char* object, const char* perm) {
    pp_acl_entry_t *ae;
    char * s;
    int s_size, ret = PP_ERR;

    assert(acl && object && perm);

    /* FIXME: lock */

    /* FIXME: use static string */
    s_size = strlen(object) + strlen(perm) + 2;
    s = malloc(s_size);
    snprintf(s, s_size, "%s:%s", object, perm);

    /* get entry for deleting */
    if ((ae = pp_hash_get_entry(acl->acl, s)) == NULL) {
	errno = PP_ACL_ERR_ACL_DOESNT_EXIST; /* FIXME */
	goto bail;
    }
    
    pp_hash_delete_entry(acl->acl, s);
    ret = PP_SUC;

 bail:
    /* FIXME: unlock */
    free(s);
    return ret;
}

int pp_acl_del_all_entries(pp_acl_t* acl, const char* object) {
    vector_t *perms;
    
    assert(acl);
    assert(object);
    
    if(NULL != (perms = pp_acl_objects_get_possible_perms(object, NULL))) {
        int i, perms_sz = vector_size(perms);
        char *perm;
            
        for (i = 0; i < perms_sz; ++i) {
            perm = (char*)vector_get(perms, i);
            pp_acl_del_entry(acl, object, perm);
        }
        
        vector_delete(perms);
    }
    
    /* TODO: when does "delete all" fail? */
    return PP_SUC;
}

pp_acl_t* pp_acl_clone(pp_acl_t* acl) {
    pp_acl_t* newacl;
    pp_acl_entry_t *ae, *newae;    
    const char* ae_key;
    
    newacl = pp_acl_create();

    for (ae_key = pp_hash_get_first_key(acl->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(acl->acl)) {
	
	// duplicate acl entry
	ae = pp_hash_get_entry(acl->acl, ae_key);
	newae = malloc(sizeof(pp_acl_entry_t));
	*newae = *ae;
	// add acl entry
	pp_hash_set_entry(newacl->acl, ae_key, newae, free);
    }
    return newacl;
}

void
pp_acl_destroy(pp_acl_t* acl) {
    if (acl) {
	pp_hash_delete(acl->acl);
	free(acl);
    }
}

/* this is useful for visually browsing through an acl,        *
 * i.e. for displaying or serialization purposes,              *
 * it returns an hashtable, where the keys are the objectnames *
 * and the values are the vectors containing all the
 * acl-entries for that object
 * ATTENTION, the acl-entries will not be copied, so do not
 * delete them!
 */
pp_hash_t* pp_acl_sort_perms_by_obj(pp_acl_t* acl) {
    pp_hash_t*      allobjs;
    vector_t*       perms;
    pp_acl_entry_t* ae;

    allobjs = pp_hash_create(50);

    for (ae = pp_hash_get_first_entry(acl->acl); ae != NULL;
	 ae = pp_hash_get_next_entry(acl->acl)) {
	if (NULL == (perms = pp_hash_get_entry(allobjs, ae->object->name))) {
	    perms = vector_new(NULL, 2, NULL);
	    pp_hash_set_entry(allobjs, ae->object->name, perms,
			      (void(*)(void*))vector_delete);
	}
	vector_add(perms, ae);
    }
    return allobjs;
}

/* TODO: RAASIP! */
#if 0 // tmp RAASIP

int pp_acl_accumulate_all_permissions(pp_acl_t* usr_acl, vector_t* grp_acls,
				 const char* object, const char* perm) {
    return pp_acl_accumulate_all_permissions_biased(usr_acl, grp_acls,
						    object, perm, 0, 0, 0, 0);
}

int pp_acl_accumulate_all_permissions_biased(pp_acl_t* usr_acl,
					     vector_t* grp_acls,
					 const char* object, const char* perm,
					 int pos_usr_bias, int neg_usr_bias,
					 int pos_grp_bias, int neg_grp_bias) {

    int pos_user = pos_usr_bias, neg_user = neg_usr_bias;
    int pos_group = pos_grp_bias, neg_group = neg_grp_bias;
    unsigned int i;
    pp_acl_t* grp_acl;
    
    if (usr_acl) {
	pp_acl_accumulate_permission(usr_acl, object, perm,
				     &pos_user, &neg_user);
    }

    if (grp_acls) {
	for (i = 0; i < vector_size(grp_acls); i++) {
            grp_acl = vector_get(grp_acls, i);
	    pp_acl_accumulate_permission(grp_acl, object, perm,
					 &pos_group, &neg_group);
	}
    }

    return ((pos_user + (pos_group - neg_user)) -
	    (neg_user + (neg_group - pos_user)) > 0);

}

void pp_acl_accumulate_permission(const pp_acl_t* acl,
				 const char* object, const char* perm,
				 int* pos, int* neg) {
    pp_acl_entry_t* ae;
    
    assert(acl);
    
    /* acl-object is "", this is anyone */
    if(object && *object == '\0') {
        *pos |= 1;
    } else {
        /* for all acl entries that match the object and permission */
        for (ae = pp_hash_get_first_entry(acl->acl); ae != NULL;
             ae = pp_hash_get_next_entry(acl->acl)) {
            
            if (!strcmp(ae->object->name, object) &&
                !strcmp(ae->perm->name, perm)) {
                if (ae->is_negative) {
                    *neg |= 1;
                } else {
                    *pos |= 1;
                }
            }
        }
    }
}

#endif //0

#ifndef NDEBUG
// useful for debugging
void pp_acl_print(pp_acl_t* acl) {
    const char* ae_key;
    pp_acl_entry_t *ae;
    
    MUTEX_LOCK(&acl->objects->mtx);
    for (ae_key = pp_hash_get_first_key(acl->acl); ae_key != NULL;
	 ae_key = pp_hash_get_next_key(acl->acl)) {
	ae = pp_hash_get_entry(acl->acl, ae_key);
	printf("%s: %s%s\n", ae->object->name,
		ae->is_negative ? "-" : "+", ae->perm->name);
    }
    MUTEX_UNLOCK(&acl->objects->mtx);
}
#endif /* !NDEBUG */

