/**
 * Access Control Lists for BMC (IPMI)
 *
 * Author: Thomas Weber <tweb@raritan.com>
 * Copyright 2006 Peppercon AG
 */

// system includes

// fw includes
#include <pp/cfg.h>
#include <pp/xdefs.h>

// local includes
#include "cfg_intern.h"

/**
 * define a static matrix for privilege-level : acl-permission mapping.
 *
 * implementation (for now):
 * - affected acls are stored in acl_ipmi_objects char array.
 * - minimum privilege level for corresponding acl is stored in 
 *   acl_ipmi_min_priv_level int array.
 * - if privilege level is set to "no access" (0), the group becomes KVM-only
 *   and ACLs are managed explicitely via web frontend
 * - if privilege level > 0 is set and minimum privilege level required for a
 *   a permission specified in acl_ipmi_objects is
 *   - greater or equal, permission will be set to "yes"
 *   - smaller, permission will be set to "no"
 *
 * ATTENTION: this will only work for user/admin-permissions!!!
 *            think of full qualified mapping? 
 *
 *               \acl | net_s |  pc | time_s |  um | um_pw
 *  privilege level\  |       |     |        |     |
 * -------------------+-------+-----+--------+-----+-------
 *  no access (0)     |     - |   - |     -  |  -  |     -
 *  callback (1)      |    no |  no |     no |  no |    no
 *  user (2)          |    no |  no |     no |  no |    no
 *  operator (3)      |    no | yes |    yes |  no |    no
 *  administrator (4) |   yes | yes |    yes | yes |   yes
 *  OEM (5)           |   yes | yes |    yes | yes |   yes
 */
static const char *acl_ipmi_objects[]    = { "net_s", "pc", "time_s", "um", "um_pw" };
static int acl_ipmi_min_priv_level[] = {    4 ,   3 ,       3 ,   4 ,      4  };

static int acl_ipmi_set_perm_by_priv_level(int gid, int priv_level);
 
int acl_ipmi_set_perm_by_priv_level_cb(pp_cfg_chg_ctx_t *ctx) {
    int ret = PP_SUC;
    vector_t groups, *key_comps;
    unsigned int vec_sz, groups_sz = 0, u, v;
    int gid;
    
    vec_sz = vector_size(ctx->key_comp_vec);
    vector_new(&groups, vec_sz, NULL);
    
    /* at first, get groups with changed privilege level */
    for (u = 0; u < vec_sz; ++u) {
        key_comps = (vector_t *)vector_get(ctx->key_comp_vec, u);
        if(PP_SUCCED(pp_cfg_is_group_key(&gid, (const vector_t *)key_comps)) &&
           vector_size(key_comps) == 6 && /* groups._e_.7.acl._e_.ipmi_priv */
           !strcmp("ipmi_priv", pp_cfg_get_key_comp_name_at_idx(key_comps, 5))){
            for (v = 0; v < groups_sz; ++v) {
                if (gid == vector_get_int(&groups, v)) {
                    /* gid already in list, should occure only once */
                    break;
                }
	    }
            if (v == groups_sz) {
                /* gid not in groups vector, add it */
                vector_add_int(&groups, gid);
                ++groups_sz;
            }
        }
    }
    
    /* apply changes to all groups affected */
    for (u = 0; u < groups_sz; ++u) {
        /* priv level changed, apply changes for corresponding acls */
        char *priv_level = NULL;
        
        gid = vector_get_int(&groups, u);
        
        if(PP_ERR == pp_cfg_get_nodflt(&priv_level,
                                       "group[%u].acl[ipmi_priv]", gid)) {
            pp_log("error getting privilege level for group %d\n", gid);
            abort(); // should never happen
            break;
        }
        
        /* TODO: this is nearly equal to pp_bmc_get_privlevel_value
                 think of storing priv level IDs instead of strings? */
        if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_NO_ACCESS_STR)) {
            /* leave acls untouched, they are KVM-only now! */
        } else if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_CALLBACK_STR)) {
            acl_ipmi_set_perm_by_priv_level(gid, 1);
        } else if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_USER_STR)) {
            acl_ipmi_set_perm_by_priv_level(gid, 2);
        } else if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_OPERATOR_STR)) {
            acl_ipmi_set_perm_by_priv_level(gid, 3);
        } else if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_ADMINISTRATOR_STR)) {
            acl_ipmi_set_perm_by_priv_level(gid, 4);
        } else if(!strcmp(priv_level, PP_CD_IPMIPRIVLEVEL_OEM_STR)) {
            acl_ipmi_set_perm_by_priv_level(gid, 5);
        } else {
            /* this should not happen! */
            pp_log("error setting privilege level %s\n", priv_level);
            ret = PP_ERR;
            abort();
        }
        
        free(priv_level);
    }
    
    vector_delete(&groups);
    return ret;
}

static int acl_ipmi_set_perm_by_priv_level(int gid, int priv_level) {
    unsigned int u;

    for(u = 0; u < sizeof(acl_ipmi_min_priv_level) / sizeof(int); ++u) {
        assert(acl_ipmi_objects[u] && *acl_ipmi_objects[u]);
        if(priv_level >= acl_ipmi_min_priv_level[u]) {
            /* minimum priv level is smaller or equal current priv level */
            pp_cfg_set(pp_acl_raasip_yes_str, "group[%u].acl[%s]", 
                       gid, acl_ipmi_objects[u]);
        } else {
            /* current priv level is smaller than minimum */
            pp_cfg_set(pp_acl_raasip_no_str, "group[%u].acl[%s]", 
                       gid, acl_ipmi_objects[u]);
        }
    }
    
    return PP_SUC;
}

