/*
 * The basic config library providing
 * config transactions
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#define PRINT_KEY(key_comps) \
            int __idx, __kcsz = vector_size(key_comps); \
            if(__kcsz > 0) { \
                printf("%s", ((key_comp_t*)vector_get2(key_comps, 0))->name); \
                for(__idx = 1; __idx < __kcsz; __idx++) \
                    printf(".%s", ((key_comp_t*)vector_get2(key_comps, __idx))->name); \
                printf("\n"); \
            }

#ifdef PP_BUILD_TYPE_DEBUG
# define D(fmt, args...)	{ printf(fmt, ##args); }
#else
# define D(fmt, args...)	{ }
#endif
            
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <pp/dict.h>
#include <pp/profile.h>

#include <pp/cfg.h>
#include <pp/cfgx.h>

#include <liberic_config.h>

#include "cfg_intern.h"

/* cfg transaction structures */
/******************************/
struct cfg_tx_log_entry_s {
    cfg_tx_op_type_t    op_type;
    pp_profile_type_t   prof_type;
    vector_t*           key_comps;
    char*               value;
};

/**
 * Thread specific transaction context key
 */
pthread_key_t pp_cfg_tx_thread_ctx_key;

/*
 * Config transaction API
 * =================================================================
 */

cfg_tx_log_entry_t* cfg_tx_log_entry_create(cfg_tx_op_type_t  op_type,
                                            pp_profile_type_t prof_type,
					    vector_t* key_comps,
					    const char* value) {
    cfg_tx_log_entry_t* txle = malloc(sizeof(cfg_tx_log_entry_t));
    txle->op_type = op_type;
    txle->prof_type = prof_type;
    txle->key_comps = key_comps;
    txle->value = op_type == CFG_TX_OP_SET ? strdup(value) : NULL;
    return txle;
}

static void cfg_tx_log_entry_destroy(cfg_tx_log_entry_t* txle) {
    pp_cfg_key_comps_destroy(txle->key_comps);
    free(txle->value);
    free(txle);
}

static pp_cfg_tx_t* cfg_tx_create(void) {
    pp_cfg_tx_t* tx;

    tx = malloc(sizeof(pp_cfg_tx_t));
    tx->chg_log = vector_new(NULL, 10,
		 (void(*)(void*))cfg_tx_log_entry_destroy);
    MUTEX_CREATE_ERRORCHECKING(&tx->mtx);
    tx->commit_in_progress = 0;

    return tx;
}

static void cfg_tx_destroy(pp_cfg_tx_t* tx) {
    assert(tx);
    pp_cfg_tx_release_thread_ctx(tx);
    vector_delete(tx->chg_log);
    pthread_mutex_destroy(&tx->mtx);
    free(tx);
}

void cfg_tx_set_va(pp_cfg_tx_t* tx, const char* value,
		   pp_profile_type_t type, const char* key, va_list ap) {
    vector_t* key_comps;

    assert(type != PP_PROFILE_NONE && type != PP_PROFILE_EFFECTIVE);
    assert(tx);
    
    key_comps = cfg_key_expand(key, ap);

    /* do not destroy key comps, cfg_tx_log_entry_destroy cares about! */
    cfg_tx_set_key_comps(tx, value, type, key_comps);
}

void cfg_tx_set_key_comps(pp_cfg_tx_t* tx, const char* value,
			  pp_profile_type_t type, vector_t* key_comps) {
    cfg_tx_log_entry_t* txle;
    cfg_tx_op_type_t op = value == NULL ? CFG_TX_OP_REM : CFG_TX_OP_SET;

    assert(type != PP_PROFILE_NONE && type != PP_PROFILE_EFFECTIVE);
    assert(tx);
    
    /* do not destroy key comps, cfg_tx_log_entry_destroy cares about! */
    txle = cfg_tx_log_entry_create(op, type, key_comps, value);
    MUTEX_LOCK(&tx->mtx);
    vector_add(tx->chg_log, txle);
    MUTEX_UNLOCK(&tx->mtx);
}

void cfg_tx_remove_va(pp_cfg_tx_t* tx, pp_profile_type_t type,
	    	      const char* key, va_list ap)
{
    cfg_tx_set_va(tx, NULL, type, key, ap);
}

int cfg_tx_get_key_comps(pp_cfg_tx_t* tx, char **value,
                         pp_profile_type_t type, const vector_t *key_comps) {
    cfg_tx_log_entry_t* txle;
    unsigned int tx_log_sz = 0, u = 0;
    int ret = PP_ERR;

    assert(type != PP_PROFILE_NONE);
    assert(tx);
    assert(key_comps);
    
    MUTEX_LOCK(&tx->mtx);
    tx_log_sz = vector_size(tx->chg_log);
    if(type != PP_PROFILE_EFFECTIVE) {
        /* we want a special profile type, got get it! */
        for(u = 0; u < tx_log_sz; ++u) {
            txle = (cfg_tx_log_entry_t*)vector_get(tx->chg_log, u);
            assert(txle->key_comps);
            if(txle->prof_type == type &&
               pp_cfg_key_compare(txle->key_comps, key_comps)) {
                if(txle->op_type != CFG_TX_OP_REM) {
                    /* there is a value, go copy it! */
                    *value = strdup(txle->value);
                } else {
                    /* no value, set *value to NULL */
                    *value = NULL;
                }
                ret = txle->prof_type;
                break;
            }
        }
    } else {
        /* same for PP_PROFILE_EFFECTIVE, do not break! */
        pp_profile_type_t effective = PP_PROFILE_NONE;
        for(u = 0; u < tx_log_sz; ++u) {
            txle = (cfg_tx_log_entry_t*)vector_get(tx->chg_log, u);
            assert(txle->key_comps);
            if(pp_cfg_key_compare(txle->key_comps, key_comps) &&
               txle->prof_type > effective) {
                if(effective != PP_PROFILE_NONE) {
                    free(*value);
                }
                if(txle->op_type != CFG_TX_OP_REM) {
                    /* there is a value, go copy it! */
                    *value = strdup(txle->value);
                } else {
                    /* no value, set *value to NULL */
                    *value = NULL;
                }
                ret = txle->prof_type;
                effective = txle->prof_type;
            }
        }
    }
    MUTEX_UNLOCK(&tx->mtx);
    
    return ret;
}

/* public fuctions
 * ================
 */

/**
 * Start a config transaction.
 * @return the transaction context
 */
pp_cfg_tx_t* pp_cfg_tx_begin(int assign_to_thread) {
    pp_cfg_tx_t *tx = cfg_tx_create();
    if(assign_to_thread) {
        pp_cfg_tx_set_thread_ctx(tx);
    }
    return tx;
}

/**
 * Commits a config transaction
 *
 * The transaction context will be destroyed.
 *
 * in case there are listeners registered for any of the config keys
 * that was set during the transaction, their callback will be fired
 * exactly once per commit
 */
int pp_cfg_tx_commit(pp_cfg_tx_t* tx) {
    return pp_cfg_tx_commit_core(tx, DO_FLUSH, 1, 1, NULL);
}

int pp_cfg_tx_commit_core(pp_cfg_tx_t* parent_tx, pp_cfg_flush_t flush, 
                          int flush_lock, int threaded, char **retstr) {
    unsigned int i, u, cl_sz, chg_log_sz;
    int ret = PP_SUC;
    cfg_tx_log_entry_t* txle;
    // 1 slot for each profile type
    pp_profile_t* profs[PP_PROFILE_COUNT];
    int prof_hits[PP_PROFILE_COUNT];
    int dirty = 0;
    pp_profile_type_t prty;
    vector_t *cl, *chglisteners = NULL, *chglistener_keys;
    cfg_notify_thr_t* notify = NULL;
    cfgl_listener_t *cfgl_listener;
    pp_hash_i_t *chglistener_hash = NULL;
    cfgl_keys_t *cfgl_keys = NULL;
    pp_cfg_tx_t *tx;

    /* init */
    memset(&profs, 0, sizeof(pp_profile_t*) * PP_PROFILE_COUNT); 
    memset(prof_hits, 0, sizeof(int) * PP_PROFILE_COUNT);
    
    /* generate new transaction
       * assure in usercode, that tx_commit is the last operation on tx, you
         will get a segfault otherwhise...
       * we are the only one, working with it */
    MUTEX_LOCK(&parent_tx->mtx);
    tx = malloc(sizeof(pp_cfg_tx_t));
    tx->chg_log = parent_tx->chg_log;
    tx->mtx = parent_tx->mtx;
    tx->commit_in_progress = 1;
    free(parent_tx); /* do not destroy... chg_log will be distroyed by tx */
    parent_tx = NULL;
    MUTEX_UNLOCK(&tx->mtx); /* unlock old parent_tx mutex! */
    /* bind tx to thread, free parent_tx */
    pp_cfg_tx_set_thread_ctx(tx);

    chg_log_sz = vector_size(tx->chg_log);

    /* if we don't want to modify anything, happily return */
    if(chg_log_sz == 0) {
       cfg_tx_destroy(tx);
       return ret;
    }
    
    /* no file access from now -> commit is pseudo-atomar */
    if(flush_lock) {
        eric_config_set_flush_lock();
    }
    
    // get all profiles and lock them
    for (i = 0; i < chg_log_sz; ++i) {
	txle = vector_get(tx->chg_log, i);
	prty = txle->prof_type;
	assert(prty != PP_PROFILE_NONE && prty != PP_PROFILE_EFFECTIVE);
	if (profs[prty] == NULL) {
	    if (PP_ERR == (ret = pp_cfg_get_profile(&profs[prty], prty, 
						    NULL, NULL))) {
		// TODO: warning, pp_cfg_get_profile needs to get the key!!!
		goto bailout;
	    }
	    pp_mutex_lock(&profs[prty]->mtx);
	}
    }

    // we need to lock the listeners at this point already, as we are
    // going to collect them during our bunch of set operations. If somebody
    // would remove them during that time, they were not valid anymore, so
    // this is what we prevent with this early lock
    MUTEX_LOCK(&cfg_listeners_mtx);
    
    // collect change listeners
    for (i = 0; i < chg_log_sz; ++i) {
	txle = vector_get(tx->chg_log, i);

	if (NULL != (cl = cfg_lookup_change_listeners(txle->key_comps))) {
            if (chglisteners == NULL) {
                chglisteners = vector_new2(NULL, 20, sizeof(cfgl_listener_t),
                                           NULL);
                chglistener_hash = pp_hash_create_i(20);
            }
            cl_sz = vector_size(cl);
            for(u = 0; u < cl_sz; ++u) {
                cfgl_listener = vector_get2(cl, u);
                if(NULL == (chglistener_keys =
                                pp_hash_get_entry_i(chglistener_hash, 
                                                    cfgl_listener->uid))) {
                    chglistener_keys = 
                        vector_new(NULL, 5,
                                   threaded ?
                                   (void(*)(void*))pp_cfg_key_comps_destroy :
                                   NULL);
                    pp_hash_set_entry_i(chglistener_hash, cfgl_listener->uid,
                                        chglistener_keys,
                                        (void(*)(void*))vector_delete);
                }

                vector_add(chglistener_keys, 
                           threaded ? cfg_key_duplicate(txle->key_comps) :
                                      txle->key_comps);
            }
            vector_addvec2(chglisteners, cl, NULL);
            vector_delete(cl);
        }
    }
    MUTEX_UNLOCK(&cfg_listeners_mtx);

    // unify chglistener vector (remove duplicates) and notify
    if (chglisteners != NULL) {
        cfgl_keys = malloc(sizeof(cfgl_keys_t));
        cfgl_keys->type = CFGL_KEYS_HASH;
        cfgl_keys->value = chglistener_hash;

	cfg_unify_change_listeners(chglisteners, cfgl_keys);

        /* do not call within-tx-handlers threaded! */
	ret = cfg_notify_change_listeners(tx, chglisteners, PP_CFG_CHG_UNDEF,
                                          cfgl_keys, &notify,
                                          CFGL_TYPE_WITHIN_TX);

#ifndef PP_BOARD_PEMX
        notify->retstr = retstr;
        cfg_notify_thr_func((void*)notify);
        ret = notify->retval;
        free(notify);
        notify = NULL;
#else /* PP_BOARD_PEMX */
        ret = PP_ERR;
        pp_log("cfg_fire_change: can't execute notify\n");
        abort();
#endif /* PP_BOARD_PEMX */
    }

    /* size may differ due to changes, change handlers applied! */
    chg_log_sz = vector_size(tx->chg_log);

    // finally, apply all the wonderful changes, we made
    for (i = 0; i < chg_log_sz; ++i) {
	txle = vector_get(tx->chg_log, i);
        switch(txle->op_type) {
	    case CFG_TX_OP_SET:
                cfg_set_to_prof_core(txle->value, profs[txle->prof_type],
                                     txle->key_comps);
                break;
            case CFG_TX_OP_REM:
                cfg_remove_from_prof_core(profs[txle->prof_type],
                                          txle->key_comps, 
                                          vector_size(txle->key_comps) - 1);
                break;
            default:
                pp_log("Undefined cfg tx operation %u\n", txle->op_type);
                assert(0);
        }
        ++prof_hits[txle->prof_type];
    }

    /* save configs and flush if needed */
    for(i = 1; i < PP_PROFILE_COUNT; ++i) {
        if(prof_hits[i] > 0) {
            cfg_save_profile(profs[i], flush, 0);
            dirty = 1;
        }
    }
    if(dirty && flush == DO_FLUSH) {
        eric_config_flush();
    }
    
 bailout:
    // release all locks
    for (i = 1; i < PP_PROFILE_COUNT; ++i) {
	if (profs[i] != NULL) {
	    pp_mutex_unlock(&profs[i]->mtx);
            pp_profile_release(profs[i]);
	}
    }
    if(flush_lock) {
        eric_config_release_flush_lock();
    }

    /* call post tx change handlers */
    if (ret != PP_ERR && chglisteners != NULL) {
       	ret = cfg_notify_change_listeners(NULL, chglisteners, PP_CFG_CHG_UNDEF,
               	                          cfgl_keys, threaded ? NULL : &notify,
		                          CFGL_TYPE_POST_TX);
#ifndef PP_BOARD_PEMX
	if(!threaded) {
            notify->retstr = retstr;
            cfg_notify_thr_func((void*)notify);
            ret = notify->retval;
            free(notify);
            notify = NULL;
            /* flush again for post tx handlers to be saved */
            /* TODO! 
               To we need that? Should post tx handler do that on their own? */
            if(flush == DO_FLUSH) {
                eric_config_flush();
            }
	}
#else /* PP_BOARD_PEMX */
        ret = PP_ERR;
        pp_log("cfg_fire_change: can't execute notify\n");
        abort();
#endif /* PP_BOARD_PEMX */
    } else {
        /* we did not call cfg_notify_change_listeners!
           if we run threaded, care about cleaning up! */
        threaded = 0; /* TODO: check that!!! */
    }
    
    if(!threaded) {
        /* clean up if we do not run threaded */
        vector_delete(chglisteners);
        pp_hash_delete_i(chglistener_hash);
        free(cfgl_keys);
    }

    free(notify); /* sanity - maybe this can be removed now? */
    cfg_tx_destroy(tx);
    
    return ret;
}

/**
 * Discards all changes since the begin of this transaction,
 * i.e. they will not be written through to the profile
 *
 * The transaction context will be destroyed.
 */
void pp_cfg_tx_rollback(void* tx) {
    cfg_tx_destroy((pp_cfg_tx_t*)tx);
}

/**
 * Sets active tx in thread context
 * @param tx transaction to set
 */
void pp_cfg_tx_set_thread_ctx(pp_cfg_tx_t* tx) {
    pthread_setspecific(pp_cfg_tx_thread_ctx_key, (void*)tx);
}

/**
 * Gets active tx from thread context
 * @return current transaction set in thread context
 */
pp_cfg_tx_t* pp_cfg_tx_get_thread_ctx(void) {
    return (pp_cfg_tx_t*)pthread_getspecific(pp_cfg_tx_thread_ctx_key);
}

/**
 * Releases active tx in thread context
 * @param tx transaction to be released, NULL for any tx set
 */
void pp_cfg_tx_release_thread_ctx(pp_cfg_tx_t* tx) {
    if(!tx || tx == pp_cfg_tx_get_thread_ctx()) {
        pp_cfg_tx_set_thread_ctx(NULL);
    }
}    

