#include <fcntl.h>
#include <dirent.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/ipc.h>
#include <sys/mount.h>
#include <sys/sem.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/vfs.h>
#include <pp/base.h>
#include <syslog.h>
#include <liberic_config.h>
#include <pp/sem_keys.h>

// make it compile.. ;-)
#define pp_log printf
#define pp_log_err(fmt, args...)        { printf(fmt "\n", ##args); \
                                         perror("error");}
#define pp_system           system

/* ------------------------------------------------------------------------- *
 * global definitions
 * ------------------------------------------------------------------------- */

#define CONFIG_DIR	"."

/* ------------------------------------------------------------------------- *
 * global variables
 * ------------------------------------------------------------------------- */

/* eric_config_flash_write_mtx *MUST* be recursive */
pthread_mutex_t eric_config_flash_write_mtx = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static pthread_mutex_t flush_mtx = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t flush_cond = PTHREAD_COND_INITIALIZER;
static pthread_t flusher_thread;
static int flusher_die = 0;
static int flush = 0;
static flush_type_t flush_type;
static int avoid_flushing = 0;
static int flush_sem_id;
int eric_config_ext_change_sem_id = -1;
int eric_config_ext_change_ntfy; /* external change notify flag */
static struct sembuf flush_lock[2] = {
    {
	sem_num: 0,
	sem_op: 0,
	sem_flg: 0
    },
    {
	sem_num: 0,
	sem_op: 1,
	sem_flg: SEM_UNDO
    }
};
static struct sembuf flush_unlock[1] = {
    {
	sem_num: 0,
	sem_op: -1,
	sem_flg: (IPC_NOWAIT | SEM_UNDO)
    }
};

static int initialized = 0;

/* ------------------------------------------------------------------------- *
 * prototypes
 * ------------------------------------------------------------------------- */

static void * flusher(void * arg);
static void do_flush(void);
static int config_file_exists(const char * filename);
static int config_file_modified(const char * filename, time_t * mtime);
static int config_file_select(const struct dirent * dirp);

/* ------------------------------------------------------------------------- *
 * initialization / cleanup
 * ------------------------------------------------------------------------- */

int
eric_config_init(flush_type_t _flush_type)
{
    const char * fn = ___F;

    flush_type = _flush_type;

    if ((flush_sem_id = semget(FLUSH_SEM_KEY, 1, IPC_CREAT | 0666)) == -1) {
	pp_log_err("%s(): semget()", fn);
	return -1;
    }

    /*
    if (flush_type == FLUSH_IN_BACKGROUND) {
	if (pthread_create(&flusher_thread, 64 * 1024, flusher, NULL)){
	    pp_log("%s(): Cannot create flusher-thread.\n", fn);
	    return -1;
	}
    }
    */
    initialized = 1;

    return 0;
}

void
eric_config_cleanup(void)
{
    int cond_err = 0;

    if (initialized) {
	/* stop flusher if necessary */
	if (flush_type == FLUSH_IN_BACKGROUND) {
	    MUTEX_LOCK(&flush_mtx);
	    flusher_die = 1;
	    if (pthread_cond_signal(&flush_cond) != 0) {
		pp_log("%s(): pthread_cond_signal() failed\n", ___F);
		cond_err = 1;
	    }
	    MUTEX_UNLOCK(&flush_mtx);
	    if (!cond_err) {
//		pthread_join(flusher_thread, NULL);
	    }
	}
    }
    initialized = 0;
}

/* ------------------------------------------------------------------------- *
 * API routines
 * ------------------------------------------------------------------------- */

int
eric_config_file_changed_since(const char * name, time_t * mtime)
{
    char filepath[MAXPATHLEN + 1];
    int ret;
    
    snprintf(filepath, sizeof(filepath), "%s/%s", CONFIG_DIR, name);

    ret = config_file_exists(filepath) && config_file_modified(filepath,mtime);

    return ret;
}

void
eric_config_install_defaults(void)
{
    pp_sem_lock(flush_sem_id);

    MUTEX_LOCK(&eric_config_flash_write_mtx);
    pp_system("/bin/default_config_be.sh > /dev/null");
    MUTEX_UNLOCK(&eric_config_flash_write_mtx);

    pp_sem_unlock(flush_sem_id);
}

int
eric_config_file_write(const char * name,
		       eric_config_option_t ** options, int create)
{
    return eric_config_file_write_core(name, options, create, 1);
}

int
eric_config_file_write_core(const char * name,
		            eric_config_option_t ** options,
                            int create, int lock)
{
    const char * fn = ___F;
    struct statfs statfs_buf;
    struct stat stat_buf;
    char filepath[MAXPATHLEN + 1], filetemp[MAXPATHLEN + 1];
    int open_flags = O_CREAT|O_TRUNC|O_RDWR;
    int fd = -1;
    FILE * fp = NULL;
    u_int i;
    int r = -1;
    struct flock fl = {
	l_type:   F_WRLCK,
	l_whence: SEEK_SET,
	l_start:  0,
	l_len:    0,
	l_pid:    getpid()
    };

    snprintf(filepath, sizeof(filepath), "%s/%s", CONFIG_DIR, name);
    snprintf(filetemp, sizeof(filetemp), "%s/%s_tmp", CONFIG_DIR, name);
    
    if (create) {
	open_flags |= O_EXCL;
    }

    if (lock) pp_sem_lock(flush_sem_id);

    if (create && !stat(filepath, &stat_buf)) {
	r = -2;
	goto bail;
    }
    
    if ((fd = open(filetemp, open_flags, 0644)) == -1) {
	pp_log_err("%s(): open() of '%s'", fn, filetemp);
	goto bail;
    }

    if (fstatfs(fd, &statfs_buf) == -1) {
	pp_log_err("%s(): fstatfs()", fn);
	goto bail;
    }

    /* do not lock on nfs volume - used for development */
    if (statfs_buf.f_type != 0x6969 && fcntl(fd, F_SETLKW, &fl) == -1) {
	pp_log_err("%s(): fcntl()", fn);
	goto bail;
    }

    if ((fp = fdopen(fd, "w+")) == NULL) {
	pp_log_err("%s(): fdopen() of '%s'", fn, filetemp);
	goto bail_unlock;
    }
	
    for (i = 0; options[i] != NULL; i++) {
	if (options[i]->key   != NULL && options[i]->key[0] &&
	    options[i]->value != NULL && options[i]->value[0]) {
	    
	    if (fprintf(fp, "%s=%s\n", options[i]->key, options[i]->value) !=
		(int)(strlen(options[i]->key) + strlen(options[i]->value) + 2)) {
		pp_log_err("%s(): fprintf() failed", fn);
		goto bail_unlock;
	    }

	}
    }

    if (rename(filetemp, filepath)) {
	pp_log_err("%s(): rename() failed", fn);
	goto bail_unlock;
    }
    
    fflush(fp);

    r = 0;

 bail_unlock:
    /* do not unlock on nfs volume - used for development */
    fl.l_type = F_UNLCK;
    if (statfs_buf.f_type != 0x6969 && fcntl(fd, F_SETLKW, &fl) == -1) {
	pp_log_err("%sfcntl()", fn);
    }

 bail:
    if (fp != NULL) {
	fclose(fp);
    } else if (fd != -1) {
	close(fd);
    }

    if (lock) pp_sem_unlock(flush_sem_id);

    return r;
}

int
eric_config_file_delete(const char * name)
{
    char filepath[MAXPATHLEN + 1];

    snprintf(filepath, sizeof(filepath), "%s/%s", CONFIG_DIR, name);

    pp_sem_lock(flush_sem_id);

    if (unlink(filepath) == -1) {
	pp_log_err("%s(): unlink() of '%s' failed", ___F, filepath);
	return -1;
    }

    pp_sem_unlock(flush_sem_id);

    return 0;
}

int
eric_config_file_for_each_option(const char * name,
				 for_each_option_cb_t cb, ...)
{
    const char * fn = ___F;
    va_list args;
    struct statfs statfs_buf;
    char filepath[MAXPATHLEN + 1];
    int r = -1;
    FILE * fp;
    u_int line;
    char key[MAX_OPT_KEY_LEN];
    char value[MAX_OPT_VALUE_LEN];
    eric_config_option_t option = { key, value };
    char nl[2];
    struct flock fl = {
	l_type:   F_RDLCK,
	l_whence: SEEK_SET,
	l_start:  0,
	l_len:    0,
	l_pid:    getpid()
    };

    snprintf(filepath, sizeof(filepath), "%s/%s", CONFIG_DIR, name);

    if ((fp = fopen(filepath, "r")) == NULL) {
	pp_log_err("%s(): fopen() of '%s'", fn, filepath);
	goto bail;
    }

    if (fstatfs(fileno(fp), &statfs_buf) == -1) {
	pp_log_err("%s(): fstatfs()", fn);
	goto bail;
    }

    /* do not lock on nfs volume - used for development */
    if (statfs_buf.f_type != 0x6969 && fcntl(fileno(fp), F_SETLKW, &fl) == -1){
	pp_log_err("%s(): fcntl()", fn);
	goto bail;
    }

    for (line = 0; !feof(fp) && !ferror(fp); line++) {
	int n = fscanf(fp, "%63[^= \t\n]=%2047[^\n]%1[\n]", key, value, nl);
	if (n >= 0 && n < 3) {
	    pp_log("parse error in line %u of '%s'\n", line + 1, filepath);
	    while (!feof(fp) && !ferror(fp) &&
		   fgetc(fp) != '\n');
	} else if (n == 3) {
	    va_start(args, cb);
	    cb(&option, args);
	    va_end(args);
	}
    }

    r = 0;

    /* do not unlock on nfs volume - used for development */
    fl.l_type = F_UNLCK;
    if (statfs_buf.f_type != 0x6969 && fcntl(fileno(fp), F_SETLKW, &fl) == -1){
	pp_log_err("%s(): fcntl()", fn);
    }

 bail:
    if (fp != NULL) {
	fclose(fp);
    }

    return r;
}
 
int
eric_config_for_all_files(for_all_files_cb_t cb, ...)
{
    struct dirent ** user_config_dirents;
    va_list args;
    int i, n;

    if (cb == NULL) {
	return -1;
    }

    if ((n = scandir(CONFIG_DIR, &user_config_dirents,
		     config_file_select, alphasort)) == -1) {
	pp_log_err("%s(): scandir() failed", ___F);
	return -1;
    }

    for (i = 0; i < n; i++) {
	va_start(args, cb);
	cb(user_config_dirents[i]->d_name, args);
	va_end(args);
	free(user_config_dirents[i]);
    }

    free(user_config_dirents);

    return 0;
}

int
eric_config_flush(void)
{
    int ret = 0;

    switch (flush_type) {
      case FLUSH_IN_FOREGROUND:
	  eric_config_flush_in_foreground();
	  break;
      case FLUSH_IN_BACKGROUND:
	  MUTEX_LOCK(&flush_mtx);
	  flush = 1;
	  if (pthread_cond_signal(&flush_cond) != 0) {
	      pp_log("%s(): pthread_cond_signal() failed\n", ___F);
	      ret = -1;
	  }
	  MUTEX_UNLOCK(&flush_mtx);
	  break;
    }

    return ret;
}

void
eric_config_flush_in_foreground(void)
{
    do_flush();
}

void
eric_config_avoid_flushing(int avoid)
{
    pp_sem_lock(flush_sem_id);
    avoid_flushing = avoid;
    pp_sem_unlock(flush_sem_id);
}

eric_config_option_t*
eric_config_option_create(const char* key, const char* value)
{
    eric_config_option_t* o = malloc(sizeof(eric_config_option_t));
    o->key = strdup(key);
    o->value = strdup(value);
    return o;
}

void
eric_config_option_destroy(void* opt)
{
    if (opt) {
	eric_config_option_t* o = (eric_config_option_t*)opt;
	free(o->key);
	free(o->value);
	free(o);
    }
}


/* ------------------------------------------------------------------------- *
 * internal routines
 * ------------------------------------------------------------------------- */

static void *
flusher(void * arg UNUSED)
{
    while (!flusher_die) {
	MUTEX_LOCK(&flush_mtx);
	while(!flush && !flusher_die) {
	    pthread_cond_wait(&flush_cond, &flush_mtx);
	}
	MUTEX_UNLOCK(&flush_mtx);

	if (flush) {
	    flush = 0;
	    do_flush();
	}
    }

    flusher_die = 0;
    return NULL;
}

static void
do_flush(void)
{
    pp_sem_lock(flush_sem_id);
    MUTEX_LOCK(&eric_config_flash_write_mtx);
    if (!avoid_flushing) {
	pp_system("/bin/flash_config.sh > /dev/null");
    }
    MUTEX_UNLOCK(&eric_config_flash_write_mtx);
    pp_sem_unlock(flush_sem_id);
}

static int
config_file_exists(const char * filename)
{
    struct stat st;

    return (stat(filename, &st) != -1 || errno != ENOENT);
}


static int
config_file_modified(const char * filename, time_t * mtime)
{
    struct stat st;

    if (stat(filename, &st) == -1) { return 1; }

    if ((time_t)*mtime != (time_t)st.st_mtime) {
	*mtime = st.st_mtime;
	return 1;
    }

    return 0;
}

static int
config_file_select(const struct dirent * dirp)
{
    if (strcmp(dirp->d_name, ".") && strcmp(dirp->d_name, "..")) {
	return 1;
    }
	
    return 0;
}

int
eric_config_set_flush_lock(void)
{
    return pp_sem_lock(flush_sem_id);
}

int
eric_config_release_flush_lock(void)
{
    return pp_sem_unlock(flush_sem_id);
}

int
eric_config_ext_changed(void)
{
    return 0; 
}   

