#include <pp/cim.h>
#include <liberic_pthread.h>

#include <pp/clp.h>

#include "cim_common.h"
#include "namespace.h"
#include "provider_common.h"

int pp_cim_init()
{
    pp_cim_ns_init();
    return PP_SUC;
}

void pp_cim_cleanup()
{
    pp_cim_ns_deinit();
}

/*
 * Add up to n levels of children of a given CIM instance to an
 * instance vector.
 *
 * @param instance   Add children of this instance
 * @param instances  Instance vector, element type: pp_cim_instance_t
 * @param level      Maximum recursion depth, -1: unlimited
 * @param ufct       Add only instances matching this UFcT, NULL: all classes
 */
static void add_children(pp_cim_instance_t *instance, vector_t *instances,
    int level, const char *ufct)
{
    pp_cim_instance_t *i;
    pthread_mutex_lock(&instance->mutex);
    i = instance->children;
    while (i) {
        if (i->cim_class->clp_update) {
            i->cim_class->clp_update(i->cim_class);
        }
        if (!ufct || pp_cim_qualifies_as_ufct(i, ufct)) {
            vector_add(instances, i);
        }
        if (level > 1 || level < 0)
            add_children(i, instances, level - 1, ufct);
        i = i->next;
    }
    pthread_mutex_unlock(&instance->mutex);
}

/*
 * Lookup a CLP target structure.
 *
 * @param target          CLP target
 * @param default_target  Default target UFiP; element type: pp_clp_ufit_t
 *
 * @return  Instance vector; element type: pp_cim_instance_t
 *
 * Note: All used instances will be locked and automatically released when
 *       deleting the result vector.
 */
vector_t *pp_cim_clp_lookup(pp_clp_target_t *target, vector_t *default_target)
{
    pp_cim_instance_t *instance = NULL;
    pp_cim_instance_t *instance2 = NULL;
    vector_t *instances;
    pp_cim_class_t *class = NULL;
    unsigned int i, j;
    vector_t *result = NULL;

    pthread_rwlock_rdlock(&pp_cim_repository_lock);

    switch (target->type) {
        case CLP_TARGET_DEFAULT:
            instance = pp_cim_lookup_absolute(default_target);
            break;
        case CLP_TARGET_INSTANCE:
        case CLP_TARGET_SET:
            if (target->ufip1_relative) {
                instance = pp_cim_lookup_absolute(default_target);
                if (!instance)
                    goto bail;
                instance = pp_cim_lookup_relative(instance, target->ufip1);
            } else {
                instance = pp_cim_lookup_absolute(target->ufip1);
            }
            break;
        case CLP_TARGET_ASSOCIATION:
            if (target->ufip1_relative) {
                instance = pp_cim_lookup_absolute(default_target);
                if (!instance)
                    goto bail;
                instance = pp_cim_lookup_relative(instance, target->ufip1);
            } else {
                instance = pp_cim_lookup_absolute(target->ufip1);
            }
            if (target->ufip2) {
                if (target->ufip2_relative) {
                    instance2 = pp_cim_lookup_absolute(default_target);
                    if (!instance2)
                        goto bail;
                    instance2 = pp_cim_lookup_relative(instance2, target->ufip2);
                } else {
                    instance2 = pp_cim_lookup_absolute(target->ufip2);
                }
                if (!instance2)
                    goto bail;
            }
            break;
        default:
            goto bail;
    }

    if (!instance)
        goto bail;

    result = vector_new(NULL, 0,
        (vector_elem_del_func_simple)pp_cim_instance_untag);
    switch (target->type) {
        case CLP_TARGET_DEFAULT:
        case CLP_TARGET_INSTANCE:
            if (instance->cim_class->clp_update) {
                instance->cim_class->clp_update(instance->cim_class);
            }
            vector_add(result, instance);
            if (target->level > 1 || target->level < 0) {
                add_children(instance, result, target->level - 1, NULL);
            }
            break;
        case CLP_TARGET_SET:
            if (instance->cim_class->clp_update) {
                instance->cim_class->clp_update(instance->cim_class);
            }
            add_children(instance, result, target->level, target->ufct);
            break;
        case CLP_TARGET_ASSOCIATION:
            class = pp_cim_class_lookup(target->ufct);
            if (!class)
                goto bail;
            if (!class->association) {
                pp_cim_class_release(class);
                goto bail;
            }
            if (class->update)
                class->update(class);
            instances = pp_cim_get_instances(class);
            pp_cim_class_release(class);
            for (i = 0; i < vector_size(instances); i++) {
                pp_cim_instance_t *assoc = vector_get(instances, i);
                vector_t *properties;
                pp_cim_instance_t *ref1 = NULL, *ref2 = NULL;

                properties = pp_cim_get_properties(assoc);
                for (j = 0; j < vector_size(properties); j++) {
                    pp_cim_propval_t *pv = vector_get(properties, j);
                    if (pv->property->type == PP_CIM_REFERENCE) {
                        if (!ref1) {
                            ref1 = pv->data.null ? NULL : pv->data.types.reference;
                        } else {
                            ref2 = pv->data.null ? NULL : pv->data.types.reference;
                            break;
                        }
                    }
                }
                vector_delete(properties);

                if (!ref1 || !ref2) {
                    printf("Warning: CIM association %s does not contain enough references.\n",
                        class->cim_name);
                }

                if (!target->ufip2) {
                    if (ref1 == instance || ref2 == instance) {
                        vector_add(result, assoc);
                    }
                } else {
                    if ((ref1 == instance && ref2 == instance2) ||
                        (ref2 == instance && ref1 == instance2)) {
                        vector_add(result, assoc);
                    }
                }
            }
            vector_delete(instances);
            break;
        default:
            goto bail;
    }

    for (i = 0; i < vector_size(result); i++) {
        instance = vector_get(result, i);
        pp_cim_instance_tag(instance);
    }

    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return result;

bail:
    if (result)
        vector_delete(result);
    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return NULL;
}

/*
 * Find associations referencing a given instance.
 *
 * @param instance  CIM instance
 * @param name      Association class name (NULL for all)
 *
 * @return  Association instance vector; element type: pp_cim_instance_t
 *
 * Note: All used instances will be locked and automatically released when
 *       deleting the result vector.
 */
vector_t *pp_cim_find_associations(pp_cim_instance_t *instance,
    const char *name)
{
    vector_t *result = NULL;
    pp_hash_iterator_t it;
    pp_cim_class_t *assoc = NULL, *class = NULL;
    unsigned int i, j;

    pthread_rwlock_rdlock(&pp_cim_repository_lock);

    if (name) {
        assoc = pp_cim_class_lookup(name);
        if (!assoc)
            goto bail;
        pp_cim_class_release(assoc);
    }

    result = vector_new(NULL, 1,
        (vector_elem_del_func_simple)pp_cim_instance_untag);

    class = pp_hash_get_first_entry_r(pp_cim_repository, &it);
    while (class) {
        if (class->association &&
            (!assoc || pp_cim_is_subclass(class, assoc))) {
            pp_cim_class_lock(class);
            for (i = 0; i < vector_size(class->instances); i++) {
                pp_cim_instance_t *inst = vector_get(class->instances, i);

                vector_t *properties = pp_cim_get_properties(inst);
                for (j = 0; j < vector_size(properties); j++) {
                    pp_cim_propval_t *pv = vector_get(properties, j);
                    if (pv->property->type == PP_CIM_REFERENCE
                        && !pv->data.null
                        && pv->data.types.reference == instance) {
                        pp_cim_instance_tag(inst);
                        vector_add(result, inst);
                        break;
                    }
                }
                vector_delete(properties);
            }
            pp_cim_class_release(class);
        }
        class = pp_hash_get_next_entry_r(pp_cim_repository, &it);
    }

bail:
    pthread_rwlock_unlock(&pp_cim_repository_lock);
    return result;
}

/*
 * Check whether a CIM instance belongs to or is derived from a given class.
 *
 * @param instance  CIM instance
 * @param name      CIM class name (e.g. CIM_NumericSensor)
 *
 * @return  0 (no match), 1 (match)
 */
int pp_cim_qualifies_as(pp_cim_instance_t *instance, const char *name)
{
    pp_cim_class_t *c = instance->cim_class;
    while (c) {
        if (strcasecmp(name, c->cim_name) == 0)
            return 1;
        c = c->superclass;
    }
    return 0;
}

/*
 * Check whether a CIM instance belongs to or is derived from a given class.
 *
 * @param instance  CIM instance
 * @param ufct      User Friendly Class Tag
 *
 * @return  0 (no match), 1 (match)
 */
int pp_cim_qualifies_as_ufct(pp_cim_instance_t *instance, const char *ufct)
{
    pp_cim_class_t *c = instance->cim_class;
    if (instance->alias && strcasecmp(ufct, instance->alias) == 0)
        return 1;
    while (c) {
        if (c->ufct && strcasecmp(ufct, c->ufct) == 0)
            return 1;
        c = c->superclass;
    }
    return 0;
}

/*
 * Check whether one class is a subclass of another.
 *
 * @param sub    Subclass
 * @param super  Superclass
 *
 * @return  0 (no match), 1 (match)
 */
int pp_cim_is_subclass(pp_cim_class_t *sub, pp_cim_class_t *super)
{
    while (sub) {
        if (sub == super)
            return 1;
        sub = sub->superclass;
    }
    return 0;
}

/*
 * Read CIM instance property.
 *
 * @param instance  CIM instance
 * @param property  Property name
 *
 * @return  Property reference, NULL in case of error
 *
 * Note: The property must be freed (pp_cim_property_delete) by the caller.
 */
pp_cim_propval_t *pp_cim_get_property(pp_cim_instance_t *instance,
    const char *property)
{
    pp_cim_propval_t *p;

    assert(instance->provider);
    assert(instance->provider->get_property);

    pthread_mutex_lock(&instance->mutex);
    p = instance->provider->get_property(instance, property);
    pthread_mutex_unlock(&instance->mutex);

    return p;
}

/*
 * Set CIM instance property.
 *
 * @param instance  CIM instance
 * @param property  Property name
 * @param data      New data
 *
 * @return  PP_SUC or PP_ERR
 *
 * Note: The data structure must be created and deleted by the caller.
 */
int pp_cim_set_property(pp_cim_instance_t *instance,
    const char *property, pp_cim_data_t data)
{
    int ret;

    assert(instance->provider);
    assert(instance->provider->get_property);

    pthread_mutex_lock(&instance->mutex);
    ret = instance->provider->set_property(instance, property, data, 0);
    pthread_mutex_unlock(&instance->mutex);

    return ret;
}

/*
 * Get all properties from instance.
 *
 * @param instance  CIM instance
 *
 * @return  Property vector or NULL, element type: pp_cim_property_t
 *
 * Note: The result vector must be freed by the caller.
 */
vector_t *pp_cim_get_properties(pp_cim_instance_t *instance)
{
    vector_t *p;

    assert(instance->provider);
    assert(instance->provider->get_properties);

    pthread_mutex_lock(&instance->mutex);
    p = instance->provider->get_properties(instance);
    pthread_mutex_unlock(&instance->mutex);

    return p;
}

/*
 * Set all instance properties at once.
 *
 * @param instance    CIM instance
 * @param names       vector with property names
 * @param properties  vector with property values
 *
 * @return  PP_ERR or PP_SUC
 */
int pp_cim_set_properties(pp_cim_instance_t *instance,
    vector_t *names, vector_t *properties)
{
    int ret;

    assert(instance->provider);
    assert(instance->provider->set_properties);

    pthread_mutex_lock(&instance->mutex);
    ret = instance->provider->set_properties(instance, names, properties);
    pthread_mutex_unlock(&instance->mutex);

    return ret;
}

/*
 * Get CIM method by name.
 *
 * @param instance  CIM instance
 * @param method    Method name
 *
 * @return  Method pointer, NULL in case of error
 *
 * Note: The method pointer must not be changed or freed by the caller.
 */
pp_cim_methodptr_t *pp_cim_get_method(pp_cim_instance_t *instance,
    const char *method)
{
    assert(instance->provider);
    assert(instance->provider->get_method);

    return instance->provider->get_method(instance, method);
}

/*
 * Update CIM properties.
 *
 * @param instance  CIM instance
 */
void pp_cim_update_properties(pp_cim_instance_t *instance)
{
    assert(instance->provider->update);

    pthread_mutex_lock(&instance->mutex);
    instance->provider->update(instance);
    pthread_mutex_unlock(&instance->mutex);
}

/*
 * Check CIM authorization.
 *
 * @param instance  CIM instance
 * @param auth_string remote client web authorization
 * @return PP_SUC if allowed, PP_ERR if denied.
 */
int pp_cim_get_authorization(pp_cim_instance_t *instance, const char *auth_string)
{
    int rtn;

    assert(instance->provider->authorize);

    pthread_mutex_lock(&instance->mutex);
    rtn = instance->provider->authorize(instance, auth_string);
    pthread_mutex_unlock(&instance->mutex);
    return rtn;
}

/*
 * Commit CIM property changes.
 *
 * @param instance  CIM instance
 */
void pp_cim_commit_properties(pp_cim_instance_t *instance)
{
    assert(instance->provider->commit);

    pthread_mutex_lock(&instance->mutex);
    instance->provider->commit(instance);
    pthread_mutex_unlock(&instance->mutex);
}

/*
 * Get value map entry for index.
 *
 * @param valmap  Value map pointer
 * @param index   Value map index
 *
 * @return  Value map entry
 *
 * Note: The result string must not be freed or changed by the caller.
 */
const char *pp_cim_map_value(pp_cim_valmap_t *valmap, int idx)
{
    if (!valmap)
        return NULL;
    while (valmap->value && idx > valmap->max)
        valmap++;
    if (valmap->value && idx >= valmap->min)
        return valmap->value;
    return NULL;
}

/*
 * Check whether a CIM instance supports the CLP Reset verb.
 *
 * @param instance  CIM instance
 *
 * @return  0 (not supported), 1 (supported)
 */
int pp_cim_clp_reset_supported(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    return (instance->provider->clp_map_reset != NULL);
}

/*
 * Perform CLP-to-CIM mapping for Reset verb.
 *
 * @param instance  CIM instance
 *
 * @return  CIM method call structure to perform Reset command.
 */
pp_cim_method_call_t *pp_cim_clp_map_reset(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    assert(instance->provider->clp_map_reset);
    return instance->provider->clp_map_reset(instance);
}

/*
 * Check whether a CIM instance supports the CLP Set verb.
 *
 * @param instance  CIM instance
 *
 * @return  0 (not supported), 1 (supported)
 */
int pp_cim_clp_set_supported(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    return (instance->provider->clp_map_set != NULL);
}

/*
 * Perform CLP-to-CIM mapping for Set verb.
 *
 * @param instance  CIM instance
 * @param prop      Property/value pair
 *
 * @return  CIM method call structure to perform Reset command.
 */
pp_cim_method_call_t *pp_cim_clp_map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    assert(instance->provider);
    assert(instance->provider->clp_map_set);
    return instance->provider->clp_map_set(instance, prop);
}

/*
 * Check whether a CIM instance supports the CLP Start verb.
 *
 * @param instance  CIM instance
 *
 * @return  0 (not supported), 1 (supported)
 */
int pp_cim_clp_start_supported(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    return (instance->provider->clp_map_start != NULL);
}

/*
 * Perform CLP-to-CIM mapping for Start verb.
 *
 * @param instance  CIM instance
 *
 * @return  CIM method call structure to perform Start command.
 */
pp_cim_method_call_t *pp_cim_clp_map_start(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    assert(instance->provider->clp_map_start);
    return instance->provider->clp_map_start(instance);
}

/*
 * Check whether a CIM instance supports the CLP Stop verb.
 *
 * @param instance  CIM instance
 *
 * @return  0 (not supported), 1 (supported)
 */
int pp_cim_clp_stop_supported(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    return (instance->provider->clp_map_stop != NULL);
}

/*
 * Perform CLP-to-CIM mapping for Stop verb.
 *
 * @param instance  CIM instance
 *
 * @return  CIM method call structure to perform Stop command.
 */
pp_cim_method_call_t *pp_cim_clp_map_stop(pp_cim_instance_t *instance)
{
    assert(instance->provider);
    assert(instance->provider->clp_map_stop);
    return instance->provider->clp_map_stop(instance);
}

/*
 * Check whether a CIM instance supports the CLP Create verb.
 *
 * @param cim_class  CIM class
 *
 * @return  0 (not supported), 1 (supported)
 */
pp_cim_method_t *pp_cim_clp_create_supported(pp_cim_class_t *cim_class)
{
    assert(cim_class);
    return pp_cim_class_get_method(cim_class, "CLP_Create");
}

/*
 * Check whether a CIM instance supports the CLP Delete verb.
 *
 * @param instance  CIM instance
 *
 * @return  0 (not supported), 1 (supported)
 */
int pp_cim_clp_delete_supported(pp_cim_instance_t *instance)
{
    assert(instance);
    return ((void *)provider_common_get_method(instance, "CLP_Delete") != NULL);
    // return (instance->provider->deinit != NULL);
//	return 0;
}

/*
 * Map method from clp to cim
 *
 */

pp_cim_methodptr_t *pp_cim_clp_map_delete(pp_cim_instance_t *instance)
{
    assert(instance);
    return provider_common_get_method(instance, "CLP_Delete");
//	assert(instance->provider);
//	assert(instance->provider->deinit);
//	return instance->provider->deinit(instance);
}



/*
 * Execute a CIM method call.
 *
 * @param call    CIM method call as returned by the CLP mapping functions
 * @param result  Result buffer, may be NULL
 *
 * @return  PP_SUC or PP_ERR
 *
 * Note: The result buffer must be cleaned by the caller.
 */
int pp_cim_method_call_exec(pp_cim_method_call_t *call, pp_cim_data_t *result)
{
    assert(call->instance);
    if (!call->methodptr)
        return PP_SUC; // NOP
    if (!call->methodptr->ptr) {
        printf("Warning: Method %s::%s() not implemented.\n",
            call->instance->cim_class->cim_name, call->methodptr->method->name);
        return PP_ERR;
    }
    return call->methodptr->ptr(call->instance, call->args, result);
}

/*
 * Get CLP help message for class.
 *
 * @param class    CIM class
 * @param verbose  Verbose flag
 *
 * @return  Help message buffer (or NULL if no help available); must be
 *          freed by caller.
 */
char *pp_cim_clp_help_class(pp_cim_class_t *class, int verbose)
{
    if (!class->clp_help)
        return NULL;
    return class->clp_help(class, verbose);
}

/*
 * Get CLP help message for instance.
 *
 * @param instance  CIM instance
 * @param verbose   Verbose flag
 *
 * @return  Help message buffer (or NULL if no help available); must be
 *          freed by caller.
 */
char *pp_cim_clp_help_instance(pp_cim_instance_t *instance, int verbose)
{
    assert(instance->provider);
    if (!instance->provider->clp_help)
        return NULL;
    return instance->provider->clp_help(instance, verbose);
}

