#ifndef _PP_CIM_PROVIDER_H
#define _PP_CIM_PROVIDER_H

#include <pp/cim.h>
#include <pp/cimTypes.h>

// CIM provider interface
struct pp_cim_provider_s {

    /*
     * Instance deinitialization.
     *
     * @param instance  CIM instance
     *
     * Note: This function only performs some deinitialization, but it does
     *       not delete the instance from memory. Use instance_delete() to
     *       free the memory afterwards.
     */
    void (*deinit)(pp_cim_instance_t *instance);

    /*
     * Update an instance's data, e.g. via IPMI.
     *
     * @param instance   CIM instance
     */
    void (*update)(pp_cim_instance_t *instance);

    /*
     * Commit an instance's data, e.g. via IPMI (e.g. setting thresholds).
     *
     * @param instance   CIM instance
     */
    void (*commit)(pp_cim_instance_t *instance);

    /*
     * Authorize an instance's data.  This simply determines if the
     * instance is visible based on the remote user'd identity.  The
     * actual implimentation is provider specific and the default
     * always authorizes.
     *
     * @return PP_SUC if authorized, PP_ERR if not. 
     * @param instance  CIM instance
     * @param auth_string   remote web client authorization string
     */
    int (*authorize)(pp_cim_instance_t *instance, const char *auth_string);

    /*
     * Read CIM instance property.
     *
     * @param instance  CIM instance
     * @param name      Property name
     *
     * @return  Property reference, NULL in case of error.
     */
    pp_cim_propval_t *(*get_property)(pp_cim_instance_t *instance,
	    const char *name);

    /*
     * Set CIM instance property.
     *
     * @param instance  CIM instance
     * @param name      Property name
     * @param data      New property data
     * @param force     Set even if property is write-protected
     *
     * @return  PP_ERR or PP_SUC
     */
    int (*set_property)(pp_cim_instance_t *instance,
	    const char *name, pp_cim_data_t data, int force);

    /*
     * Get all CIM instance properties.
     *
     * @param instance  CIM instance
     *
     * @return  Property vector, element type: pp_cim_property_t.
     */
    vector_t *(*get_properties)(pp_cim_instance_t *instance);

    /*
     * Set all CIM instance properties at once.
     *
     * @param instance    CIM instance
     * @param names       vector with property names
     * @param properties  vector with property values
     *
     * @return  PP_ERR or PP_SUC
     */
    int (*set_properties)(pp_cim_instance_t *instance,
    	    vector_t *names, vector_t *properties);

    /*
     * Get CIM method by name.
     *
     * @param instance  CIM instance
     * @param name      Method name
     *
     * @return  Method reference, NULL in case of error.
     */
    pp_cim_methodptr_t *(*get_method)(pp_cim_instance_t *instance,
	    const char *name);

    /*
     * Call method by name.
     *
     * @param instance  CIM instance
     * @param name      Method name
     * @param args      Method arguments, element type: pp_cim_method_arg_t
     * @param result    Result buffer, may be NULL
     *
     * @return  PP_ERR or PP_SUC
     *
     * Note: The result buffer must be cleaned by the caller.
     */
    int (*call_method)(pp_cim_instance_t *instance,
	    const char *name, vector_t *args, pp_cim_data_t *result);

    /*
     * CLP to CIM mapping: Map Reset verb; may be NULL.
     *
     * @param instance  CIM instance
     *
     * @return  CIM method call for Reset command, NULL in case of error.
     */
    pp_cim_method_call_t *(*clp_map_reset)(pp_cim_instance_t *instance);

    /*
     * CLP to CIM mapping: Map Set verb; may be NULL.
     *
     * @param instance  CIM instance
     * @param prop      Property/value pair
     *
     * @return  CIM method call for Set command, NULL in case of error.
     */
    pp_cim_method_call_t *(*clp_map_set)(pp_cim_instance_t *instance,
	    pp_clp_property_value_t *prop);

    /*
     * CLP to CIM mapping: Map Start verb; may be NULL.
     *
     * @param instance  CIM instance
     *
     * @return  CIM method call for Start command, NULL in case of error.
     */
    pp_cim_method_call_t *(*clp_map_start)(pp_cim_instance_t *instance);

    /*
     * CLP to CIM mapping: Map Stop verb; may be NULL.
     *
     * @param instance  CIM instance
     *
     * @return  CIM method call for Stop command, NULL in case of error.
     */
    pp_cim_method_call_t *(*clp_map_stop)(pp_cim_instance_t *instance);

    /*
     * Get CLP help message for instance.
     *
     * @param instance  CIM instance
     * @param verbose   Verbose flag
     *
     * @return  Pointer to help buffer; must be freed by caller.
     */
    char *(*clp_help)(pp_cim_instance_t *instance, int verbose);

};

#endif // _PP_CIM_PROVIDER_H

