#include "cim_common.h"
#include "provider_common.h"
#include "instance.h"

/*
 * Generic set of CIM provider functions. Any provider may either use
 * these functions for property access and method invocation or, if
 * necessary, bring its own functions. Specialized functions may
 * delegate "standard" cases to their generic counterpart, though.
 */

// all generic provider structure
pp_cim_provider_t provider_common =
{
    .deinit = provider_common_deinit,
    .update = provider_common_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = NULL // no help available
};

// generic deinit(): NOP
void provider_common_deinit(pp_cim_instance_t *instance UNUSED)
{
}

// generic update(): NOP
void provider_common_update(pp_cim_instance_t *instance UNUSED)
{
}

// generic commit(): NOP
void provider_common_commit(pp_cim_instance_t *instance UNUSED)
{
}

// generic authorize(): PP_SUC
int provider_common_authorize(pp_cim_instance_t *instance UNUSED,
    const char *auth_string UNUSED)
{
    return PP_SUC;
}

// generic get_property()
pp_cim_propval_t *provider_common_get_property(pp_cim_instance_t *instance,
    const char *name)
{
    unsigned int i;
    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *p = vector_get(instance->prop_table, i);
        if (strcasecmp(p->property->name, name) == 0) {
            return pp_cim_propval_dup(p);
        }
    }
    return NULL;
}

// generic set_property()
int provider_common_set_property(pp_cim_instance_t *instance,
    const char *name, pp_cim_data_t data, int force)
{
    unsigned int i;
    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *p = vector_get(instance->prop_table, i);
        if (strcasecmp(p->property->name, name) == 0) {
            if (!(p->property->writable || force))
                return PP_ERR;
            pp_cim_data_null(p->property->type, p->property->array, &p->data);
	    if (p->property->type == PP_CIM_STRING_CONST) p->property->type = PP_CIM_STRING;
            pp_cim_data_copy(p->property->type, p->property->array, &p->data, &data);
            p->changed = 1;
            return PP_SUC;
        }
    }
    return PP_ERR;
}

// generic get_properties()
vector_t *provider_common_get_properties(pp_cim_instance_t *instance)
{
    vector_t *result;
    unsigned int i;

    result = vector_new(NULL, vector_size(instance->prop_table),
        (vector_elem_del_func_simple)pp_cim_propval_delete);

    for (i = 0; i < vector_size(instance->prop_table); i++) {
        pp_cim_propval_t *p = vector_get(instance->prop_table, i);
        if (!p->property->priv) {
            pp_cim_propval_t *q = instance->provider->get_property(instance,
                p->property->name);
            vector_add(result, q);
        }
    }

    return result;
}

// generic set_properties()
int provider_common_set_properties(pp_cim_instance_t *instance,
    vector_t *names, vector_t *props)
{
    unsigned int i;
    int ret = 0;

    assert(vector_size(names) == vector_size(props));

    for (i = 0; i < vector_size(props); i++) {
        pp_cim_data_t *data = vector_get(props, i);
        char *name = vector_get(names, i);
        ret += instance->provider->set_property(instance, name, *data, 0);
    }

    return ret;
}

// generic get_method()
pp_cim_methodptr_t *provider_common_get_method(pp_cim_instance_t *instance,
    const char *name)
{
    unsigned int i;
    for (i = 0; i < vector_size(instance->method_table); i++) {
        pp_cim_methodptr_t *m = vector_get(instance->method_table, i);
        if (strcasecmp(m->method->name, name) == 0)
            return m;
    }
    return NULL;
}

// generic call_method()
int provider_common_call_method(pp_cim_instance_t *instance,
    const char *name, vector_t *args, pp_cim_data_t *result)
{
    unsigned int i;
    for (i = 0; i < vector_size(instance->method_table); i++) {
        pp_cim_methodptr_t *m = vector_get(instance->method_table, i);
        if (strcasecmp(m->method->name, name) == 0) {
            if (!m->ptr)
                return PP_ERR;
            m->ptr(instance, args, result);
            return PP_SUC;
        }
    }
    return PP_ERR;
}

