#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_fan.h"
#include "provider_coolingdevice.h"

static pp_cim_method_call_t *map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop);
static char *fan_clp_help_instance(pp_cim_instance_t *instance, int verbose);
static char *fan_clp_help_class(pp_cim_class_t *cim_class, int verbose);
static int fan_setspeed(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = provider_common_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = map_set,
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = fan_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"VariableSpeed", PP_CIM_BOOLEAN, 0, NULL, 0, 0, 1, 0, {0, {.boolean = 1}}},
    {"DesiredSpeed", PP_CIM_UNSIGNED, 0, NULL, 0, 0, 1, 1, {1, {0}}},
    {"MaxSpeed", PP_CIM_UNSIGNED, 0, NULL, 0, 0, 0, 0, {1, {0}}},
    {.name = NULL}
};

// CIM_Fan::SetSpeed() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t fan_setspeed_args[] =
{
    {"DesiredSpeed", PP_CIM_UNSIGNED, 0, NULL, 1, 0},
};

// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"SetSpeed", PP_CIM_UNSIGNED, 1, fan_setspeed_args, 0, fan_setspeed},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_fan_desc =
{
    .cim_name = "CIM_Fan",
    .ufct = "fan",
    .dispname = "CIM Fan",
    .superclass = "CIM_CoolingDevice",
    .assoc = 0,
    .properties = properties,
    .methods = methods,
    .update = NULL,
    .clp_update = NULL,
    .clp_help = fan_clp_help_class
};

// Create a new CIM_Fan instance
pp_cim_instance_t *pp_cim_fan_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_Fan", &provider);
    pp_cim_data_t d = {0, {.unsigned_int = 10000}};
    provider_common_set_property(i, "MaxSpeed", d, 1);
    return i;
}

static pp_cim_method_call_t *map_set(pp_cim_instance_t *instance,
    pp_clp_property_value_t *prop)
{
    pp_cim_data_t *arg;
    pp_cim_methodptr_t *methodptr;
    pp_cim_method_call_t *call;
    unsigned int speed;

    if (strcasecmp(prop->property, "DesiredSpeed") == 0) {
        char *ptr;
        if (!prop->value || prop->value[0] == '\0')
            return NULL;
        speed = strtol(prop->value, &ptr, 10);
        if (*ptr != '\0')
            return NULL;

        methodptr = provider_common_get_method(instance, "SetSpeed");
        call = pp_cim_method_call_new(methodptr, instance);
        arg = malloc(sizeof(pp_cim_data_t));
        arg->null = 0;
        arg->types.unsigned_int = speed;
        vector_add(call->args, arg);
        return call;
    }

    return NULL;
}

static const char clp_help_message[] =
    "The CIM_Fan class is used to control a variable speed fan. It supports\r\n"
    "the following CLP operations:\r\n"
    "  - Changing the fan speed:   SET DesiredSpeed=2000\r\n";

static char *fan_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static char *fan_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static int fan_setspeed(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result)
{
    pp_cim_propval_t *pv;
    pp_cim_data_t *arg;
    unsigned int speed, maxspeed;
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    if (vector_size(args) != 1)
        return PP_ERR;

    pv = provider_common_get_property(instance, "MaxSpeed");
    maxspeed = pv->data.types.unsigned_int;
    pp_cim_propval_delete(pv);

    arg = vector_get(args, 0);
    if (arg->types.unsigned_int > maxspeed)
        arg->types.unsigned_int = maxspeed;
    speed = arg->types.unsigned_int;
    pp_cim_debug("CIM_Fan::SetSpeed(%d)\n", speed);
    provider_common_set_property(instance, "DesiredSpeed", *arg, 0);

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.oem_pp_set_pwm.pwm_id = 3;
    ipmi_param.data.oem_pp_set_pwm.duty_cycle = (speed * 100 / maxspeed);
    ret = pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP,
        PP_IPMI_OEM_PP_SET_PWM, &ipmi_param, NULL,
        &err, NULL);

    if (result) {
        result->null = 0;
        result->types.unsigned_int = 0;
    }

    return PP_SUC;
}

