#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include "cim_common.h"
#include "instance.h"
#include "namespace.h"
#include "provider_common.h"
#include "provider_logrecord.h"
#include "provider_recordinlog.h"
#include "provider_recordlog.h"
#include "provider_sensor.h"
#include <pp/ipmi.h>

// forward declarations
static char *logrecord_clp_help_instance(pp_cim_instance_t *instance,
    int verbose);
static char *logrecord_clp_help_class(pp_cim_class_t *cim_class, int verbose);

static void logrecord_class_update(pp_cim_class_t *cim_class);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = provider_common_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = NULL, // 'set' command not supported
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL, // 'stop' command not supported
    .clp_help = logrecord_clp_help_instance
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"LogCreationClassName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {0, {.string_const = "CIM_RecordLog"}}},
    {"LogName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {0, {.string_const = "IPMI SEL"}}},
    {"CreationClassName", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"RecordID", PP_CIM_STRING_CONST, 0, NULL, 1, 0, 1, 0, {1, {0}}},
//    { "MessageTimeStamp",     PP_CIM_DATETIME, 0, NULL, 1, 0, 1, 0, { 1, { 0 }                         } },
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_logrecord_desc =
{
    .cim_name = "CIM_LogRecord",
    .ufct = "record",
    .dispname = "CIM Log Record",
    .superclass = "CIM_RecordForLog",
    .assoc = 0,
    .properties = properties,
    .methods = NULL, // no methods
    .update = logrecord_class_update,
    .clp_update = NULL,
    .clp_help = logrecord_clp_help_class
};

// Create a new CIM_LogRecord instance
pp_cim_instance_t *pp_cim_logrecord_new()
{
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_LogRecord", &provider);
    return i;
}

static time_t last_timestamp = -1;
static int last_recordnum = -1;

static pp_cim_instance_t *find_ipmi_log(void)
{
    pp_cim_class_t *cim_class = pp_cim_class_lookup("CIM_RecordLog");
    vector_t *instances = pp_cim_get_instances(cim_class);
    pp_cim_data_t d;
    pp_cim_instance_t *result = NULL;
    unsigned int i;

    d.null = 0;
    d.types.string_const = "IPMI: Host SEL Log";

    for (i = 0; i < vector_size(instances); i++) {
        pp_cim_instance_t *inst = vector_get(instances, i);
        pp_cim_propval_t *pv = provider_common_get_property(inst, "InstanceID");
        if (pp_cim_data_equal(PP_CIM_STRING_CONST, pv->data, d)) {
            pp_cim_instance_tag(inst);
            result = inst;
        }
        pp_cim_propval_delete(pv);
        if (result)
            break;
    }

    vector_delete(instances);
    pp_cim_class_release(cim_class);

    return result;
}

static const char clp_help_message[] =
    "The CIM_LogRecord class is used to store log entries of a CIM_RecordLog.\r\n"
    "A new logrecord instance is generated for each log entry and associated\r\n"
    "with the recordlog via a CIM_RecordInLog association instance.\r\n";

static char *logrecord_clp_help_instance(pp_cim_instance_t *instance UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static char *logrecord_clp_help_class(pp_cim_class_t *cim_class UNUSED,
    int verbose UNUSED)
{
    return strdup(clp_help_message);
}

static void logrecord_class_update(pp_cim_class_t *cim_class UNUSED)
{
    unsigned int i;
    pp_cim_instance_t *log_inst = find_ipmi_log();
    vector_t *records;

    records = pp_cim_logrecord_discovery();

    if (records) {
        pp_cim_class_t *class;
        pp_cim_instance_t *inst;
        pp_cim_data_t d;

        // delete all CIM_LogRecord instances
        inst = log_inst->children;
        while (inst) {
            pp_cim_instance_t *next = inst->next;
            pp_cim_instance_unregister(inst);
            inst = next;
        }
        // delete all CIM_RecordInLog instances
        class = pp_cim_class_lookup("CIM_RecordInLog");
        while (vector_size(class->instances) > 0) {
            inst = vector_get(class->instances, 0);
            pp_cim_instance_unregister(inst);
        }
        pp_cim_class_release(class);

        log_inst->children = NULL;
        for (i = 0; i < vector_size(records); i++) {
            inst = vector_get(records, i);
            pp_cim_add_instance(inst, log_inst);
            pp_cim_recordinlog_new(log_inst, inst);
        }

        d.null = 0;
        d.types.unsigned_int = vector_size(records);
        provider_common_set_property(log_inst, "CurrentNumberOfRecords", d, 1);

        vector_delete(records);
    }

    pp_cim_instance_untag(log_inst);
}

// SEL Generic or Sensor Specific State Code (IPMI-to-CIM mapping, 4.5.2.8)
static char *get_statecode(pp_ipmi_sel_list_entry_t *entry)
{
    pp_strstream_t *str = pp_strstream_init(NULL);

    if (entry->event_direction_code == PP_IPMI_EVENT_DIRECTION_DEASSERTION) {
        pp_strappend(str, "Deassert: ");
    } else {
        pp_strappend(str, "Assert: ");
    }

    pp_strappend(str, pp_strstream_buf(&entry->description));

    return pp_strstream_buf_and_free(str);
}

// CIM_LogRecord::Caption (IPMI-to-CIM mapping, 4.5.2.8)
static char *get_caption(pp_ipmi_sel_list_entry_t *entry)
{
    pp_strstream_t *str = pp_strstream_init(NULL);
    char *c = get_statecode(entry);
    pp_strappend(str, pp_strstream_buf(&entry->sensor_type));
    pp_strappendchr(str, ' ');
    pp_strappend(str, c);
    free(c);
    return pp_strstream_buf_and_free(str);
}

// CIM_LogRecord::Description (IPMI-to-CIM mapping, 4.5.2.9)
static char *get_description(pp_ipmi_sel_list_entry_t *entry)
{
    pp_strstream_t *str = pp_strstream_init(NULL);
    char *statecode = get_statecode(entry);
    char *sensor_name = NULL, *entity = NULL, *sensor_type = NULL;
    unsigned int entity_inst = 0;

    pp_cim_instance_t *sensor = pp_cim_sensor_getbyid(entry->sensor_num);
    if (sensor) {
        pp_cim_propval_t *pv;
        pv = provider_common_get_property(sensor, "Name");
        sensor_name = strdup(pv->data.types.string_const);
        pp_cim_propval_delete(pv);
        pv = provider_common_get_property(sensor, "SensorType");
        if (pv->data.types.unsigned_int != 1) {
            sensor_type = strdup(pp_cim_map_value(
                pv->property->valmap, pv->data.types.unsigned_int));
        } else {
            // SensorType == "Other" -> get OtherSensorTypeDescription
            pp_cim_propval_t *pv2 = provider_common_get_property(
                sensor, "OtherSensorTypeDescription");
            sensor_type = strdup(pv2->data.types.string_const);
            pp_cim_propval_delete(pv2);
        }
        pp_cim_propval_delete(pv);
        pv = provider_common_get_property(sensor, "_EntityName");
        entity = strdup(pv->data.types.string_const);
        pp_cim_propval_delete(pv);
        pv = provider_common_get_property(sensor, "_EntityInst");
        entity_inst = pv->data.types.unsigned_int;
        pp_cim_propval_delete(pv);
        pp_cim_instance_untag(sensor);
    }

    if (sensor_name) {
        pp_strappendf(str, "%s: %s for %s %d; %s",
            sensor_name, sensor_type,
            entity, entity_inst, statecode);
    } else {
        pp_strappend(str, "Unknown sensor event");
    }

    free(statecode);
    free(sensor_name);
    free(sensor_type);
    free(entity);

    return pp_strstream_buf_and_free(str);
}

vector_t *pp_cim_logrecord_discovery()
{
    pp_ipmi_parameter_t ipmi_param;
    pp_ipmi_return_t ipmi_ret, ipmi_ret_sensor;
    vector_t *result = NULL;
    unsigned int sel_size, i, record_id;
    int ret, err; //, max_timestamp;
    time_t max_timestamp;

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_INFO,
	NULL, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return NULL;
    }
    max_timestamp = 0;
    printf("last_add_time = %ld\n", ipmi_ret.data.sel_info.last_add_time.timestamp);
    if (ipmi_ret.data.sel_info.last_add_time.is_valid) {
	max_timestamp = ipmi_ret.data.sel_info.last_add_time.timestamp;
    }
    printf("last_del_time = %ld\n", ipmi_ret.data.sel_info.last_del_time.timestamp);
    if (ipmi_ret.data.sel_info.last_del_time.is_valid &&
	    ipmi_ret.data.sel_info.last_del_time.timestamp > max_timestamp) {
	max_timestamp = ipmi_ret.data.sel_info.last_del_time.timestamp;
    }

    printf("last_timestamp = %ld, max_timestamp = %ld, last_recordnum = %d, entries = %d\n",
	    last_timestamp, max_timestamp, last_recordnum, ipmi_ret.data.sel_info.entries);
    if (last_timestamp == max_timestamp
        && last_recordnum == ipmi_ret.data.sel_info.entries) {
        // No new log entries
        goto bail;
    }

    last_timestamp = max_timestamp;
    last_recordnum = ipmi_ret.data.sel_info.entries;

    pp_ipmi_cleanup_ret(&ipmi_ret);
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_LIST,
        NULL, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return NULL;
    }
    sel_size = vector_size(ipmi_ret.data.sel_list);

    // fetch associated sensor information
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.sensor_get_by_id_list = vector_new2(NULL, sel_size, sizeof(int), NULL);
    for (i = 0; i < sel_size; i++) {
	int found = 0;
	pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);
	unsigned int j;
	for (j = 0; j < vector_size(ipmi_param.data.sensor_get_by_id_list); j++) {
	    unsigned int *num_ptr = vector_get2(ipmi_param.data.sensor_get_by_id_list, j);
	    if (*num_ptr == entry->sensor_num) {
		found = 1;
		break;
	    }
	}
	if (!found) {
	    vector_add2(ipmi_param.data.sensor_get_by_id_list, &entry->sensor_num);
	}
    }
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SENSOR, PP_IPMI_SENSOR_SUBCMD_GET_BY_ID,
	&ipmi_param, &ipmi_ret_sensor, &err, NULL);
    vector_delete(ipmi_param.data.sensor_get_by_id_list);
    if (ret) {
	printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
	goto bail;
    }

    record_id = 0;
    result = vector_new(NULL, vector_size(ipmi_ret.data.sel_list), NULL);
    for (i = 0; i < sel_size; i++) {
	unsigned int j;
	int found;
        pp_strstream_t *str;
        pp_cim_data_t d;
        d.null = 0;
        pp_ipmi_sdr_list_entry_t *sdr_entry = NULL;
        pp_ipmi_sel_list_entry_t *entry = vector_get(ipmi_ret.data.sel_list, i);
        int owner_lun = 0, owner_id = 0;

        if (entry->record_type != 2 || entry->evm_revision != 4
            || (entry->event_type_code != PP_IPMI_EVENT_TYPE_THRESHOLD &&
            entry->event_type_code != PP_IPMI_EVENT_TYPE_GENERIC_DISCRETE)) {
            // SEL type not supported by IPMI-to-CIM mapping
            continue;
        }

        record_id++;

        // get associated sensor device id
	found = 0;
	for (j = 0; j < vector_size(ipmi_ret_sensor.data.sensor_get_list); j++) {
	    sdr_entry = vector_get(ipmi_ret_sensor.data.sensor_get_list, j);
	    if (sdr_entry->sdr_id == (int)entry->sensor_num) {
		found = 1;
		break;
	    }
	}
	if (!found) continue;

        switch (sdr_entry->type) {
            case PP_IPMI_SENSOR_TYPE_FULL:
                owner_lun = sdr_entry->data.full.owner_lun;
                owner_id = sdr_entry->data.full.owner_id;
                break;
            case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
                owner_lun = sdr_entry->data.compact.owner_lun;
                owner_id = sdr_entry->data.compact.owner_id;
                break;
            case PP_IPMI_SENSOR_TYPE_EVENTONLY:
                owner_lun = sdr_entry->data.event_only.owner_lun;
                owner_id = sdr_entry->data.event_only.owner_id;
                break;
            default:
                printf("Warning: Unhandled sensor type %d.", sdr_entry->type);
        }

        pp_cim_instance_t *inst = pp_cim_instance_new("CIM_LogRecord", &provider);

        asprintf(&d.types.string, "%d", record_id);
        provider_common_set_property(inst, "RecordID", d, 1);
        free(d.types.string);

        d.types.string_const = "*string CIM_Sensor.DeviceID*uint8[2] IPMI_RecordID"
            "*uint8 IPMI_RecordType*uint8[4] IPMI_Timestamp"
            "*uint8[2] IPMI_GeneratorID*uint8 IPMI_EvMRev"
            "*uint8 IPMI_SensorType*uint8 IPMI_SensorNumber"
            "*boolean IPMI_AssertionEvent*uint8 IPMI_EventType"
            "*uint8 IPMI_EventData1*uint8 IPMI_EventData2"
            "*uint8 IPMI_EventData3*uint32 IANA*";
        provider_common_set_property(inst, "RecordFormat", d, 1);

        str = pp_strstream_init(NULL);
        pp_strappendf(str, "*%d.%d.%d", sdr_entry->sdr_id, owner_lun, owner_id);
        pp_strappendf(str, "*%d %d", entry->record_id & 0xff,
            (entry->record_id >> 8) & 0xff);
        pp_strappendf(str, "*%d", entry->record_type & 0xff);
        pp_strappendf(str, "*%d %d %d %d",
            (int)entry->timestamp.timestamp & 0xff,
            ((int)entry->timestamp.timestamp >> 8) & 0xff,
            ((int)entry->timestamp.timestamp >> 16) & 0xff,
            ((int)entry->timestamp.timestamp >> 24) & 0xff);
        pp_strappendf(str, "*%d %d", entry->generator_id & 0xff,
            (entry->generator_id >> 8) & 0xff);
        pp_strappendf(str, "*%d", entry->evm_revision & 0xff);
        pp_strappendf(str, "*%d", entry->sensor_type_code);
        pp_strappendf(str, "*%d", entry->sensor_num & 0xff);
        if (entry->event_type_code == PP_IPMI_EVENT_TYPE_THRESHOLD) {
            pp_strappend(str, "*true");
        } else {
            pp_strappend(str, "*false");
        }
        pp_strappendf(str, "*%d", entry->event_type_code & 0xff);
        pp_strappendf(str, "*%d", entry->event_data[0] & 0xff);
        pp_strappendf(str, "*%d", entry->event_data[1] & 0xff);
        pp_strappendf(str, "*%d", entry->event_data[2] & 0xff);
        pp_strappendf(str, "*%d*", 0); // IANA?

        d.types.string_const = pp_strstream_buf(str);
        provider_common_set_property(inst, "RecordData", d, 1);
        pp_strstream_free(str);

        d.types.string = get_caption(entry);
        provider_common_set_property(inst, "Caption", d, 1);
        free(d.types.string);

        d.types.string = get_description(entry);
        provider_common_set_property(inst, "Description", d, 1);
        free(d.types.string);

        vector_add(result, inst);
    }
    pp_cim_debug("%d new CIM_LogRecord instances created.\n", vector_size(result));

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return result;

bail:
    pp_ipmi_cleanup_ret(&ipmi_ret);
    pp_ipmi_cleanup_ret(&ipmi_ret_sensor);
    if (result)
        vector_delete(result);
    return NULL;
}

