#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_memberofcollection.h"
#include "provider_account.h"
#include "provider_group.h"

static void memberofcollection_update(pp_cim_instance_t *instance);

static void memberofcollection_class_update(pp_cim_class_t *class);

static int memberofcollection_clp_setgroup(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static int memberofcollection_clp_create(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result);
static int memberofcollection_clp_delete(pp_cim_instance_t *instance, vector_t *args UNUSED,
    pp_cim_data_t *result UNUSED);

static pp_cim_method_call_t *memberofcollection_clp_map_set(
    pp_cim_instance_t *instance, pp_clp_property_value_t *prop);

// provider API functions
static pp_cim_provider_t provider =
{
    .deinit = provider_common_deinit,
    .update = memberofcollection_update,
    .commit = provider_common_commit,
    .authorize = provider_common_authorize,
    .get_property = provider_common_get_property,
    .set_property = provider_common_set_property,
    .get_properties = provider_common_get_properties,
    .set_properties = provider_common_set_properties,
    .get_method = provider_common_get_method,
    .call_method = provider_common_call_method,
    .clp_map_reset = NULL, // 'reset' command not supported
    .clp_map_set = memberofcollection_clp_map_set,
    .clp_map_start = NULL, // 'start' command not supported
    .clp_map_stop = NULL // 'stop' command not supported
};

// class properties
// { <name>, <type>, <array>, <valmap>, <key>, <priv>, <required>, <writable>, { <null>, { <defvalue> } } }
static pp_cim_property_t properties[] =
{
    {"Collection", PP_CIM_REFERENCE, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {"Member", PP_CIM_REFERENCE, 0, NULL, 1, 0, 1, 0, {1, {0}}},
    {.name = NULL}
};

// CIM_MemberOfCollection::_CLP_SetGroup() arguments
// { <name>, <type>, <array>, <valmap>, <in>, <out> }
static pp_cim_method_arg_t memberofcollection_clp_setgroup_args[] =
{
    {"Group", PP_CIM_UNSIGNED, 0, NULL, 1, 0}
};
/*
static pp_cim_method_arg_t memberofcollection_clp_create_args[] =
{
        {"Member", PP_CIM_STRING, 0, NULL, 1, 0}
};
*/
// class methods
// { <name>, <result>, <num_args>, <args>, <priv>, <defptr> }
static pp_cim_method_t methods[] =
{
    {"_CLP_SetGroup", PP_CIM_VOID, 1, memberofcollection_clp_setgroup_args, 1, memberofcollection_clp_setgroup},
    {"CLP_Create", PP_CIM_VOID, 0, NULL, 0, memberofcollection_clp_create},
    {"CLP_Delete", PP_CIM_VOID, 0, NULL, 0, memberofcollection_clp_delete},
    {.name = NULL}
};

// class description
pp_cim_class_desc_t pp_cim_memberofcollection_desc =
{
    .cim_name = "CIM_MemberOfCollection",
    .ufct = "memberofcollection",
    .dispname = "CIM Member Of Collection",
    .superclass = NULL,
    .assoc = 1,
    .properties = properties,
    .methods = methods,
    .update = memberofcollection_class_update,
    .clp_update = NULL
};

// Create a new CIM_MemberOfCollection (or subclass) instance
pp_cim_instance_t *pp_cim_memberofcollection_new(
    pp_cim_instance_t *collection, pp_cim_instance_t *member)
{
    pp_cim_data_t d;
    //printf("%s\n", __FUNCTION__);
    pp_cim_instance_t *i = pp_cim_instance_new("CIM_MemberOfCollection", &provider);
    d.null = collection ? 0 : 1;
    d.types.reference = collection;
    provider_common_set_property(i, "Collection", d, 1);
    d.null = member ? 0 : 1;
    d.types.reference = member;
    provider_common_set_property(i, "Member", d, 1);
    return i;
}

static void memberofcollection_update(pp_cim_instance_t *instance)
{
    pp_cim_instance_t *member = NULL;
    pp_cim_propval_t *pv = NULL;
    printf("%s\n", __FUNCTION__);

    provider_common_update(instance);

    pv = provider_common_get_property(instance, "Member");
    if (pv && !pv->data.null)
        member = pv->data.types.reference;
    pp_cim_propval_delete(pv);
    if (!member)
        return;

    if (strcasecmp(member->cim_class->cim_name, "CIM_Account") == 0) {
        pp_ipmi_parameter_t ipmi_param;
        pp_ipmi_return_t ipmi_ret;
        int ret, err;
        pp_cim_data_t d;

        memset(&ipmi_param, 0, sizeof(ipmi_param));
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        ipmi_param.data.usr_info.chan = 0x0E; // current channel
        ipmi_param.data.usr_info.user_id = pp_cim_account_get_id(member);
        ret = pp_ipmi_send_command(PP_IPMI_CMD_USER, PP_IPMI_USER_SUBCMD_INFO,
            &ipmi_param, &ipmi_ret, &err, NULL);
        if (ret) {
            printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
            return;
        }

        d.null = 0;
        d.types.reference = pp_cim_group_getbyid(ipmi_ret.data.usr_info.priv_limit);
        provider_common_set_property(instance, "Collection", d, 1);
        
        pp_ipmi_cleanup_ret(&ipmi_ret);
    }
}

static void memberofcollection_class_update(pp_cim_class_t *class)
{
    printf("%s\n", __FUNCTION__);

    vector_t *instances = pp_cim_get_instances(class);
    unsigned int i;
    for (i = 0; i < vector_size(instances); i++) {
        pp_cim_instance_t *inst = vector_get(instances, i);
        pp_cim_update_properties(inst);
    }
    vector_delete(instances);
}

// Method: CIM_MemberOfCollection::_CLP_SetGroup
static int memberofcollection_clp_setgroup(pp_cim_instance_t *instance,
    vector_t *args, pp_cim_data_t *result UNUSED)
{
    pp_cim_instance_t *account = NULL, *group;
    pp_cim_propval_t *pv;
    pp_cim_data_t *data;
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    data = vector_get(args, 0);
    if (data->null)
        return PP_ERR;
    group = data->types.reference;

    pv = provider_common_get_property(instance, "Member");
    if (pv && !pv->data.null)
        account = pv->data.types.reference;
    pp_cim_propval_delete(pv);
    if (!account)
        return PP_ERR;

    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.channel_setaccess.channel = 0x0e;
    ipmi_param.data.channel_setaccess.user_id = pp_cim_account_get_id(account);
    ipmi_param.data.channel_setaccess.privilege = pp_cim_group_get_id(group);
    ipmi_param.data.channel_setaccess.ipmi = 1;
    ret = pp_ipmi_send_command(PP_IPMI_CMD_CHANNEL, PP_IPMI_CHANNEL_SUBCMD_SET_ACCESS,
        &ipmi_param, NULL, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return PP_ERR;
    }

    args = args;
    return PP_SUC;
}

static pp_cim_method_call_t *memberofcollection_clp_map_set(
    pp_cim_instance_t *instance, pp_clp_property_value_t *prop)
{
    pp_cim_methodptr_t *ptr;
    pp_cim_method_call_t *call;
    pp_cim_instance_t *group;
    pp_cim_data_t *data;

    if (sscanf(prop->value, "%p", &group) != 1)
        return NULL;
    if (strcasecmp(prop->property, "Collection"))
        return NULL;
    if (strcasecmp(group->cim_class->cim_name, "CIM_Group"))
        return NULL;

    ptr = provider_common_get_method(instance, "_CLP_SetGroup");
    call = pp_cim_method_call_new(ptr, instance);
    data = malloc(sizeof(pp_cim_data_t));
    data->null = 0;
    data->types.reference = group;
    vector_add(call->args, data);

    return call;
}

/*
 * Create
 *
 */
static int memberofcollection_clp_create(pp_cim_instance_t *instance, vector_t *args,
    pp_cim_data_t *result)
{
    pp_cim_instance_t *moc, *member = NULL;
    //printf("%s\n", __FUNCTION__);

    result->null = 0;
    result->types.reference = NULL;

    if (vector_size(args) != 1) {
        printf("%s: vector size of args != 1\n", __FUNCTION__);
        return PP_ERR;
    }
    pp_clp_property_value_t *pv = vector_get(args, 0);
    //printf("%s:%s = %s\n", __FUNCTION__, pv->property, pv->value);
    if (strcasecmp(pv->property, "member")) {
        return PP_ERR;
    }

    // lookup the instance from the args
    vector_t *ufip;
    int relative, inst, isclass;
    ufip = vector_new(NULL, 10,
        (vector_elem_del_func_simple)pp_clp_ufit_delete);
    if (PP_SUCCED(parse_ufip(pv->value, ufip, &relative, &inst, &isclass))) {
        //printf("parse ok\n");
        member = pp_cim_lookup_absolute(ufip);
        //printf("got member\n");
    } else
        printf("parse fail\n");
    free(ufip);
    if (!member)
        return PP_ERR;

    // Check instances
    //pp_cim_instance_dump(instance);
    //pp_cim_instance_dump(member);
    pp_cim_class_t *cc = pp_cim_class_lookup("CIM_Collection");
    pp_cim_class_t *cm = pp_cim_class_lookup("CIM_ManagedElement");
    if (!pp_cim_is_subclass(member->cim_class, cm)) {
        //printf("Member is not the subclass of CIM_ManagedElement\n");
        return PP_ERR;
    }
    if (!pp_cim_is_subclass(instance->cim_class, cc)) {
        //printf("Collecton is not the subclass of CIM_Collection\n");
        return PP_ERR;
    }

    // ask collection
    pp_cim_method_t *method = pp_cim_class_get_method(instance->cim_class, "addMember");
    if (method != NULL) {
        vector_t *v;
        int ans;
        pp_cim_data_t *data;
        v = vector_new(NULL, 1, NULL);
        data = malloc(sizeof(pp_cim_data_t));
        data->null = 0;
        data->types.reference = member;
        vector_add(v, data);
        printf("method->defptr\n");
        ans = method->defptr(instance, v, NULL);
        vector_delete(v);
        if (ans == PP_ERR)
            return PP_ERR;
    }
    // check is this relation is alreay exist
    // remove CIM_MemberOfCollection
    pp_cim_class_t *sd = pp_cim_class_lookup("CIM_MemberOfCollection");
    vector_t *v = pp_cim_get_instances(sd);
    //vector_t *v = sd->instances;
    pp_cim_instance_t *p;
    pp_cim_propval_t *pvm, *pvc;
    unsigned int i = 0;

    for (i = 0; i < vector_size(v); i++) {
        p = vector_get(v, i);
        pvm = provider_common_get_property(p, "Member");
        pvc = provider_common_get_property(p, "Collection");
        if (pvm && pvc) {
            //pp_cim_instance_dump(pvm->data.types.reference);
            //pp_cim_instance_dump(pvc->data.types.reference);
            if (pvm->data.types.reference == member &&
                pvc->data.types.reference == instance) {
                //pp_cim_instance_dump(p);
                vector_delete(v);
                pp_cim_class_release(sd);
                return PP_ERR;
            }
        }
    }
    vector_delete(v);
    pp_cim_class_release(sd);

    // new instance
    moc = pp_cim_memberofcollection_new(instance, member);
    //printf("moc %x\n", moc);
    //pp_cim_add_instance(moc, instance);
    result->types.reference = member;
    return PP_SUC;
}
/*
 * Delete memberofcollection
 *
 */
static int memberofcollection_clp_delete(pp_cim_instance_t *instance, vector_t *args UNUSED,
    pp_cim_data_t *result UNUSED)
{
    // ask collection
    pp_cim_instance_t *collection, *member;
    pp_cim_propval_t *pvc = provider_common_get_property(instance, "Collection");
    pp_cim_propval_t *pvm = provider_common_get_property(instance, "Member");
    collection = pvc->data.types.reference;
    member = pvm->data.types.reference;

    pp_cim_method_t *method = pp_cim_class_get_method(collection->cim_class, "delMember");
    if (method != NULL) {
        vector_t *v;
        int ans;
        pp_cim_data_t *data;
        v = vector_new(NULL, 1, NULL);
        data = malloc(sizeof(pp_cim_data_t));
        data->null = 0;
        data->types.reference = member;
        vector_add(v, data);
        printf("method->defptr\n");
        ans = method->defptr(collection, v, NULL);
        vector_delete(v);
        if (ans == PP_ERR)
            return PP_ERR;
    }


    // unregister instance
    pp_cim_instance_unregister(instance);

    return PP_SUC;
}
