#include <pp/cim.h>
#include <pp/cimTypes.h>
#include <pp/cim_provider.h>
#include <pp/ipmi.h>
#include "cim_common.h"
#include "sensor_discovery.h"
#include "instance.h"
#include "provider_common.h"
#include "provider_sensor.h"
#include "provider_numericsensor.h"

// IPMI-to-CIM mapping, 3.4.2.6 -- incomplete!
sensor_type_t sensor_types[] =
{
    {PP_IPMI_SPEC_SENSOR_TYPE_TEMPERATURE, 2, NULL, "tempsensor"},
    {PP_IPMI_SPEC_SENSOR_TYPE_VOLTAGE, 3, NULL, "voltsensor"},
    {PP_IPMI_SPEC_SENSOR_TYPE_CURRENT, 4, NULL, "currsensor"},
    {PP_IPMI_SPEC_SENSOR_TYPE_FAN, 5, NULL, "tachsensor"},
    {PP_IPMI_SPEC_SENSOR_TYPE_PHYS_SECURITY, 1, "ChassisIntrusion", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_PROCESSOR, 1, "Processor", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_POWER_SUPPLY, 1, "PowerSupply", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_POWER_UNIT, 1, "PowerUnit", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_WATCHDOG, 1, "Watchdog1", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_SYSEVENT, 1, "SystemEvent", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_CRIT_INT, 1, "CriticalInterrupt", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_BUTTON, 1, "Button", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_CHASSIS, 1, "Chassis", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_CHIPSET, 1, "ChipSet", NULL},
    {PP_IPMI_SPEC_SENSOR_TYPE_ACPI_STATE, 1, "SystemACPIPowerState", NULL}
};

vector_t *pp_cim_sensor_discovery()
{
    pp_ipmi_return_t ipmi_ret;
    pp_cim_instance_t *inst;
    vector_t *result;
    unsigned int i;
    int ret, err;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));

    ret = pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_LIST,
        NULL, &ipmi_ret, &err, NULL);
    if (ret) {
        printf("IPMI failure: %s.\n", pp_ipmi_get_error_string(err));
        return NULL;
    }

    result = vector_new(NULL, vector_size(ipmi_ret.data.sdr_list), NULL);

    for (i = 0; i < vector_size(ipmi_ret.data.sdr_list); i++) {
        const char *alias = NULL;
        const char *other_type = NULL;
        int j, type = -1;
        const char *sensortype_name = NULL;
        pp_ipmi_sdr_list_entry_t *entry = vector_get(ipmi_ret.data.sdr_list, i);
        pp_cim_propval_t *pv;

        if (entry->type != PP_IPMI_SENSOR_TYPE_FULL
            && entry->type != PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR
            && entry->type != PP_IPMI_SENSOR_TYPE_EVENTONLY) {
            continue;
        }

        for (j = 0; j < (int)(sizeof(sensor_types) / sizeof(sensor_type_t)); j++) {
            if (entry->spec_type == sensor_types[j].ipmi_type) {
                alias = sensor_types[j].alias;
                type = sensor_types[j].sensor_type;
                other_type = sensor_types[j].other_type;
            }
        }

        if (type == -1) {
            printf("Warning: Unknown IPMI sensor spec type %d for sensor %s.\n",
                entry->spec_type, pp_strstream_buf(&entry->sdr_name));
        } else {
            pp_cim_data_t data;
            pp_strstream_t *str;
            data.null = 0;

            if (entry->type == PP_IPMI_SENSOR_TYPE_FULL
                && entry->data.full.type == PP_IPMI_SENSOR_FULL_TYPE_ANALOG) {
                inst = pp_cim_numericsensor_new();
                if (alias) {
                    str = pp_strstream_init(NULL);
                    pp_strappendchr(str, 'n');
                    pp_strappend(str, alias);
                    inst->alias = pp_strstream_buf_and_free(str);
                }
            } else {
                inst = pp_cim_sensor_new();
                inst->alias = alias ? strdup(alias) : NULL;
            }

            data.types.unsigned_int = entry->sdr_id;
            provider_common_set_property(inst, "_IPMI_ID", data, 1);

            data.types.string_const = pp_strstream_buf(&entry->sdr_name);
            provider_common_set_property(inst, "_IPMI_Name", data, 1);

            data.types.string_const = pp_strstream_buf(&entry->entity.string);
            provider_common_set_property(inst, "_EntityName", data, 1);

            data.types.unsigned_int = entry->entity.instance;
            provider_common_set_property(inst, "_EntityInst", data, 1);

            // init DeviceID property
            str = pp_strstream_init(NULL);
            pp_strappendf(str, "%d.%d.%d",
                entry->sdr_id, entry->data.full.owner_lun, entry->data.full.owner_id);
            data.types.string_const = pp_strstream_buf(str);
            provider_common_set_property(inst, "DeviceID", data, 1);
            pp_strstream_free(str);

            // init ElementName/Name properties
            str = pp_strstream_init(NULL);
            pp_strappendf(str, "%s(%d.%d.%d)", pp_strstream_buf(&entry->sdr_name),
                entry->sdr_id, entry->data.full.owner_lun, entry->data.full.owner_id);
            data.types.string_const = pp_strstream_buf(str);
            provider_common_set_property(inst, "ElementName", data, 1);
            provider_common_set_property(inst, "Name", data, 1);
            pp_strstream_free(str);

            // init SensorType/OtherSensorTypeDescription properties
            data.types.unsigned_int = type;
            provider_common_set_property(inst, "SensorType", data, 1);

            if (other_type) {
                data.types.string_const = other_type;
                provider_common_set_property(inst, "OtherSensorTypeDescription", data, 1);
                sensortype_name = other_type;
            } else {
                pv = provider_common_get_property(inst, "SensorType");
                sensortype_name = pp_cim_map_value(pv->property->valmap, type);
                pp_cim_propval_delete(pv);
            }

            // init Caption/Description properties
            str = pp_strstream_init(NULL);
            pp_strappendf(str, "%s(%d.%d.%d)", sensortype_name,
                entry->sdr_id, entry->data.full.owner_lun, entry->data.full.owner_id);
            data.types.string_const = pp_strstream_buf(str);
            provider_common_set_property(inst, "Caption", data, 1);
            pp_strstream_free(str);

            str = pp_strstream_init(NULL);
            pp_strappendf(str, "%s(%d.%d.%d): ", pp_strstream_buf(&entry->sdr_name),
                entry->sdr_id, entry->data.full.owner_lun, entry->data.full.owner_id);
            pp_strappendf(str, "%s for %s %d", sensortype_name,
                pp_strstream_buf(&entry->entity.string), entry->entity.instance);
            data.types.string_const = pp_strstream_buf(str);
            provider_common_set_property(inst, "Description", data, 1);
            pp_strstream_free(str);

            vector_add(result, inst);
        }
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return result;
}

