#include "clp_common.h"
#include "clpParser.h"
#include <string.h>

/* allocate a new command structure */
pp_clp_cmd_t *pp_clp_cmd_new()
{
	pp_clp_cmd_t *cmd;

	cmd = (pp_clp_cmd_t *)malloc(sizeof(pp_clp_cmd_t));

	vector_new(&cmd->options, 20,
		(vector_elem_del_func_simple)pp_clp_optval_delete);
	vector_new(&cmd->properties, 20,
		(vector_elem_del_func_simple)pp_clp_propval_delete);

	cmd->target = pp_clp_target_new();

	cmd->display.associations = 1;
	vector_new(&cmd->display.association_names, 5, free);
	cmd->display.properties = 1;
	vector_new(&cmd->display.property_names, 5, free);
	cmd->display.filter_property = NULL;
	cmd->display.filter_value = NULL;
	cmd->display.targets = 1;
	vector_new(&cmd->display.target_names, 5, free);
	cmd->display.verbs = 1;

	cmd->output.format = CLP_OUTPUT_TEXT;
	cmd->output.level = CLP_OUTPUT_TERSE;
	strcpy(cmd->output.language, "eng");
	cmd->output.end = 0;
	cmd->output.reverse = 0;
	cmd->output.count = -1;
	cmd->output.range_min = -1;
	cmd->output.range_max = -1;

	cmd->opt_all = 0;
	cmd->opt_default = 0;
	cmd->opt_examine = 0;
	cmd->opt_help = 0;
	cmd->opt_version = 0;

	return cmd;
}

/* free a command structure */
void pp_clp_cmd_delete(pp_clp_cmd_t *cmd)
{
	if (!cmd)
		return;

	vector_delete(&cmd->options);
	vector_delete(&cmd->properties);
	vector_delete(&cmd->display.association_names);
	vector_delete(&cmd->display.property_names);
	vector_delete(&cmd->display.target_names);

	pp_clp_target_delete(cmd->target);

	free(cmd);
}

/* allocate an option/argument structure */
pp_clp_option_value_t *pp_clp_optval_new(pp_clp_option_t opt)
{
	pp_clp_option_value_t *ov;
	ov = malloc(sizeof(pp_clp_option_value_t));
	ov->option = opt;
	vector_new(&ov->values, 10, free);
	return ov;
}

/* free an option/argument structure */
void pp_clp_optval_delete(pp_clp_option_value_t *ov)
{
	if (!ov)
		return;
	vector_delete(&ov->values);
	free(ov);
}

/* allocate a UFiT structure */
pp_clp_ufit_t *pp_clp_ufit_new()
{
	pp_clp_ufit_t *ufit = malloc(sizeof(pp_clp_ufit_t));
	ufit->ufct = NULL;
	ufit->instance_id = -1;
	return ufit;
}

/* free a UFiT structure */
void pp_clp_ufit_delete(pp_clp_ufit_t *ufit)
{
	if (!ufit)
		return;
	free(ufit->ufct);
	free(ufit);
}

/* allocate a target structure */
pp_clp_target_t *pp_clp_target_new()
{
	pp_clp_target_t *target = malloc(sizeof(pp_clp_target_t));
	target->type = CLP_TARGET_DEFAULT;
	target->ufip1 = NULL;
	target->ufip1_relative = 0;
	target->ufct = NULL;
	target->ufip2 = NULL;
	target->ufip2_relative = 0;
	target->level = 1;
	return target;
}

/* free a target structure */
void pp_clp_target_delete(pp_clp_target_t *target)
{
	if (!target)
		return;
	if (target->ufip1)
		vector_delete(target->ufip1);
	if (target->ufip2)
		vector_delete(target->ufip2);
	free(target->ufct);
	free(target);
}

/* allocate a poperty/value structure */
pp_clp_property_value_t *pp_clp_propval_new()
{
	pp_clp_property_value_t *pv = malloc(sizeof(pp_clp_property_value_t));
	pv->property = NULL;
	pv->value = NULL;
	return pv;
}

/* free a property/value structure */
void pp_clp_propval_delete(pp_clp_property_value_t *pv)
{
	assert(NULL != pv);
	free(pv->property);
	free(pv->value);
	free(pv);
}

/* convert a UFiP vector into a string */
char *pp_clp_print_ufip(vector_t *ufip, int relative)
{
	int len = 0;
	unsigned int i;
	char *path;
	if (vector_size(ufip) == 0)
		return strdup("/");
	for (i = 0; i < vector_size(ufip); i++) {
		pp_clp_ufit_t *ufit = vector_get(ufip, i);
		if (!relative || i > 0)
			len++;
		if (strcmp(ufit->ufct, ".") == 0) {
			len += 1;
		} else if (strcmp(ufit->ufct, "..") == 0) {
			len += 2;
		} else {
			len += snprintf(NULL, 0, "%s%d", ufit->ufct, ufit->instance_id);
		}
	}
	path = malloc(len + 1);
	path[0] = '\0';
	for (i = 0; i < vector_size(ufip); i++) {
		pp_clp_ufit_t *ufit = vector_get(ufip, i);
		if (!relative || i > 0)
			strcat(path, "/");
		if (strcmp(ufit->ufct, ".") == 0) {
			strcat(path, ".");
		} else if (strcmp(ufit->ufct, "..") == 0) {
			strcat(path, "..");
		} else {
			int tlen = snprintf(NULL, 0, "%s%d", ufit->ufct, ufit->instance_id);
			char *tag = malloc(tlen + 1);
			sprintf(tag, "%s%d", ufit->ufct, ufit->instance_id);
			strcat(path, tag);
			free(tag);
		}
	}
	return path;
}

#ifdef PP_CLP_DEBUG
/* dump an option/argument structure */
void pp_clp_optval_dump(pp_clp_option_value_t *pv)
{
	unsigned int i;
	assert(NULL != pv);

	printf("   - Option ");
	for (i = 0; clp_option_names[i].option_name; i++) {
		if (clp_option_names[i].option_id == pv->option) {
			printf("%s\n", clp_option_names[i].option_name);
			break;
		}
	}

	if (vector_size(&pv->values) > 0) {
		printf("   - Option Values: \n");
		for (i = 0; i < vector_size(&pv->values); i++) {
			printf("      %s\n", (char *)vector_get(&pv->values, i));
		}
	}
}

/* dump a target structure */
void pp_clp_target_dump(pp_clp_target_t *target)
{
	char *ufip;
	printf(" * Target type: %s\n",
		target->type == CLP_TARGET_DEFAULT ? "default" :
		target->type == CLP_TARGET_INSTANCE ? "instance" :
		target->type == CLP_TARGET_SET ? "set" :
		target->type == CLP_TARGET_ASSOCIATION ? "association" :
		target->type == CLP_TARGET_CLASS ? "class" :
		target->type == CLP_TARGET_VERB ? "verb" :
		"unknown");

	if (target->type == CLP_TARGET_INSTANCE ||
		target->type == CLP_TARGET_SET ||
		target->type == CLP_TARGET_CLASS ||
		target->type == CLP_TARGET_ASSOCIATION) {
		ufip = pp_clp_print_ufip(target->ufip1, target->ufip1_relative);
		printf("   - First UFiP: %s (%s)\n", ufip,
			target->ufip1_relative ? "relative" : "absolute");
		free(ufip);
	}

	if (target->type == CLP_TARGET_CLASS ||
		target->type == CLP_TARGET_SET) {
		printf("   - Class name: %s\n", target->ufct);
	} else if (target->type == CLP_TARGET_VERB) {
		printf("   - CLP verb: %s\n", clp_verb_names[target->verb].verb_name);
	}

	if (target->type == CLP_TARGET_ASSOCIATION) {
		printf("   - Associtation class: %s\n", target->ufct);
		if (target->ufip2) {
			ufip = pp_clp_print_ufip(target->ufip2, target->ufip2_relative);
			printf("   - Second UFiP: %s (%s)\n", ufip,
				target->ufip2_relative ? "relative" : "absolute");
			free(ufip);
		}
	}

	if (target->level < 0) {
		printf("   - Level: all\n");
	} else {
		printf("   - Level: %d\n", target->level);
	}
}

/* dump a property/value structure */
void pp_clp_propval_dump(pp_clp_property_value_t *pv)
{
	assert(NULL != pv);
	printf("   - %s=%s\n", pv->property, pv->value);
}

/* dump display options */
void pp_clp_display_options_dump(pp_clp_display_options_t *display)
{
	printf(" * Display options:\n");
	if (display->associations) {
		printf("   - Show associations\n");
		if (vector_size(&display->association_names) > 0) {
			unsigned int i;
			printf("   - Association names:");
			for (i = 0; i < vector_size(&display->association_names); i++) {
				char *c = vector_get(&display->association_names, i);
				printf(" %s", c);
			}
			printf("\n");
		}
	}
	if (display->properties) {
		printf("   - Show properties\n");
		if (vector_size(&display->property_names) > 0) {
			unsigned int i;
			printf("   - Property names:");
			for (i = 0; i < vector_size(&display->property_names); i++) {
				char *c = vector_get(&display->property_names, i);
				printf(" %s", c);
			}
			printf("\n");
			if (display->filter_property) {
				printf("   - Property filter: %s==%s\n",
					display->filter_property, display->filter_value);
			}
		}
	}
	if (display->targets) {
		printf("   - Show targets\n");
		if (vector_size(&display->target_names) > 0) {
			unsigned int i;
			printf("   - Target names:");
			for (i = 0; i < vector_size(&display->target_names); i++) {
				char *c = vector_get(&display->target_names, i);
				printf(" %s", c);
			}
			printf("\n");
		}
	}
	if (display->verbs) {
		printf("   - Show verbs\n");
	}
}

/* dump output options */
void pp_clp_output_options_dump(pp_clp_output_options_t *options)
{
	printf(" * Output options:\n");
	printf("   - Format: %s\n",
		options->format == CLP_OUTPUT_TEXT ? "text" :
		options->format == CLP_OUTPUT_CLPCSV ? "clpcsv" :
		options->format == CLP_OUTPUT_KEYWORD ? "keyword" :
		options->format == CLP_OUTPUT_CLPXML ? "clpxml" :
		"unknown");
	printf("   - Level: %s\n",
		options->level == CLP_OUTPUT_ERROR ? "error" :
		options->level == CLP_OUTPUT_TERSE ? "terse" :
		options->level == CLP_OUTPUT_VERBOSE ? "verbose" :
		"unknown");
	printf("   - Language: %s\n", options->language);
	printf("   - Beginning at: %s item\n", options->end ? "last" : "first");
	printf("   - Result order: %s\n", options->reverse ? "reverse" : "default");
	printf("   - Count: %d\n", options->count);
	printf("   - Number: %d - %d\n", options->range_min, options->range_max);
}

/* dump a command structure */
void pp_clp_cmd_dump(pp_clp_cmd_t *cmd)
{
	unsigned int i;
	assert(NULL != cmd);

	printf("CLP command dump:%c\n", '\r');

	printf(" * Verb: %s\n", clp_verb_names[cmd->verb].verb_name);

	if (vector_size(&cmd->options) > 0) {
		printf(" * Options:\n");
		for (i = 0; i < vector_size(&cmd->options); i++) {
			pp_clp_optval_dump(vector_get(&cmd->options, i));
		}
	}

	pp_clp_target_dump(cmd->target);

	if (vector_size(&cmd->properties) > 0) {
		printf(" * Properties:\n");
		for (i = 0; i < vector_size(&cmd->properties); i++) {
			pp_clp_propval_dump(vector_get(&cmd->properties, i));
		}
	}

	if (cmd->verb == CLP_VERB_SHOW) {
		pp_clp_display_options_dump(&cmd->display);
	}
	pp_clp_output_options_dump(&cmd->output);

	printf(" * Option flags:");
	if (cmd->opt_all)
		printf(" all");
	if (cmd->opt_default)
		printf(" default");
	if (cmd->opt_examine)
		printf(" examine");
	if (cmd->opt_help)
		printf(" help");
	if (cmd->opt_version)
		printf(" version");
	printf("\n");
}

#endif // PP_CLP_DEBUG

