#include "clpParser.h"
#include "clp_common.h"
#include "command.h"
#include <pp/cim.h>

static int check_display_instance(pp_clp_cmd_t *cmd,
	pp_cim_instance_t *instance)
{
	unsigned int i;

	// check property==value filter condition
	if (cmd->display.filter_property) {
		int ret = 1;
		const char *mapped = NULL;
		pp_cim_propval_t *prop = pp_cim_get_property(instance,
			cmd->display.filter_property);
		pp_cim_datatype_t type;
		pp_cim_data_t *data;
		if (!prop)
			return 0;
		type = prop->property->type;
		data = pp_cim_data_parse(type, cmd->display.filter_value);
		if (!prop->property->array) {
			ret = 0;
			if (data && pp_cim_data_equal(type, prop->data, *data)) {
				ret = 1;
			}
			if (type == PP_CIM_UNSIGNED && !prop->data.null) {
				mapped = pp_cim_map_value(prop->property->valmap,
					prop->data.types.unsigned_int);
				if (mapped && strcmp(mapped, cmd->display.filter_value) == 0) {
					ret = 1;
				}
			}
		} else {
			// the filter condition is met if one of the array elements
			// equals the specified value
			ret = 0;
			for (i = 0; i < vector_size(prop->data.types.array); i++) {
				pp_cim_data_t *d = vector_get(prop->data.types.array, i);
				if (data && pp_cim_data_equal(type, *d, *data)) {
					ret = 1;
					break;
				}
				if (type == PP_CIM_UNSIGNED && !d->null) {
					mapped = pp_cim_map_value(prop->property->valmap,
						d->types.unsigned_int);
					if (mapped && strcmp(mapped,
						cmd->display.filter_value) == 0) {
						ret = 1;
						break;
					}
				}
			}
		}
		pp_cim_propval_delete(prop);
		pp_cim_data_delete(type, 0, data);
		if (ret == 0)
			return 0;
	}

	// check for existence of requested properties
	for (i = 0; i < vector_size(&cmd->display.property_names); i++) {
		char *name = vector_get(&cmd->display.property_names, i);
		pp_cim_propval_t *p = pp_cim_get_property(instance, name);
		if (!p)
			return 0;
		pp_cim_propval_delete(p);
	}

	// check for existence of requested associations
	for (i = 0; i < vector_size(&cmd->display.association_names); i++) {
		char *name = vector_get(&cmd->display.association_names, i);
		vector_t *a = pp_cim_find_associations(instance, name);
		int size = a ? vector_size(a) : 0;
		if (a)
			vector_delete(a);
		if (size == 0)
			return 0;
	}

	return 1;
}

static void print_property(pp_clp_session_t *session, pp_clp_cmd_t *cmd UNUSED,
	pp_cim_propval_t *p)
{
	if (p->data.null) {
		eric_term_printf(session->term, "    %s is undefined\r\n",
			p->property->name);
	} else if (!p->property->array) {
		char *value = pp_cim_data_print(p->property->type, p->data);
		const char *mapped = NULL;
		if (p->property->type == PP_CIM_UNSIGNED) {
			mapped = pp_cim_map_value(p->property->valmap,
				p->data.types.unsigned_int);
		}
		if (mapped) {
			eric_term_printf(session->term, "    %s is %s (%s)\r\n",
				p->property->name, value, mapped);
		} else if (p->property->type == PP_CIM_STRING) {
			eric_term_printf(session->term, "    %s is \"%s\"\r\n",
				p->property->name, value);
		} else if (p->property->type == PP_CIM_REFERENCE) {
			vector_t *ufip = vector_new(NULL, 10,
				(vector_elem_del_func_simple)pp_clp_ufit_delete);
			if (PP_SUCCED(pp_cim_get_ufip(p->data.types.reference, ufip))) {
				;
				char *path = pp_clp_print_ufip(ufip, 0);
				eric_term_printf(session->term, "    %s is %s\r\n",
					p->property->name, path);
				free(path);
			} else {
				eric_term_printf(session->term, "    %s is an anonymous %s instance\r\n",
					p->property->name, p->data.types.reference->cim_class->cim_name);
			}
			vector_delete(ufip);
		} else {
			eric_term_printf(session->term, "    %s is %s\r\n",
				p->property->name, value);
		}
		free(value);
	} else {
		unsigned int i;
		eric_term_printf(session->term, "    %s is [", p->property->name);
		for (i = 0; i < vector_size(p->data.types.array); i++) {
			pp_cim_data_t *d = vector_get(p->data.types.array, i);
			char *value = pp_cim_data_print(p->property->type, *d);
			const char *mapped = NULL;
			if (p->property->type == PP_CIM_UNSIGNED) {
				mapped = pp_cim_map_value(p->property->valmap,
					d->types.unsigned_int);
			}
			if (mapped) {
				eric_term_printf(session->term, "%s (%s)",
					value, mapped);
			} else if (p->property->type == PP_CIM_STRING) {
				eric_term_printf(session->term, "\"%s\"",
					value);
			} else {
				eric_term_printf(session->term, "%s",
					value);
			}
			if (i < vector_size(p->data.types.array) - 1) {
				eric_term_printf(session->term, ", ");
			}
			free(value);
		}
		eric_term_printf(session->term, "]\r\n");
	}
}

static void print_associations(pp_clp_session_t *session,
	pp_cim_instance_t *instance, vector_t *associations)
{
	unsigned int i, j;
	for (i = 0; i < vector_size(associations); i++) {
		pp_cim_instance_t *inst = vector_get(associations, i);
		pp_cim_instance_t *ref = NULL;
		vector_t *properties = pp_cim_get_properties(inst);

		if (!properties)
			return;
		for (j = 0; j < vector_size(properties); j++) {
			pp_cim_propval_t *pv = vector_get(properties, j);
			if (pv->property->type == PP_CIM_REFERENCE
				&& !pv->data.null
				&& pv->data.types.reference != instance) {
				ref = pv->data.types.reference;
				break;
			}
		}
		vector_delete(properties);

		if (ref) {
			vector_t *ufip = vector_new(NULL, 10,
				(vector_elem_del_func_simple)pp_clp_ufit_delete);
			if (PP_SUCCED(pp_cim_get_ufip(ref, ufip))) {
				char *path = pp_clp_print_ufip(ufip, 0);
				eric_term_printf(session->term, "    %s => %s\r\n",
					inst->cim_class->cim_name, path);
				free(path);
			} else {
				eric_term_printf(session->term, "    %s => anonymous %s instance\r\n",
					inst->cim_class->cim_name,
					ref->cim_class->cim_name);
			}
			vector_delete(ufip);
		}
	}
}

static void display_instance(pp_clp_session_t *session, pp_clp_cmd_t *cmd,
	pp_cim_instance_t *instance)
{
	char *path;
	int has_ufip = 0;
	vector_t *ufip = vector_new(NULL, 10,
		(vector_elem_del_func_simple)pp_clp_ufit_delete);
	if (PP_SUCCED(pp_cim_get_ufip(instance, ufip))) {
		path = pp_clp_print_ufip(ufip, 0);
		eric_term_printf(session->term, "%s\r\n", path);
		free(path);
		has_ufip = 1;
	} else {
		eric_term_printf(session->term, "%s\r\n",
			instance->cim_class->cim_name);
	}
	vector_delete(ufip);

	if (cmd->display.properties) {
		unsigned int i;
		unsigned int num = vector_size(&cmd->display.property_names);
		eric_term_printf(session->term, "  Properties:\r\n");
		if (num == 0) {
			// list all (required) properties
			vector_t *properties = pp_cim_get_properties(instance);
			if (properties) {
				for (i = 0; i < vector_size(properties); i++) {
					pp_cim_propval_t *p = vector_get(properties, i);
					if (p && (p->property->required || cmd->opt_all)) {
						print_property(session, cmd, p);
					} else if (!p) {
						eric_term_printf(session->term, "  Error reading property\r\n");
					}
				}
				vector_delete(properties);
			}
		} else {
			// list requested properties
			for (i = 0; i < num; i++) {
				char *name = vector_get(&cmd->display.property_names, i);
				pp_cim_propval_t *p = pp_cim_get_property(instance, name);
				if (p) {
					print_property(session, cmd, p);
					pp_cim_propval_delete(p);
				} else {
					eric_term_printf(session->term, "  Error reading %s\r\n", name);
				}
			}
		}
	}

	if (cmd->display.targets) {
		if (instance->children) {
			pp_cim_instance_t *i = instance->children;
			eric_term_printf(session->term, "  Targets:\r\n");
			while (i) {
				pp_clp_ufit_t *ufit = pp_cim_get_ufit(i);
				eric_term_printf(session->term, "    %s%d\r\n",
					ufit->ufct, ufit->instance_id);
				pp_clp_ufit_delete(ufit);
				i = i->next;
			}
		} else {
			eric_term_printf(session->term, "  No targets.\r\n");
		}
	}

	if (cmd->display.associations) {
		int num = 0;
		if (vector_size(&cmd->display.association_names) == 0) {
			vector_t *associations = pp_cim_find_associations(instance, NULL);
			if (associations && vector_size(associations) > 0) {
				eric_term_printf(session->term, "  Associations:\r\n");
				print_associations(session, instance, associations);
				num += vector_size(associations);
			}
			if (associations)
				vector_delete(associations);
		} else {
			unsigned int i;
			for (i = 0; i < vector_size(&cmd->display.association_names); i++) {
				char *name = vector_get(&cmd->display.association_names, i);
				vector_t *associations = pp_cim_find_associations(instance, name);
				if (associations && vector_size(associations) > 0) {
					if (num == 0) {
						eric_term_printf(session->term, "  Associations:\r\n");
					}
					print_associations(session, instance, associations);
					num += vector_size(associations);
				}
				if (associations)
					vector_delete(associations);
			}
		}
		if (num == 0) {
			eric_term_printf(session->term, "  No associations.\r\n");
		}
	}

	if (cmd->display.verbs) {
		eric_term_printf(session->term, "  Supported verbs:");
		if (has_ufip) {
			eric_term_printf(session->term, " CD");
		}
		eric_term_printf(session->term, " HELP");
		if (pp_cim_clp_reset_supported(instance)) {
			eric_term_printf(session->term, " RESET");
		}
		if (pp_cim_clp_set_supported(instance)) {
			eric_term_printf(session->term, " SET");
		}
		eric_term_printf(session->term, " SHOW");
		if (pp_cim_clp_start_supported(instance)) {
			eric_term_printf(session->term, " START");
		}
		if (pp_cim_clp_stop_supported(instance)) {
			eric_term_printf(session->term, " STOP");
		}

		if (pp_cim_clp_create_supported(instance->cim_class)) {
			eric_term_printf(session->term, " CREATE");
		}

		if (pp_cim_clp_delete_supported(instance)) {
			eric_term_printf(session->term, " DELETE");
		}

		eric_term_printf(session->term, "\r\n");
	}

	if (cmd->display.associations || cmd->display.properties ||
		cmd->display.targets || cmd->display.verbs) {
		eric_term_printf(session->term, "\r\n");
	}
}

int clp_verb_handler_show(pp_clp_session_t *session, pp_clp_cmd_t *cmd)
{
	int ret = PP_SUC;
	unsigned int i, j;
	vector_t *instances, *list;

	if (cmd->opt_help) {
		int verbose = (cmd->output.level == CLP_OUTPUT_VERBOSE);
		clp_help_verb(session->term, CLP_VERB_SHOW, verbose);
		return PP_SUC;
	} else if (cmd->opt_version) {
		eric_term_printf(session->term, "SHOW %s\r\n", VERSION);
		return PP_SUC;
	}

	instances = pp_cim_clp_lookup(cmd->target, &session->default_target);
	if (!instances) {
		eric_term_printf(session->term, "Target not found.\r\n");
		return PP_ERR;
	}
	list = vector_new(NULL, 10,
		(vector_elem_del_func_simple)pp_cim_instance_untag);

	if (cmd->opt_examine) {
		eric_term_printf(session->term, "Command validity check successful.\r\n");
		eric_term_printf(session->term, "Run without -examine option to perform command.\r\n");
		goto bail;
	}

	if (vector_size(&cmd->display.target_names) == 0) {
		for (i = 0; i < vector_size(instances); i++) {
			pp_cim_instance_t *instance = vector_get(instances, i);
			pp_cim_update_properties(instance);
			if (check_display_instance(cmd, instance)) {
				pp_cim_instance_tag(instance);
				vector_add(list, instance);
			}
		}
	} else {
		for (j = 0; j < vector_size(&cmd->display.target_names); j++) {
			char *ufct = vector_get(&cmd->display.target_names, j);
			for (i = 0; i < vector_size(instances); i++) {
				pp_cim_instance_t *instance = vector_get(instances, i);
				pp_cim_update_properties(instance);
				if (!pp_cim_qualifies_as_ufct(instance, ufct))
					continue;
				if (check_display_instance(cmd, instance)) {
					pp_cim_instance_tag(instance);
					vector_add(list, instance);
				}
			}
		}
	}

	if (vector_size(list) == 0) {
		eric_term_printf(session->term, "No matching targets.\r\n");
	} else {
		int min = 0;
		int max = vector_size(list) - 1;
		int n;
		int count = 0;

		if (cmd->output.range_min > 0 && cmd->output.range_max > 0) {
			min = (cmd->output.range_min >= 1 ?
				cmd->output.range_min : 1) - 1;
			max = (cmd->output.range_max <= (int)vector_size(list) ?
				cmd->output.range_max : (int)vector_size(list)) - 1;
		} else if (cmd->output.reverse) {
			if (!cmd->output.end)
				max = min;
		} else {
			if (cmd->output.end)
				min = max;
		}

		if (cmd->output.reverse) {
			for (n = max; n >= min; n--) {
				pp_cim_instance_t *instance = vector_get(list, n);
				display_instance(session, cmd, instance);
				count++;
				if (cmd->output.count > 0 && count >= cmd->output.count)
					break;
			}
		} else {
			for (n = min; n <= max; n++) {
				pp_cim_instance_t *instance = vector_get(list, n);
				display_instance(session, cmd, instance);
				count++;
				if (cmd->output.count > 0 && count >= cmd->output.count)
					break;
			}
		}
	}

bail:
	vector_delete(instances);
	vector_delete(list);
	return ret;
}

