#include <pp/flash_access.h>
#include <sys/vfs.h>
#include <dirent.h>
#include <fcntl.h> // for O_CREAT
#include <string.h>
#include <stdio.h>

#ifdef __USE_FANCY_WRITE_METHODS__
#include <pp/base.h>
#include <pp/mutex.h>
#include <errno.h>
#include <pp/sem_keys.h>
#include <unistd.h>

static PP_FA_FILE *fa_create_fa_stream(const char *filename, const char *mode);
static int init_placeholders();
static int placeholder_resize(const char *filename, long size);

static int fa_write_sem = 0; // semaphore id to sync access
/* fopen-mode referring to truncate mode */
static const char *fa_trunc_str[] = {"r+", "w", "r+", "r+", "r+"}; 

/* error handling */
static const char *fa_errors[] = {
    /*0*/  "The Semaphore could not be initialized.",
    /*1*/  "The flashdisk root path is invalid.",
    /*2*/  "The filename or path is not on flashdisk.",
};

static int errno_base = 1;
inline int pp_fa_errno_base() { return errno_base; }

static const char* get_error_string(const int error) {
    return fa_errors[error - errno_base];
}

/* flash access file handle struct, defined as PP_FA_FILE */
struct fa_file_s {
    FILE*                       stream;
    long                        bytes_reserved;
    long                        bytes_written;
    long                        max_pos;
    pp_fa_truncate_mode_t       truncate_mode;
};

/**
 * opens a file for reading
 * access to file is locked, i.e. no one can read or write to file until it is
 * closed
 * perhaps we should allow multiple reads, but this may block pp_fopen_write()
 * calls
 * see fopen();
 */
PP_FA_FILE *pp_fa_open_read(const char *filename);

/**
 * opens a file for writing
 * access to file is locked, i.e. no one can read or write to file until it is
 * closed
 * number of bytes to be reserved for reading have to be specified, -1 for no
 * reservation
 * returns NULL if flashdisk-space could not be reserved
 * see fopen();
 */
PP_FA_FILE *pp_fa_open_write(const char *filename, long reserve, 
                             pp_fa_truncate_mode_t truncate_mode) {
    PP_FA_FILE *fa_stream;
    long free_size, placeholder_size, ph;
    char *dirstr = NULL, *tmp;
   
    assert(filename && *filename);
    assert(reserve > 0 || truncate_mode != RESERVED_LENGTH);

    if(strncmp(PP_FA_ROOT, filename, strlen(PP_FA_ROOT))) {
        errno = PP_FA_ERR_PATH_INVALID;
        return NULL; // do not bail out as we don't want to unlock / free
    }
    
    pp_sema_lock(fa_write_sem);
    
    if(NULL == (fa_stream = fa_create_fa_stream(filename,
                                                fa_trunc_str[truncate_mode])))
        goto bailout;

    /* get the directoy name */
    if(!(tmp = strchr((char*)filename + strlen(PP_FA_ROOT), '/'))) {
        errno = PP_FA_ERR_PATH_INVALID;
        goto bailout;
    }
    
    ph = tmp - (char*)filename - strlen(PP_FA_ROOT);
    dirstr = (char*)malloc(ph + 1);
    strncpy(dirstr, (char*)filename + strlen(PP_FA_ROOT), (size_t)ph);
    
    placeholder_size = pp_fa_du(dirstr);
    
    if(reserve > 0) {
        fa_stream->bytes_reserved = reserve;
        if((free_size = pp_fa_df(0)) == -1)
            goto bailout;
        /* when do we consider the reservation to be to large?
           * the free space is free_size + placeholder_size
           * perhaps we reserve to write a new file? then the resulting
             placeholder_size would be placeholder_size (which is the current
             size of the folder) + the reservation size.
           * reservation may be at most free_size / 2
           comments? */
        if(reserve > free_size / 2) {
            errno = ENOSPC;
            goto bailout;
        }
    }

    fa_stream->truncate_mode = truncate_mode;
    
    free(dirstr);
    return fa_stream;
 
 bailout:
    free(dirstr);
    pp_sema_unlock(fa_write_sem);
    return NULL;
}

/**
 * write char pointer to file
 * see pp_fa_write();
 */
size_t pp_fa_print(PP_FA_FILE *fa_stream, const char *str) {
    return(pp_fa_write(fa_stream, (void*)str, 1, strlen(str)));
}

/**
 * generic write to file
 * returns 0 if number of bytes written exceedes reserved space, return value
 * of fwrite() otherwhise
 * see fwrite();
 */
size_t pp_fa_write(PP_FA_FILE *fa_stream, const void *ptr, 
                   size_t size, size_t nitems) {
    long written, pos;
    
    assert(ptr);
    assert(fa_stream);
    
    if(fa_stream->bytes_reserved > 0 &&
       (long)(size * nitems) > fa_stream->bytes_reserved -
                               fa_stream->bytes_written)
       return 0;
    
    written = fwrite(ptr, size, nitems, fa_stream->stream);
    fa_stream->bytes_written += written;
    
    pos = ftell(fa_stream->stream);
    if(pos > fa_stream->max_pos)
        fa_stream->max_pos = pos;
    
    return written;
}

/**
 * closes file
 * truncates file as requested from pp_fopen_write()
 * removes locks
 * see fclose(); truncate();
 */
int pp_fa_close(PP_FA_FILE *fa_stream) {
    int ret;
    
    assert(fa_stream);
    assert(fa_stream->stream);
    
    /* truncate */
    switch(fa_stream->truncate_mode) {
        case RESERVED_LENGTH:
            ftruncate(fileno(fa_stream->stream), fa_stream->bytes_reserved);
            break;
            
        case LAST_POS:
            ftruncate(fileno(fa_stream->stream), ftell(fa_stream->stream));
            break;
            
        case MAX_POS:
            ftruncate(fileno(fa_stream->stream), fa_stream->max_pos);
            break;
            
        default:
            /* NONE, BEFORE_WRITE */
            break;
    }

    ret = fclose(fa_stream->stream);

    pp_sema_unlock(fa_write_sem);
    
    free(fa_stream);
    
    return ret;
}

/* library management */
int pp_fa_init() {
    errno_base = pp_register_errnos(sizeof(fa_errors) / sizeof(*fa_errors),
				    get_error_string);

//    fa_write_sem = pp_sema_init(FA_SEM_KEY);
    fa_write_sem = semget(FA_SEM_KEY, 1, IPC_CREAT | 0666);
//    fa_write_sem = semget(FA_SEM_KEY, 1, 0666);
    if(fa_write_sem < 0) {
        errno = PP_FA_ERR_INIT_SEM;
        return PP_ERR;
    }
    
    return init_placeholders();
}

void pp_fa_cleanup() {
    pp_unregister_errnos(errno_base);
}

/* static methods */
static PP_FA_FILE *fa_create_fa_stream(const char *filename, const char *mode) {
    PP_FA_FILE *fa_stream;
    
    assert(filename && *filename);
    assert(mode && *mode);
    
    fa_stream = (PP_FA_FILE*)malloc(sizeof(PP_FA_FILE));
    if(NULL == (fa_stream->stream = fopen(filename, mode)))
        if(errno != ENOENT ||
           NULL == (fa_stream->stream = fopen(filename, "w"))) {
            // retry, if we got ENOENT
            free(fa_stream);
            return NULL;
        }
    fa_stream->bytes_reserved = -1;
    fa_stream->bytes_written = 0;
    fa_stream->max_pos = 0;
    return fa_stream;
}

static int init_placeholders() {
    DIR *dir;
    struct dirent *entry;
    struct stat statbuf;
    char dirname[PATH_MAX], filename[PATH_MAX];
    long size;
    int ret = PP_SUC;
    
    pp_sema_lock(fa_write_sem);

    dir = opendir(PP_FA_ROOT);
    if(!dir) {
        ret = PP_ERR;
        errno = PP_FA_ERR_ROOT_INVALID;
        goto bailout;
    }

    while((entry = readdir(dir))) {
        if((strcmp(entry->d_name, "..") == 0) || 
           (strcmp(entry->d_name, ".") == 0))
	    continue;
        snprintf(dirname, PATH_MAX, "%s%s", PP_FA_ROOT, entry->d_name);
	if((lstat(dirname, &statbuf)) != 0) { 
            ret = PP_ERR;
            goto bailout;
        }
        if(S_ISDIR(statbuf.st_mode)) {
            snprintf(filename, PATH_MAX, "%s.tmp", dirname);
            if((size = pp_fa_du(dirname)) != pp_fa_du(filename) &&
               (ret = placeholder_resize(filename, size)) == PP_ERR)
                goto bailout;
        }
    }

 bailout:
    closedir(dir);
    pp_sema_unlock(fa_write_sem);
    return ret;
}

static int placeholder_resize(const char *filename, long size) {
    struct stat statbuf;
    FILE *file = NULL, *urandom = NULL;
    char buf[BUFSIZ];
    long read_sz = size;
    
    if((lstat(filename, &statbuf)) == 0) {
        if(statbuf.st_size > size) {
            // tmpfile is larger than it should, truncate
            truncate(filename, size);
            size = 0;
        } else if(statbuf.st_size < size) {
            // tmpfile is smaller than it should, append
            size -= statbuf.st_size;
        }
    }
    
    if(size > 0) {
        // need to write size bytes
        if((file = fopen(filename, "a")) == NULL ||
           (urandom = fopen("/dev/urandom", "r")) == NULL)
            goto error;
        while(size > 0) {
            read_sz = size > BUFSIZ ? BUFSIZ : size;
            if(fread(buf, 1, read_sz, urandom) != (size_t)read_sz ||
               fwrite(buf, 1, read_sz, file) != (size_t)read_sz)
                goto error;
            size -= BUFSIZ;
        }
        fclose(urandom);
        fclose(file);
    }
    return PP_SUC;
    
 error:
    fclose(urandom);
    fclose(file);
    return PP_ERR;
}
#endif /* __USE_FANCY_WRITE_METHODS__ */

/* generic methods, not fancy enough to be IFDEFed */

long pp_fa_df(int optimistic) {
    struct statfs statfsbuf;

    if (statfs(PP_FA_ROOT, &statfsbuf) != 0) {
#ifdef __USE_FANCY_WRITE_METHODS__
        errno = PP_FA_ERR_ROOT_INVALID;
#endif /* __USE_FANCY_WRITE_METHODS__ */
        return -1;
    }

    /* the _real_ free space is f_bavail * statfsbuf.f_bsize,
       but JFFS2 is not reliable in these things ;-)
       TODO: fix that when we can calculate real available space */
    if(optimistic)
        return (long)(statfsbuf.f_bavail * statfsbuf.f_bsize);
    return (long)(statfsbuf.f_bavail * statfsbuf.f_bsize * 0.75);
}

/* tiny recursive du - mainly from busybox*/
long pp_fa_du(const char *filename)
{
    struct stat statbuf;
    long sum;
    char newfile[PATH_MAX];

    if ((lstat(filename, &statbuf)) != 0) {
        return 0;
    }

    sum = statbuf.st_size;

    /* Don't add in stuff pointed to by symbolic links */
    if (S_ISLNK(statbuf.st_mode)) {
        sum = 0L;
    }
    if (S_ISDIR(statbuf.st_mode)) {
        DIR *dir;
        struct dirent *entry;

        dir = opendir(filename);
        if (!dir)
            return 0;

        while ((entry = readdir(dir))) {
            char *name = entry->d_name;

            if ((strcmp(name, "..") == 0) || (strcmp(name, ".") == 0)) {
                continue;
            }
            snprintf(newfile, PATH_MAX, "%s/%s", filename, name);
            sum += pp_fa_du(newfile);
        }
        closedir(dir);
    }

    return sum;
}

