#ifndef __PP_IPMI_H
#define __PP_IPMI_H

#ifdef WIN32

#define PACKED_STRUCT

#include <time.h>
#include <pp/strstream.h>
#include <pp/win32.h>

#else

#include <pp/base.h>
#include <pp_byte_order.h>
#define PACKED_STRUCT __attribute__ ((packed))

#endif

#include <pp/vector.h>
#if defined (PP_FEAT_IPMI_SERVER)
#include <pp/bmc/loopi_adapter.h> /* for loopi channel numbers */
#endif

#define DEFAULT_IPMI_PORT	623

typedef struct {
    unsigned char id;  // the index of the destination
    unsigned char acked;
    unsigned char type;
    unsigned char timeout;
    unsigned char retries;

    unsigned char addr_format;
    unsigned char backup_gateway;
    unsigned char ip_addr[4];
    unsigned char mac_addr[6];

    /* OEM fields */
    char email[65];
    char community[65];
} pp_ipmi_lanp_dest_t;

#pragma pack(1)

struct pp_ipmi_event_filter_entry_s {
    unsigned char id;  // the index of the filter, not part of the filter data
  
    // according to table 17-2
    BITFIELD3(unsigned, \
             record_type:5, \
             configuration_type:2, \
             enable:1);
        
    struct {
        BITFIELD8(unsigned,  \
                 alert:1, \
                 poweroff:1, \
                 reset:1, \
                 power_cycle:1, \
                 OEM:1, \
                 diagnostic_interrupt:1, \
                 group_control:1, \
                 dummy2:1);
    } PACKED_STRUCT action;

    BITFIELD3(unsigned,
             alert_policy_number:4, \
             alert_policy_group_control_selector:3, \
             dummy3:1);
    
    unsigned char event_severity;
    unsigned char generatorID1;
    unsigned char generatorID2;
    unsigned char sensor_type;
    unsigned char sensor_number;
    unsigned char event_trigger;
    
    unsigned char offset_mask1;
    unsigned char offset_mask2;
    struct {
        unsigned char mask;
        unsigned char exact;
        unsigned char data;
    } PACKED_STRUCT data_filter[3];
} PACKED_STRUCT;
typedef struct pp_ipmi_event_filter_entry_s pp_ipmi_pef_filter_t;

struct pp_ipmi_alert_policy_entry_s {
    unsigned char id;  // the index of the policy, not part of the policy data

    BITFIELD3(unsigned, \
             policy:3, \
             enable:1, \
             number:4);
    
    BITFIELD2(unsigned, \
             destination:4, \
             channel:4);
    
    unsigned char alert_string;
} PACKED_STRUCT;
typedef struct pp_ipmi_alert_policy_entry_s pp_ipmi_pef_policy_t;

#pragma pack()

/* ------------------------------------------------------------------------- */

/* Select which type of command should be sent to ipmi tool.
   First step is to choose the main command (the lib-plugins
   in ipmitool). */
typedef enum {
    PP_IPMI_CMD_NONE = 0,
    PP_IPMI_CMD_BMC,
    PP_IPMI_CMD_CHASSIS,
    PP_IPMI_CMD_SEL,
    PP_IPMI_CMD_SDR,
    PP_IPMI_CMD_SENSOR,
    PP_IPMI_CMD_PEF,
    PP_IPMI_CMD_LAN,
    PP_IPMI_CMD_RAW,
    /* below not yet changed to our firmware needs!
       (means everything is put on the console) */
    PP_IPMI_CMD_LANP,
    PP_IPMI_CMD_EVENT,
    PP_IPMI_CMD_FRU,
    PP_IPMI_CMD_USER,
    PP_IPMI_CMD_CHANNEL,
    PP_IPMI_CMD_SESSION,
    PP_IPMI_CMD_KVM,
    /* OEM commands */
#ifdef PRODUCT_AMDDC
    PP_IPMI_CMD_OEM_OPMA,
#endif
#ifdef PP_FEAT_ESB2_TPT
    PP_IPMI_CMD_OEM_INTEL_USR_CFG,
    PP_IPMI_CMD_OEM_INTEL_TPT,
#endif
#ifdef PRODUCT_PDU
    PP_IPMI_CMD_OEM_PP_RPC,
#endif
#ifdef OEM_RACKABLE
    PP_IPMI_CMD_OEM_RACKABLE,
#endif
    PP_IPMI_CMD_OEM_PP,
    PP_IPMI_CMD_OEM_RARITAN,
    PP_IPMI_CMD_SELF_TEST,	/* this is a raritan specific OEM command set */
} pp_ipmi_command_t;

/* Second step is to choose the sub-command for the selected command.
   This depends on the selected main command.
   
   The params comment shows what must be passed as the params
   parameter of pp_ipmi_send_command(). If none specified, no
   parameter is needed (set it to NULL). */

/* bmc commands */
typedef enum {
    PP_IPMI_BMC_SUBCMD_INFO,
    PP_IPMI_BMC_SUBCMD_GET_ENABLES,
    PP_IPMI_BMC_SUBCMD_SET_ENABLES,	/* params: pp_ipmi_bmc_set_enables_param_t */
    PP_IPMI_BMC_SUBCMD_RESET,		/* params: pp_ipmi_bmc_reset_param_t */
    PP_IPMI_BMC_SUBCMD_GET_DEVICE_GUID,
    PP_IPMI_BMC_SUBCMD_GET_SYSTEM_GUID,
    PP_IPMI_BMC_SUBCMD_GET_ACPI_STATE,
    
} pp_ipmi_bmc_subcommand_t;

/* chassis commands */
typedef enum {
    PP_IPMI_CHASSIS_SUBCMD_STATUS,
    PP_IPMI_CHASSIS_SUBCMD_POWER_STATUS,
    PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL,	/* params: pp_ipmi_chassis_power_param_t */
    PP_IPMI_CHASSIS_SUBCMD_IDENTIFY,	/* params: int, seconds, 0 is off, -1 is always on */
    PP_IPMI_CHASSIS_SUBCMD_POH,
    PP_IPMI_CHASSIS_SUBCMD_RESTART_CAUSE,
    PP_IPMI_CHASSIS_SUBCMD_GET_POLICIES,
    PP_IPMI_CHASSIS_SUBCMD_SET_POLICY,	/* params: pp_ipmi_chassis_set_pol_param_t */
    PP_IPMI_CHASSIS_SUBCMD_GET_BOOTPARAM,	/* params: unsigned char, bootparameter as number */
    PP_IPMI_CHASSIS_SUBCMD_GET_BOOTDEV,
    PP_IPMI_CHASSIS_SUBCMD_SET_BOOTDEV,	/* params: pp_ipmi_chassis_bflag_param_t */
} pp_ipmi_chassis_subcommand_t;

/* sel commands */
typedef enum {
    PP_IPMI_SEL_SUBCMD_INFO,
    PP_IPMI_SEL_SUBCMD_CLEAR,
    PP_IPMI_SEL_SUBCMD_LIST,		/* params: none (-> all) or pp_ipmi_sel_list_param_t */
    PP_IPMI_SEL_SUBCMD_GET,		/* params: vector_t, all entries to delete */
    PP_IPMI_SEL_SUBCMD_DELETE,		/* params: vector_t, all entries to delete */
    PP_IPMI_SEL_SUBCMD_SET_TIME,	/* params: time_t */
    PP_IPMI_SEL_SUBCMD_GET_TIME,
} pp_ipmi_sel_subcommand_t;

/* sdr commands */
typedef enum {
    PP_IPMI_SDR_SUBCMD_INFO,
    PP_IPMI_SDR_SUBCMD_LIST,		/* params: none or pp_ipmi_sdr_list_param_t */
    PP_IPMI_SDR_SUBCMD_UPDATE,		/* params: sdr_idx */
} pp_ipmi_sdr_subcommand_t;

/* sensor commands */
typedef enum {
    PP_IPMI_SENSOR_SUBCMD_LIST,
    PP_IPMI_SENSOR_SUBCMD_THRESHOLD,	/* params: vector_t with pp_ipmi_sensor_set_th_param_t */
    PP_IPMI_SENSOR_SUBCMD_GET_BY_NAME,	/* params: vector_t, one or more sensor(s) by name */
    PP_IPMI_SENSOR_SUBCMD_GET_BY_ID,	/* params: vector_t, one or more sensor(s) by ID */
    PP_IPMI_SENSOR_SUBCMD_REARM_EVENTS,  /* params: pp_ipmi_sensor_rearm_param_t */
    PP_IPMI_SENSOR_SUBCMD_GET_RAW_BY_ID, /* params: vector_t, one or more sensor(s) by ID */
    PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID, /* params: vector_t, one or more sensor(s) by ID */
    PP_IPMI_SENSOR_SUBCMD_SET_RAW_TH,	/* params: vector_t with pp_ipmi_sensor_set_raw_th_param_t */
} pp_ipmi_sensor_subcommand_t;

/* lanp commands */
typedef enum {
    PP_IPMI_LANP_SUBCMD_GET,	    /* params: pp_ipmi_lanp_get_param_t */
    PP_IPMI_LANP_SUBCMD_SET,
    PP_IPMI_LANP_SUBCMD_DESTS,
    PP_IPMI_LANP_SUBCMD_SET_DEST,   /* params: pp_ipmi_lanp_set_dest_param_t */
    PP_IPMI_LANP_SUBCMD_SET_IP_SOURCE,		/* params: pp_ipmi_lanp_set_ip_source_t */
    PP_IPMI_LANP_SUBCMD_SET_SNMP_STRING,	/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_IP_ADDRESS,		/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_NETMASK,		/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_MAC_ADDRESS,	/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_DEF_GW_IP,		/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_DEF_GW_MAC,		/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_BAK_GW_IP,		/* params: pp_ipmi_lanp_set_string_t */
    PP_IPMI_LANP_SUBCMD_SET_BAK_GW_MAC,		/* params: pp_ipmi_lanp_set_string_t */
} pp_ipmi_lanp_subcommand_t;

/* lan commands */
typedef enum {
    PP_IPMI_LAN_DESTINATION_SET,
    PP_IPMI_LAN_DESTINATION_GET,
} pp_ipmi_lan_subcommand_t;

/* raw command */
typedef enum {
    PP_IPMI_RAW_SUBCMD_RAW,
} pp_ipmi_raw_subcommand_t;

/* event subcommands */
typedef enum {
    PP_IPMI_EVENT_SUBCMD_GENERATE,
} pp_ipmi_event_subcommand_t;

/* fru subcommands */
typedef enum {
    PP_IPMI_FRU_SUBCMD_PRINT,
    PP_IPMI_FRU_SUBCMD_BOARD,
} pp_ipmi_fru_subcommand_t;

/* user subcommands */
typedef enum {
    PP_IPMI_USER_SUBCMD_SUMMARY,
    PP_IPMI_USER_SUBCMD_INFO,
    PP_IPMI_USER_SUBCMD_LIST,
    PP_IPMI_USER_SUBCMD_SET_NAME,
    PP_IPMI_USER_SUBCMD_SET_PASSWORD,
    PP_IPMI_USER_SUBCMD_TEST_PASSWORD,
    PP_IPMI_USER_SUBCMD_DISABLE,
    PP_IPMI_USER_SUBCMD_ENABLE
} pp_ipmi_user_subcommand_t;

/* channel subcommands */
typedef enum {
    PP_IPMI_CHANNEL_SUBCMD_AUTHCAP,
    PP_IPMI_CHANNEL_SUBCMD_GET_ACCESS,
    PP_IPMI_CHANNEL_SUBCMD_SET_ACCESS,
    PP_IPMI_CHANNEL_SUBCMD_INFO,
    PP_IPMI_CHANNEL_SUBCMD_GET_CIPHERS,
} pp_ipmi_channel_subcommand_t;

/* session subcommands */
typedef enum {
    PP_IPMI_SESSION_SUBCMD_INFO,
} pp_ipmi_session_subcommand_t;

/* pef subcommands */
typedef enum {
    PP_IPMI_PEF_FILTER_GET,
    PP_IPMI_PEF_FILTER_SET,
    PP_IPMI_PEF_POLICY_GET,
    PP_IPMI_PEF_POLICY_SET,
} pp_ipmi_pef_subcommand_t;

/* kvm commands */
typedef enum {
    PP_IPMI_KVM_SUBCMD_RESET,
} pp_ipmi_kvm_subcommand_t;

#ifdef PRODUCT_AMDDC
typedef enum {
    PP_IPMI_OEM_OPMA_CLEAR_CMOS,
    PP_IPMI_OEM_OPMA_SET_LOCAL_LOCKOUT,
    PP_IPMI_OEM_OPMA_GET_LOCAL_LOCKOUT,
} pp_ipmi_OEM_opma_subcommand_t;
#endif /* PRODUCT_AMDDC */

#ifdef PP_FEAT_ESB2_TPT
/* oem intel esb2 usr cfg commands */
typedef enum {
    PP_IPMI_OEM_INTEL_GET_USR_CFG,
    PP_IPMI_OEM_INTEL_SET_USR_CFG,
} pp_ipmi_OEM_intel_esb2_usr_cfg_subcommand_t;

/* oem intel esb2 tpt commands */
typedef enum {
    PP_IPMI_OEM_INTEL_GET_TPT_STATUS,
    PP_IPMI_OEM_INTEL_TPT_LISTEN,
    PP_IPMI_OEM_INTEL_GET_TPT_CFG,
    PP_IPMI_OEM_INTEL_SET_TPT_CFG,
} pp_ipmi_oem_intel_esb2_tpt_subcommand_t;
#endif

#ifdef OEM_RACKABLE
typedef enum {
    PP_IPMI_OEM_RACKABLE_LCD_SET_SLOT,
    PP_IPMI_OEM_RACKABLE_LCD_GET_SLOT,
    PP_IPMI_OEM_RACKABLE_LCD_SET_STATE,
    PP_IPMI_OEM_RACKABLE_LCD_GET_STATE
} pp_ipmi_OEM_rackable_subcommand_t;
#endif /* OEM_RACKABLE */

typedef enum {
    PP_IPMI_OEM_PP_GET_PWM,
    PP_IPMI_OEM_PP_SET_PWM
} pp_ipmi_oem_pp_subcommand_t;

#ifdef PRODUCT_PDU
typedef enum {
    PP_IPMI_OEM_PP_RPC_SET_POWER_ON_DELAY,
    PP_IPMI_OEM_PP_RPC_GET_POWER_ON_DELAY,
    PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_STATE,
    PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE,
    PP_IPMI_OEM_PP_RPC_SET_GROUP_STATE,
    PP_IPMI_OEM_PP_RPC_SET_GROUP_MEMBERSHIP,
    PP_IPMI_OEM_PP_RPC_GET_GROUP_MEMBERSHIP,
    PP_IPMI_OEM_PP_RPC_SET_GROUP_POWER_ON_DELAY,
    PP_IPMI_OEM_PP_RPC_GET_GROUP_POWER_ON_DELAY,
    PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_ACL,
    PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_ACL,
}pp_ipmi_oem_pp_rpc_subcommand_t;
#endif /*PRODUCT_PDU*/

// OEM raritan specific
typedef enum {
    PP_IPMI_OEM_RARITAN_FW_VERSION,
    PP_IPMI_OEM_RARITAN_FW_UPGRADE,	// params: pp_ipmi_oem_raritan_fw_upgrade_param_t
    PP_IPMI_OEM_RARITAN_CFG_BACKUP,	// params: pp_ipmi_oem_raritan_cfg_backup_param_t
    PP_IPMI_OEM_RARITAN_CFG_RESTORE,	// params: pp_ipmi_oem_raritan_cfg_restore_param_t
    PP_IPMI_OEM_RARITAN_SN_GET,
    PP_IPMI_OEM_RARITAN_SN_SET,		// params: pp_ipmi_oem_raritan_sn_set_param_t
    PP_IPMI_OEM_RARITAN_DEFAULTS,
    PP_IPMI_OEM_RARITAN_GET_EXT_USER_PARAMS,
    PP_IPMI_OEM_RARITAN_LIST_EXT_USER_PARAMS,
} pp_ipmi_oem_raritan_subcommand_t;

typedef enum {
    PP_IPMI_SELF_TEST_SUBCMD_DDC_INFO,
    PP_IPMI_SELF_TEST_SUBCMD_VIDEO_STATUS,
    PP_IPMI_SELF_TEST_SUBCMD_VIDEO_CRC,
    PP_IPMI_SELF_TEST_SUBCMD_IPMB_BMC_STATUS,
    PP_IPMI_SELF_TEST_SUBCMD_FML_ESB2_STATUS,
    PP_IPMI_SELF_TEST_SUBCMD_USB_OPERATION,
    PP_IPMI_SELF_TEST_SUBCMD_NIC_STATUS,
    PP_IPMI_SELF_TEST_SUBCMD_NIC_LOOPBACK,
    PP_IPMI_SELF_TEST_SUBCMD_NIC_PING,
    PP_IPMI_SELF_TEST_SUBCMD_NIC_BROADCAST_PING,
} pp_ipmi_self_test_subcommand_t;

/* ------------------------------------------------------------------------- */

/* All ipmi commands have their own parameters which differ from each
   command. All possible parameter structures are defined below.
   Some commands have no parameter (e.g. reset).
   Some commands have not been ported to the new parameter structure,
   they use the "old" ipmistyle interface (argc/argv). For these
   functions, just use a char* string, it is converted internally.
   To determine at runtime whether the old char* interface is used,
   the member is_cmdline of the parameter structure must be set
   accordingly.
*/

/* commonly used things */
typedef enum {
    // order is important!
    PP_IPMI_PRIV_UNSPECIFIED,
    PP_IPMI_PRIV_NO_ACCESS,
    PP_IPMI_PRIV_CALLBACK,
    PP_IPMI_PRIV_USER,
    PP_IPMI_PRIV_OPERATOR,
    PP_IPMI_PRIV_ADMIN,
    PP_IPMI_PRIV_OEM,
} pp_ipmi_priv_levels;


/* bmc reset */
typedef enum {
    PP_IPMI_BMC_RESET_COLD,
    PP_IPMI_BMC_RESET_WARM,
} pp_ipmi_bmc_reset_param_t;

/* bmc get enables */
typedef struct {
    int			receive_msg_int_enabled;
    int			event_msg_buf_int_enabled;
    int			event_msg_buf_enabled;
    int			sel_enabled;
    int			oem0_enabled;
    int			oem1_enabled;
    int			oem2_enabled;
} pp_ipmi_bmc_set_enables_param_t;

/* chassis power */
typedef enum {
    PP_IPMI_CHASSIS_POWER_DOWN,
    PP_IPMI_CHASSIS_POWER_UP,
    PP_IPMI_CHASSIS_POWER_CYCLE,
    PP_IPMI_CHASSIS_RESET,
    PP_IPMI_CHASSIS_POWER_DIAG,
    PP_IPMI_CHASSIS_POWER_SOFT,
} pp_ipmi_chassis_power_param_t;

/* chassis set policy */
typedef enum {
    PP_IPMI_CHASSIS_POLICY_UNKNOWN,
    PP_IPMI_CHASSIS_POLICY_ALWAYS_ON,
    PP_IPMI_CHASSIS_POLICY_ALWAYS_OFF,
    PP_IPMI_CHASSIS_POLICY_PREVIOUS,
} pp_ipmi_chassis_restart_policy_t;

typedef pp_ipmi_chassis_restart_policy_t pp_ipmi_chassis_set_pol_param_t;

/* chassis set bootflag */
typedef enum {
    PP_IPMI_CHASSIS_SET_BOOTFLAG_NO_CHANGE,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_PXE,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DISK,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_SAFE_DISK,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_DIAG,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_CDROM,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_BIOS,
    PP_IPMI_CHASSIS_SET_BOOTFLAG_FORCE_FLOPPY,
} pp_ipmi_chassis_bootflag_t;

typedef struct {
    pp_ipmi_chassis_bootflag_t	device;
    int				clear_cmos;
} pp_ipmi_chassis_bflag_param_t;

/* set user access privileges */
typedef struct {
    unsigned char channel;
    unsigned char user_id;
    int callin; /* user restricted to callback; -1=off, 1=on, 0=no change */
    int link;   /* user link authentication;    -1=off, 1=on, 0=no change */
    int ipmi;   /* user ipmi messaging;         -1=off, 1=on, 0=no change */
    int privilege;
} pp_ipmi_chan_setaccess_param_t;

/* user summary */
typedef struct {
    unsigned char chan;
} pp_ipmi_usr_summary_param_t;

/* user info */
typedef struct {
    unsigned char chan;
    unsigned char user_id;
} pp_ipmi_channel_user_param_t;

typedef pp_ipmi_channel_user_param_t pp_ipmi_usr_info_param_t;

/* user list */
typedef struct {
    unsigned char chan;
} pp_ipmi_usr_list_param_t;

/* set user name */
typedef struct {
    unsigned char uid;
    char name[17];
} pp_ipmi_usr_set_name_param_t;

/* user set/test password */
typedef struct {
    unsigned char uid;
    unsigned char size20; /* boolean */
    char pwd[20];
} pp_ipmi_usr_password_param_t;

typedef struct {
    unsigned char uid;
} pp_ipmi_usr_uid_param_t;

/* sel list */
typedef enum {
    PP_IPMI_SEL_LIST_RANGE_ALL,
    PP_IPMI_SEL_LIST_RANGE_FROMTO,
    PP_IPMI_SEL_LIST_RANGE_LASTDAY,
    PP_IPMI_SEL_LIST_RANGE_LAST3DAYS,
    PP_IPMI_SEL_LIST_RANGE_LASTWEEK,
    PP_IPMI_SEL_LIST_RANGE_LASTMONTH,
    PP_IPMI_SEL_LIST_RANGE_COUNT_LAST,
    PP_IPMI_SEL_LIST_RANGE_COUNT_FIRST,
} pp_ipmi_sel_range_t;

typedef struct {
    pp_ipmi_sel_range_t range;
    /* only valid if range == PP_IPMI_SEL_LIST_RANGE_FROMTO */
    time_t		from;
    time_t		to;
    /* only valid if range == PP_IPMI_SEL_LIST_RANGE_COUNT_LAST or PP_IPMI_SEL_LIST_RANGE_COUNT_FIRST */
    unsigned int	count;
} pp_ipmi_sel_list_param_t;

/* sdr list */
typedef enum {
    PP_IPMI_SDR_LIST_ALL,
    PP_IPMI_SDR_LIST_FULL,
    PP_IPMI_SDR_LIST_COMPACT,
    PP_IPMI_SDR_LIST_EVENT,
    PP_IPMI_SDR_LIST_FRU,
    PP_IPMI_SDR_LIST_MCLOC,
} pp_ipmi_sdr_list_param_t;

/* sensor set threshold */
typedef enum {
    PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV,
    PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV,
    PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT,
    PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT,
    PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT,
    PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT,
} pp_ipmi_threshold_type_t;

typedef struct {
    char*		sensor_name;
    pp_ipmi_threshold_type_t threshold_type;
    float		value;
    int			use_cim;
    int			cim_value;
} pp_ipmi_sensor_set_th_param_t;

typedef struct {
    int		                sensor_id;
    pp_ipmi_threshold_type_t    threshold_type;
    unsigned char               raw_value;
} pp_ipmi_sensor_set_raw_th_param_t;

typedef struct {
    unsigned unc_high : 1;
    unsigned unc_low  : 1;
    unsigned lnr_high : 1;
    unsigned lnr_low  : 1;
    unsigned lcr_high : 1;
    unsigned lcr_low  : 1;
    unsigned lnc_high : 1;
    unsigned lnc_low  : 1;
    unsigned unr_high : 1;
    unsigned unr_low  : 1;
    unsigned ucr_high : 1;
    unsigned ucr_low  : 1;
} pp_ipmi_threshold_events_t;

typedef struct {
    int sensor_id;
    int all_events; // 0: selected events only ; 1: all events
    int discrete;   // 0: threshold sensor; 1: discrete sensor
    // all subsequent fields are ignored if all_events != 0:
    union {
	pp_ipmi_threshold_events_t threshold;
	int discrete;
    } assertions;
    union {
	pp_ipmi_threshold_events_t threshold;
	int discrete;
    } deassertions;
} pp_ipmi_sensor_rearm_param_t;

typedef struct {
    unsigned char chan;
    unsigned char set;           // this is the same as dest->id
    pp_ipmi_lanp_dest_t dest;
} pp_ipmi_lanp_set_dest_param_t;

/* get lan parameters */
typedef struct {
    unsigned char chan;
    // in which parameters are we interested?
    int need_set_in_progress;
    int need_auth_types;
    int need_ip_addr_source;
    int need_ip_addr;
    int need_subnet_mask;
    int need_mac_addr;
    int need_snmp_string;
    int need_ip_header;
    int need_arp_config;
    int need_def_gw_ip;
    int need_def_gw_mac;
    int need_bak_gw_ip;
    int need_bak_gw_mac;
    int need_cipher_suites;
    int need_cipher_priv_max;
} pp_ipmi_lanp_get_param_t;

/* set lan parameters */
typedef enum {
    PP_IPMI_LANP_IP_ADDR_SOURCE_UNSPECIFIED,
    PP_IPMI_LANP_IP_ADDR_SOURCE_STATIC,
    PP_IPMI_LANP_IP_ADDR_SOURCE_DHCP,
    PP_IPMI_LANP_IP_ADDR_SOURCE_BIOS_ASSIGNED,
    PP_IPMI_LANP_IP_ADDR_SOURCE_OTHER
} pp_ipmi_lanp_ip_addr_source;

typedef struct {
    unsigned char chan;
    pp_ipmi_lanp_ip_addr_source source;
} pp_ipmi_lanp_set_ip_source_t;

typedef struct {
    unsigned char chan;
    char *value;
} pp_ipmi_lanp_set_string_t;

/* raw */
typedef struct {
    unsigned char net_fn;
    unsigned char cmd;
    vector_t *data;	/* vector of unsigned int */
} pp_ipmi_raw_param_t;

#ifdef PRODUCT_AMDDC

typedef struct {
    unsigned char lock;
} pp_ipmi_oem_opma_set_local_lockout_param_t;

#endif /* PRODUCT_AMDDC */

#ifdef PP_FEAT_ESB2_TPT

typedef enum {
    PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE    = 1 << 2,
    PP_IPMI_OEM_INTEL_USR_CFG_ALL           = 0xffffffff,
} pp_ipmi_oem_intel_usr_cfg_mask_t;

typedef struct {
    unsigned char kvm_enable;
} pp_ipmi_oem_intel_usr_cfg_t;

typedef struct {
    unsigned char chan;
    unsigned char uid;
    pp_ipmi_oem_intel_usr_cfg_mask_t mask;
    pp_ipmi_oem_intel_usr_cfg_t cfg;
} pp_ipmi_oem_intel_set_usr_cfg_param_t;

typedef struct {
    unsigned char chan;
    unsigned char uid;
    pp_ipmi_oem_intel_usr_cfg_mask_t mask;
} pp_ipmi_oem_intel_get_usr_cfg_param_t;

typedef struct {
    unsigned char chan;
} pp_ipmi_oem_intel_get_tpt_status_param_t;

#define PP_IPMI_OEM_INTEL_PORT_DEFAULT ((unsigned short)0)

typedef enum {
    PP_IPMI_OEM_INTEL_PROTO_DEFAULT = 0,
    PP_IPMI_OEM_INTEL_PROTO_SSLTLS = 1,
    PP_IPMI_OEM_INTEL_PROTO_SSLTLS_SPECIAL_CIPHER = 2,
    PP_IPMI_OEM_INTEL_PROTO_TCPIP_ONLY = 3,
} pp_ipmi_oem_intel_protocol_t;

typedef enum {
    PP_IPMI_OEM_INTEL_LINGER_DEFAULT = 0,
    PP_IPMI_OEM_INTEL_LINGER_ON = 1,
    PP_IPMI_OEM_INTEL_LINGER_OFF = 2,
} pp_ipmi_oem_intel_linger_t;

typedef enum {
    PP_IPMI_OEM_INTEL_COALESCING_DEFAULT = 0,
    PP_IPMI_OEM_INTEL_COALESCING_ON = 1,
    PP_IPMI_OEM_INTEL_COALESCING_OFF = 2,
} pp_ipmi_oem_intel_coalescing_t;

typedef struct {
    unsigned char chan;
    unsigned short port;
    int listen; /* boolean */
    pp_ipmi_oem_intel_protocol_t protocol;
    unsigned char cipher_suite;
    pp_ipmi_oem_intel_linger_t linger;
    pp_ipmi_oem_intel_coalescing_t coalescing;
} pp_ipmi_oem_intel_tpt_listen_param_t;

typedef enum {
    PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES           = 1 << 1,
    PP_IPMI_OEM_INTEL_TPT_CFG_ADDIN_FML_SADDR   = 1 << 2,
    PP_IPMI_OEM_INTEL_TPT_CFG_PORT              = 1 << 3,
    PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE       = 1 << 4,
    PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_IN_SIZE   = 1 << 5,
    PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_OUT_SIZE  = 1 << 6,
    PP_IPMI_OEM_INTEL_TPT_CFG_FML_OUTPUT_LIMIT  = 1 << 7,
    PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN          = 1 << 8,
    PP_IPMI_OEM_INTEL_TPT_CFG_TIMEOUTS          = 1 << 9,
    PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_SUPP    = 1 << 10,
    PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_DEF     = 1 << 11,
    PP_IPMI_OEM_INTEL_TPT_CFG_ALL               = 0xffffffff,
} pp_ipmi_oem_intel_tpt_cfg_mask_t;

#define PP_IPMI_OEM_INTEL_SUPP_SSL30    0x04
#define PP_IPMI_OEM_INTEL_SUPP_TLS10    0x01

#define PP_IPMI_OEM_INTEL_PROTO_SSL30   0x00
#define PP_IPMI_OEM_INTEL_PROTO_TLS10   0x01

typedef struct {
    // PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES
    unsigned char enabled;
    unsigned char in_standby;
    // PP_IPMI_OEM_INTEL_TPT_CFG_ADDIN_FML_SADDR
    unsigned char addin_fml_saddr;
    // PP_IPMI_OEM_INTEL_TPT_CFG_PORT
    unsigned short port;
    // PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE
    unsigned char fml_multi_push_mode;
    // PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_IN_SIZE (readonly)
    unsigned short fml_max_in_size;
    // PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_OUT_SIZE (readonly)
    unsigned short fml_max_out_size;
    // PP_IPMI_OEM_INTEL_TPT_CFG_FML_OUTPUT_LIMIT
    unsigned short fml_output_limit;
    // PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN
    unsigned char def_ssl;
    unsigned char def_linger;
    unsigned char def_coalescing;
    // PP_IPMI_OEM_INTEL_TPT_CFG_TIMEOUTS
    unsigned char recv_timeout;
    unsigned char xmit_timeout;
    // PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_SUPP (readonly)
    unsigned char ssl_supported;
    // PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_DEF
    unsigned char def_ssl_proto;
} pp_ipmi_oem_intel_tpt_cfg_t;

typedef struct {
    pp_ipmi_oem_intel_tpt_cfg_mask_t mask;
    unsigned char chan;
    pp_ipmi_oem_intel_tpt_cfg_t cfg;
} pp_ipmi_oem_intel_set_tpt_cfg_param_t;

typedef struct {
    pp_ipmi_oem_intel_tpt_cfg_mask_t mask;
    unsigned char chan;
} pp_ipmi_oem_intel_get_tpt_cfg_param_t;

#endif

#ifdef OEM_RACKABLE
typedef struct {
    int slot;
    int active;
    const char *message;
} pp_ipmi_oem_rackable_lcd_set_slot_param_t;

typedef struct {
    int slot;
} pp_ipmi_oem_rackable_lcd_get_slot_param_t;

typedef struct {
    int active;
    char *message;
} pp_ipmi_oem_rackable_lcd_get_slot_result_t;

typedef enum {
    PP_IPMI_OEM_RACKABLE_LCD_DISABLED = 0,
    PP_IPMI_OEM_RACKABLE_LCD_ROTATE = 1,
    PP_IPMI_OEM_RACKABLE_LCD_PAUSE = 2,
    PP_IPMI_OEM_RACKABLE_LCD_IMMEDIATE = 3
} pp_ipmi_oem_rackable_lcd_mode_t;

typedef struct {
    int set_mode;
    pp_ipmi_oem_rackable_lcd_mode_t mode;
    int set_slot;
    int slot;
    int set_interval;
    int interval;
    int set_immediate;
    const char *immediate;
} pp_ipmi_oem_rackable_lcd_set_state_param_t;

typedef struct {
    pp_ipmi_oem_rackable_lcd_mode_t mode;
    int current_slot;
    int active_slots;
    int interval;
    char *immediate;
} pp_ipmi_oem_rackable_lcd_get_state_result_t;
#endif

#ifdef PRODUCT_PDU

typedef struct {
    unsigned char delay;
} pp_ipmi_oem_pp_rpc_set_power_on_delay_param_t;

typedef struct {
    unsigned char receptacle;
    unsigned char status;
} pp_ipmi_oem_pp_rpc_set_receptacle_state_param_t;

typedef struct {
    unsigned char receptacle;
} pp_ipmi_oem_pp_rpc_get_receptacle_state_param_t;

typedef struct {
    unsigned char group;
    unsigned char status;
} pp_ipmi_oem_pp_rpc_set_group_state_param_t;

typedef struct {
    unsigned char group;
    unsigned char enable;
    unsigned char mem[3];
} pp_ipmi_oem_pp_rpc_set_group_membership_param_t;

typedef struct {
    unsigned char group;
} pp_ipmi_oem_pp_rpc_get_group_membership_param_t;

typedef struct {
    unsigned char group;
    unsigned char delay;
} pp_ipmi_oem_pp_rpc_set_group_power_on_delay_param_t;

typedef struct {
    unsigned char group;
} pp_ipmi_oem_pp_rpc_get_group_power_on_delay_param_t;

typedef struct {
    unsigned char receptacle;
    unsigned char num_acl;
    unsigned char acl[256];
} pp_ipmi_oem_pp_rpc_set_receptacle_acl_param_t;

typedef struct {
    unsigned char receptacle;
} pp_ipmi_oem_pp_rpc_get_receptacle_acl_param_t;

#endif/*PRODUCT_PDU*/

typedef struct {
    unsigned short pwm_id;
} pp_ipmi_oem_pp_get_pwm_param_t;

typedef struct {
    unsigned short pwm_id;
    unsigned short duty_cycle;
} pp_ipmi_oem_pp_set_pwm_param_t;

// OEM raritan specific
#define PP_IPMI_OEM_RARITAN_FW_UPGRAGE_PROGRESS_START_FLASHING	-1
typedef void (* pp_ipmi_progress_cb_func)(void * param, int progress /* percentage or special value, see above */);

typedef struct {
    const char *filename;
    int validate_only;
    int cross_oem;
    int cross_hwid;
    int max_chunk_size;
    pp_ipmi_progress_cb_func progress_cb;
    void *progress_cb_param;
} pp_ipmi_oem_raritan_fw_upgrade_param_t;

typedef struct {
    const char *filename;
    int max_chunk_size;
} pp_ipmi_oem_raritan_cfg_backup_param_t;

typedef struct {
    const char *filename;
    int max_chunk_size;
} pp_ipmi_oem_raritan_cfg_restore_param_t;

typedef struct {
    char *serial;
} pp_ipmi_oem_raritan_sn_set_param_t;

// self test specific stuff
typedef struct {
    int channel;
} pp_ipmi_self_test_usb_operation_param_t;

typedef struct {
    int channel;
} pp_ipmi_self_test_nic_param_t;

typedef struct {
    int channel;
    const char *host_to_ping;
} pp_ipmi_self_test_nic_ping_param_t;

/* all parameters in one structure/union */
/* TODO: remove cmdline interface */
typedef struct {
    int			is_cmdline;
    union {
    	char*				cmdline;		/* old style interface */
        unsigned char                   buffer[1];              /* plain byte buffer */
    	pp_ipmi_bmc_reset_param_t	bmc_reset;		/* PP_IPMI_BMC_SUBCMD_RESET */
    	pp_ipmi_bmc_set_enables_param_t	bmc_enables;		/* PP_IPMI_BMC_SUBCMD_SET_ENABLES */
    	pp_ipmi_chassis_power_param_t	chassis_power;		/* PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL */
    	pp_ipmi_chassis_set_pol_param_t	chassis_set_policy;	/* PP_IPMI_CHASSIS_SUBCMD_SET_POLICY */
    	unsigned char			chassis_get_bparam;	/* PP_IPMI_CHASSIS_SUBCMD_GET_BOOTPARAM */
    	pp_ipmi_chassis_bflag_param_t	chassis_set_bootdev;	/* PP_IPMI_CHASSIS_SUBCMD_SET_BOOTDEV */
    	int				chassis_identify;	/* PP_IPMI_CHASSIS_SUBCMD_INDENTIFY */
	pp_ipmi_chan_setaccess_param_t  channel_setaccess;      /* PP_IPMI_CHANNEL_SUBCMD_SET_ACCESS */
        pp_ipmi_usr_summary_param_t     usr_summary;            /* PP_IPMI_USER_SUBCMD_SUMMARY */
	pp_ipmi_usr_info_param_t        usr_info;               /* PP_IPMI_USER_SUBCMD_INFO */
        pp_ipmi_usr_list_param_t        usr_list;               /* PP_IPMI_USER_SUBCMD_LIST */
	pp_ipmi_usr_set_name_param_t    usr_set_name;           /* PP_IPMI_USER_SUBCMD_SET_NAME */
        pp_ipmi_usr_password_param_t    usr_set_pwd;            /* PP_IPMI_USER_SUBCMD_SET_PASSWORD */
        pp_ipmi_usr_password_param_t    usr_test_pwd;           /* PP_IPMI_USER_SUBCMD_TEST_PASSWORD */
	pp_ipmi_usr_uid_param_t         usr_enable;             /* PP_IPMI_USER_SUBCMD_ENABLE */
	pp_ipmi_usr_uid_param_t         usr_disable;            /* PP_IPMI_USER_SUBCMD_DISABLE */
    	vector_t*			sel_delete_list;	/* PP_IPMI_SEL_SUBCMD_DELETE */
    								/* vector of unsigned int, use vector_add/get2! */
    	pp_ipmi_sel_list_param_t	sel_list;		/* PP_IPMI_SEL_SUBCMD_LIST */
    	vector_t*			sel_get_list;		/* PP_IPMI_SEL_SUBCMD_GET */
    								/* vector of unsigned int, use vector_add/get2! */
    	time_t				sel_set_time;		/* PP_IPMI_SEL_SUBCMD_SET_TIME */
    	pp_ipmi_sdr_list_param_t	sdr_list;		/* PP_IPMI_SDR_SUBCMD_LIST */
	size_t				sdr_idx;		/* PP_IPMI_SDR_SUBCMD_UPDATE */
    	vector_t*			sensor_get_by_name_list;	/* PP_IPMI_SENSOR_SUBCMD_GET_BY_NAME */
    								/* vector of char* */
    	vector_t*			sensor_get_by_id_list;	/* PP_IPMI_SENSOR_SUBCMD_GET_BY_ID */
    								/* vector of unsigned int, use vector_add/get2! */
    	vector_t*			sensor_threshold_list;	/* PP_IPMI_SENSOR_SUBCMD_THRESHOLD */
    								/* vector of pp_ipmi_sensor_set_th_param_t, use vector_add/get2! */
    	vector_t*			sensor_raw_threshold_list;	/* PP_IPMI_SENSOR_SUBCMD_SET_RAW_TH */
    								/* vector of pp_ipmi_sensor_set_raw_th_param_t, use vector_add/get2! */
	pp_ipmi_sensor_rearm_param_t    rearm_sensor_events;    /* PP_IPMI_SENSOR_SUBCMD_REARM_EVENTS */
        pp_ipmi_lanp_set_dest_param_t   lanp_set_dest;          /* PP_IPMI_LANP_SUBCMD_SET_DEST */
	pp_ipmi_lanp_get_param_t	lanp_get;		/* PP_IPMI_LANP_SUBCMD_GET */
	pp_ipmi_lanp_set_ip_source_t	lanp_set_ip_source;	/* PP_IPMI_LANP_SUBCMD_SET_IP_SOURCE */
	pp_ipmi_lanp_set_string_t	lanp_set_snmp_string;	/* PP_IPMI_LANP_SUBCMD_SET_SNMP_STRING */
	pp_ipmi_lanp_set_string_t	lanp_set_ip_address;	/* PP_IPMI_LANP_SUBCMD_SET_IP_ADDRESS */
	pp_ipmi_lanp_set_string_t	lanp_set_netmask;	/* PP_IPMI_LANP_SUBCMD_SET_NETMASK */
	pp_ipmi_lanp_set_string_t	lanp_set_mac_address;	/* PP_IPMI_LANP_SUBCMD_SET_MAC_ADDRESS */
	pp_ipmi_lanp_set_string_t	lanp_set_def_gw_ip;	/* PP_IPMI_LANP_SUBCMD_SET_DEF_GW_IP */
	pp_ipmi_lanp_set_string_t	lanp_set_def_gw_mac;	/* PP_IPMI_LANP_SUBCMD_SET_DEF_GW_MAC */
	pp_ipmi_lanp_set_string_t	lanp_set_bak_gw_ip;	/* PP_IPMI_LANP_SUBCMD_SET_BAK_GW_IP */
	pp_ipmi_lanp_set_string_t	lanp_set_bak_gw_mac;	/* PP_IPMI_LANP_SUBCMD_SET_BAK_GW_MAC */
        pp_ipmi_pef_filter_t            pef_filter_set;         /* PP_IPMI_PEF_FILTER_SET */
        pp_ipmi_pef_policy_t            pef_policy_set;         /* PP_IPMI_PEF_POLICY_SET */
        pp_ipmi_lanp_dest_t             lan_destination_set;    /* PP_IPMI_LAN_DESTINATION_SET */
	pp_ipmi_raw_param_t		raw_data;		/* PP_IPMI_RAW_SUBCMD_RAW */

#ifdef PRODUCT_AMDDC
	pp_ipmi_oem_opma_set_local_lockout_param_t  oem_opma_set_local_lockout;
#endif
		
#ifdef PP_FEAT_ESB2_TPT
        pp_ipmi_oem_intel_set_usr_cfg_param_t       oem_intel_set_usr_cfg;
        pp_ipmi_oem_intel_get_usr_cfg_param_t       oem_intel_get_usr_cfg;
        pp_ipmi_oem_intel_get_tpt_status_param_t    oem_intel_get_tpt_status;
        pp_ipmi_oem_intel_tpt_listen_param_t        oem_intel_tpt_listen;
        pp_ipmi_oem_intel_set_tpt_cfg_param_t       oem_intel_set_tpt_cfg;
        pp_ipmi_oem_intel_get_tpt_cfg_param_t       oem_intel_get_tpt_cfg;
#endif

#ifdef OEM_RACKABLE
	pp_ipmi_oem_rackable_lcd_set_slot_param_t             oem_rackable_lcd_set_slot;
	pp_ipmi_oem_rackable_lcd_get_slot_param_t             oem_rackable_lcd_get_slot;
	pp_ipmi_oem_rackable_lcd_set_state_param_t            oem_rackable_lcd_set_state;
#endif

#ifdef PRODUCT_PDU
	pp_ipmi_oem_pp_rpc_set_power_on_delay_param_t         oem_pp_rpc_set_power_on_delay;
	pp_ipmi_oem_pp_rpc_set_receptacle_state_param_t       oem_pp_rpc_set_recep_state; 
	pp_ipmi_oem_pp_rpc_get_receptacle_state_param_t       oem_pp_rpc_get_recep_state;
	pp_ipmi_oem_pp_rpc_set_group_state_param_t            oem_pp_rpc_set_group_state;
	pp_ipmi_oem_pp_rpc_set_group_membership_param_t       oem_pp_rpc_set_group_membership;
	pp_ipmi_oem_pp_rpc_get_group_membership_param_t       oem_pp_rpc_get_group_membership;
	pp_ipmi_oem_pp_rpc_set_group_power_on_delay_param_t   oem_pp_rpc_set_group_power_on_delay;
	pp_ipmi_oem_pp_rpc_get_group_power_on_delay_param_t   oem_pp_rpc_get_group_power_on_delay;
	pp_ipmi_oem_pp_rpc_set_receptacle_acl_param_t  	      oem_pp_rpc_set_recep_acl;
	pp_ipmi_oem_pp_rpc_get_receptacle_acl_param_t         oem_pp_rpc_get_recep_acl;
#endif/*PRODUCT_PDU*/

	pp_ipmi_oem_pp_get_pwm_param_t              oem_pp_get_pwm; /* PP_IPMI_OEM_PP_GET_PWM */
	pp_ipmi_oem_pp_set_pwm_param_t              oem_pp_set_pwm; /* PP_IPMI_OEM_PP_SET_PWM */

	pp_ipmi_oem_raritan_fw_upgrade_param_t	    oem_raritan_fw_upgrade; /* PP_IPMI_OEM_RARITAN_FW_UPGRADE */
	pp_ipmi_oem_raritan_cfg_backup_param_t	    oem_raritan_cfg_backup; /* PP_IPMI_OEM_RARITAN_CFG_BACKUP */
	pp_ipmi_oem_raritan_cfg_restore_param_t	    oem_raritan_cfg_restore; /* PP_IPMI_OEM_RARITAN_CFG_RESTORE */
	pp_ipmi_oem_raritan_sn_set_param_t	    oem_raritan_sn_set;	    /* PP_IPMI_OEM_RARITAN_SN_SET */
	unsigned char	    oem_raritan_get_ext_user_params;	/* user ID */ /* PP_IPMI_OEM_RARITAN_GET_EXT_USER_PARAMS */

	pp_ipmi_self_test_usb_operation_param_t	    self_test_usb;	/* PP_IPMI_SELF_TEST_SUBCMD_USB_OPERATION */
	pp_ipmi_self_test_nic_param_t		    self_test_nic_status;	/* PP_IPMI_SELF_TEST_SUBCMD_NIC_STATUS */ 
	pp_ipmi_self_test_nic_param_t		    self_test_nic_loopback;	/* PP_IPMI_SELF_TEST_SUBCMD_NIC_LOOPBACK */ 
	pp_ipmi_self_test_nic_ping_param_t	    self_test_nic_ping;		/* PP_IPMI_SELF_TEST_SUBCMD_NIC_PING */ 
	pp_ipmi_self_test_nic_param_t		    self_test_nic_bcast_ping;	/* PP_IPMI_SELF_TEST_SUBCMD_NIC_BROADCAST_PING */
    } data;
} pp_ipmi_parameter_t;

/* ------------------------------------------------------------------------- */

/* All ipmi commands used have a return structure which differs from each
   command. All possible structures are defined below. */

/* a time type; we hold here both a string and a timestamp, in case we need both */
typedef struct {
    int			is_valid;
    time_t		timestamp;
    int			is_pre_init;
    int			is_unspecified;
    pp_strstream_t	string_time;
    pp_strstream_t	string_date;
} pp_ipmi_time_t;

/* bmc information type */
typedef struct {
    int			device_id;
    int			device_revision;
    pp_strstream_t	firmware_revision;
    pp_strstream_t	ipmi_version;
    unsigned long	manufacturer_id;
    unsigned long	product_id;
    int			device_available;
    int			provides_sdrs;
    vector_t*		additional_device_support;	/* a vector of char* */
    unsigned char	aux_fw_info[4];
} pp_ipmi_bmc_info_t;

/* bmc get enables */
typedef struct {
    int			receive_msg_int_enabled;
    int			event_msg_buf_int_enabled;
    int			event_msg_buf_enabled;
    int			sel_enabled;
    int			oem0_enabled;
    int			oem1_enabled;
    int			oem2_enabled;
} pp_ipmi_bmc_get_enables_t;

/* bmc guid */
typedef struct {
    unsigned char guid[16];
} pp_ipmi_bmc_guid_t;

/* bmc get acpi state */
typedef enum {
    PP_IPMI_ACPI_SYS_STATE_S0 = 0,
    PP_IPMI_ACPI_SYS_STATE_S1,
    PP_IPMI_ACPI_SYS_STATE_S2,
    PP_IPMI_ACPI_SYS_STATE_S3,
    PP_IPMI_ACPI_SYS_STATE_S4,
    PP_IPMI_ACPI_SYS_STATE_S5,
    PP_IPMI_ACPI_SYS_STATE_S45,
    PP_IPMI_ACPI_SYS_STATE_MECHOFF,
    PP_IPMI_ACPI_SYS_STATE_S123,
    PP_IPMI_ACPI_SYS_STATE_S1234,
    PP_IPMI_ACPI_SYS_STATE_S5OVR,
    PP_IPMI_ACPI_SYS_STATE_LEGON = 0x20,
    PP_IPMI_ACPI_SYS_STATE_LEGOFF = 0x21,
    PP_IPMI_ACPI_SYS_STATE_UNKNOWN = 0x2A,
} pp_ipmi_bmc_acpi_sys_state_t;

typedef enum {
    PP_IPMI_ACPI_DEV_STATE_D0 = 0,
    PP_IPMI_ACPI_DEV_STATE_D1,
    PP_IPMI_ACPI_DEV_STATE_D2,
    PP_IPMI_ACPI_DEV_STATE_D3,
    PP_IPMI_ACPI_DEV_STATE_UNKNOWN = 0x2A,
} pp_ipmi_bmc_acpi_dev_state_t;

typedef struct {
    pp_ipmi_bmc_acpi_sys_state_t sys_state;
    pp_ipmi_bmc_acpi_dev_state_t dev_state;
    pp_strstream_t	sys_state_string;
    pp_strstream_t	dev_state_string;
} pp_ipmi_bmc_acpi_state_t;

/* chassis status */
typedef enum {
    PP_IPMI_CHASSIS_POWER_STATUS_UNKNOWN = 0,
    PP_IPMI_CHASSIS_POWER_STATUS_ON,
    PP_IPMI_CHASSIS_POWER_STATUS_OFF,
} pp_ipmi_power_status_t;

typedef struct {
    pp_ipmi_power_status_t	power_status;
    pp_strstream_t	power_status_string;
    int			power_overload;
    int			power_interlock_active;
    int			main_power_fault;
    int			power_control_fault;
    struct {
    	pp_ipmi_chassis_restart_policy_t	policy;
    	pp_strstream_t	string;
    } power_restore_policy;
    pp_strstream_t	last_power_event;
    int			chassis_intrusion_active;
    int			front_panel_lockout_active;
    int			drive_fault;
    int			cooling_fan_fault;
    int			front_panel_control_info_valid;
    /* only valid when front_panel_control_info_valid is true */
    int			front_panel_control;
    int			sleep_allowed;
    int			diag_allowed;
    int			reset_allowed;
    int			power_allowed;
    int			sleep_disabled;
    int			diag_disabled;
    int			reset_disabled;
    int			power_disabled;
} pp_ipmi_chassis_status_t;

/* chassis power status */
typedef struct {
    pp_ipmi_power_status_t	status;
    pp_strstream_t		status_string;
} pp_ipmi_chassis_power_status_t;

/* chassis supported power-fault policies */
typedef struct {
    int			always_on_supported;
    int			always_off_supported;
    int			previous_supported;
} pp_ipmi_chassis_get_policies_t;

/* chassis bootparam */
typedef struct {
    unsigned int	version;
    unsigned int	boot_param;
    int			invalid_or_locked;
    int			data_size;
    char *		data;
} pp_ipmi_chassis_get_bparam_t;

/* chassis bootflag */
typedef struct {
    int				valid;
    pp_ipmi_chassis_bootflag_t	flag;
} pp_ipmi_chassis_get_bflag_return_t;

typedef enum {
    PP_IPMI_USR_ACCESS_IPMI_MESSAGING = 1 << 1,
    PP_IPMI_USR_ACCESS_LINK_AUTH      = 1 << 2,
    PP_IPMI_USR_ACCESS_CALL_IN        = 1 << 3,
} pp_ipmi_usr_access_t;

/* user summary */
typedef struct {
    int max_users;
    int enabled_users;
    int fixed_names;
} pp_ipmi_usr_summary_t;

/* user list */
typedef struct {  /* limits by IPMI spec */
    pp_ipmi_priv_levels priv_limit;
    pp_ipmi_usr_access_t access;
    char name[17];
} pp_ipmi_usr_info_t;

/* user list */
typedef struct {  /* limits by IPMI spec */
    pp_ipmi_priv_levels priv_limit[64];
    pp_ipmi_usr_access_t access[64];
    char name[64][17];
} pp_ipmi_usr_list_t;

/* allocation info */
typedef struct {
    struct {
    	int		specified;
    	/* only valid if specified is set */
    	int		units;
    } alloc_units;
    struct {
    	int		specified;
    	/* only valid if specified is set */
    	int		unit_size;
    } alloc_unit_size;
    int			free_units;
    int			largest_free_block;
    int			max_record_size;
} pp_ipmi_allocation_info_t;

/* sel information */
typedef struct {
    int			version;
    int			entries;
    int			free_space;
    int			percent_used;
    pp_ipmi_time_t	last_add_time;
    pp_ipmi_time_t	last_del_time;
    int			overflow;
    int			delete_supported;
    int			partial_add_supported;
    int			reserve_supported;
    int			get_alloc_info_supported;
    /* only valid when get_alloc_info_supported is set */
    pp_ipmi_allocation_info_t	allocation_info;
} pp_ipmi_sel_info_t;

/* sensor reading/tolerance type */
typedef struct {
    int valid;
    float value;
    int cim_value;
} pp_ipmi_sensor_value_t;

typedef enum {
    PP_IPMI_EVENT_TYPE_UNSPECIFIED,
    PP_IPMI_EVENT_TYPE_THRESHOLD,
    PP_IPMI_EVENT_TYPE_GENERIC_DISCRETE,
    PP_IPMI_EVENT_TYPE_SENSOR_SPECIFIC_DISCRETE,
    PP_IPMI_EVENT_TYPE_OEM,
    PP_IPMI_EVENT_TYPE_RESERVED,
} pp_ipmi_sel_event_type_t;

typedef enum {
    PP_IPMI_EVENT_DIRECTION_ASSERTION,
    PP_IPMI_EVENT_DIRECTION_DEASSERTION
} pp_ipmi_sel_event_direction_t;

/* one sel entry; the sel table consists of a vector of entries */
typedef struct {
    unsigned int	record_id;
    unsigned int	record_type;
    pp_strstream_t	record_type_string;
    pp_ipmi_time_t	timestamp;
    int			further_data_is_valid;
    unsigned int	generator_id;
    unsigned int	evm_revision;
    pp_strstream_t	sensor_type;
    unsigned char	sensor_type_code;
    unsigned int	sensor_num;
    pp_strstream_t	sensor_name;
    pp_strstream_t	event_type;
    pp_ipmi_sel_event_type_t	event_type_code;
    pp_strstream_t	event_direction;
    pp_ipmi_sel_event_direction_t	event_direction_code;
    unsigned char	event_data[3];
    pp_strstream_t	description;
    /* sensor threshold assertion event (event_type_code == PP_IPMI_EVENT_TYPE_THRESHOLD) */
    pp_ipmi_sensor_value_t	trigger_reading;
    pp_ipmi_sensor_value_t	threshold_reading;
    int			above_threshold;
    pp_strstream_t	unit;
} pp_ipmi_sel_list_entry_t;

#define PP_IPMI_FRU_INTERNAL_INFO   (1 << 0)
#define PP_IPMI_FRU_CHASSIS_INFO    (1 << 1)
#define PP_IPMI_FRU_BOARD_INFO      (1 << 2)
#define PP_IPMI_FRU_PRODUCT_INFO    (1 << 3)
#define PP_IPMI_FRU_MULTI_INFO      (1 << 4)

/* fru information */
typedef struct {
    int ver;
    int info_mask; /* PP_IPMI_FRU_XXXX */
    /* TODO: add internal use area here */
    struct {
        int             ver;
        pp_strstream_t  type;
        pp_strstream_t  model;
        pp_strstream_t  serial;
    } chassis;
    struct {
        int             ver;
        int             lang;
        pp_ipmi_time_t  mfg_time;
        pp_strstream_t  manufacturer;
        pp_strstream_t  name;
        pp_strstream_t  serial;
        pp_strstream_t  model;
        pp_strstream_t  fru_id;
    } board;
    struct {
        int             ver;
        int             lang;
        pp_strstream_t  manufacturer;
        pp_strstream_t  name;
        pp_strstream_t  model;
        pp_strstream_t  version;
        pp_strstream_t  serial;
        pp_strstream_t  asset;
        pp_strstream_t  fru_id;
    } product;
    /* TODO: add multi record area here */
} pp_ipmi_fru_info_t;

/* sdr information */
typedef struct {
    int			version;
    int			record_count;
    struct {
    	int		space;
    	pp_strstream_t	string;
    } free_space;
    pp_ipmi_time_t	most_resent_addition;
    pp_ipmi_time_t	most_resent_erase;
    int			overflow;
    pp_strstream_t	modal_update_support;
    int			delete_supported;
    int			partial_add_supported;
    int			reserve_supported;
    int			get_alloc_info_supported;
    /* only valid when get_alloc_info_supported is set */
    pp_ipmi_allocation_info_t	allocation_info;
} pp_ipmi_sdr_info_t;

/* enum of sensor types */
typedef enum {
    PP_IPMI_SENSOR_TYPE_FULL,
    PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR,
    PP_IPMI_SENSOR_TYPE_EVENTONLY,
    PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR,
    PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR,
    PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR,
    PP_IPMI_SENSOR_TYPE_OTHER,
} pp_ipmi_sensor_type_t;

/* more specific sensor types */
typedef enum {
    PP_IPMI_SPEC_SENSOR_TYPE_UNKNOWN = 0,
    PP_IPMI_SPEC_SENSOR_TYPE_RESERVED,
    PP_IPMI_SPEC_SENSOR_TYPE_OEM_RESERVED,
    PP_IPMI_SPEC_SENSOR_TYPE_TEMPERATURE,
    PP_IPMI_SPEC_SENSOR_TYPE_VOLTAGE,
    PP_IPMI_SPEC_SENSOR_TYPE_CURRENT,
    PP_IPMI_SPEC_SENSOR_TYPE_FAN,
    PP_IPMI_SPEC_SENSOR_TYPE_PHYS_SECURITY,
    PP_IPMI_SPEC_SENSOR_TYPE_VIOL_ATTEMPT,
    PP_IPMI_SPEC_SENSOR_TYPE_PROCESSOR,
    PP_IPMI_SPEC_SENSOR_TYPE_POWER_SUPPLY,
    PP_IPMI_SPEC_SENSOR_TYPE_POWER_UNIT,
    PP_IPMI_SPEC_SENSOR_TYPE_COOLING_DEVICE,
    PP_IPMI_SPEC_SENSOR_TYPE_OTHER,
    PP_IPMI_SPEC_SENSOR_TYPE_MEMORY,
    PP_IPMI_SPEC_SENSOR_TYPE_DRIVE_SLOT,
    PP_IPMI_SPEC_SENSOR_TYPE_POST,
    PP_IPMI_SPEC_SENSOR_TYPE_FIRMWARE,
    PP_IPMI_SPEC_SENSOR_TYPE_EVENT_LOG_DISABLED,
    PP_IPMI_SPEC_SENSOR_TYPE_WATCHDOG,
    PP_IPMI_SPEC_SENSOR_TYPE_SYSEVENT,
    PP_IPMI_SPEC_SENSOR_TYPE_CRIT_INT,
    PP_IPMI_SPEC_SENSOR_TYPE_BUTTON,
    PP_IPMI_SPEC_SENSOR_TYPE_MODULE,
    PP_IPMI_SPEC_SENSOR_TYPE_MICROCONTROLLER,
    PP_IPMI_SPEC_SENSOR_TYPE_ADD_IN,
    PP_IPMI_SPEC_SENSOR_TYPE_CHASSIS,
    PP_IPMI_SPEC_SENSOR_TYPE_CHIPSET,
    PP_IPMI_SPEC_SENSOR_TYPE_OTHER_FRU,
    PP_IPMI_SPEC_SENSOR_TYPE_CABLE,
    PP_IPMI_SPEC_SENSOR_TYPE_TERMINATOR,
    PP_IPMI_SPEC_SENSOR_TYPE_BOOT_INITIATED,
    PP_IPMI_SPEC_SENSOR_TYPE_BOOT_ERROR,
    PP_IPMI_SPEC_SENSOR_TYPE_OS_BOOT,
    PP_IPMI_SPEC_SENSOR_TYPE_OS_STOP,
    PP_IPMI_SPEC_SENSOR_TYPE_SLOT,
    PP_IPMI_SPEC_SENSOR_TYPE_ACPI_STATE,
    PP_IPMI_SPEC_SENSOR_TYPE_PLATFORM_ALERT,
    PP_IPMI_SPEC_SENSOR_TYPE_ENTITY,
    PP_IPMI_SPEC_SENSOR_TYPE_MONITOR_IC,
    PP_IPMI_SPEC_SENSOR_TYPE_LAN,
    PP_IPMI_SPEC_SENSOR_TYPE_MANAGEMENT_HEALTH,
    PP_IPMI_SPEC_SENSOR_TYPE_BATTERY,
    PP_IPMI_SPEC_SENSOR_TYPE_FRU_DEVICE_LOCATOR,
    PP_IPMI_SPEC_SENSOR_TYPE_MC_DEVICE_LOCATOR,
    PP_IPMI_SPEC_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR,
} pp_ipmi_specific_sensor_type_t;

typedef enum {
    PP_IPMI_SENSOR_STATUS_NO_READING = 0,
    PP_IPMI_SENSOR_STATUS_OK,
    PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL,
    PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL,
    PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL,
    PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL,
    PP_IPMI_SENSOR_STATUS_LOWER_NON_RECOVERABLE,
    PP_IPMI_SENSOR_STATUS_UPPER_NON_RECOVERABLE,
    PP_IPMI_SENSOR_STATUS_ASSERTED,		/* for discrete sensors */
} pp_ipmi_sensor_status_t;

typedef enum {
    PP_IPMI_SENSOR_FULL_TYPE_ANALOG,
    PP_IPMI_SENSOR_FULL_TYPE_DISCRETE,
} pp_ipmi_full_sensor_type_t;

typedef enum {
    PP_IPMI_SENSOR_READING_OK,
    PP_IPMI_SENSOR_READING_ERROR,
    PP_IPMI_SENSOR_READING_SCANNING_DISABLED,
    PP_IPMI_SENSOR_READING_NOT_PRESENT,
    PP_IPMI_SENSOR_READING_NOT_LINEAR,
} pp_ipmi_full_sensor_reading_present_t;

typedef enum {
    PP_IPMI_SENSOR_EVT_CTRL_PRE_TRESHOLD,
    PP_IPMI_SENSOR_EVT_CTRL_ENTIRE,
    PP_IPMI_SENSOR_EVT_CTRL_GLOBAL,
    PP_IPMI_SENSOR_EVT_CTRL_NONE,
} pp_ipmi_full_sensor_evt_control_t;

typedef struct {
    int upper_non_recoverable;
    int upper_critical;
    int upper_non_critical;
    int lower_non_recoverable;
    int lower_critical;
    int lower_non_critical;
} pp_ipmi_full_sensor_threshold_boolean_t;

/* one sdr entry; the sdr table consists of a vector of entries */
typedef struct {
    pp_ipmi_sensor_type_t	type;
    pp_ipmi_specific_sensor_type_t	spec_type;
    int			sdr_id;
    pp_strstream_t	sdr_name;
    struct {
    	int		is_valid;
    	unsigned int	id;
    	unsigned int	instance;
    	pp_strstream_t	string;
    } entity;
    pp_strstream_t	sdr_type_main;	/* Analog, discrete, compact, event, fru locator, mc locator, oem */
    pp_strstream_t	sdr_type_sub;	/* voltage, fan, ... */
    uint8_t		raw[64];
    union {
    	struct {
    	    pp_ipmi_full_sensor_type_t type;
    	    pp_ipmi_full_sensor_reading_present_t reading_present;
    	    pp_ipmi_sensor_value_t	reading;
    	    pp_ipmi_sensor_status_t	status;
    	    pp_strstream_t	status_string;
    	    int			owner_id;
    	    int			owner_lun;
    	    pp_strstream_t		assertion_events;
    	    pp_strstream_t		deassertion_events;
    	    pp_strstream_t		assertion_event_mask;
    	    pp_strstream_t		deassertion_event_mask;
    	    pp_strstream_t		assertions_enabled;
    	    pp_strstream_t		deassertions_enabled;
    	    /* low-level data for CIM */
    	    int			cim_base_unit;
    	    int			cim_unit_modifier;
    	    int			cim_rate_units;
    	    pp_ipmi_full_sensor_evt_control_t	evt_msg_control;
    	    pp_strstream_t		evt_msg_control_string;
    	    /* only valid for analog sensors (PP_IPMI_SENSOR_FULL_TYPE_ANALOG) */
    	    pp_ipmi_sensor_value_t	tolerance;
    	    pp_strstream_t	unit;
    	    pp_ipmi_sensor_value_t	nominal_reading;
    	    pp_ipmi_sensor_value_t	normal_minimum;
    	    pp_ipmi_sensor_value_t	normal_maximum;
    	    pp_ipmi_sensor_value_t	upper_non_recoverable;
    	    pp_ipmi_sensor_value_t	upper_critical;
    	    pp_ipmi_sensor_value_t	upper_non_critical;
    	    pp_ipmi_sensor_value_t	lower_non_recoverable;
    	    pp_ipmi_sensor_value_t	lower_critical;
    	    pp_ipmi_sensor_value_t	lower_non_critical;
    	    pp_ipmi_sensor_value_t	minimum_sensor_range;
    	    pp_ipmi_sensor_value_t	maximum_sensor_range;
    	    pp_ipmi_sensor_value_t	hysteresis_pos;
    	    pp_ipmi_sensor_value_t	hysteresis_neg;
    	    pp_ipmi_full_sensor_threshold_boolean_t	readable_thresholds;
    	    pp_ipmi_full_sensor_threshold_boolean_t	settable_thresholds;
    	    pp_ipmi_full_sensor_threshold_boolean_t	threshold_read_mask;
    	    pp_strstream_t	cim_current_state;
    	    vector_t		cim_possible_states;
            /* sensor value conversion */
            int                 m;
            int                 b;
            int                 b_exp;
            int                 r_exp;
            uint8_t             analog;
            uint8_t             linearization;
    	} full;
    	struct {
    	    pp_ipmi_sensor_status_t status;
    	    int			owner_id;
    	    int			owner_lun;
    	    int			scanning_disabled;
    	    unsigned int	event_type;
    	    pp_strstream_t	description;
    	    pp_strstream_t	assertion_events;
    	    pp_strstream_t	deassertion_events;
    	    pp_strstream_t	assertion_event_mask;
    	    pp_strstream_t	deassertion_event_mask;
    	    pp_strstream_t	assertions_enabled;
    	    pp_strstream_t	deassertions_enabled;
    	    pp_strstream_t	cim_current_state;
    	    vector_t		cim_possible_states;
    	} compact;
    	struct {
    	    int			owner_id;
    	    int			owner_lun;
    	    unsigned int	event_type;
    	} event_only;
    	struct {
    	    unsigned int	dev_slave_addr;
    	    unsigned int	channel_num;
    	    int			acpi_system_ps_notif_required;
    	    int			acpi_device_ps_notif_required;
    	    int			controler_presence_static;	/* dynamic otherwise */
    	    int			logs_init_agent_errors;
    	    pp_strstream_t	event_message_gen;
    	    int			cap_chassis_device;
    	    int			cap_bridge;
    	    int			cap_ipmb_event_generator;
    	    int			cap_ipmi_event_receiver;
    	    int			cap_fru_inventory_support;
    	    int			cap_sel_device;
    	    int			cap_sdr_repository;
    	    int			cap_sensor_device;
    	} mc_locator;
    	struct {
    	    unsigned int	dev_slave_addr;
    	    unsigned int	dev_id;
    	    pp_strstream_t	dev_id_description;
    	    unsigned int	lun;
    	    unsigned int	bus;
    	    unsigned int	channel_num;
    	    unsigned int	dev_type;
    	    unsigned int	dev_type_modifier;
    	    pp_strstream_t	dev_type_string;
    	} fru_locator;
    	struct {
    	    unsigned int	dev_access_addr;
    	    unsigned int	dev_slave_addr;
    	    unsigned int	address_span;
    	    unsigned int	lun;
    	    unsigned int	bus;
    	    unsigned int	channel_num;
    	    unsigned int	dev_type;
    	    unsigned int	dev_type_modifier;
    	    pp_strstream_t	dev_type_string;
    	} generic_locator;
    } data;
} pp_ipmi_sdr_list_entry_t;

typedef struct {
    int                                    id; // the sensor id
    pp_ipmi_full_sensor_reading_present_t  reading_present; // reading present?
    unsigned char                          raw_value; // raw sensor reading
    union {
        unsigned char	                    data; // threshold comparison data
        struct {
            BITFIELD7(unsigned char,        below_non_critical:1,
                                            below_critical:1,
                                            below_non_recoverable:1,
                                            above_non_critical:1,
                                            above_critical:1,
                                            above_non_recoverable:1,
                                            reserved:2);
        } bits;
    } threshold;
} pp_ipmi_raw_reading_list_entry_t;

typedef struct {
    int                                    id; // the sensor id
    union {
        unsigned char	                    data;
        struct {
            BITFIELD7(unsigned char,        lower_non_critical:1,
                                            lower_critical:1,
                                            lower_non_recoverable:1,
                                            upper_non_critical:1,
                                            upper_critical:1,
                                            upper_non_recoverable:1,
                                            reserved:2);
        } bits;
    } threshold_present;
    unsigned char                          lower_non_critical;
    unsigned char                          lower_critical;
    unsigned char                          lower_non_recoverable;
    unsigned char                          upper_non_critical;
    unsigned char                          upper_critical;
    unsigned char                          upper_non_recoverable;
} pp_ipmi_raw_thresholds_list_entry_t;

typedef pp_ipmi_lanp_dest_t pp_ipmi_lanp_dest_list_entry_t;

/* lan parameters */
typedef enum {
    PP_IPMI_LANP_SET_COMPLETE,
    PP_IPMI_LANP_SET_IN_PROGRESS,
    PP_IPMI_LANP_SET_COMMIT_WRITE,
    PP_IPMI_LANP_SET_RESERVED,
    PP_IPMI_LANP_SET_UNKNOWN
} pp_ipmi_lanp_set_in_progress_t;

typedef struct {
    int none;
    int md2;
    int md5;
    int password;
    int oem;
} pp_ipmi_lanp_auth_types;

typedef struct {
    int valid;
    pp_strstream_t address;
} pp_ipmi_lanp_ip_or_mac_address;

typedef struct {
    struct {
	int valid;
	pp_ipmi_lanp_set_in_progress_t val;
	const char * string;
    } set_in_progress;
    struct {
	int valid;
	pp_ipmi_lanp_auth_types auth_types;
    } auth_type_support;
    struct {
	int valid;
	pp_ipmi_lanp_auth_types callback;
	pp_ipmi_lanp_auth_types user;
	pp_ipmi_lanp_auth_types _operator;
	pp_ipmi_lanp_auth_types admin;
	pp_ipmi_lanp_auth_types oem;
    } auth_type_enables;
    struct {
	int valid;
	pp_ipmi_lanp_ip_addr_source val;
	const char *string;
    } ip_addr_source;
    pp_ipmi_lanp_ip_or_mac_address ip_address;
    pp_ipmi_lanp_ip_or_mac_address subnet_mask;
    pp_ipmi_lanp_ip_or_mac_address mac_address;
    struct {
	int valid;
	pp_strstream_t string;
    } snmp_string;
    struct {
	int valid;
	unsigned ttl;
	unsigned flags;
	unsigned precedence;
	unsigned tos;
    } ip_header;
    struct {
	int valid;
	int arp_responses_enabled;
	int gratuitous_arp_enabled;
    } arp_control;
    struct {
	int valid;
	float timeout;
    } gratuitous_arp_timeout;
    pp_ipmi_lanp_ip_or_mac_address def_gw_ip;
    pp_ipmi_lanp_ip_or_mac_address def_gw_mac;
    pp_ipmi_lanp_ip_or_mac_address bak_gw_ip;
    pp_ipmi_lanp_ip_or_mac_address bak_gw_mac;
    struct {
	int valid;
	vector_t *ciphers;  // vector containing int elements
    } rcmp_cipher_suites;
    struct {
	int valid;
	pp_ipmi_priv_levels levels[15];
    } cipher_priv_max;
} pp_ipmi_lanp_get_t;

/* raw data */
/* NOTE: if a completion code != 0 arrises, the IPMI command function does
         not return -1 in case of raw data. Instead, the completion_code
	 member in the return structure is set and 0 is returned. In case
	 of an error, -1 is returned, though. */
typedef struct {
    unsigned char completion_code;
    vector_t *data;	/* vector of unsigned int */
} pp_ipmi_raw_t;

#ifdef PRODUCT_AMDDC

typedef struct {
    unsigned char locked;
} pp_ipmi_oem_opma_get_local_lockout_result_t;

#endif /* PRODUCT_AMDDC */

#ifdef PP_FEAT_ESB2_TPT

typedef struct {
    unsigned char revision;
    pp_ipmi_oem_intel_usr_cfg_mask_t mask;
    pp_ipmi_oem_intel_usr_cfg_t cfg;
} pp_ipmi_oem_intel_get_usr_cfg_result_t;

typedef struct {
    unsigned char status;
    unsigned short port;
    unsigned char remote_ip[4];
} pp_ipmi_oem_intel_get_tpt_status_result_t;

typedef struct {
    pp_ipmi_oem_intel_tpt_cfg_mask_t mask;
    pp_ipmi_oem_intel_tpt_cfg_t cfg;
} pp_ipmi_oem_intel_get_tpt_cfg_result_t;

#endif /* PP_FEAT_ESB2_TPT */

#ifdef PRODUCT_PDU

typedef struct {
    unsigned char delay;
} pp_ipmi_oem_pp_rpc_get_power_on_delay_result_t;

typedef struct {
    unsigned char status;
    unsigned char soft_breaker;
    unsigned char red;
    unsigned char green;
    unsigned char blinking;
} pp_ipmi_oem_pp_rpc_get_receptacle_state_result_t;

typedef struct {
    unsigned char enable;
    unsigned char mem[3];
} pp_ipmi_oem_pp_rpc_get_group_membership_result_t;

typedef struct {
    unsigned char delay;
} pp_ipmi_oem_pp_rpc_get_group_power_on_delay_result_t;

typedef struct {
    unsigned char num_acl;
    unsigned char acl[256];
} pp_ipmi_oem_pp_rpc_get_recep_acl_result_t;

#endif /* PRODUCT_PDU */

typedef struct {
    unsigned short duty_cycle;
} pp_ipmi_oem_pp_get_pwm_result_t;

typedef struct {
    pp_strstream_t version;
    u_int32_t build_no;
    u_int32_t hw_id;
    pp_strstream_t tag;
    pp_strstream_t oem;
} pp_ipmi_oem_raritan_fw_version_t;

typedef struct {
    int dev_uid;	// -1 if no internal UID is set
} pp_ipmi_oem_raritan_get_ext_user_params_t;

// note: user 0 is not valid!
typedef pp_ipmi_oem_raritan_get_ext_user_params_t pp_ipmi_oem_raritan_list_ext_user_params_t[64];

/* self test results */
typedef struct {
    unsigned char ddc_data[128];
} pp_ipmi_self_test_ddc_t;

typedef struct {
    unsigned int x_res;
    unsigned int y_res;
    unsigned int hor_line_freq;
    unsigned int vert_frame_freq;
} pp_ipmi_self_test_video_status_t;

typedef struct {
    uint32_t image_crc;
} pp_ipmi_self_test_video_crc_t;

typedef struct {
    int have_link;	/* 1 - yes, 0 - no */
    enum {
	PP_IPMI_NIC_STATUS_DUPLEX_UNKNOWN,
	PP_IPMI_NIC_STATUS_DUPLEX_HALF,
	PP_IPMI_NIC_STATUS_DUPLEX_FULL
    } duplex;
    enum {
	PP_IPMI_NIC_STATUS_SPEED_UNKNOWN,
	PP_IPMI_NIC_STATUS_SPEED_10_MBIT,
	PP_IPMI_NIC_STATUS_SPEED_100_MBIT,
	PP_IPMI_NIC_STATUS_SPEED_1_GBIT,
	PP_IPMI_NIC_STATUS_SPEED_10_GBIT
    } speed;
} pp_ipmi_self_test_nic_status_t;

typedef struct {
    int responding_hosts;
} pp_ipmi_self_test_nic_bcast_ping_t;

/* all possible return structures are packed into one return type */
typedef struct {
    unsigned int type;	/* used for the cleanup function to determine what must be released */
    
    union {
    	pp_ipmi_bmc_info_t	bmc_info;		/* PP_IPMI_BMC_SUBCMD_INFO */
    	pp_ipmi_bmc_get_enables_t	bmc_enables;	/* PP_IPMI_BMC_SUBCMD_GET_ENABLES */
    	pp_ipmi_bmc_guid_t		bmc_device_guid;	/* PP_IPMI_BMC_SUBCMD_GET_DEVICE_GUID */
    	pp_ipmi_bmc_guid_t		bmc_system_guid;	/* PP_IPMI_BMC_SUBCMD_GET_SYSTEM_GUID */
	pp_ipmi_bmc_acpi_state_t	bmc_acpi_state;/* PP_IPMI_BMC_SUBCMD_GET_ACPI_STATUS */
    	pp_ipmi_chassis_status_t	chassis_status;	/* PP_IPMI_CHASSIS_SUBCMD_STATUS */
    	pp_ipmi_chassis_power_status_t	chassis_power_status;	/* PP_IPMI_CHASSIS_SUBCMD_POWER_STATUS */
    	pp_strstream_t		chassis_poh;		/* PP_IPMI_CHASSIS_SUBCMD_POH */
    	pp_strstream_t		chassis_restart_cause;	/* PP_IPMI_CHASSIS_SUBCMD_RESTART_CAUSE */
	pp_ipmi_chassis_get_bflag_return_t	chassis_bootflag;	/* PP_IPMI_CHASSIS_SUBCMD_GET_BOOTDEV */
    	pp_ipmi_chassis_get_policies_t	chassis_policies;	/* PP_IPMI_CHASSIS_SUBCMD_GET_POLICIES */
    	pp_ipmi_chassis_get_bparam_t	chassis_bootparam;	/* PP_IPMI_CHASSIS_SUBCMD_GET_BOOTPARAM */
	pp_ipmi_usr_summary_t   usr_summary;            /* PP_IPMI_USER_SUBCMD_SUMMARY */
	pp_ipmi_usr_info_t      usr_info;               /* PP_IPMI_USER_SUBCMD_INFO */
        pp_ipmi_usr_list_t      usr_list;               /* PP_IPMI_USER_SUBCMD_LIST */
    	pp_ipmi_sel_info_t	sel_info;		/* PP_IPMI_SEL_SUBCMD_INFO */
    	pp_ipmi_time_t		sel_get_time;		/* PP_IPMI_SEL_SUBCMD_GET_TIME */
    	vector_t *		sel_list;		/* PP_IPMI_SEL_SUBCMD_LIST, PP_IPMI_SEL_SUBCMD_GET */
    							/* vector of pp_ipmi_sel_list_entry_t */
	pp_ipmi_fru_info_t	fru_info;		/* PP_IPMI_FRU_SUBCMD_BOARD */
    	pp_ipmi_sdr_info_t	sdr_info;		/* PP_IPMI_SDR_SUBCMD_INFO */
    	vector_t *		sdr_list;		/* PP_IPMI_SDR_SUBCMD_LIST */
    							/* vector of pp_ipmi_sdr_list_entry_t */
    	vector_t *		sensor_list;		/* PP_IPMI_SENSOR_LIST */
    							/* vector of pp_ipmi_sdr_list_entry_t */
    	vector_t *		sensor_get_list;	/* PP_IPMI_SENSOR_GET */
    							/* vector of pp_ipmi_sdr_list_entry_t */
    	vector_t *		sensor_get_raw_list;	/* PP_IPMI_SENSOR_GET_RAW_BY_ID */
    							/* vector of pp_ipmi_raw_reading_list_entry_t */
    	vector_t *		sensor_get_thresh_list;	/* PP_IPMI_SENSOR_GET_THRESH_BY_ID */
    							/* vector of pp_ipmi_raw_thresholds_list_entry_t */
	pp_ipmi_lanp_get_t	lanp_get;		/* PP_IPMI_LANP_SUMCMD_GET */
        vector_t *              lanp_dest_list;         /* PP_IPMI_LANP_SUBCMD_DESTS */
                                                        /* vector of pp_ipmi_lanp_dest_list_entry_t */
        vector_t*               pef_filter_list;        /* PP_IPMI_PEF_FILTER_GET */
                                                        /* vector of pp_ipmi_pef_filter_t */
        vector_t*               pef_policy_list;        /* PP_IPMI_PEF_POLICY_GET */
                                                        /* vector of pp_ipmi_pef_policy_t */
        vector_t*               lan_destination_list;   /* PP_IPMI_LAN_DESTINATION_GET */
                                                        /* vector of pp_ipmi_lanp_dest_t */
	pp_ipmi_raw_t		raw_data;		/* PP_IPMI_RAW_SUBCMD_RAW */

#ifdef PRODUCT_AMDDC
	pp_ipmi_oem_opma_get_local_lockout_result_t oem_opma_get_local_lockout;
#endif
	
#ifdef PP_FEAT_ESB2_TPT
        pp_ipmi_oem_intel_get_usr_cfg_result_t      oem_intel_get_usr_cfg;
        pp_ipmi_oem_intel_get_tpt_status_result_t   oem_intel_get_tpt_status;
        pp_ipmi_oem_intel_get_tpt_cfg_result_t      oem_intel_get_tpt_cfg;
#endif

#ifdef PRODUCT_PDU
	pp_ipmi_oem_pp_rpc_get_power_on_delay_result_t       oem_pp_rpc_get_power_on_delay_rs;
	pp_ipmi_oem_pp_rpc_get_receptacle_state_result_t     oem_pp_rpc_get_recep_state_rs;
	pp_ipmi_oem_pp_rpc_get_group_membership_result_t     oem_pp_rpc_get_group_membership_rs;
	pp_ipmi_oem_pp_rpc_get_group_power_on_delay_result_t oem_pp_rpc_get_group_power_on_delay_rs;
	pp_ipmi_oem_pp_rpc_get_recep_acl_result_t 	     oem_pp_rpc_get_recep_acl_rs;
#endif

	pp_ipmi_oem_pp_get_pwm_result_t             oem_pp_get_pwm;

	pp_ipmi_oem_raritan_fw_version_t	oem_raritan_fw_version;	/* PP_IPMI_OEM_RARITAN_FW_VERSION */
	pp_strstream_t				oem_raritan_sn;	/* PP_IPMI_OEM_RARITAN_SN_GET */
	pp_ipmi_oem_raritan_get_ext_user_params_t	oem_raritan_get_ext_user_params; /* PP_IPMI_OEM_RARITAN_GET_EXT_USER_PARAMS */
	pp_ipmi_oem_raritan_list_ext_user_params_t	oem_raritan_list_ext_user_params; /* PP_IPMI_OEM_RARITAN_LIST_EXT_USER_PARAMS */

	pp_ipmi_self_test_ddc_t		self_test_ddc;		/* PP_IPMI_SELF_TEST_SUBCMD_DDC_INFO */
	pp_ipmi_self_test_video_status_t	self_test_video_status;	/* PP_IPMI_SELF_TEST_SUBCMD_VIDEO_STATUS */
	pp_ipmi_self_test_video_crc_t		self_test_video_crc;	/* PP_IPMI_SELF_TEST_SUBCMD_VIDEO_CRC */
	pp_ipmi_self_test_nic_status_t		self_test_nic_status;	/* PP_IPMI_SELF_TEST_SUBCMD_NIC_STATUS */
	pp_ipmi_self_test_nic_bcast_ping_t	self_test_bcast_ping;	/* PP_IPMI_SELF_TEST_SUBCMD_NIC_BROADCAST_PING */

#ifdef OEM_RACKABLE
	pp_ipmi_oem_rackable_lcd_get_slot_result_t	oem_rackable_lcd_get_slot;
	pp_ipmi_oem_rackable_lcd_get_state_result_t	oem_rackable_lcd_get_state;
#endif
    } data;
} pp_ipmi_return_t;

/* ------------------------------------------------------------------------- */

/* NOTE: The following commands don't use the return structure to return
         their values and don't use the parameter structure for their
         parameters (they use a string in the parameter structure instead).
         This must be implemented when it is needed. The data
         is put on stdout inbetween. All other commands don't have a
         return structure, they just return whether the execution was
         successful or not (e.g. power, reset, ...)

   PP_IPMI_CMD_BMC
   	none
   
   PP_IPMI_CMD_CHASSIS
   	none
   
   PP_IPMI_CMD_SEL
   	none
   
   PP_IPMI_CMD_SDR
   	none
   
   PP_IPMI_CMD_SENSOR
   	none
   
   PP_IPMI_CMD_LANP
	PP_IPMI_LANP_SUBCMD_SET (only used for a subset of SET commands, others already done)
   
   PP_IPMI_CMD_EVENT
   	all
   
   PP_IPMI_CMD_FRU
   	none
   
   PP_IPMI_CMD_USER
   	none
   
   PP_IPMI_CMD_CHANNEL
   	all
   
   PP_IPMI_CMD_SESSION
   	all
   
   PP_IPMI_CMD_KVM
   	all
   
*/

/* ------------------------------------------------------------------------- */

struct ipmi_intf;

#ifdef __cplusplus
extern "C" {
#endif

/* Initialize and cleanup the library */
#ifdef WIN32
int pp_ipmi_init(int do_init_winsock);
#else
int pp_ipmi_init(void);
#endif
void pp_ipmi_cleanup(void);

/* set a logging function; by default, all logging
   is done using printf() */
void pp_ipmi_set_logfunc(void (*logfunc)(const char*));

/* Send one ipmi command to the ipmi library; the data
   returned is encapsulated in the ret structure.
   This function opens the desired interface, executes the
   defined command, closes the interface and returns.
   The function returns != 0 when an error occurs. The
   error code can be obtained by the optional error
   parameter.
   Uses the default loopi channel for communication within
   eric. Use pp_ipmi_send_command_pipe for internal
   communication from a different process. */
int pp_ipmi_send_command(pp_ipmi_command_t cmd_group,
			 int subcommand,
			 pp_ipmi_parameter_t* params,
			 pp_ipmi_return_t* ret,
			 int * error,
                         const char * current_user);
int pp_ipmi_send_command_pipe(pp_ipmi_command_t cmd_group,
                         int subcommand,
                         pp_ipmi_parameter_t* params,
                         pp_ipmi_return_t* ret,
                         int * error,
                         const char * current_user,
                         int loopi_chan);

/* The following functions allow to open a BMC connection,
   send several IPMI commands over the connection (including
   a keep-alive), abort and close the connection. This must
   be used for a interactive IPMI session. */
struct ipmi_intf * pp_ipmi_connect_lan(const char *host, int port,
                                       const char *username, const char *password,
                                       const char *oemtype,
                                       int use_lanplus, int * error);
#if defined(PP_FW_TYPE_KIRATOOL)
struct ipmi_intf * pp_ipmi_connect_ppscsi(const char *device, const char *ppscsi_product,
					  const char *username, const char *password,
					  const char *oemtype, int *error);
struct ipmi_intf * pp_ipmi_connect_local(const char *username, const char *password,
					 int use_asmi, const char *oemtype, int * error);
#endif

int pp_ipmi_open_interface(struct ipmi_intf * iface, int * error);
void pp_ipmi_abort(struct ipmi_intf * iface, int * error);
void pp_ipmi_disconnect(struct ipmi_intf * iface, int force, int * error);

int pp_ipmi_cmd_run(struct ipmi_intf * iface, pp_ipmi_command_t cmd_group,
                    int subcommand, pp_ipmi_parameter_t *params,
                    pp_ipmi_return_t *ret, int * error);
int pp_ipmi_keepalive(struct ipmi_intf * iface, int * error);

/* release memory allocated for the pp_ipmi_return_t structure */
void pp_ipmi_cleanup_ret(pp_ipmi_return_t* ret);

/* get an error string from an error code */
const char * pp_ipmi_get_error_string(int error_code);
int pp_ipmi_is_auth_failed(int error_code);
int pp_ipmi_is_self_test_failed(int error_code);
int pp_ipmi_is_not_supported(int error_code);

/* clean up single sdr table entry */
void pp_ipmi_delete_sdr_table_entry(void * void_entry);
void pp_ipmi_reset_sdr_table_entry(void * void_entry);
void pp_ipmi_clone_sdr_list(pp_ipmi_return_t *dst,pp_ipmi_return_t *src,size_t idx);

/**
 * convert raw sensor value to real sensor reading
 *
 * @returns real sensor reading
 */
double pp_ipmi_sdr_convert_sensor_reading_core(int m, int b, int k1, int k2,
                                               uint8_t	 analog, 
                                               uint8_t	 linearization,
                                               uint16_t val);

/* pp_ipmi_sdr_convert_sensor_value_to_raw  -  convert sensor reading back to raw
 *
 * @sensor:     sensor record
 * @val:        converted sensor reading
 *
 * returns raw sensor reading
 */
uint8_t
pp_ipmi_sdr_convert_sensor_value_to_raw(int m, int b, int k1, int k2,
                                        uint8_t analog, double val);

#ifdef __cplusplus
}
#endif

#endif /* __PP_IPMI_H */
