/* this is a wrapper around the ipmitool which is used as our
   IPMI client library
 */

#include <stdio.h>
#include <memory.h>
#include <malloc.h>
#include <string.h>

#ifdef _WIN32
#include <winsock2.h>
#include <windows.h>
#include <pp/strstream.h>
#else
#include <unistd.h>
#include <pthread.h>
#include <pp/base.h>
#include <sys/time.h>
#endif

#include <pp/ipmi.h>

#include <config.h>

#include <ipmitool/ipmi_print.h>

#include <ipmitool.h>
#include <ipmi_params.h>
#include <ipmi_test.h>
/* loopi makes only sense in case we are also BMC */
#if defined (PP_FEAT_IPMI_SERVER)
#include <pp/bmc/loopi_adapter.h>
#endif /* PP_FEAT_IPMI_SERVER */

/******************************************************************************
* connection mutex (libpp_ipmi is not thread safe!)                           *
******************************************************************************/

#ifdef _WIN32

static CRITICAL_SECTION ipmitool_mutex;

static void ipmi_create_mutex(void) {
    InitializeCriticalSection(&ipmitool_mutex);
}

static void ipmi_destroy_mutex(void) {
    DeleteCriticalSection(&ipmitool_mutex);
}

static void ipmi_lock_mutex(void) {
    EnterCriticalSection(&ipmitool_mutex);
}

static void ipmi_unlock_mutex(void) {
    LeaveCriticalSection(&ipmitool_mutex);
}

#else // _WIN32

static pthread_mutex_t ipmitool_mutex;

static void ipmi_create_mutex(void) {
	pthread_mutex_init(&ipmitool_mutex, NULL);
}

static void ipmi_destroy_mutex(void) {
	pthread_mutex_destroy(&ipmitool_mutex);
}

static void ipmi_lock_mutex(void) {
    pthread_mutex_lock(&ipmitool_mutex);
}

static void ipmi_unlock_mutex(void) {
    pthread_mutex_unlock(&ipmitool_mutex);
}

#endif // _WIN32

/******************************************************************************
* initialization and cleanup stuff                                            *
******************************************************************************/

#ifdef _WIN32

#include <winsock2.h>
#include <windows.h>

static int winsock_initialized = 0;

static int init_winsock(void) {
	WORD wVersionRequested;
	WSADATA wsaData;
	
	int	err;
 
	wVersionRequested = MAKEWORD( 2, 2 );
 
	err	= WSAStartup(wVersionRequested, &wsaData);
	if (err != 0) {
		ipmi_printf("No usable WinSock.dll found!\n");
		return -1;
	}

	if (LOBYTE(wsaData.wVersion) != 2 ||
        HIBYTE(wsaData.wVersion) != 2) {
		ipmi_printf("No usable WinSock.dll found!\n");
		return -1;
	}
	return 0;
}

static void cleanup_winsock(void) {
	WSACleanup();
}

int pp_ipmi_init(int do_init_winsock) {
	ipmi_create_mutex();
	if (do_init_winsock && init_winsock()) {
		return -1;
	}

	return 0;
}

void pp_ipmi_cleanup(void) {
	if (winsock_initialized) {
		cleanup_winsock();
		winsock_initialized = 0;
	}
	ipmi_destroy_mutex();
}

#else
#ifdef PRODUCT_ASMIDC
static int pp_ipmi_sel_get_time(void) {
    pp_ipmi_return_t ipmi_ret;
    struct timeval tv;
    struct timezone tz;
    int ret, err;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&tz, 0, sizeof(timezone));
    ret = 0;
    err = 0;
    
    ret = pp_ipmi_send_command(PP_IPMI_CMD_SEL, PP_IPMI_SEL_SUBCMD_GET_TIME, NULL, &ipmi_ret, &err, NULL);
    if (ret) {
        pp_log("IPMI: Error in sending SEL GET_TIME command (%s).\n", pp_ipmi_get_error_string(err));
    } else {
        pp_log("[SEL] get SEL time: %s %s (0x%x).\n", ipmi_ret.data.sel_get_time.string_date.buf,
    	       ipmi_ret.data.sel_get_time.string_time.buf, (unsigned int)ipmi_ret.data.sel_get_time.timestamp);
        tv.tv_sec = ipmi_ret.data.sel_get_time.timestamp;
        tv.tv_usec = 0;
        settimeofday(&tv, &tz);
    }
    pp_ipmi_cleanup_ret(&ipmi_ret);
    return 0;
}
#endif /* PRODUCT_ASMIDC */
int pp_ipmi_init() {
	ipmi_create_mutex();
	ipmi_test();
#ifdef PRODUCT_ASMIDC
    pp_ipmi_sel_get_time();
#endif /* PRODUCT_ASMIDC */     
	return 0;
}

void pp_ipmi_cleanup(void) {
	ipmi_destroy_mutex();
}

#endif

/******************************************************************************
* build the settings for ipmitool                                             *
******************************************************************************/

static void cleanup_config(ipmitool_config_t *config) {
    if (!config)		return;
    if (config->hostname)	free(config->hostname);
    if (config->device)		free(config->device);
    if (config->password)	free(config->password);
    if (config->username)	free(config->username);
    if (config->oemtype)	free(config->oemtype);

    free(config);
}

#if defined (_WIN32) || defined (PP_FW_TYPE_KIRATOOL)

// this is just a dummy implementation!
// use the pp_ipmi_connect_*() and pp_ipmi_cmd_run() functions!
static ipmitool_config_t* get_config(const char * current_user) {
    ipmitool_config_t *ret = NULL;

    (void) current_user;

    ret = malloc(sizeof(ipmitool_config_t));
    if (!ret) return NULL;
    
    memset(ret, 0, sizeof(ipmitool_config_t));

    ret->intfname = "lan";
    ret->hostname = strdup("lara-thre");
    ret->username = strdup("super");
    ret->password = strdup("pass");

#ifdef PP_IPMI_DEBUG
    ret->verbose = 2;
#else
    ret->verbose = 0;
#endif // PP_IPMI_DEBUG

	return ret;
}

#else /* !_WIN32 && !PP_FW_TYPE_KIRATOOL */

#include <pp/features.h>
#include <pp/cfg.h>

static ipmitool_config_t* get_config(const char * current_user) {
    ipmitool_config_t *ret;
    char * acc_mode;
    char *bmc_addr_string;
    u_int bmc_addr_uint;
    
    ret = malloc(sizeof(ipmitool_config_t));
    if (!ret) return NULL;
    memset(ret, 0, sizeof(ipmitool_config_t));

#if defined(PP_FEAT_IPMI_SERVER) && !defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    acc_mode = strdup("loopi");
#else // !PP_FEAT_IPMI_SERVER
    pp_cfg_get(&acc_mode, "ipmi.medium._c_");
#endif // !PP_FEAT_IPMI_SERVER

#ifdef PRODUCT_INTELDC
    ret->own_addr = 0x33 << 1; // ASICS I2C slave addr
#else
    ret->own_addr = 0x55 << 1; // PPC I2C slave addr
#endif
    
    ret->bmc_addr = 0x20;
    pp_cfg_get(&bmc_addr_string, "ipmi.medium.i2c.bmc_address");
    bmc_addr_uint = pp_strtoul(bmc_addr_string, 0x20, 16, NULL);
    ret->bmc_addr = (bmc_addr_uint <= 0xFF && bmc_addr_uint > 0) ? (u_char)bmc_addr_uint : 0x20;

/* FIXME: make oemtype configurable */
    if (!strcmp(acc_mode, "serial")) {
    	char* parity_str;
    	char* hs_str;
    	int stop_bits;
    	int kbaud;
    	int port = -1;
    	char *port1_s, *port2_s;
        
        ret->intfname = "serial";
        pp_cfg_get(&ret->password, "ipmi.medium.serial.password");
        
        pp_cfg_get(&port1_s, "serialport[0]._c_");
        pp_cfg_get(&port2_s, "serialport[1]._c_");
        
        if (!strcmp(port1_s, "ipmi")) {
            port = 0;
        } else if (!strcmp(port2_s, "ipmi")) {
            port = 1;
        }
        
        free(port1_s);
        free(port2_s);
        
        if (port < 0) {
            pp_log("Error: none of the serial ports is configured for IPMI!\n");
            goto fail;
        }

	pp_cfg_get_int(&kbaud, "serialport[0].ipmi.speed");
	pp_cfg_get_int(&ret->bits, "serialport[0].ipmi.data");
	pp_cfg_get(&parity_str, "serialport[0].ipmi.parity");
	pp_cfg_get_int(&stop_bits, "serialport[0].ipmi.stop");
	pp_cfg_get(&hs_str, "serialport[0].ipmi.handshake");
	if (!strcmp(hs_str, "sw")) {
	    ret->hwf = 0; ret->swf = 1;
	} else if (!strcmp(hs_str, "hw")) {
	    ret->hwf = 1; ret->swf = 0;
	} else {
	    ret->hwf = ret->swf = 0;
	}
	free(hs_str);

    	if (!strcmp(parity_str, "odd")) {
	    ret->parity[0]='O';
    	} else if (!strcmp(parity_str, "even")) {
	    ret->parity[0]='E';
    	} else {
	    ret->parity[0]='\0';
    	}
    	ret->parity[1]='\0';
    	free(parity_str);
    
    	ret->stop2 = stop_bits == 2 ? 1 : 0;
    	ret->baud_rate = kbaud / 100;
    } else if (!strcmp(acc_mode, "lan")) {
        ret->intfname = "lan";
        pp_cfg_get(&ret->hostname, "ipmi.medium.lan.ipaddr");
        pp_cfg_get(&ret->username, "ipmi.medium.lan.username");
        pp_cfg_get(&ret->password, "ipmi.medium.lan.password");
    } else if (!strcmp(acc_mode, "i2c")) {
        ret->intfname = "i2c";
#if defined(PP_FEAT_IPMI_SERVER)
    } else if (!strcmp(acc_mode, "loopi")) {
        ret->intfname = "loopi";
        if (current_user) ret->username = strdup(current_user);
#endif // PP_FEAT_IPMI_SERVER
    } else {
        ret->intfname = "disabled";
    }

#ifdef PP_IPMI_DEBUG
    ret->verbose = 2;
#else
    ret->verbose = 0;
#endif // PP_IPMI_DEBUG

    free(acc_mode);
    free(bmc_addr_string);

#if !defined(PP_FEAT_IPMI_SERVER)
    (void) current_user;
#endif // PP_FEAT_IPMI_SERVER
    return ret;

 fail:
     cleanup_config(ret);
     return NULL;
}

#endif /* !_WIN32 */

/******************************************************************************
* sending an IPMI command via ipmitool                                        *
******************************************************************************/

int pp_ipmi_send_command(pp_ipmi_command_t cmd,
			 int subcommand,
			 pp_ipmi_parameter_t *params,
			 pp_ipmi_return_t* ipmi_ret,
			 int * error,
                         const char * current_user)
{
    return pp_ipmi_send_command_pipe(cmd, subcommand, params, ipmi_ret,
				     error, current_user, 
#if defined (PP_FEAT_IPMI_SERVER)
				    PP_BMC_LOOPI_ERIC
#else 
				    0
#endif
                                    );
}

int pp_ipmi_send_command_pipe(pp_ipmi_command_t cmd,
                         int subcommand,
                         pp_ipmi_parameter_t* params,
                         pp_ipmi_return_t* ipmi_ret,
                         int * error,
                         const char * current_user,
                         int loopi_chan)
{
    int ret = -1;
    ipmitool_config_t *config;

    ipmi_lock_mutex();

    config = get_config(current_user);
    if (!config) {
    	ipmi_printf("Error: could not get IPMI config.\n");
    	goto bail;
    }

    config->loopi_chan = loopi_chan;

    ret = ipmitool_main(config, cmd, subcommand,
    			params, ipmi_ret, error);
    
    cleanup_config(config);

bail:
    ipmi_unlock_mutex();
    return ret;
}
