#include <config.h>
#include <stdio.h>
#include <memory.h>

#include <pp/ipmi.h>
#ifdef WIN32
#include <pp/strstream.h>
#else
#include <pp/base.h>
#endif
#include <pp/vector.h>
#include <pp/intl.h> // date and time conversion

#include <ipmitool/ipmi_print.h>
#include <ipmi_return.h>

/******************************************************************************
* function prototypes                                                         *
******************************************************************************/

static void delete_sel_table_entry(void *);
static void delete_raw_reading_list_entry(void *);
static void delete_raw_thresholds_list_entry(void *);

/******************************************************************************
* all possible return types                                                   *
******************************************************************************/

typedef enum {
    IPMI_RET_UNUSED = 0,
    IPMI_RET_BMC_INFO,
    IPMI_RET_BMC_GET_ENABLES,
    IPMI_RET_BMC_GET_DEVICE_GUID,
    IPMI_RET_BMC_GET_SYSTEM_GUID,
    IPMI_RET_BMC_GET_ACPI_STATE,
    IPMI_RET_CHASSIS_STATUS,
    IPMI_RET_CHASSIS_POWER_STATUS,
    IPMI_RET_CHASSIS_POH,
    IPMI_RET_CHASSIS_RESTART_CAUSE,
    IPMI_RET_CHASSIS_GET_BOOTPARAM,
    IPMI_RET_CHASSIS_GET_BOOTFLAG,
    IPMI_RET_SEL_INFO,
    IPMI_RET_SEL_GET_TIME,
    IPMI_RET_SEL_LIST,
    IPMI_RET_FRU_INFO,
    IPMI_RET_SDR_INFO,
    IPMI_RET_SDR_LIST,
    IPMI_RET_SENSOR_GET_LIST,
    IPMI_RET_SENSOR_GET_RAW_LIST,
    IPMI_RET_SENSOR_GET_THRESH_LIST,
    IPMI_RET_LANP_GET,
    IPMI_RET_LANP_DEST_LIST,
    IPMI_RET_PEF_FILTER_LIST,
    IPMI_RET_PEF_POLICY_LIST,
    IPMI_RET_LAN_DESTINATION_LIST,
    IPMI_RET_OEM_RARITAN_FW_VERSION,
    IPMI_RET_OEM_RARITAN_SN,
    IPMI_RET_RAW_DATA,
    IPMI_RET_OEM_RACKABLE_GET_SLOT,
    IPMI_RET_OEM_RACKABLE_GET_STATE,
} ipmi_return_value_types;

/******************************************************************************
* helper functions                                                            *
******************************************************************************/

/*
static void cleanup_vector_char_dynamic(void* elem) {
    free(elem);
}
*/

void ipmi_set_timestamp(time_t timestamp, pp_ipmi_time_t* pp_time) {
    char tbuf[40];
    
    pp_time->is_valid = 1;
    pp_time->timestamp = timestamp;
#ifdef WIN32
	if (timestamp < 0) {
		timestamp = 0;
	}
#endif
    pp_time->is_pre_init = (timestamp < 0x20000000) ? 1 : 0;
    pp_time->is_unspecified = ((timestamp == 0) || (timestamp == (time_t)-1)) ? 1 : 0;

    strftime(tbuf, sizeof(tbuf), "%Y-%m-%d", localtime(&timestamp));
    tbuf[sizeof(tbuf) - 1] = '\0';
    pp_strappend(&pp_time->string_date, tbuf);

    strftime(tbuf, sizeof(tbuf), "%H:%M:%S", localtime(&timestamp));
    tbuf[sizeof(tbuf) - 1] = '\0';
    pp_strappend(&pp_time->string_time, tbuf);
}

/******************************************************************************
* init functions                                                              *
******************************************************************************/

static void init_ret(pp_ipmi_return_t* ret) {
    memset(ret, 0, sizeof(pp_ipmi_return_t));
}

static void ipmi_init_timestamp(pp_ipmi_time_t *timestamp) {
    pp_strstream_init(&timestamp->string_time);
    pp_strstream_init(&timestamp->string_date);
}

int ipmi_init_bmc_info(pp_ipmi_return_t* ret) {
    pp_ipmi_bmc_info_t *info = &ret->data.bmc_info;
    
    init_ret(ret);
    
    pp_strstream_init(&info->firmware_revision);
    pp_strstream_init(&info->ipmi_version);
    info->additional_device_support = vector_new(NULL, 0, NULL);
    
    ret->type = IPMI_RET_BMC_INFO;
    
    return 0;
}

int ipmi_init_bmc_get_enables(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_BMC_GET_ENABLES;
    return 0;
}

int ipmi_init_bmc_get_device_guid(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_BMC_GET_DEVICE_GUID;
    return 0;
}

int ipmi_init_bmc_get_system_guid(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_BMC_GET_SYSTEM_GUID;
    return 0;
}

int ipmi_init_bmc_get_acpi_state(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_BMC_GET_ACPI_STATE;
    pp_strstream_init(&ret->data.bmc_acpi_state.sys_state_string);
    pp_strstream_init(&ret->data.bmc_acpi_state.dev_state_string);
    return 0;
}

int ipmi_init_chassis_status(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_STATUS;
    pp_strstream_init(&ret->data.chassis_status.power_status_string);
    pp_strstream_init(&ret->data.chassis_status.power_restore_policy.string);
    pp_strstream_init(&ret->data.chassis_status.last_power_event);
    return 0;
}

int ipmi_init_chassis_power_status(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_POWER_STATUS;
    pp_strstream_init(&ret->data.chassis_power_status.status_string);
    return 0;
}

int ipmi_init_chassis_poh(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_POH;
    pp_strstream_init(&ret->data.chassis_poh);
    return 0;
}

int ipmi_init_chassis_restart_cause(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_RESTART_CAUSE;
    pp_strstream_init(&ret->data.chassis_restart_cause);
    return 0;
}

int ipmi_init_chassis_get_bootparam(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_GET_BOOTPARAM;
    return 0;
}

int ipmi_init_chassis_get_bootflag(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_CHASSIS_GET_BOOTFLAG;
    return 0;
}

int ipmi_init_sel_info(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SEL_INFO;
    ipmi_init_timestamp(&ret->data.sel_info.last_add_time);
    ipmi_init_timestamp(&ret->data.sel_info.last_del_time);
    return 0;
}

int ipmi_init_sel_get_time(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SEL_GET_TIME;
    ipmi_init_timestamp(&ret->data.sel_get_time);
    return 0;
}

int ipmi_init_sel_list(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SEL_LIST;
    ret->data.sel_list = vector_new(NULL, 0, delete_sel_table_entry);
    return ret->data.sel_list ? 0 : 1;
}

pp_ipmi_sel_list_entry_t * ipmi_init_sel_list_entry(pp_ipmi_return_t* ret) {
    pp_ipmi_sel_list_entry_t *entry;
    
    if (!ret->data.sel_list) {
    	return NULL;
    }
    
    entry = malloc(sizeof(pp_ipmi_sel_list_entry_t));
    
    if (!entry) {
    	return NULL;
    }
    
    memset(entry, 0, sizeof(pp_ipmi_sel_list_entry_t));
    
    ipmi_init_timestamp(&entry->timestamp);
    pp_strstream_init(&entry->record_type_string);
    pp_strstream_init(&entry->sensor_type);
    pp_strstream_init(&entry->sensor_name);
    pp_strstream_init(&entry->event_type);
    pp_strstream_init(&entry->event_direction);
    pp_strstream_init(&entry->description);
    pp_strstream_init(&entry->unit);
    
    vector_add(ret->data.sel_list, entry);
    
    return entry;
}

int ipmi_init_fru_info(pp_ipmi_return_t* ret) {
    ret->type = IPMI_RET_FRU_INFO;
    pp_strstream_init(&ret->data.fru_info.chassis.type);
    pp_strstream_init(&ret->data.fru_info.chassis.model);
    pp_strstream_init(&ret->data.fru_info.chassis.serial);
    ipmi_init_timestamp(&ret->data.fru_info.board.mfg_time);
    pp_strstream_init(&ret->data.fru_info.board.manufacturer);
    pp_strstream_init(&ret->data.fru_info.board.name);
    pp_strstream_init(&ret->data.fru_info.board.model);
    pp_strstream_init(&ret->data.fru_info.board.serial);
    pp_strstream_init(&ret->data.fru_info.board.fru_id);
    pp_strstream_init(&ret->data.fru_info.product.manufacturer);
    pp_strstream_init(&ret->data.fru_info.product.name);
    pp_strstream_init(&ret->data.fru_info.product.model);
    pp_strstream_init(&ret->data.fru_info.product.version);
    pp_strstream_init(&ret->data.fru_info.product.serial);
    pp_strstream_init(&ret->data.fru_info.product.asset);
    pp_strstream_init(&ret->data.fru_info.product.fru_id);
    return 0;
}

int ipmi_init_sdr_info(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SDR_INFO;
    ipmi_init_timestamp(&ret->data.sdr_info.most_resent_addition);
    ipmi_init_timestamp(&ret->data.sdr_info.most_resent_erase);
    pp_strstream_init(&ret->data.sdr_info.free_space.string);
    pp_strstream_init(&ret->data.sdr_info.modal_update_support);
    return 0;
}

int ipmi_init_sdr_list(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SDR_LIST;
    ret->data.sdr_list = vector_new(NULL, 0, pp_ipmi_delete_sdr_table_entry);
    return ret->data.sdr_list ? 0 : 1;
}

int ipmi_init_sensor_list(pp_ipmi_return_t* ret) {
    return ipmi_init_sensor_get_list(ret);
}

int ipmi_init_sensor_get_list(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SENSOR_GET_LIST;
    ret->data.sensor_get_list = vector_new(NULL, 0,
                                           pp_ipmi_delete_sdr_table_entry);
    ret->data.sensor_list = ret->data.sensor_get_list;
    return ret->data.sensor_get_list ? 0 : 1;
}

int ipmi_init_sensor_get_raw_list(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SENSOR_GET_RAW_LIST;
    ret->data.sensor_get_raw_list = vector_new(NULL, 0,
                                               delete_raw_reading_list_entry);
    ret->data.sensor_list = ret->data.sensor_get_raw_list;
    return ret->data.sensor_get_raw_list ? 0 : 1;
}

int ipmi_init_sensor_get_thresh_list(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_SENSOR_GET_THRESH_LIST;
    ret->data.sensor_get_thresh_list = vector_new(NULL, 0,
                                              delete_raw_thresholds_list_entry);
    ret->data.sensor_list = ret->data.sensor_get_thresh_list;
    return ret->data.sensor_get_thresh_list ? 0 : 1;
}

static pp_ipmi_sdr_list_entry_t * ipmi_init_sensor(pp_ipmi_sensor_type_t type) {
    pp_ipmi_sdr_list_entry_t *entry;
    
    entry = malloc(sizeof(pp_ipmi_sdr_list_entry_t));
    
    if (!entry) {
    	return NULL;
    }
    
    memset(entry, 0, sizeof(pp_ipmi_sdr_list_entry_t));
    
    entry->type = type;
    
    pp_strstream_init(&entry->sdr_name);
    pp_strstream_init(&entry->entity.string);
    pp_strstream_init(&entry->sdr_type_main);
    pp_strstream_init(&entry->sdr_type_sub);
    
    switch (type) {
    	case PP_IPMI_SENSOR_TYPE_FULL:
    	    pp_strstream_init(&entry->data.full.unit);
    	    pp_strstream_init(&entry->data.full.status_string);
    	    pp_strstream_init(&entry->data.full.assertion_events);
    	    pp_strstream_init(&entry->data.full.deassertion_events);
    	    pp_strstream_init(&entry->data.full.assertion_event_mask);
    	    pp_strstream_init(&entry->data.full.deassertion_event_mask);
    	    pp_strstream_init(&entry->data.full.assertions_enabled);
    	    pp_strstream_init(&entry->data.full.deassertions_enabled);
    	    pp_strstream_init(&entry->data.full.evt_msg_control_string);
    	    pp_strstream_init(&entry->data.full.cim_current_state);
    	    vector_new(&entry->data.full.cim_possible_states, 0, (vector_elem_del_func_simple)free);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
    	    pp_strstream_init(&entry->data.compact.description);
    	    pp_strstream_init(&entry->data.compact.assertion_events);
    	    pp_strstream_init(&entry->data.compact.deassertion_events);
    	    pp_strstream_init(&entry->data.compact.assertion_event_mask);
    	    pp_strstream_init(&entry->data.compact.deassertion_event_mask);
    	    pp_strstream_init(&entry->data.compact.assertions_enabled);
    	    pp_strstream_init(&entry->data.compact.deassertions_enabled);
    	    pp_strstream_init(&entry->data.compact.cim_current_state);
    	    vector_new(&entry->data.compact.cim_possible_states, 0, (vector_elem_del_func_simple)free);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_EVENTONLY:
    	    break;
    	case PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR:
    	    pp_strstream_init(&entry->data.fru_locator.dev_id_description);
    	    pp_strstream_init(&entry->data.fru_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR:
    	    pp_strstream_init(&entry->data.generic_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR:
    	    pp_strstream_init(&entry->data.mc_locator.event_message_gen);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_OTHER:
    	    break;
    }
    
    return entry;
}

void pp_ipmi_clone_sdr_list(pp_ipmi_return_t *dst,pp_ipmi_return_t *src,size_t idx) {
    pp_ipmi_sdr_list_entry_t *src_entry=vector_get(src->data.sdr_list,idx);
    ipmi_init_sdr_list(dst);
    pp_ipmi_sdr_list_entry_t *entry=ipmi_init_sdr_list_entry(src_entry->type,dst);
    memcpy(entry->raw,src_entry->raw,sizeof(entry->raw));
}

pp_ipmi_sdr_list_entry_t * ipmi_init_sdr_list_entry(pp_ipmi_sensor_type_t type, pp_ipmi_return_t* ret) {
    pp_ipmi_sdr_list_entry_t *entry;
    
    if (!ret->data.sdr_list) {
    	return NULL;
    }
    
    entry = ipmi_init_sensor(type);
    
    if (entry) {
    	vector_add(ret->data.sdr_list, entry);
    }
    
    return entry;
}

pp_ipmi_sdr_list_entry_t * ipmi_init_sensor_get_list_entry(pp_ipmi_sensor_type_t type, pp_ipmi_return_t* ret) {
    pp_ipmi_sdr_list_entry_t *entry;
    
    if (!ret->data.sensor_get_list) {
    	return NULL;
    }
    
    entry = ipmi_init_sensor(type);
    
    if (entry) {
    	vector_add(ret->data.sensor_get_list, entry);
    }
    
    return entry;
}

pp_ipmi_raw_reading_list_entry_t*
ipmi_init_sensor_get_raw_list_entry(int sensor_num, pp_ipmi_return_t* ret) {
    pp_ipmi_raw_reading_list_entry_t *entry;
    
     if (!ret->data.sensor_get_raw_list) {
    	return NULL;
    }
    
    entry = malloc(sizeof(pp_ipmi_raw_reading_list_entry_t));
    memset(entry, 0, sizeof(pp_ipmi_raw_reading_list_entry_t));
    entry->id = sensor_num;
    vector_add(ret->data.sensor_get_raw_list, entry);
    
    return entry;
}

pp_ipmi_raw_thresholds_list_entry_t*
ipmi_init_sensor_get_thresholds_list_entry(int sensor_num,
                                           pp_ipmi_return_t* ret) {
    pp_ipmi_raw_thresholds_list_entry_t *entry;
    
     if (!ret->data.sensor_get_thresh_list) {
    	return NULL;
    }
    
    entry = malloc(sizeof(pp_ipmi_raw_thresholds_list_entry_t));
    memset(entry, 0, sizeof(pp_ipmi_raw_thresholds_list_entry_t));
    entry->id = sensor_num;
    vector_add(ret->data.sensor_get_thresh_list, entry);
    
    return entry;
}

int ipmi_init_lanp_get(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_LANP_GET;
    pp_strstream_init(&ret->data.lanp_get.ip_address.address);
    pp_strstream_init(&ret->data.lanp_get.subnet_mask.address);
    pp_strstream_init(&ret->data.lanp_get.mac_address.address);
    pp_strstream_init(&ret->data.lanp_get.snmp_string.string);
    pp_strstream_init(&ret->data.lanp_get.def_gw_ip.address);
    pp_strstream_init(&ret->data.lanp_get.def_gw_mac.address);
    pp_strstream_init(&ret->data.lanp_get.bak_gw_ip.address);
    pp_strstream_init(&ret->data.lanp_get.bak_gw_mac.address);
    ret->data.lanp_get.rcmp_cipher_suites.ciphers = vector_new(NULL, 0, NULL);
    return 0;
}

int ipmi_init_lanp_dest_list(pp_ipmi_return_t* ret)
{
    init_ret(ret);
    ret->type = IPMI_RET_LANP_DEST_LIST;
    ret->data.lanp_dest_list = vector_new(NULL, 0, free);
    return ret->data.lanp_dest_list ? 0 : 1;
}

pp_ipmi_lanp_dest_list_entry_t* ipmi_init_lanp_dest_list_entry(uint8_t id, pp_ipmi_return_t* ret)
{
    pp_ipmi_lanp_dest_list_entry_t *e = NULL;

    if (ret->data.lanp_dest_list) {
        e = malloc(sizeof(pp_ipmi_lanp_dest_list_entry_t));
        if (e) {
            memset(e, 0, sizeof(pp_ipmi_lanp_dest_list_entry_t));
            e->id = id;
            vector_add(ret->data.lanp_dest_list, e);
        }
    }
    return e;
}

int ipmi_init_pef_filter_list(pp_ipmi_return_t* ret)
{
    init_ret(ret);
    ret->type = IPMI_RET_PEF_FILTER_LIST;
    ret->data.pef_filter_list = vector_new(NULL, 0, free);
    return ret->data.pef_filter_list ? 0 : 1;
}

pp_ipmi_pef_filter_t* ipmi_init_pef_filter_list_entry(pp_ipmi_return_t* ret)
{
    pp_ipmi_pef_filter_t *e = NULL;

    if (ret->data.pef_filter_list) {
        e = malloc(sizeof(pp_ipmi_pef_filter_t));
        if (e) {
            memset(e, 0, sizeof(pp_ipmi_pef_filter_t));
            vector_add(ret->data.pef_filter_list, e);
        }
    }
    return e;
}

int ipmi_init_pef_policy_list(pp_ipmi_return_t* ret)
{
    init_ret(ret);
    ret->type = IPMI_RET_PEF_POLICY_LIST;
    ret->data.pef_policy_list = vector_new(NULL, 0, free);
    return ret->data.pef_policy_list ? 0 : 1;
}

pp_ipmi_pef_policy_t* ipmi_init_pef_policy_list_entry(pp_ipmi_return_t* ret)
{
    pp_ipmi_pef_policy_t *e = NULL;

    if (ret->data.pef_policy_list) {
        e = malloc(sizeof(pp_ipmi_pef_policy_t));
        if (e) {
            memset(e, 0, sizeof(pp_ipmi_pef_policy_t));
            vector_add(ret->data.pef_policy_list, e);
        }
    }
    return e;
}

int ipmi_init_lan_destination_list(pp_ipmi_return_t* ret)
{
    init_ret(ret);
    ret->type = IPMI_RET_LAN_DESTINATION_LIST;
    ret->data.lan_destination_list = vector_new(NULL, 0, free);
    return ret->data.lan_destination_list ? 0 : 1;
}

pp_ipmi_lanp_dest_t* ipmi_init_lan_destination_list_entry(pp_ipmi_return_t* ret)
{
    pp_ipmi_lanp_dest_t *e = NULL;

    if (ret->data.lan_destination_list) {
        e = malloc(sizeof(pp_ipmi_lanp_dest_t));
        if (e) {
            memset(e, 0, sizeof(pp_ipmi_lanp_dest_t));
            vector_add(ret->data.lan_destination_list, e);
        }
    }
    return e;
}

int ipmi_init_oem_raritan_fw_version(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_OEM_RARITAN_FW_VERSION;
    pp_strstream_init(&ret->data.oem_raritan_fw_version.version);
    pp_strstream_init(&ret->data.oem_raritan_fw_version.oem);
    pp_strstream_init(&ret->data.oem_raritan_fw_version.tag);
    return 0;
}

int ipmi_init_oem_raritan_sn(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_OEM_RARITAN_SN;
    pp_strstream_init(&ret->data.oem_raritan_sn);
    return 0;
}

int ipmi_init_raw_data(pp_ipmi_return_t* ret) {
    init_ret(ret);
    ret->type = IPMI_RET_RAW_DATA;
    ret->data.raw_data.data = vector_new(NULL, 0, NULL);
    return ret->data.raw_data.data ? 0 : 1;
}

#ifdef OEM_RACKABLE
int ipmi_init_oem_rackable_get_slot(pp_ipmi_return_t* ret) {
    ret->type = IPMI_RET_OEM_RACKABLE_GET_SLOT;
    return 0;
}

int ipmi_init_oem_rackable_get_state(pp_ipmi_return_t* ret) {
    ret->type = IPMI_RET_OEM_RACKABLE_GET_STATE;
    return 0;
}
#endif

/******************************************************************************
* cleanup functions - low level                                               *
******************************************************************************/

static void ipmi_cleanup_timestamp(pp_ipmi_time_t *timestamp) {
    pp_strstream_free(&timestamp->string_time);
    pp_strstream_free(&timestamp->string_date);
}

static void ipmi_cleanup_bmc_info(pp_ipmi_return_t* ret) {
    pp_ipmi_bmc_info_t *info = &ret->data.bmc_info;
    
    pp_strstream_free(&info->firmware_revision);
    pp_strstream_free(&info->ipmi_version);
    if (info->additional_device_support) {
    	vector_delete(info->additional_device_support);
    }
}

static void ipmi_cleanup_bmc_get_acpi_state(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.bmc_acpi_state.sys_state_string);
    pp_strstream_free(&ret->data.bmc_acpi_state.dev_state_string);
}

static void ipmi_cleanup_chassis_status(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.chassis_status.power_status_string);
    pp_strstream_free(&ret->data.chassis_status.power_restore_policy.string);
    pp_strstream_free(&ret->data.chassis_status.last_power_event);
}

static void ipmi_cleanup_chassis_power_status(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.chassis_power_status.status_string);
}

static void ipmi_cleanup_chassis_poh(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.chassis_poh);
}

static void ipmi_cleanup_chassis_restart_cause(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.chassis_restart_cause);
}

static void  ipmi_cleanup_chassis_get_bootparam(pp_ipmi_return_t* ret) {
    if (ret->data.chassis_bootparam.data) {
    	free(ret->data.chassis_bootparam.data);
    }
}

static void ipmi_cleanup_sel_info(pp_ipmi_return_t* ret) {
    ipmi_cleanup_timestamp(&ret->data.sel_info.last_add_time);
    ipmi_cleanup_timestamp(&ret->data.sel_info.last_del_time);
}

static void ipmi_cleanup_sel_get_time(pp_ipmi_return_t* ret) {
    ipmi_cleanup_timestamp(&ret->data.sel_get_time);
}

static void ipmi_cleanup_sel_list(pp_ipmi_return_t* ret) {
    if (ret->data.sel_list) {
    	vector_delete(ret->data.sel_list);
    }
}

static void delete_sel_table_entry(void * void_entry) {
    pp_ipmi_sel_list_entry_t *entry = (pp_ipmi_sel_list_entry_t *) void_entry;
    
    ipmi_cleanup_timestamp(&entry->timestamp);
    pp_strstream_free(&entry->record_type_string);
    pp_strstream_free(&entry->sensor_type);
    pp_strstream_free(&entry->sensor_name);
    pp_strstream_free(&entry->event_type);
    pp_strstream_free(&entry->event_direction);
    pp_strstream_free(&entry->description);
    pp_strstream_free(&entry->unit);

    free(entry);
}

static void ipmi_cleanup_fru_info(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.fru_info.chassis.type);
    pp_strstream_free(&ret->data.fru_info.chassis.model);
    pp_strstream_free(&ret->data.fru_info.chassis.serial);
    ipmi_cleanup_timestamp(&ret->data.fru_info.board.mfg_time);
    pp_strstream_free(&ret->data.fru_info.board.manufacturer);
    pp_strstream_free(&ret->data.fru_info.board.name);
    pp_strstream_free(&ret->data.fru_info.board.model);
    pp_strstream_free(&ret->data.fru_info.board.serial);
    pp_strstream_free(&ret->data.fru_info.board.fru_id);
    pp_strstream_free(&ret->data.fru_info.product.manufacturer);
    pp_strstream_free(&ret->data.fru_info.product.name);
    pp_strstream_free(&ret->data.fru_info.product.model);
    pp_strstream_free(&ret->data.fru_info.product.version);
    pp_strstream_free(&ret->data.fru_info.product.serial);
    pp_strstream_free(&ret->data.fru_info.product.asset);
    pp_strstream_free(&ret->data.fru_info.product.fru_id);
}

static void ipmi_cleanup_sdr_info(pp_ipmi_return_t* ret) {
    ipmi_cleanup_timestamp(&ret->data.sdr_info.most_resent_addition);
    ipmi_cleanup_timestamp(&ret->data.sdr_info.most_resent_erase);
    pp_strstream_free(&ret->data.sdr_info.free_space.string);
    pp_strstream_free(&ret->data.sdr_info.modal_update_support);
}

static void ipmi_cleanup_sdr_list(pp_ipmi_return_t* ret) {
    if (ret->data.sdr_list) {
    	vector_delete(ret->data.sdr_list);
    }
}

static void ipmi_cleanup_sensor_list(pp_ipmi_return_t* ret) {
    if (ret->data.sensor_get_list) {
    	vector_delete(ret->data.sensor_get_list);
    }
}

static void ipmi_cleanup_sensor_raw_list(pp_ipmi_return_t* ret) {
    if (ret->data.sensor_get_raw_list) {
    	vector_delete(ret->data.sensor_get_raw_list);
    }
}

static void ipmi_cleanup_sensor_thresh_list(pp_ipmi_return_t* ret) {
    if (ret->data.sensor_get_thresh_list) {
    	vector_delete(ret->data.sensor_get_thresh_list);
    }
}

void pp_ipmi_reset_sdr_table_entry(void * void_entry) {
    pp_ipmi_sdr_list_entry_t *entry = (pp_ipmi_sdr_list_entry_t *) void_entry;

    switch (entry->type) {
    	case PP_IPMI_SENSOR_TYPE_FULL:
    	    pp_strstream_reset(&entry->data.full.unit);
    	    pp_strstream_reset(&entry->data.full.status_string);
    	    pp_strstream_reset(&entry->data.full.assertion_events);
    	    pp_strstream_reset(&entry->data.full.deassertion_events);
    	    pp_strstream_reset(&entry->data.full.assertion_event_mask);
    	    pp_strstream_reset(&entry->data.full.deassertion_event_mask);
    	    pp_strstream_reset(&entry->data.full.assertions_enabled);
    	    pp_strstream_reset(&entry->data.full.deassertions_enabled);
    	    pp_strstream_reset(&entry->data.full.evt_msg_control_string);
    	    pp_strstream_reset(&entry->data.full.cim_current_state);
    	    vector_delete(&entry->data.full.cim_possible_states);
    	    vector_new(&entry->data.full.cim_possible_states, 0, (vector_elem_del_func_simple)free);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
    	    pp_strstream_reset(&entry->data.compact.description);
    	    pp_strstream_reset(&entry->data.compact.assertion_events);
    	    pp_strstream_reset(&entry->data.compact.deassertion_events);
    	    pp_strstream_reset(&entry->data.compact.assertion_event_mask);
    	    pp_strstream_reset(&entry->data.compact.deassertion_event_mask);
    	    pp_strstream_reset(&entry->data.compact.assertions_enabled);
    	    pp_strstream_reset(&entry->data.compact.deassertions_enabled);
    	    pp_strstream_reset(&entry->data.compact.cim_current_state);
    	    vector_delete(&entry->data.compact.cim_possible_states);
    	    vector_new(&entry->data.compact.cim_possible_states, 0, (vector_elem_del_func_simple)free);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_EVENTONLY:
    	    break;
    	case PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR:
    	    pp_strstream_reset(&entry->data.fru_locator.dev_id_description);
    	    pp_strstream_reset(&entry->data.fru_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR:
    	    pp_strstream_reset(&entry->data.generic_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR:
    	    pp_strstream_reset(&entry->data.mc_locator.event_message_gen);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_OTHER:
    	    break;
    }


    pp_strstream_reset(&entry->sdr_name);
    pp_strstream_reset(&entry->entity.string);
    pp_strstream_reset(&entry->sdr_type_main);
    pp_strstream_reset(&entry->sdr_type_sub);
}

void pp_ipmi_delete_sdr_table_entry(void * void_entry) {
    pp_ipmi_sdr_list_entry_t *entry = (pp_ipmi_sdr_list_entry_t *) void_entry;

    switch (entry->type) {
    	case PP_IPMI_SENSOR_TYPE_FULL:
    	    pp_strstream_free(&entry->data.full.unit);
    	    pp_strstream_free(&entry->data.full.status_string);
    	    pp_strstream_free(&entry->data.full.assertion_events);
    	    pp_strstream_free(&entry->data.full.deassertion_events);
    	    pp_strstream_free(&entry->data.full.assertion_event_mask);
    	    pp_strstream_free(&entry->data.full.deassertion_event_mask);
    	    pp_strstream_free(&entry->data.full.assertions_enabled);
    	    pp_strstream_free(&entry->data.full.deassertions_enabled);
    	    pp_strstream_free(&entry->data.full.evt_msg_control_string);
    	    pp_strstream_free(&entry->data.full.cim_current_state);
    	    vector_delete(&entry->data.full.cim_possible_states);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_COMPACT_SENSOR:
    	    pp_strstream_free(&entry->data.compact.description);
    	    pp_strstream_free(&entry->data.compact.assertion_events);
    	    pp_strstream_free(&entry->data.compact.deassertion_events);
    	    pp_strstream_free(&entry->data.compact.assertion_event_mask);
    	    pp_strstream_free(&entry->data.compact.deassertion_event_mask);
    	    pp_strstream_free(&entry->data.compact.assertions_enabled);
    	    pp_strstream_free(&entry->data.compact.deassertions_enabled);
    	    pp_strstream_free(&entry->data.compact.cim_current_state);
    	    vector_delete(&entry->data.compact.cim_possible_states);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_EVENTONLY:
    	    break;
    	case PP_IPMI_SENSOR_TYPE_FRU_DEVICE_LOCATOR:
    	    pp_strstream_free(&entry->data.fru_locator.dev_id_description);
    	    pp_strstream_free(&entry->data.fru_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_GENERIC_DEVICE_LOCATOR:
    	    pp_strstream_free(&entry->data.generic_locator.dev_type_string);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_MC_DEVICE_LOCATOR:
    	    pp_strstream_free(&entry->data.mc_locator.event_message_gen);
    	    break;
    	case PP_IPMI_SENSOR_TYPE_OTHER:
    	    break;
    }

    pp_strstream_free(&entry->sdr_name);
    pp_strstream_free(&entry->entity.string);
    pp_strstream_free(&entry->sdr_type_main);
    pp_strstream_free(&entry->sdr_type_sub);
    
    free(entry);
}

void delete_raw_reading_list_entry(void * void_entry) {
    pp_ipmi_raw_reading_list_entry_t *entry =
        (pp_ipmi_raw_reading_list_entry_t*) void_entry;
                                          
    free(entry);
}

void delete_raw_thresholds_list_entry(void * void_entry) {
    pp_ipmi_raw_thresholds_list_entry_t *entry =
        (pp_ipmi_raw_thresholds_list_entry_t*) void_entry;
                                          
    free(entry);
}

static void ipmi_cleanup_lanp_get(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.lanp_get.ip_address.address);
    pp_strstream_free(&ret->data.lanp_get.subnet_mask.address);
    pp_strstream_free(&ret->data.lanp_get.mac_address.address);
    pp_strstream_free(&ret->data.lanp_get.snmp_string.string);
    pp_strstream_free(&ret->data.lanp_get.def_gw_ip.address);
    pp_strstream_free(&ret->data.lanp_get.def_gw_mac.address);
    pp_strstream_free(&ret->data.lanp_get.bak_gw_ip.address);
    pp_strstream_free(&ret->data.lanp_get.bak_gw_mac.address);
    vector_delete(ret->data.lanp_get.rcmp_cipher_suites.ciphers);
}

static void ipmi_cleanup_lanp_dest_list(pp_ipmi_return_t* ret)
{
    if (ret->data.lanp_dest_list) {
        vector_delete(ret->data.lanp_dest_list);
        ret->data.lanp_dest_list = NULL;
    }
}

static void ipmi_cleanup_pef_filter_list(pp_ipmi_return_t* ret) {
    if (ret->data.pef_filter_list) {
        vector_delete(ret->data.pef_filter_list);
        ret->data.pef_filter_list = NULL;
    }
}

static void ipmi_cleanup_pef_policy_list(pp_ipmi_return_t* ret) {
    if (ret->data.pef_policy_list) {
        vector_delete(ret->data.pef_policy_list);
        ret->data.pef_policy_list = NULL;
    }
}

static void ipmi_cleanup_lan_destination_list(pp_ipmi_return_t* ret) {
    if (ret->data.lan_destination_list) {
        vector_delete(ret->data.lan_destination_list);
        ret->data.lan_destination_list = NULL;
    }
}

static void ipmi_cleanup_oem_raritan_fw_version(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.oem_raritan_fw_version.version);
    pp_strstream_free(&ret->data.oem_raritan_fw_version.oem);
    pp_strstream_free(&ret->data.oem_raritan_fw_version.tag);
}

static void ipmi_cleanup_oem_raritan_sn(pp_ipmi_return_t* ret) {
    pp_strstream_free(&ret->data.oem_raritan_sn);
}

static void ipmi_cleanup_raw_data(pp_ipmi_return_t* ret) {
    vector_delete(ret->data.raw_data.data);
}

#ifdef OEM_RACKABLE
static void ipmi_cleanup_oem_rackable_get_slot(pp_ipmi_return_t* ret) {
    free(ret->data.oem_rackable_lcd_get_slot.message);
}

static void ipmi_cleanup_oem_rackable_get_state(pp_ipmi_return_t* ret) {
    free(ret->data.oem_rackable_lcd_get_state.immediate);
}
#endif

/******************************************************************************
* cleanup functions - high level                                              *
******************************************************************************/

void pp_ipmi_cleanup_ret(pp_ipmi_return_t* ret) {
    if (!ret) return;
    
    switch ((ipmi_return_value_types) ret->type) {
    	case IPMI_RET_UNUSED:
    	    /* nothing allocated */
    	    break;
    	case IPMI_RET_CHASSIS_STATUS:
    	    ipmi_cleanup_chassis_status(ret);
    	    break;
    	case IPMI_RET_CHASSIS_POWER_STATUS:
    	    ipmi_cleanup_chassis_power_status(ret);
    	    break;
    	case IPMI_RET_BMC_INFO:
    	    ipmi_cleanup_bmc_info(ret);
    	    break;
    	case IPMI_RET_BMC_GET_ENABLES:
    	    break;
        case IPMI_RET_BMC_GET_ACPI_STATE:
	    ipmi_cleanup_bmc_get_acpi_state(ret);
	    break;
    	case IPMI_RET_CHASSIS_POH:
    	    ipmi_cleanup_chassis_poh(ret);
    	    break;
    	case IPMI_RET_CHASSIS_RESTART_CAUSE:
    	    ipmi_cleanup_chassis_restart_cause(ret);
    	    break;
    	case IPMI_RET_CHASSIS_GET_BOOTPARAM:
    	    ipmi_cleanup_chassis_get_bootparam(ret);
    	    break;
        case IPMI_RET_CHASSIS_GET_BOOTFLAG:
	    break;
    	case IPMI_RET_SEL_INFO:
    	    ipmi_cleanup_sel_info(ret);
    	    break;
    	case IPMI_RET_SEL_GET_TIME:
    	    ipmi_cleanup_sel_get_time(ret);
    	    break;
    	case IPMI_RET_SEL_LIST:
    	    ipmi_cleanup_sel_list(ret);
    	    break;
    	case IPMI_RET_FRU_INFO:
    	    ipmi_cleanup_fru_info(ret);
    	    break;
    	case IPMI_RET_SDR_INFO:
    	    ipmi_cleanup_sdr_info(ret);
    	    break;
    	case IPMI_RET_SDR_LIST:
    	    ipmi_cleanup_sdr_list(ret);
    	    break;
    	case IPMI_RET_SENSOR_GET_LIST:
    	    ipmi_cleanup_sensor_list(ret);
    	    break;
    	case IPMI_RET_SENSOR_GET_RAW_LIST:
    	    ipmi_cleanup_sensor_raw_list(ret);
    	    break;
    	case IPMI_RET_SENSOR_GET_THRESH_LIST:
    	    ipmi_cleanup_sensor_thresh_list(ret);
    	    break;
	case IPMI_RET_LANP_GET:
	    ipmi_cleanup_lanp_get(ret);
	    break;
        case IPMI_RET_LANP_DEST_LIST:
            ipmi_cleanup_lanp_dest_list(ret);
            break;
        case IPMI_RET_PEF_FILTER_LIST:
            ipmi_cleanup_pef_filter_list(ret);
            break;
        case IPMI_RET_PEF_POLICY_LIST:
            ipmi_cleanup_pef_policy_list(ret);
            break;
        case IPMI_RET_LAN_DESTINATION_LIST:
            ipmi_cleanup_lan_destination_list(ret);
            break;
	case IPMI_RET_OEM_RARITAN_FW_VERSION:
	    ipmi_cleanup_oem_raritan_fw_version(ret);
	    break;
	case IPMI_RET_OEM_RARITAN_SN:
	    ipmi_cleanup_oem_raritan_sn(ret);
	    break;
	case IPMI_RET_RAW_DATA:
	    ipmi_cleanup_raw_data(ret);
	    break;
#ifdef OEM_RACKABLE
	case IPMI_RET_OEM_RACKABLE_GET_SLOT:
	    ipmi_cleanup_oem_rackable_get_slot(ret);
	    break;
	case IPMI_RET_OEM_RACKABLE_GET_STATE:
	    ipmi_cleanup_oem_rackable_get_state(ret);
	    break;
#endif
	default:
	    ;
    }

    // avoid double-free
    ret->type = IPMI_RET_UNUSED;
}
