/*
 * Copyright (c) 2003, 2004 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include "ipmitool.h"

#include <stdio.h>

#ifdef WIN32
#include <pp/win32.h>
#else
#include <unistd.h>
#endif

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <pp/ipmi.h>

#include <ipmitool/helper.h>
#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_session.h>
#include <ipmitool/ipmi_sdr.h>
#include <ipmitool/ipmi_sel.h>
#include <ipmitool/ipmi_fru.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_chassis.h>
#include <ipmitool/ipmi_bmc.h>
#include <ipmitool/ipmi_sensor.h>
#include <ipmitool/ipmi_channel.h>
#include <ipmitool/ipmi_session.h>
#include <ipmitool/ipmi_event.h>
#include <ipmitool/ipmi_user.h>
#include <ipmitool/ipmi_kvm.h>
#include <ipmitool/ipmi_pef.h>
#include <ipmitool/ipmi_lan.h>
#include <ipmitool/ipmi_raw.h>
#include <ipmitool/ipmi_oem.h>
#include <ipmitool/ipmi_raritanoem.h>
#include <ipmitool/ipmi_selftest.h>
#ifdef PRODUCT_PDU
#include <ipmitool/ipmi_oem_pp_rpc.h>
#endif
#include <ipmitool/ipmi_oem_pp.h>

#ifdef PRODUCT_AMDDC
#include <ipmitool/ipmi_oem_opma.h>
#endif

#ifdef PP_FEAT_ESB2_TPT
#include <ipmitool/ipmi_oem_intel.h>
#endif

#ifdef OEM_RACKABLE
#include <ipmitool/ipmi_oem_rackable.h>
#endif

#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>


#include <config.h>

int verbose = 0;

static struct ipmi_cmd {
	int (*func)(struct ipmi_intf *, int, pp_ipmi_parameter_t *, pp_ipmi_return_t *, int *);
	pp_ipmi_command_t cmd;
} ipmi_cmd_list[] = {
	{ ipmi_bmc_main,		PP_IPMI_CMD_BMC },
	{ ipmi_chassis_main,		PP_IPMI_CMD_CHASSIS },
	{ ipmi_sel_main,		PP_IPMI_CMD_SEL },
	{ ipmi_sdr_main,		PP_IPMI_CMD_SDR },
	{ ipmi_sensor_main,		PP_IPMI_CMD_SENSOR },
	{ ipmi_lanp_main,		PP_IPMI_CMD_LANP },
	{ ipmi_channel_main,		PP_IPMI_CMD_CHANNEL },
        { ipmi_pef_main,        	PP_IPMI_CMD_PEF },
        { ipmi_lan_main,        	PP_IPMI_CMD_LAN },
        { ipmi_user_main,       	PP_IPMI_CMD_USER },
	{ ipmi_fru_main,	        PP_IPMI_CMD_FRU },
	{ ipmi_raw_main,       		PP_IPMI_CMD_RAW },
#if defined BUILD_UNNEEDED_PARTS
	{ ipmi_event_main,		PP_IPMI_CMD_EVENT },
	{ ipmi_kvm_main,		PP_IPMI_CMD_KVM },
	{ ipmi_session_main,		PP_IPMI_CMD_SESSION },
#endif /* BUILD_UNNEEDED_PARTS */
#ifdef PRODUCT_AMDDC
	{ ipmi_oem_opma_main,		PP_IPMI_CMD_OEM_OPMA },
#endif /* PRODUCT_AMDDC */
#ifdef PP_FEAT_ESB2_TPT
        { ipmi_oem_intel_usr_cfg_main,	PP_IPMI_CMD_OEM_INTEL_USR_CFG },
        { ipmi_oem_intel_tpt_main,	PP_IPMI_CMD_OEM_INTEL_TPT },
#endif /* PP_FEAT_ESB2_TPT */
#ifdef PRODUCT_PDU
        { ipmi_oem_pp_rpc_main, 	PP_IPMI_CMD_OEM_PP_RPC },
#endif /* PRODUCT_PDU */
#ifdef OEM_RACKABLE
	{ ipmi_oem_rackable_main,	PP_IPMI_CMD_OEM_RACKABLE },
#endif /* OEM_RACKABLE */
	{ ipmi_oem_pp_main,		PP_IPMI_CMD_OEM_PP },
	{ ipmi_raritanoem_main,		PP_IPMI_CMD_OEM_RARITAN },
	{ ipmi_selftest_main,		PP_IPMI_CMD_SELF_TEST },
	{ NULL, PP_IPMI_CMD_NONE },
};

/*
 * Run a command from ipmi_cmd_list based on parameters.
 */
int pp_ipmi_cmd_run(struct ipmi_intf * intf, pp_ipmi_command_t cmd, int subcommand, pp_ipmi_parameter_t *params, pp_ipmi_return_t *ipmi_ret, int * error)
{
	struct ipmi_cmd * command;
	int ret;

	ipmi_reset_error(error);

	if (!intf) {
		ipmi_set_error(error, IPMI_ERROR_NOT_CONNECTED);
		return -1;
	}

	// open the interface in case it isn't yet
	if ((ret = pp_ipmi_open_interface(intf, error)) != 0) {
		return ret;
	}
	for (command = ipmi_cmd_list; command->func; command++) {
		if (command->cmd == cmd)
			break;
	}

	if (!command->func) {
		ipmi_printf("Invalid command: %d\n", cmd);
		return -1;
	}

	return command->func(intf, subcommand, params, ipmi_ret, error);
}

int ipmitool_main(ipmitool_config_t *config,
		  pp_ipmi_command_t cmd,
		  int subcommand,
		  pp_ipmi_parameter_t *params,
		  pp_ipmi_return_t *ret, int * error)
{
	struct ipmi_intf * intf = NULL;
	uint8_t privlvl = 0;
	uint8_t target_addr = 0;
	uint8_t my_addr = 0;
	uint8_t authtype = 0;
	char * hostname = NULL;
	char * device = NULL;
	char * username = NULL;
	char * password = NULL;
	char * intfname = NULL;
	char * oemtype = NULL;
	int port = 0;
	int cipher_suite_id = 3; /* See table 22-19 of the IPMIv2 spec */
	int rc = -1;
	
	ipmi_reset_error(error);

	if (config) {
		if (config->intfname) {
			intfname = strdup(config->intfname);
		}
		if (config->privlvl) {
			privlvl = (uint8_t)str2val(config->privlvl, ipmi_privlvl_vals);
			if (!privlvl)
				ipmi_printf("Invalid privilege level %s!\n", config->privlvl);
		}
		if (config->authtype) {
			authtype = (int)str2val(config->authtype, ipmi_authtype_session_vals);
		}
		if (config->verbose) {
			verbose = config->verbose;
		}
		if (config->port) {
			port = config->port;
		}
		if (config->bmc_addr) {
			target_addr = config->bmc_addr;
		}
		if (config->own_addr) {
			my_addr = config->own_addr;
		}
		if (config->hostname) {
			hostname = strdup(config->hostname);
		}
		if (config->device) {
			device = strdup(config->device);
		}
		if (config->password) {
			password = strdup(config->password);
		}
		if (config->username) {
			username = strdup(config->username);
		}
		if (config->oemtype) {
			oemtype = strdup(config->oemtype);
		}
	}

	/* load interface */
	intf = ipmi_intf_load(intfname, error);
	if (!intf) {
	    ipmi_printf("%s(): Error loading interface '%s'\n", ___F, intfname);
	    goto out_free;
	}

	/* run OEM setup if found */
	if (oemtype != NULL &&
	    ipmi_oem_setup(intf, oemtype) < 0) {
		ipmi_printf("%s(): OEM setup for \"%s\" failed", ___F);
		goto out_free;
	}

	/* set session variables */
        intf->loopi_chan = config->loopi_chan;
	if (hostname)
		ipmi_intf_session_set_hostname(intf, hostname);
	if (device)
		ipmi_intf_session_set_device(intf, device);
	if (username)
		ipmi_intf_session_set_username(intf, username);
	if (password)
		ipmi_intf_session_set_password(intf, password);
	if (port)
		ipmi_intf_session_set_port(intf, port);
	if (authtype)
		ipmi_intf_session_set_authtype(intf, authtype);
	if (privlvl)
		ipmi_intf_session_set_privlvl(intf, privlvl);
	else
		ipmi_intf_session_set_privlvl(intf,
		      IPMI_SESSION_PRIV_ADMIN);	/* default */
	ipmi_intf_session_set_cipher_suite_id(intf, cipher_suite_id);
	ipmi_intf_session_set_serial(intf, config->serial_port, config->baud_rate, config->parity,
		config->bits, config->stop2, config->hwf, config->swf);

	/* setup IPMB local and target address if given */
	intf->target_addr = target_addr ? target_addr : IPMI_BMC_SLAVE_ADDR;
	intf->bmc_addr = intf->target_addr;
	intf->my_addr = my_addr ? my_addr : IPMI_REMOTE_SWID;

	/* now we finally run the command */
	rc = pp_ipmi_cmd_run(intf, cmd, subcommand, params, ret, error);

	/* clean repository caches */
	ipmi_cleanup(intf);

	if (intf->opened && intf->close)
		intf->close(intf, 0, error);

 out_free:
	if (intfname)
		free(intfname);
	if (hostname)
		free(hostname);
	if (device)
		free(device);
	if (username)
		free(username);
	if (password)
		free(password);
	if (intf) {
		if (intf->session) {
			/* clean up session allocated by ipmi_{lan,loopi}_setup */
			free(intf->session);
		}
		free(intf);
	}

	return rc;
}

static struct ipmi_intf * pp_ipmi_connect_internal(const char *intf_name,
                                                   const char *hostname, int port,
                                                   const char *device,
						   const char *ppscsi_product,
                                                   const char *username, const char *password,
                                                   const char *oemtype, int * error) {
	struct ipmi_intf * intf;

	ipmi_reset_error(error);

	intf = ipmi_intf_load(intf_name, error);
	if (!intf) {
	    ipmi_printf("%s(): Error loading interface '%s'\n", ___F, intf_name);
	    goto out;
	}

	/* run OEM setup if found */
	if (oemtype != NULL &&
	    ipmi_oem_setup(intf, oemtype) < 0) {
	    
	    ipmi_printf("%s(): OEM setup for \"%s\" failed", ___F, oemtype);
	    free(intf);
	    intf = NULL;
	    goto out;
	}

	/* set session variables */
	if (hostname)
		ipmi_intf_session_set_hostname(intf, hostname);
	if (username)
		ipmi_intf_session_set_username(intf, username);
	if (password)
		ipmi_intf_session_set_password(intf, password);
	if (port)
		ipmi_intf_session_set_port(intf, port);
	if (device)
		ipmi_intf_session_set_device(intf, device);
	if (ppscsi_product)
		ipmi_intf_session_set_ppscsi_product(intf, ppscsi_product);
	ipmi_intf_session_set_privlvl(intf, IPMI_SESSION_PRIV_ADMIN);

out:
	return intf;
}

struct ipmi_intf * pp_ipmi_connect_lan(const char *hostname, int port,
                                       const char *username, const char *password,
                                       const char *oemtype,
                                       int use_lanplus, int * error) {
	struct ipmi_intf * intf = pp_ipmi_connect_internal(use_lanplus ? "lanplus" : "lan",
	    hostname, port, NULL, NULL, username, password, oemtype, error);
	int cipher_suite_id = 3; /* See table 22-19 of the IPMIv2 spec */

	ipmi_reset_error(error);

	if (intf != 0) {
		ipmi_intf_session_set_cipher_suite_id(intf, cipher_suite_id);
	}

	return intf;
}

#if defined (PP_FW_TYPE_KIRATOOL)

struct ipmi_intf * pp_ipmi_connect_ppscsi(const char *device, const char *ppscsi_product,
					  const char *username, const char *password,
					  const char *oemtype, int *error) {
	return pp_ipmi_connect_internal("ppscsi", NULL, 0, device, ppscsi_product,
	    username, password, oemtype, error);
}

#ifdef WIN32
# define LOCAL_INTERFACE    "imb"
#else
# define LOCAL_INTERFACE    "open"
#endif

// TODO: this will fail until local is implemented!
struct ipmi_intf * pp_ipmi_connect_local(const char *username, const char *password,
					 int use_asmi, const char *oemtype, int * error) {
	return pp_ipmi_connect_internal(LOCAL_INTERFACE, NULL, 0,
	    use_asmi ? "\\\\.\\AsmiImb" : "\\\\.\\Imb",
	    NULL, username, password, oemtype, error);
}

#endif /* PP_FW_TYPE_KIRATOOL */

int pp_ipmi_open_interface(struct ipmi_intf * intf, int * error) {
	int ret = 0;

	ipmi_reset_error(error);

	if (!intf->opened && intf->open && intf->open(intf, error) < 0) {
		ret = -1;
		if (intf->close) {
			intf->close(intf, 0, error);
		}
	}

	return ret;
}

void pp_ipmi_disconnect(struct ipmi_intf * intf, int force, int * error) {
	ipmi_reset_error(error);

	/* clean repository caches */
	if (intf) 
		ipmi_cleanup(intf);

	if (intf && intf->opened && intf->close)
		intf->close(intf, force, error);

	if (intf)
		free(intf);
}

void pp_ipmi_abort(struct ipmi_intf * intf, int * error) {
	ipmi_reset_error(error);

	if (intf && intf->opened && intf->do_abort)
		intf->do_abort(intf, error);
}

int pp_ipmi_keepalive(struct ipmi_intf * intf, int * error) {
	int ret = 0;

	ipmi_reset_error(error);

	if (intf && intf->keepalive) {
		ret = intf->keepalive(intf, error);
	}

	return ret;
}

