/*
 * Copyright (c) 2003 Sun Microsystems, Inc.  All Rights Reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistribution of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 * 
 * Redistribution in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * 
 * This software is provided "AS IS," without a warranty of any kind.
 * ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES,
 * INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED.
 * SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL NOT BE LIABLE
 * FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.  IN NO EVENT WILL
 * SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA,
 * OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR
 * PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY OF
 * LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * You acknowledge that this software is not designed or intended for use
 * in the design, construction, operation or maintenance of any nuclear
 * facility.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#ifdef WIN32

#include <winsock2.h>
#include <windows.h>

#else // WIN32

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>

#endif // WIN32

#include <errno.h>
#include <sys/types.h>
#include <signal.h>
#include <setjmp.h>

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/helper.h>
#include <ipmitool/ipmi_constants.h>
#include <ipmitool/ipmi_strings.h>
#include <ipmitool/ipmi_lanp.h>
#include <ipmitool/ipmi_channel.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmi_params.h>
#include <ipmi_return.h>

#ifdef WIN32
#define strncasecmp _strnicmp
#define bzero(buf, size)	memset(buf, 0, size)
#endif

static struct lan_param {
	int cmd;
	int size;
	char desc[24];
	uint8_t * data;
	int data_len;
} ipmi_lan_params[] = {
	{ IPMI_LANP_SET_IN_PROGRESS,	1,	"Set in Progress",		NULL,	0 },
	{ IPMI_LANP_AUTH_TYPE,		1,	"Auth Type Support",		NULL,	0 },
	{ IPMI_LANP_AUTH_TYPE_ENABLE,	5,	"Auth Type Enable",		NULL,	0 },
	{ IPMI_LANP_IP_ADDR,		4,	"IP Address",			NULL,	0 },
	{ IPMI_LANP_IP_ADDR_SRC,	1,	"IP Address Source",		NULL,	0 },
	{ IPMI_LANP_MAC_ADDR,		6,	"MAC Address",			NULL,	0 }, /* 5 */
	{ IPMI_LANP_SUBNET_MASK,	4,	"Subnet Mask",			NULL,	0 },
	{ IPMI_LANP_IP_HEADER,		3,	"IP Header",			NULL,	0 },
	{ IPMI_LANP_PRI_RMCP_PORT,	2,	"Primary RMCP Port",		NULL,	0 },
	{ IPMI_LANP_SEC_RMCP_PORT,	2,	"Secondary RMCP Port",		NULL,	0 },
	{ IPMI_LANP_BMC_ARP,		1,	"BMC ARP Control",		NULL,	0 }, /* 10 */
	{ IPMI_LANP_GRAT_ARP,		1,	"Gratituous ARP Intrvl",	NULL,	0 },
	{ IPMI_LANP_DEF_GATEWAY_IP,	4,	"Default Gateway IP",		NULL,	0 },
	{ IPMI_LANP_DEF_GATEWAY_MAC,	6,	"Default Gateway MAC",		NULL,	0 },
	{ IPMI_LANP_BAK_GATEWAY_IP,	4,	"Backup Gateway IP",		NULL,	0 },
	{ IPMI_LANP_BAK_GATEWAY_MAC,	6,	"Backup Gateway MAC",		NULL,	0 }, /* 15 */
	{ IPMI_LANP_SNMP_STRING,	18,	"SNMP Community String",	NULL,	0 },
	{ IPMI_LANP_NUM_DEST,		1,	"Number of Destinations",	NULL,	0 },
	{ IPMI_LANP_DEST_TYPE,		4,	"Destination Type",		NULL,	0 },
	{ IPMI_LANP_DEST_ADDR,		13,	"Destination Addresses",	NULL,	0 },
	{ IPMI_LANP_VLAN_ID,		2,	"802.1q VLAN ID",		NULL,	0 }, /* 20 */
	{ IPMI_LANP_VLAN_PRIORITY,	1,	"802.1q VLAN Priority",		NULL,	0 },
	{ IPMI_LANP_RMCP_CIPHER_SUPPORT,1,	"RMCP+ Cipher Suite Count",	NULL,	0 },
	{ IPMI_LANP_RMCP_CIPHERS,	16,	"RMCP+ Cipher Suites",		NULL,	0 },
	{ IPMI_LANP_RMCP_PRIV_LEVELS,	9,	"Cipher Suite Priv Max",	NULL,	0 },
	{ IPMI_LANP_OEM_ALERT_STRING,	28,	"OEM Alert String",		NULL,	0 }, /* 25 */
	{ IPMI_LANP_ALERT_RETRY,	1,	"Alert Retry Algorithm",	NULL,	0 },
	{ IPMI_LANP_UTC_OFFSET,		3,	"UTC Offset",			NULL,	0 },
	{ IPMI_LANP_DHCP_SERVER_IP,	4,	"DHCP Server IP",		NULL,	0 },
	{ IPMI_LANP_DHCP_SERVER_MAC,	6,	"DHDP Server MAC",		NULL,	0 }, 
	{ IPMI_LANP_DHCP_ENABLE,	1,	"DHCP Enable",			NULL,	0 }, /* 30 */
	{ IPMI_LANP_CHAN_ACCESS_MODE,	2,	"Channel Access Mode",		NULL,	0 },
	{ -1, 0, "", NULL, 0 }
};

/* get_lan_param - Query BMC for LAN parameter data
 *
 * return pointer to lan_param if successful
 * if parameter not supported then
 *   return pointer to lan_param with
 *   lan_param->data == NULL and lan_param->data_len == 0
 * return NULL on error
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 * @param:   lan parameter id
 */
static struct lan_param *
get_lan_param(struct ipmi_intf * intf, uint8_t chan, int param, int * error)
{
	struct lan_param * p;
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[4];

	p = &ipmi_lan_params[param];
	if (p == NULL)
		return NULL;

	msg_data[0] = chan;
	msg_data[1] = p->cmd;
	msg_data[2] = 0;
	msg_data[3] = 0;

	memset(&req, 0, sizeof(req));
	req.msg.netfn    = IPMI_NETFN_TRANSPORT;
	req.msg.cmd      = IPMI_LAN_GET_CONFIG;
	req.msg.data     = msg_data;
	req.msg.data_len = 4;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Get LAN Parameter command failed\n");
		return NULL;
	}
	if (rsp->ccode == 0x80) {
		/* parameter not supported
		 * return lan_param without data
		 */
		p->data = NULL;
		p->data_len = 0;
		return p;
	}
	if (rsp->ccode == 0xc9) {
		/* parameter out of range
		 * return lan_param without data
		 */
		p->data = NULL;
		p->data_len = 0;
		return p;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Get LAN Parameter command failed: %s\n",
			val2str(rsp->ccode, completion_code_vals));
		return NULL;
	}

	p->data = rsp->data + 1;
	p->data_len = rsp->data_len-1;

	return p;
}

/* set_lan_param_wait - Wait for Set LAN Parameter command to complete
 *
 * On some systems this can take unusually long so we wait for the write
 * to take effect and verify that the data was written successfully
 * before continuing or retrying.
 *
 * returns 0 on success
 * returns -1 on error
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 * @param:   lan parameter id
 * @data:    lan parameter data
 * @len:     length of lan parameter data
 */
static int
set_lan_param_wait(struct ipmi_intf * intf, uint8_t chan,
		   int param, uint8_t * data, int len, int * error)
{
	struct lan_param * p;
	int retry = 10;		/* 10 retries */

	if (verbose > 1) {
		ipmi_printf("Waiting for Set LAN Parameter to complete...\n");
		printbuf(data, len, "SET DATA");
	}

	for (;;) {
		p = get_lan_param(intf, chan, param, error);
		if (p == NULL) {
			sleep(IPMI_LANP_TIMEOUT);
			if (retry-- == 0)
				return -1;
			continue;
		}
		if (verbose > 1)
			printbuf(p->data, p->data_len, "READ DATA");
		if (p->data_len != len) {
			sleep(IPMI_LANP_TIMEOUT);
			if (retry-- == 0) {
				ipmi_printf("Mismatched data lengths: %d != %d\n",
				       p->data_len, len);
				return -1;
			}
			continue;
		}
		if (memcmp(data, p->data, len) != 0) {
			sleep(IPMI_LANP_TIMEOUT);
			if (retry-- == 0) {
				ipmi_printf("LAN Parameter Data does not match!  "
				       "Write may have failed.\n");
				return -1;
			}
			continue;
		}
		break;
	}
	return 0;
}

/* __set_lan_param - Write LAN Parameter data to BMC
 *
 * This function does the actual work of writing the LAN parameter
 * to the BMC and calls set_lan_param_wait() if requested.
 *
 * returns 0 on success
 * returns -1 on error
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 * @param:   lan parameter id
 * @data:    lan parameter data
 * @len:     length of lan parameter data
 * @wait:    whether to wait for write completion
 */
static int
__set_lan_param(struct ipmi_intf * intf, uint8_t chan,
		int param, uint8_t * data, int len, int wait, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t msg_data[32];
	
	if (param < 0)
		return -1;

	msg_data[0] = chan;
	msg_data[1] = param;

	memcpy(&msg_data[2], data, len);
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_TRANSPORT;
	req.msg.cmd = IPMI_LAN_SET_CONFIG;
	req.msg.data = msg_data;
	req.msg.data_len = len+2;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Set Lan Parameter failed\n");
		return -1;
	}
	if ((rsp->ccode > 0) && (wait != 0)) {
		ipmi_printf("Warning: Set Lan Parameter failed: %s\n",
		       val2str(rsp->ccode, completion_code_vals));
		if (rsp->ccode == 0xcc) {
			/* retry hack for invalid data field ccode */
			int retry = 2;		/* # of retries */
			ipmi_printf("Retrying...\n");
			for (;;) {
				if (retry-- == 0)
					break;
				sleep(IPMI_LANP_TIMEOUT);
				rsp = intf->sendrecv(intf, &req, error);
				if (rsp == NULL)
					continue;
				if (rsp->ccode > 0)
					continue;
				return set_lan_param_wait(intf, chan, param, data, len, error);
			}
		}
		else if (rsp->ccode != 0xff) {
			/* let 0xff ccode continue */
			return -1;
		}
	}

	if (wait == 0)
		return 0;
	return set_lan_param_wait(intf, chan, param, data, len, error);
}

/* ipmi_lanp_lock_state - Retrieve set-in-progress status
 *
 * returns one of:
 *  IPMI_LANP_WRITE_UNLOCK
 *  IPMI_LANP_WRITE_LOCK
 *  IPMI_LANP_WRITE_COMMIT
 *  -1 on error/if not supported
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 */
static int
ipmi_lanp_lock_state(struct ipmi_intf * intf, uint8_t chan, int * error)
{
	struct lan_param * p;
	p = get_lan_param(intf, chan, IPMI_LANP_SET_IN_PROGRESS, error);
	if (p == NULL)
		return -1;
	return (p->data[0] & 3);
}

/* ipmi_lanp_lock - Lock set-in-progress bits for our use
 *
 * Write to the Set-In-Progress LAN parameter to indicate
 * to other management software that we are modifying parameters.
 *
 * No meaningful return value because this is an optional
 * requirement in IPMI spec and not found on many BMCs.
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 */
static void
ipmi_lanp_lock(struct ipmi_intf * intf, uint8_t chan, int * error)
{
	uint8_t val = IPMI_LANP_WRITE_LOCK;
	int retry = 3;

	for (;;) {
		int state = ipmi_lanp_lock_state(intf, chan, error);
		if (state == -1)
			break;
		if (state == val)
			break;
		if (retry-- == 0)
			break;
		__set_lan_param(intf, chan, IPMI_LANP_SET_IN_PROGRESS,
				&val, 1, 0, error);
	}
}

/* ipmi_lanp_unlock - Unlock set-in-progress bits
 *
 * Write to the Set-In-Progress LAN parameter, first with
 * a "commit" instruction and then unlocking it.
 *
 * No meaningful return value because this is an optional
 * requirement in IPMI spec and not found on many BMCs.
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 */
static void
ipmi_lanp_unlock(struct ipmi_intf * intf, uint8_t chan, int * error)
{
	uint8_t val = IPMI_LANP_WRITE_COMMIT;
	int rc;

	rc = __set_lan_param(intf, chan, IPMI_LANP_SET_IN_PROGRESS, &val, 1, 0, error);
	if (rc < 0) {
		if (verbose)
			ipmi_printf("LAN Parameter Commit not supported\n");
		val = IPMI_LANP_WRITE_UNLOCK;
		__set_lan_param(intf, chan, IPMI_LANP_SET_IN_PROGRESS, &val, 0, 0, error);
	}
}

/* set_lan_param - Wrap LAN parameter write with set-in-progress lock
 *
 * Returns value from __set_lan_param()
 *
 * @intf:    ipmi interface handle
 * @chan:    ipmi channel
 * @param:   lan parameter id
 * @data:    lan parameter data
 * @len:     length of lan parameter data
 */
static int
set_lan_param(struct ipmi_intf * intf, uint8_t chan,
	      int param, uint8_t * data, int len, int * error)
{
	int rc;
	ipmi_lanp_lock(intf, chan, error);
	rc = __set_lan_param(intf, chan, param, data, len, 1, error);
	ipmi_lanp_unlock(intf, chan, error);
	return rc;
}


static int
lan_set_arp_interval(struct ipmi_intf * intf,
		     uint8_t chan, uint8_t * ival, int * error)
{
	struct lan_param *lp;
	uint8_t interval;
	int rc = 0;
	
	lp = get_lan_param(intf, chan, IPMI_LANP_GRAT_ARP, error);
	if (lp == NULL)
		return -1;
	if (lp->data == NULL)
		return -1;

	if (ival != 0) {
		interval = ((uint8_t)atoi((char *)ival) * 2) - 1;
		rc = set_lan_param(intf, chan, IPMI_LANP_GRAT_ARP, &interval, 1, error);
	} else {
		interval = lp->data[0];
	}

	ipmi_printf("BMC-generated Gratuitous ARP interval:  %.1f seconds\n",
	       (float)((interval + 1) / 2));

	return rc;
}

static int
lan_set_arp_generate(struct ipmi_intf * intf,
		     uint8_t chan, uint8_t ctl, int * error)
{
	struct lan_param *lp;
	uint8_t data;

	lp = get_lan_param(intf, chan, IPMI_LANP_BMC_ARP, error);
	if (lp == NULL)
		return -1;
	if (lp->data == NULL)
		return -1;
	data = lp->data[0];

	/* set arp generate bitflag */
	if (ctl == 0)
		data &= ~0x1;
	else
		data |= 0x1;

	ipmi_printf("%sabling BMC-generated Gratuitous ARPs\n", ctl ? "En" : "Dis");
	return set_lan_param(intf, chan, IPMI_LANP_BMC_ARP, &data, 1, error);
}

static int
lan_set_arp_respond(struct ipmi_intf * intf,
		    uint8_t chan, uint8_t ctl, int * error)
{
	struct lan_param *lp;
	uint8_t data;

	lp = get_lan_param(intf, chan, IPMI_LANP_BMC_ARP, error);
	if (lp == NULL)
		return -1;
	if (lp->data == NULL)
		return -1;
	data = lp->data[0];

	/* set arp response bitflag */
	if (ctl == 0)
		data &= ~0x2;
	else
		data |= 0x2;

	ipmi_printf("%sabling BMC-generated ARP responses\n", ctl ? "En" : "Dis");
	return set_lan_param(intf, chan, IPMI_LANP_BMC_ARP, &data, 1, error);
}

pp_ipmi_priv_levels priv_level_to_enum(uint8_t priv_level) {
    switch (priv_level) {
	case IPMI_SESSION_PRIV_CALLBACK:
	    return PP_IPMI_PRIV_CALLBACK;
	case IPMI_SESSION_PRIV_USER:
	    return PP_IPMI_PRIV_USER;
	case IPMI_SESSION_PRIV_OPERATOR:
	    return PP_IPMI_PRIV_OPERATOR;
	case IPMI_SESSION_PRIV_ADMIN:
	    return PP_IPMI_PRIV_ADMIN;
	case IPMI_SESSION_PRIV_OEM:
	    return PP_IPMI_PRIV_OEM;
	case IPMI_SESSION_PRIV_UNSPECIFIED:
	    return PP_IPMI_PRIV_UNSPECIFIED;
	case 0xF:
	    return PP_IPMI_PRIV_NO_ACCESS;
    }

    return PP_IPMI_PRIV_UNSPECIFIED;
}

static void ipmi_get_in_progress(uint8_t val, pp_ipmi_return_t* ipmi_ret) {
    ipmi_ret->data.lanp_get.set_in_progress.valid = 1;
    val &= 3;
    switch (val) {
	case 0:
	    ipmi_ret->data.lanp_get.set_in_progress.val = PP_IPMI_LANP_SET_COMPLETE;
	    ipmi_ret->data.lanp_get.set_in_progress.string = "Set Complete";
	    break;
	case 1:
	    ipmi_ret->data.lanp_get.set_in_progress.val = PP_IPMI_LANP_SET_IN_PROGRESS;
	    ipmi_ret->data.lanp_get.set_in_progress.string = "Set In Progress";
	    break;
	case 2:
	    ipmi_ret->data.lanp_get.set_in_progress.val = PP_IPMI_LANP_SET_COMMIT_WRITE;
	    ipmi_ret->data.lanp_get.set_in_progress.string = "Commit Write";
	    break;
	case 3:
	    ipmi_ret->data.lanp_get.set_in_progress.val = PP_IPMI_LANP_SET_RESERVED;
	    ipmi_ret->data.lanp_get.set_in_progress.string = "Reserved";
	    break;
	default:
	    ipmi_ret->data.lanp_get.set_in_progress.val = PP_IPMI_LANP_SET_UNKNOWN;
	    ipmi_ret->data.lanp_get.set_in_progress.string = "Unknown";
    }
}

static void ipmi_get_auth_type(uint8_t val, pp_ipmi_lanp_auth_types *types) {
   types->none = (val & 1<<IPMI_SESSION_AUTHTYPE_NONE) ? 1 : 0;
   types->md2 = (val & 1<<IPMI_SESSION_AUTHTYPE_MD2) ? 1 : 0;
   types->md5 = (val & 1<<IPMI_SESSION_AUTHTYPE_MD5) ? 1 : 0;
   types->password = (val & 1<<IPMI_SESSION_AUTHTYPE_PASSWORD) ? 1 : 0;
   types->oem = (val & 1<<IPMI_SESSION_AUTHTYPE_OEM) ? 1 : 0;
}

static void ipmi_get_ip_source(uint8_t val, pp_ipmi_return_t* ipmi_ret) {
    ipmi_ret->data.lanp_get.ip_addr_source.valid = 1;
    val &= 0xf;
    switch (val) {
	case 0:
	    ipmi_ret->data.lanp_get.ip_addr_source.val = PP_IPMI_LANP_IP_ADDR_SOURCE_UNSPECIFIED;
	    ipmi_ret->data.lanp_get.ip_addr_source.string = "Unspecified";
	    break;
	case 1:
	    ipmi_ret->data.lanp_get.ip_addr_source.val = PP_IPMI_LANP_IP_ADDR_SOURCE_STATIC;
	    ipmi_ret->data.lanp_get.ip_addr_source.string = "Static Address";
	    break;
	case 2:
	    ipmi_ret->data.lanp_get.ip_addr_source.val = PP_IPMI_LANP_IP_ADDR_SOURCE_DHCP;
	    ipmi_ret->data.lanp_get.ip_addr_source.string = "DHCP Address";
	    break;
	case 3:
	    ipmi_ret->data.lanp_get.ip_addr_source.val = PP_IPMI_LANP_IP_ADDR_SOURCE_BIOS_ASSIGNED;
	    ipmi_ret->data.lanp_get.ip_addr_source.string = "BIOS Assigned Address";
	    break;
	default:
	    ipmi_ret->data.lanp_get.ip_addr_source.val = PP_IPMI_LANP_IP_ADDR_SOURCE_OTHER;
	    ipmi_ret->data.lanp_get.ip_addr_source.string = "Other";
	    break;
    }
}

static void ipmi_get_ip_addr(uint8_t *source, pp_ipmi_lanp_ip_or_mac_address *ip) {
    ip->valid = 1;
    pp_strappendf(&ip->address, "%d.%d.%d.%d",
	source[0], source[1], source[2], source[3]);
}

static void ipmi_get_mac_addr(uint8_t *source, pp_ipmi_lanp_ip_or_mac_address *mac) {
    mac->valid = 1;
    pp_strappendf(&mac->address, "%02x:%02x:%02x:%02x:%02x:%02x",
	source[0], source[1], source[2], source[3], source[4], source[5]);
}

static void ipmi_get_ip_header(uint8_t *source, pp_ipmi_return_t *ret) {
    ret->data.lanp_get.ip_header.valid = 1;
    ret->data.lanp_get.ip_header.ttl = source[0];
    ret->data.lanp_get.ip_header.flags = source[1];
    ret->data.lanp_get.ip_header.precedence = source[2] & 0xe0;
    ret->data.lanp_get.ip_header.tos = source[2] & 0x1e;
}

static void ipmi_get_arp(uint8_t val, pp_ipmi_return_t *ret) {
    ret->data.lanp_get.arp_control.valid = 1;
    ret->data.lanp_get.arp_control.arp_responses_enabled = (val & 2) ? 1 : 0;
    ret->data.lanp_get.arp_control.gratuitous_arp_enabled = (val & 1) ? 1 : 0;
}

static int
ipmi_lan_get(struct ipmi_intf * intf, pp_ipmi_lanp_get_param_t *params, pp_ipmi_return_t* ipmi_ret, int * error) {
    struct lan_param * p;
    uint8_t medium;
    int rc = 0;
    u_char chan = params->chan;

    if (chan < 1 || chan > IPMI_CHANNEL_NUMBER_MAX) {
	ipmi_printf("Invalid Channel %d\n", chan);
	ipmi_set_error(error, IPMI_ERROR_INVALID_CHANNEL);
	return -1;
    }

    /* find type of channel and only accept 802.3 LAN */
    medium = ipmi_get_channel_medium(intf, chan, error);
    if (medium != IPMI_CHANNEL_MEDIUM_LAN &&
	medium != IPMI_CHANNEL_MEDIUM_LAN_OTHER) {
	    ipmi_printf("Channel %d (%s) is not a LAN channel\n",
		chan, val2str(medium, ipmi_channel_medium_vals), medium);
	    ipmi_set_error(error, IPMI_ERROR_CHANNEL_NOT_LAN_CHANNEL);
	    return -1;
    }

    if (params->need_set_in_progress) {
	p = get_lan_param(intf, chan, IPMI_LANP_SET_IN_PROGRESS, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_get_in_progress(p->data[0], ipmi_ret);
	}
    }

    if (params->need_auth_types) {
	p = get_lan_param(intf, chan, IPMI_LANP_AUTH_TYPE, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_ret->data.lanp_get.auth_type_support.valid = 1;
	    ipmi_get_auth_type(p->data[0], &ipmi_ret->data.lanp_get.auth_type_support.auth_types);
	}

	p = get_lan_param(intf, chan, IPMI_LANP_AUTH_TYPE_ENABLE, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_ret->data.lanp_get.auth_type_enables.valid = 1;
	    ipmi_get_auth_type(p->data[0], &ipmi_ret->data.lanp_get.auth_type_enables.callback);
	    ipmi_get_auth_type(p->data[1], &ipmi_ret->data.lanp_get.auth_type_enables.user);
	    ipmi_get_auth_type(p->data[2], &ipmi_ret->data.lanp_get.auth_type_enables._operator);
	    ipmi_get_auth_type(p->data[3], &ipmi_ret->data.lanp_get.auth_type_enables.admin);
	    ipmi_get_auth_type(p->data[4], &ipmi_ret->data.lanp_get.auth_type_enables.oem);
	}
    }

    if (params->need_ip_addr_source) {
	p = get_lan_param(intf, chan, IPMI_LANP_IP_ADDR_SRC, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_get_ip_source(p->data[0], ipmi_ret);
	}
    }

    if (params->need_ip_addr) {
	p = get_lan_param(intf, chan, IPMI_LANP_IP_ADDR, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_ip_addr(p->data, &ipmi_ret->data.lanp_get.ip_address);
    }

    if (params->need_subnet_mask) {
	p = get_lan_param(intf, chan, IPMI_LANP_SUBNET_MASK, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_ip_addr(p->data, &ipmi_ret->data.lanp_get.subnet_mask);
    }

    if (params->need_mac_addr) {
	p = get_lan_param(intf, chan, IPMI_LANP_MAC_ADDR, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_mac_addr(p->data, &ipmi_ret->data.lanp_get.mac_address);
    }

    if (params->need_snmp_string) {
	p = get_lan_param(intf, chan, IPMI_LANP_SNMP_STRING, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_ret->data.lanp_get.snmp_string.valid = 1;
	    pp_strappendf(&ipmi_ret->data.lanp_get.snmp_string.string, "%s", p->data);
	}
    }

    if (params->need_ip_header) {
	p = get_lan_param(intf, chan, IPMI_LANP_IP_HEADER, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_ip_header(p->data, ipmi_ret);
    }

    if (params->need_arp_config) {
	p = get_lan_param(intf, chan, IPMI_LANP_BMC_ARP, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_arp(p->data[0], ipmi_ret);

	p = get_lan_param(intf, chan, IPMI_LANP_GRAT_ARP, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL) {
	    ipmi_ret->data.lanp_get.gratuitous_arp_timeout.valid = 1;
	    ipmi_ret->data.lanp_get.gratuitous_arp_timeout.timeout = (float)((p->data[0] + 1) / 2);
	}
    }

    if (params->need_def_gw_ip) {
	p = get_lan_param(intf, chan, IPMI_LANP_DEF_GATEWAY_IP, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_ip_addr(p->data, &ipmi_ret->data.lanp_get.def_gw_ip);
    }

    if (params->need_def_gw_mac) {
	p = get_lan_param(intf, chan, IPMI_LANP_DEF_GATEWAY_MAC, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_mac_addr(p->data, &ipmi_ret->data.lanp_get.def_gw_mac);
    }

    if (params->need_bak_gw_ip) {
	p = get_lan_param(intf, chan, IPMI_LANP_BAK_GATEWAY_IP, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_ip_addr(p->data, &ipmi_ret->data.lanp_get.bak_gw_ip);
    }

    if (params->need_bak_gw_mac) {
	p = get_lan_param(intf, chan, IPMI_LANP_BAK_GATEWAY_MAC, error);
	if (p == NULL)
	    return -1;
	if (p->data != NULL)
	    ipmi_get_mac_addr(p->data, &ipmi_ret->data.lanp_get.bak_gw_mac);
    }

    /* Determine supported Cipher Suites -- Requires two calls */
    if (params->need_cipher_suites) {
	p = get_lan_param(intf, chan, IPMI_LANP_RMCP_CIPHER_SUPPORT, error);
	if (p == NULL)
	    return -1;
	else if (p->data != NULL) {
	    p = get_lan_param(intf, chan, IPMI_LANP_RMCP_CIPHERS, error);
	    if (p == NULL)
		return -1;
	    ipmi_ret->data.lanp_get.rcmp_cipher_suites.valid = 1;

	    /* Now we're dangerous.  There are only 15 fixed cipher
	    suite IDs, but the spec allows for 16 in the return data.*/
	    if ((p->data != NULL) && (p->data_len <= 16)) {
		uint8_t cipher_suite_count = p->data[0];
		uint32_t i;
		for (i = 0; (i < 16) && (i < cipher_suite_count); ++i) {
		    vector_add_int(
			ipmi_ret->data.lanp_get.rcmp_cipher_suites.ciphers,
			p->data[i + 1]);
		}
	    }
	}
    }


    /* RMCP+ Messaging Cipher Suite Privilege Levels */
    /* These are the privilege levels for the 15 fixed cipher suites */
    if (params->need_cipher_priv_max) {
	p = get_lan_param(intf, chan, IPMI_LANP_RMCP_PRIV_LEVELS, error);
	if (p == NULL)
	    return -1;
	if ((p->data != NULL) && (p->data_len == 9)) {
	    ipmi_ret->data.lanp_get.cipher_priv_max.valid = 1;

	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 0] = priv_level_to_enum(p->data[1] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 1] = priv_level_to_enum(p->data[1] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 2] = priv_level_to_enum(p->data[2] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 3] = priv_level_to_enum(p->data[2] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 4] = priv_level_to_enum(p->data[3] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 5] = priv_level_to_enum(p->data[3] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 6] = priv_level_to_enum(p->data[4] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 7] = priv_level_to_enum(p->data[4] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 8] = priv_level_to_enum(p->data[5] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[ 9] = priv_level_to_enum(p->data[5] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[10] = priv_level_to_enum(p->data[6] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[11] = priv_level_to_enum(p->data[6] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[12] = priv_level_to_enum(p->data[7] & 0x0F);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[13] = priv_level_to_enum(p->data[7] >> 4);
	    ipmi_ret->data.lanp_get.cipher_priv_max.levels[14] = priv_level_to_enum(p->data[8] & 0x0F);
	}
    }

    return rc;
}

/* Configure Authentication Types */
static int
ipmi_lan_set_auth(struct ipmi_intf * intf, uint8_t chan, char * level, char * types, int * error)
{
	uint8_t data[5];
	uint8_t authtype = 0;
	char * p;
	struct lan_param * lp;

	if (level == NULL || types == NULL)
		return -1;

	lp = get_lan_param(intf, chan, IPMI_LANP_AUTH_TYPE_ENABLE, error);
	if (lp == NULL)
		return -1;
	if (lp->data == NULL)
		return -1;

	if (verbose > 1)
		ipmi_printf("%-24s: callback=0x%02x user=0x%02x operator=0x%02x admin=0x%02x oem=0x%02x\n",
  		       lp->desc, lp->data[0], lp->data[1], lp->data[2], lp->data[3], lp->data[4]);
	memset(data, 0, 5);
	memcpy(data, lp->data, 5);

	p = types;
	while (p) {
		if (strncmp(p, "none", 4) == 0)
			authtype |= 1 << IPMI_SESSION_AUTHTYPE_NONE;
		else if (strncmp(p, "md2", 3) == 0)
			authtype |= 1 << IPMI_SESSION_AUTHTYPE_MD2;
		else if (strncmp(p, "md5", 3) == 0)
			authtype |= 1 << IPMI_SESSION_AUTHTYPE_MD5;
		else if (strncmp(p, "key", 3) == 0 ||
			 (strncasecmp(p, "key", 3) == 0))
			authtype |= 1 << IPMI_SESSION_AUTHTYPE_KEY;
		else if (strncasecmp(p, "oem", 3) == 0)
			authtype |= 1 << IPMI_SESSION_AUTHTYPE_OEM;
		else
			ipmi_printf("invalid authtype: %s\n", p);
		p = strchr(p, ',');
		if (p)
			p++;
	}

	p = level;
	while (p) {
		if (strncmp(p, "callback", 8) == 0)
			data[0] = authtype;
		else if (strncmp(p, "user", 4) == 0)
			data[1] = authtype;
		else if (strncmp(p, "operator", 8) == 0)
			data[2] = authtype;
		else if (strncmp(p, "admin", 5) == 0)
			data[3] = authtype;
		else 
			ipmi_printf("Invalid authentication level: %s\n", p);
		p = strchr(p, ',');
		if (p)
			p++;
	}

	if (verbose > 1)
		printbuf(data, 5, "authtype data");

	return set_lan_param(intf, chan, IPMI_LANP_AUTH_TYPE_ENABLE, data, 5, error);
}

static int
ipmi_lan_set_password(struct ipmi_intf * intf,
		      uint8_t userid, uint8_t * password, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t data[18];

	memset(&data, 0, sizeof(data));
	data[0] = userid & 0x3f;/* user ID */
	data[1] = 0x02;		/* set password */

	if (password != NULL)
		memcpy(data+2, password, __min(strlen((char *)password), 16));

	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 0x47;
	req.msg.data = data;
	req.msg.data_len = 18;

	rsp = intf->sendrecv(intf, &req, error);

	if (rsp == NULL) {
		ipmi_printf("Unable to Set LAN Password for user %d\n", userid);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set LAN Password for user %d failed: %s\n",
			userid, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	/* adjust our session password
	 * or we will no longer be able to communicate with BMC
	 */
	ipmi_intf_session_set_password(intf, (char *)password);
	ipmi_printf("Password %s for user %d\n",
	       (password == NULL) ? "cleared" : "set", userid);

	return 0;
}

static int
ipmi_set_channel_access(struct ipmi_intf * intf, uint8_t channel, uint8_t enable, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[3];

	memset(&req, 0, sizeof(req));
	
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 0x40;
	req.msg.data = rqdata;
	req.msg.data_len = 3;

	/* SAVE TO NVRAM */
	memset(rqdata, 0, 3);
	rqdata[0] = channel & 0xf;
	rqdata[1] = 0x60;	/* set pef disabled, per-msg auth enabled */
	if (enable != 0)
		rqdata[1] |= 0x2; /* set always available if enable is set */
	rqdata[2] = 0x44; 	/* set channel privilege limit to ADMIN */

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Set Channel Access for channel %d\n", channel);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set Channel Access for channel %d failed: %s\n",
			channel, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	/* SAVE TO CURRENT */
	memset(rqdata, 0, 3);
	rqdata[0] = channel & 0xf;
	rqdata[1] = 0xa0;	/* set pef disabled, per-msg auth enabled */
	if (enable != 0)
		rqdata[1] |= 0x2; /* set always available if enable is set */
	rqdata[2] = 0x84; 	/* set channel privilege limit to ADMIN */
  
	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Set Channel Access for channel %d\n", channel);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set Channel Access for channel %d failed: %s\n",
			channel, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	/* can't send close session if access off so abort instead */
	if (enable == 0)
		intf->abort = 1;

	return 0;
}

static int
ipmi_set_user_access(struct ipmi_intf * intf, uint8_t channel, uint8_t userid, int * error)
{
	struct ipmi_rs * rsp;
	struct ipmi_rq req;
	uint8_t rqdata[4];

	memset(rqdata, 0, sizeof(rqdata));
	rqdata[0] = 0x90 | (channel & 0xf);
	rqdata[1] = userid & 0x3f;
	rqdata[2] = 0x4;
	rqdata[3] = 0;
	
	memset(&req, 0, sizeof(req));
	req.msg.netfn = IPMI_NETFN_APP;
	req.msg.cmd = 0x43;
	req.msg.data = rqdata;
	req.msg.data_len = 4;

	rsp = intf->sendrecv(intf, &req, error);
	if (rsp == NULL) {
		ipmi_printf("Unable to Set User Access for channel %d\n", channel);
		return -1;
	}
	if (rsp->ccode > 0) {
		ipmi_printf("Set User Access for channel %d failed: %s\n",
			channel, val2str(rsp->ccode, completion_code_vals));
		return -1;
	}

	return 0;
}


static int
get_cmdline_cipher_suite_priv_data(char * arg, uint8_t * buf)
{
	int i, ret = 0;

	if (strlen(arg) != 15)
	{
		ipmi_printf("Invalid privilege specification length: %d\n",
			strlen(arg));
		return -1;
	}

	/*
	 * The first byte is reservered (0).  The resst of the buffer is setup
	 * so that each nibble holds the maximum privilege level available for
	 * that cipher suite number.  The number of nibbles (15) matches the number
	 * of fixed cipher suite IDs.  This command documentation mentions 16 IDs
	 * but table 22-19 shows that there are only 15 (0-14).
	 *
	 * data 1 - reserved
	 * data 2 - maximum priv level for first (LSN) and second (MSN) ciphers
	 * data 3 - maximum priv level for third (LSN) and fourth (MSN) ciphers
	 * data 9 - maximum priv level for 15th (LSN) cipher.
	 */
	bzero(buf, 9);

	for (i = 0; i < 15; ++i)
	{
		uint8_t priv_level = 0;

		switch (arg[i])
		{
		case 'X':
			priv_level = IPMI_SESSION_PRIV_UNSPECIFIED; /* 0 */
			break;
		case 'c':
			priv_level = IPMI_SESSION_PRIV_CALLBACK;    /* 1 */
			break;
		case 'u':
			priv_level = IPMI_SESSION_PRIV_USER;        /* 2 */
			break;
		case 'o':
			priv_level = IPMI_SESSION_PRIV_OPERATOR;    /* 3 */
			break;
		case 'a':
			priv_level = IPMI_SESSION_PRIV_ADMIN;       /* 4 */
			break;
		case 'O':
			priv_level = IPMI_SESSION_PRIV_OEM;         /* 5 */
			break;
		default:
			ipmi_printf("Invalid privilege specification char: %c\n",
				arg[i]);
			ret = -1;
			break;
		}
		
		if (ret != 0)
			break;
		else
		{
			if ((i + 1) % 2)
			{
				// Odd number cipher suites will be in the LSN
				buf[1 + (i / 2)] += priv_level;
			}
			else
			{
				// Even number cipher suites will be in the MSN
				buf[1 + (i / 2)] += (priv_level << 4);
			}
		}
	}
	
	return ret;
}


static void ipmi_lan_set_usage(void)
{
	ipmi_printf("\nusage: lan set <channel> <command> [option]\n\n");
	ipmi_printf("LAN set commands:\n");
	ipmi_printf("  password <password>            Set session password for this channel\n");
	ipmi_printf("  user                           Enable default user for this channel\n");
	ipmi_printf("  access <on|off>                Enable or disable access to this channel\n");
	ipmi_printf("  arp response <on|off>          Enable or disable BMC ARP responding\n");
	ipmi_printf("  arp generate <on|off>          Enable or disable BMC gratuitous ARP generation\n");
	ipmi_printf("  arp interval <seconds>         Set gratuitous ARP generation interval\n");
	ipmi_printf("  auth <level> <type,..>         Set channel authentication types\n");
	ipmi_printf("    level  = callback, user, operator, admin\n");
	ipmi_printf("    type   = none, md2, md5, key\n");
	ipmi_printf("  cipher_privs XXXXXXXXXXXXXXX   Set RMCP+ cipher suite privilege levels\n");
	ipmi_printf("    X = Cipher Suite Unused\n");
	ipmi_printf("    c = CALLBACK\n");
	ipmi_printf("    u = USER\n");
	ipmi_printf("    o = OPERATOR\n");
	ipmi_printf("    a = ADMIN\n");
	ipmi_printf("    O = OEM\n");
	ipmi_printf("\n");
}

static int
ipmi_lan_set(struct ipmi_intf * intf, int argc, char ** argv, int * error)
{
	uint8_t data[32];
	uint8_t chan, medium;
	int rc = 0;

	if (argc < 2) {
		ipmi_lan_set_usage();
		return 0;
	}

	chan = (uint8_t)strtol(argv[0], NULL, 0);

	if (chan < 1 || chan > IPMI_CHANNEL_NUMBER_MAX) {
		ipmi_printf("Invalid Channel %d\n", chan);
		ipmi_lan_set_usage();
		return -1;
	}

	/* find type of channel and only accept 802.3 LAN */
	medium = ipmi_get_channel_medium(intf, chan, error);
	if (medium != IPMI_CHANNEL_MEDIUM_LAN &&
	    medium != IPMI_CHANNEL_MEDIUM_LAN_OTHER) {
		ipmi_printf("Channel %d is not a LAN channel!\n", chan);
		return -1;
	}

	memset(&data, 0, sizeof(data));

	/* set user access */
	if (strncmp(argv[1], "user", 4) == 0) {
		rc = ipmi_set_user_access(intf, chan, 1, error);
	}
	/* set channel access mode */
	else if (strncmp(argv[1], "access", 6) == 0) {
		if (argc < 3 || (strncmp(argv[2], "help", 4) == 0)) {
			ipmi_printf("lan set access <on|off>\n");
		}
		else if (strncmp(argv[2], "on", 2) == 0) {
			rc = ipmi_set_channel_access(intf, chan, 1, error);
		}
		else if (strncmp(argv[2], "off", 3) == 0) {
			rc = ipmi_set_channel_access(intf, chan, 0, error);
		}
		else {
			ipmi_printf("lan set access <on|off>\n");
		}
	}
	/* set ARP control */
	else if (strncmp(argv[1], "arp", 3) == 0) {
		if (argc < 3 || (strncmp(argv[2], "help", 4) == 0)) {
			ipmi_printf("lan set <channel> arp respond <on|off>\n");
			ipmi_printf("lan set <channel> arp generate <on|off>\n");
			ipmi_printf("lan set <channel> arp interval <seconds>\n\n");
			ipmi_printf("example: lan set 7 arp gratuitous off\n");
		}
		else if (strncmp(argv[2], "interval", 8) == 0) {
			rc = lan_set_arp_interval(intf, chan, (uint8_t *)argv[3], error);
		}
		else if (strncmp(argv[2], "generate", 8) == 0) {
			if (argc < 4)
				ipmi_printf("lan set <channel> arp generate <on|off>\n");
			else if (strncmp(argv[3], "on", 2) == 0)
				rc = lan_set_arp_generate(intf, chan, 1, error);
			else if (strncmp(argv[3], "off", 3) == 0)
				rc = lan_set_arp_generate(intf, chan, 0, error);
			else
				ipmi_printf("lan set <channel> arp generate <on|off>\n");
		}
		else if (strncmp(argv[2], "respond", 7) == 0) {
			if (argc < 4)
				ipmi_printf("lan set <channel> arp respond <on|off>\n");
			else if (strncmp(argv[3], "on", 2) == 0)
				rc = lan_set_arp_respond(intf, chan, 1, error);
			else if (strncmp(argv[3], "off", 3) == 0)
				rc = lan_set_arp_respond(intf, chan, 0, error);
			else
				ipmi_printf("lan set <channel> arp respond <on|off>\n");
		}
		else {
			ipmi_printf("lan set <channel> arp respond <on|off>\n");
			ipmi_printf("lan set <channel> arp generate <on|off>\n");
			ipmi_printf("lan set <channel> arp interval <seconds>\n");
		}
	}
	/* set authentication types */
	else if (strncmp(argv[1], "auth", 4) == 0) {
		if (argc < 3 || (strncmp(argv[2], "help", 4) == 0)) {
			ipmi_printf("lan set <channel> auth <level> <type,type,...>\n");
			ipmi_printf("  level = callback, user, operator, admin\n");
			ipmi_printf("  types = none, md2, md5, key\n");
			ipmi_printf("example: lan set 7 auth admin key,md5\n");
		} else {
			rc = ipmi_lan_set_auth(intf, chan, argv[2], argv[3], error);
		}
	}
	/* session password
	 * not strictly a lan setting, but its used for lan connections */
	else if (strncmp(argv[1], "password", 8) == 0) {
		rc = ipmi_lan_set_password(intf, 1, (uint8_t *)argv[2], error);
	}
	/* RMCP+ cipher suite privilege levels */
	else if (strncmp(argv[1], "cipher_privs", 12) == 0)
	{
		if ((argc != 3)                        ||
		    (strncmp(argv[2], "help", 4) == 0) ||
		    get_cmdline_cipher_suite_priv_data(argv[2], data))
		{
			ipmi_printf("lan set <channel> cipher_privs XXXXXXXXXXXXXXX\n");
			ipmi_printf("    X = Cipher Suite Unused\n");
			ipmi_printf("    c = CALLBACK\n");
			ipmi_printf("    u = USER\n");
			ipmi_printf("    o = OPERATOR\n");
			ipmi_printf("    a = ADMIN\n");
			ipmi_printf("    O = OEM\n");
		}
		else
		{
			rc = set_lan_param(intf, chan, IPMI_LANP_RMCP_PRIV_LEVELS, data, 9, error);
		}
	}
	else {
		ipmi_printf("Unknown lan set command or parse error.\n");
	}
	
	return rc;
}

static int
ipmi_set_ip_addr_source(struct ipmi_intf * intf, uint8_t chan,
			pp_ipmi_lanp_ip_addr_source source,
			int *error) {
    u_char data;

    switch (source) {
	case PP_IPMI_LANP_IP_ADDR_SOURCE_UNSPECIFIED:
	    data = 0;
	    break;
	case PP_IPMI_LANP_IP_ADDR_SOURCE_STATIC:
	    data = 1;
	    break;
	case PP_IPMI_LANP_IP_ADDR_SOURCE_DHCP:
	    data = 2;
	    break;
	case PP_IPMI_LANP_IP_ADDR_SOURCE_BIOS_ASSIGNED:
	    data = 3;
	    break;
	default:
	    ipmi_printf("Error: wrong IP source specified.\n");
	    ipmi_set_error(error, IPMI_ERROR_INVALID_PARAMETER);
	    return -1;
    }

    return set_lan_param(intf, chan, IPMI_LANP_IP_ADDR_SRC, &data, 1, error);
}

static int
ipmi_set_snmp_string(struct ipmi_intf *intf, uint8_t chan, char *snmp, int *error) {
    uint8_t data[18];

    memset(data, 0, sizeof(data));
    memcpy(data, snmp, min(strlen(snmp), (size_t)18));
    return set_lan_param(intf, chan, IPMI_LANP_SNMP_STRING, data, 18, error);
}

static int
ipmi_set_ip_addr_internal(struct ipmi_intf *intf, uint8_t chan, char *ip, uint8_t cmd, int *error) {
    uint8_t data[4];

    if (get_cmdline_ipaddr(ip, data) != 0) {
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_PARSE_IP_ADDR);
	return -1;
    }

    return set_lan_param(intf, chan, cmd, data, 4, error);
}

static int
ipmi_set_ip_addr(struct ipmi_intf *intf, uint8_t chan, char *ip, int *error) {
    return ipmi_set_ip_addr_internal(intf, chan, ip, IPMI_LANP_IP_ADDR, error);
}

static int
ipmi_set_netmask(struct ipmi_intf *intf, uint8_t chan, char *ip, int *error) {
    return ipmi_set_ip_addr_internal(intf, chan, ip, IPMI_LANP_SUBNET_MASK, error);
}

static int
ipmi_set_def_gw_ip(struct ipmi_intf *intf, uint8_t chan, char *ip, int *error) {
    return ipmi_set_ip_addr_internal(intf, chan, ip, IPMI_LANP_DEF_GATEWAY_IP, error);
}

static int
ipmi_set_bak_gw_ip(struct ipmi_intf *intf, uint8_t chan, char *ip, int *error) {
    return ipmi_set_ip_addr_internal(intf, chan, ip, IPMI_LANP_BAK_GATEWAY_IP, error);
}

static int
ipmi_set_mac_addr_internal(struct ipmi_intf *intf, uint8_t chan, char *mac, uint8_t cmd, int *error) {
    uint8_t data[6];

    if (get_cmdline_macaddr(mac, data) != 0) {
	ipmi_set_error(error, IPMI_ERROR_COULD_NOT_PARSE_MAC_ADDR);
	return -1;
    }

    return set_lan_param(intf, chan, cmd, data, 6, error);
}

static int
ipmi_set_mac_addr(struct ipmi_intf *intf, uint8_t chan, char *mac, int *error) {
    return ipmi_set_mac_addr_internal(intf, chan, mac, IPMI_LANP_MAC_ADDR, error);
}

static int
ipmi_set_def_gw_mac(struct ipmi_intf *intf, uint8_t chan, char *mac, int *error) {
    return ipmi_set_mac_addr_internal(intf, chan, mac, IPMI_LANP_DEF_GATEWAY_MAC, error);
}

static int
ipmi_set_bak_gw_mac(struct ipmi_intf *intf, uint8_t chan, char *mac, int *error) {
    return ipmi_set_mac_addr_internal(intf, chan, mac, IPMI_LANP_BAK_GATEWAY_MAC, error);
}


/*
 *
 */

static int
ipmi_get_dest_count(struct ipmi_intf * intf, uint8_t chan,
                    uint8_t* count, int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[4];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 17; /* dest count */
    rqdata[2] = 0;
    rqdata[3] = 0;

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x02; /* Get LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode || rsp->data_len < 2) {
        ipmi_printf("Error %x: Get LAN Config Params | Dest Count\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    if (count) *count = rsp->data[3] & 0x07;
    return 0;
}

static int
ipmi_get_dest_type(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                   uint8_t* acked, uint8_t* type,
                   uint8_t* timeout, uint8_t* retries,
                   int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[4];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 18; /* dest type */
    rqdata[2] = set;
    rqdata[3] = 0;

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x02; /* Get LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode || rsp->data_len < 5) {
        ipmi_printf("Error %x: Get LAN Config Params | Dest Type\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    if (acked) *acked = rsp->data[3] & 0x80 ? 1 : 0;
    if (type) *type = rsp->data[3] & 0x07;
    if (timeout) *timeout = rsp->data[4];
    if (retries) *retries = rsp->data[5] & 0x07;
    return 0;
}

static int
ipmi_get_dest_addr(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                   uint8_t* addr_format, uint8_t* backup_gateway,
                   uint8_t* ip_addr, uint8_t* mac_addr, int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[4];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 19; /* dest addr */
    rqdata[2] = set;
    rqdata[3] = 0;

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x02; /* Get LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode || rsp->data_len < 14) {
        ipmi_printf("Error %x: Get LAN Config Params | Dest Addr\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    if (addr_format) *addr_format = rsp->data[3] >> 4;
    if (backup_gateway) *backup_gateway = rsp->data[4] & 1 ? 1 : 0;
    if (ip_addr) memcpy(ip_addr, rsp->data + 5, 4);
    if (mac_addr) memcpy(mac_addr, rsp->data + 9, 6);
    return 0;
}

static int
ipmi_get_dest_email(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                    char* email, int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[4];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 192; /* dest email (Peppercon OEM) */
    rqdata[2] = set;
    rqdata[3] = 0;

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x02; /* Get LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode || rsp->data_len < 2) {
        ipmi_printf("Error %x: Get LAN Config Params | Dest EMail\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    if (email) {
        strncpy(email, (char *)rsp->data + 3, 64);
        email[64] = '\0';
    }
    return 0;
}

static int
ipmi_lanp_dests(struct ipmi_intf* intf, uint8_t chan, pp_ipmi_return_t* ipmi_ret, int* perr)
{
    int i;
    uint8_t cnt;

    if (ipmi_init_lanp_dest_list(ipmi_ret)) {
        ipmi_printf("Error: could not allocate return structure.\n");
        return -1;
    }

    if (ipmi_get_dest_count(intf, chan, &cnt, perr)) {
        ipmi_printf("Error: failed to get LAN alert destination count.\n");
        return -1;
    }

    for (i = 1; i <= cnt; i++) {
        pp_ipmi_lanp_dest_list_entry_t *e = ipmi_init_lanp_dest_list_entry((uint8_t)i, ipmi_ret);

        if (ipmi_get_dest_type(intf, chan, (uint8_t)i, &e->acked, &e->type,
                               &e->timeout, &e->retries, perr)) {
            ipmi_printf("Error: failed to get LAN alert destination type for set %d.\n", i);
            return -1;
        }

        if (e->type == 6) {
            /* OEM type: propably Peppercon OEM email, try to read */
            if (ipmi_get_dest_email(intf, chan, (uint8_t)i, e->email, perr)) {
                ipmi_printf("Error: LAN destination %d is of type 'OEM', but not an email address.\n", i);
            }
        } else {
            /* SMNP trap */
            if (ipmi_get_dest_addr(intf, chan, (uint8_t)i, &e->addr_format, &e->backup_gateway,
                                   e->ip_addr, e->mac_addr, perr)) {
                ipmi_printf("Error: failed to get LAN alert destination address for set %d.\n", i);
                return -1;
            }
        }
    }
    return 0;
}

static int
ipmi_set_dest_type(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                   uint8_t acked, uint8_t type,
                   uint8_t timeout, uint8_t retries,
                   int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[6];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 18; /* dest type */
    rqdata[2] = set;
    rqdata[3] = (acked ? 0x80 : 0x00) | (type & 0x07);
    rqdata[4] = timeout;
    rqdata[5] = retries & 0x07;

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x01; /* Set LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode) {
        ipmi_printf("Error %x: Set LAN Config Params | Dest Type\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    return 0;
}

static int
ipmi_set_dest_addr(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                   uint8_t addr_format, uint8_t backup_gateway,
                   const uint8_t* ip_addr, const uint8_t* mac_addr,
                   int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[15];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 19; /* dest addr */
    rqdata[2] = set;
    rqdata[3] = addr_format << 4;
    rqdata[4] = backup_gateway ? 0x01 : 0x00;
    memcpy(rqdata + 5, ip_addr, 4);
    memcpy(rqdata + 9, mac_addr, 6);

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x01; /* Set LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode) {
        ipmi_printf("Error %x: Set LAN Config Params | Dest Addr\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    return 0;
}

static int
ipmi_set_dest_email(struct ipmi_intf * intf, uint8_t chan, uint8_t set,
                    const char* email, int* perr)
{
    struct ipmi_rs * rsp;
    struct ipmi_rq req;
    uint8_t rqdata[66];

    rqdata[0] = chan & 0x0f;
    rqdata[1] = 192; /* dest email (Peppercon OEM) */
    rqdata[2] = set;
    strncpy((char *)rqdata + 3, email, 64);

    req.msg.netfn = IPMI_NETFN_TRANSPORT;
    req.msg.cmd = 0x01; /* Set LAN Config Params */
    req.msg.data = rqdata;
    req.msg.data_len = sizeof(rqdata);

    rsp = intf->sendrecv(intf, &req, perr);
    if (!rsp || rsp->ccode) {
        ipmi_printf("Error %x: Set LAN Config Params | Dest EMail\n", rsp ? rsp->ccode : 0);
        return -1;
    }

    return 0;
}

static int
ipmi_lanp_set_dest(struct ipmi_intf* intf, uint8_t chan, uint8_t set,
                   const pp_ipmi_lanp_dest_t* dest,
                   int* perr)
{
    if (ipmi_set_dest_type(intf, chan, set, dest->acked, dest->type,
                           dest->timeout, dest->retries, perr)) {
        ipmi_printf("Error: failed to set LAN alert destination type for set %d.\n", set);
        return -1;
    }

    if (dest->type == 6) {
        /* Peppercon OEM type (email) */
        if (ipmi_set_dest_email(intf, chan, set, dest->email, perr)) {
            ipmi_printf("Error: failed to set LAN alert destination email for set %d.\n", set);
            return -1;
        }
    } else {
        /* SMNP trap */
        if (ipmi_set_dest_addr(intf, chan, set, dest->addr_format, dest->backup_gateway,
                                dest->ip_addr, dest->mac_addr, perr)) {
            ipmi_printf("Error: failed to set LAN alert destination address for set %d.\n", set);
            return -1;
        }
    }
    return 0;
}

int
ipmi_lanp_main(struct ipmi_intf * intf, int subcmd, pp_ipmi_parameter_t *params,
	       pp_ipmi_return_t *ipmi_ret, int * error) {
    int rc = -1;
    t_ipmitool_commandline *cmdline = get_cmdline(params);

    if (!cmdline) {
	ipmi_printf("Error: could not allocate commandline.\n");
	return -1;
    }

    switch ((pp_ipmi_lanp_subcommand_t) subcmd) {
	case PP_IPMI_LANP_SUBCMD_GET:
	    return ipmi_lan_get(intf, &params->data.lanp_get, ipmi_ret, error);
	case PP_IPMI_LANP_SUBCMD_SET_IP_SOURCE:
	    return ipmi_set_ip_addr_source(intf, params->data.lanp_set_ip_source.chan,
		params->data.lanp_set_ip_source.source, error);
	case PP_IPMI_LANP_SUBCMD_SET_SNMP_STRING:
	    return ipmi_set_snmp_string(intf, params->data.lanp_set_snmp_string.chan,
		params->data.lanp_set_snmp_string.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_IP_ADDRESS:
	    return ipmi_set_ip_addr(intf, params->data.lanp_set_ip_address.chan,
		params->data.lanp_set_ip_address.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_NETMASK:
	    return ipmi_set_netmask(intf, params->data.lanp_set_netmask.chan,
		params->data.lanp_set_netmask.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_DEF_GW_IP:
	    return ipmi_set_def_gw_ip(intf, params->data.lanp_set_def_gw_ip.chan,
		params->data.lanp_set_def_gw_ip.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_BAK_GW_IP:
	    return ipmi_set_bak_gw_ip(intf, params->data.lanp_set_bak_gw_ip.chan,
		params->data.lanp_set_bak_gw_ip.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_MAC_ADDRESS:
	    return ipmi_set_mac_addr(intf, params->data.lanp_set_mac_address.chan,
		params->data.lanp_set_mac_address.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_DEF_GW_MAC:
	    return ipmi_set_def_gw_mac(intf, params->data.lanp_set_def_gw_mac.chan,
		params->data.lanp_set_def_gw_mac.value, error);
	case PP_IPMI_LANP_SUBCMD_SET_BAK_GW_MAC:
	    return ipmi_set_bak_gw_mac(intf, params->data.lanp_set_bak_gw_mac.chan,
		params->data.lanp_set_bak_gw_mac.value, error);
	// the rest of the "set" stuff
	case PP_IPMI_LANP_SUBCMD_SET:
	    rc = ipmi_lan_set(intf, cmdline->argc, cmdline->argv, error);
	    break;
	case PP_IPMI_LANP_SUBCMD_DESTS:
	    return ipmi_lanp_dests(intf, 1, ipmi_ret, error); // FIXME: chan should be a param
	case PP_IPMI_LANP_SUBCMD_SET_DEST:
	    {
		pp_ipmi_lanp_set_dest_param_t *p;
		assert(params && !params->is_cmdline);
		p = &params->data.lanp_set_dest;
		return ipmi_lanp_set_dest(intf, p->chan, p->set, &p->dest, error);
	    }
	default:
	    ipmi_printf("Invalid LAN command: %d\n", subcmd);
	    ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
    }

    cleanup_cmdline(cmdline);
    return rc;
}
