/**
 * \file ipmi_oem_intel.c
 *
 * Description: OEM commands for Intel ESB2-BMC
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <config.h>

#include <ipmitool/ipmi.h>
#include <ipmitool/ipmi_oem_intel.h>
#include <ipmitool/ipmi_intf.h>
#include <ipmitool/ipmi_print.h>
#include <ipmitool/ipmi_error.h>
#include <ipmitool/ipmi_strings.h>

/*
static void dump_ipmi_msg(const char *txt, struct ipmi_rq *req)
{
    printf("%s: %02x %02x", txt, req->msg.netfn, req->msg.cmd);

    int i;
    for (i = 0; i < req->msg.data_len; i++) {
        printf(" %02x", req->msg.data[i]);
    }

    printf("\n");
}
*/

static int get_usr_kvm_enable(struct ipmi_intf *intf,
                              unsigned char chan,
                              unsigned char uid,
                              unsigned char *prev,
                              unsigned char *penable,
                              int *error)
{
    const char *cmd_name = "[OEM Intel] Get User Config Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[4];
    b[0] = chan & 0x7f; /* get param and revision */
    b[1] = 3; // KVM enable
    b[2] = uid;
    b[3] = 0;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_USR_CFG;
    req.msg.cmd = BMC_OEM_INTEL_GET_USR_CFG;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }
    if (rsp->data_len < 3) {
        ipmi_printf("'%s' response truncated\n", cmd_name);
        return -1;
    }

    if (prev) *prev = rsp->data[0];
    if (penable) *penable = rsp->data[2];

    return 0;
}

static int ipmi_oem_intel_get_usr_cfg(struct ipmi_intf * intf,
                                      pp_ipmi_parameter_t *params,
                                      pp_ipmi_return_t *ret,
                                      int * error)
{
    memset(&ret->data.oem_intel_get_usr_cfg, 0, sizeof(ret->data.oem_intel_get_usr_cfg));

    if ((params->data.oem_intel_get_usr_cfg.mask & PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE)
    && (-1 != get_usr_kvm_enable(intf,
                                 params->data.oem_intel_get_usr_cfg.chan,
                                 params->data.oem_intel_get_usr_cfg.uid,
                                 &ret->data.oem_intel_get_usr_cfg.revision,
                                 &ret->data.oem_intel_get_usr_cfg.cfg.kvm_enable,
                                 error))) {
        ret->data.oem_intel_get_usr_cfg.mask |= PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE;
    }

    return 0;
}

static int set_usr_kvm_enable(struct ipmi_intf *intf,
                              unsigned char chan,
                              unsigned char uid,
                              unsigned char enable,
                              int *error)
{
    const char *cmd_name = "[OEM Intel] Set User Config Params";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[4];
    b[0] = chan;
    b[1] = 3; // KVM enable
    b[2] = uid;
    b[3] = enable;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_USR_CFG;
    req.msg.cmd = BMC_OEM_INTEL_SET_USR_CFG;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}

static int ipmi_oem_intel_set_usr_cfg(struct ipmi_intf * intf,
                                      pp_ipmi_parameter_t *params,
                                      pp_ipmi_return_t *ret UNUSED,
                                      int * error)
{
    if (params->data.oem_intel_set_usr_cfg.mask & PP_IPMI_OEM_INTEL_USR_CFG_KVM_ENABLE) {
        set_usr_kvm_enable(intf,
                           params->data.oem_intel_set_usr_cfg.chan,
                           params->data.oem_intel_set_usr_cfg.uid,
                           params->data.oem_intel_set_usr_cfg.cfg.kvm_enable,
                           error);
    }

    return 0;
}

static int ipmi_oem_intel_get_tpt_status(struct ipmi_intf * intf,
                                         pp_ipmi_parameter_t *params,
                                         pp_ipmi_return_t *ret,
                                         int * error)
{
    const char *cmd_name = "[OEM Intel] Get TPT Status";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[1];
    b[0] = params->data.oem_intel_get_tpt_status.chan;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_TPT;
    req.msg.cmd = BMC_OEM_INTEL_GET_TPT_STATUS;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    ret->data.oem_intel_get_tpt_status.status = rsp->data[0];
    ret->data.oem_intel_get_tpt_status.port = rsp->data[1] | ((unsigned short)rsp->data[2] << 8);
    *(long*)ret->data.oem_intel_get_tpt_status.remote_ip = *(long*)&rsp->data[3];

    return 0;
}

static int ipmi_oem_intel_tpt_listen(struct ipmi_intf * intf,
                                     pp_ipmi_parameter_t *params,
                                     pp_ipmi_return_t *ret UNUSED,
                                     int * error)
{
    const char *cmd_name = "[OEM Intel] TPT Listen/Close";
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[6];
    b[0] = params->data.oem_intel_tpt_listen.chan & 0x0f;
    b[1] = params->data.oem_intel_tpt_listen.port & 0xff;
    b[2] = (params->data.oem_intel_tpt_listen.port >> 8) & 0xff;
    b[3] = params->data.oem_intel_tpt_listen.protocol & 0x0f;
    b[3] |= params->data.oem_intel_tpt_listen.listen ? 0x00 : 0x80;
    b[4] = params->data.oem_intel_tpt_listen.cipher_suite;
    b[5] = params->data.oem_intel_tpt_listen.coalescing & 0x03;
    b[5] |= (params->data.oem_intel_tpt_listen.linger & 0x03) << 2;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_TPT;
    req.msg.cmd = BMC_OEM_INTEL_TPT_LISTEN;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

//dump_ipmi_msg("#### ipmi_oem_intel_tpt_listen", &req);
    rsp = intf->sendrecv(intf, &req, error);
    if (!rsp) {
        ipmi_printf("'%s' failed\n", cmd_name);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s' failed: %s\n", cmd_name,
               val2str(rsp->ccode, completion_code_vals));
        return -1;
    }

    return 0;
}

static int get_tpt_param(struct ipmi_intf *intf,
                         struct ipmi_rq *get_tpt_cfg_req,
                         unsigned char idx,
                         struct ipmi_rs **prsp,
                         unsigned char data_len,
                         int *error)
{
    const char *cmd_name = "[OEM Intel] Get TPT Config Params";
    get_tpt_cfg_req->msg.data[1] = idx;

    *prsp = intf->sendrecv(intf, get_tpt_cfg_req, error);
    if (!*prsp) {
        ipmi_printf("'%s (%d)' failed\n", cmd_name, idx);
        return -1;
    }
    if ((*prsp)->ccode) {
        ipmi_printf("'%s (%d)' failed: %s\n", cmd_name,
                    idx, val2str((*prsp)->ccode, completion_code_vals));
        return -1;
    }
    if ((*prsp)->data_len > 1 && ((*prsp)->data[0] & 0x0f) > 1) {
        ipmi_printf("'%s (%d)' failed: bad param revision 0x%.2x\n", cmd_name,
                    idx, (*prsp)->data[0]);
        return -1;
    }
    if ((*prsp)->data_len < 1 + data_len) { // param revision + data
        ipmi_printf("'%s (%d)' failed: response truncated\n", cmd_name,
                    idx);
        return -1;
    }
    return 0;
}

static int ipmi_oem_intel_get_tpt_cfg(struct ipmi_intf * intf,
                                      pp_ipmi_parameter_t *params,
                                      pp_ipmi_return_t *ret,
                                      int * error)
{
    struct ipmi_rs *rsp;
    struct ipmi_rq req;

    unsigned char b[4];
    memset(b, 0, sizeof(b));
    b[0] = params->data.oem_intel_set_tpt_cfg.chan;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_TPT;
    req.msg.cmd = BMC_OEM_INTEL_GET_TPT_CFG;
    req.msg.data_len = sizeof(b);
    req.msg.data = b;

    memset(&ret->data.oem_intel_get_tpt_cfg, 0, sizeof(ret->data.oem_intel_get_tpt_cfg));

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES)
    && (-1 != get_tpt_param(intf, &req, 1, &rsp, 1, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES;
        ret->data.oem_intel_get_tpt_cfg.cfg.enabled = rsp->data[1] & 0x01 ? 1 : 0;
        ret->data.oem_intel_get_tpt_cfg.cfg.in_standby = rsp->data[1] & 0x80 ? 0 : 1;
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_ADDIN_FML_SADDR)
    && (-1 != get_tpt_param(intf, &req, 2, &rsp, 1, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_ADDIN_FML_SADDR;
        ret->data.oem_intel_get_tpt_cfg.cfg.addin_fml_saddr = rsp->data[1] >> 1;
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_PORT) 
    && (-1 != get_tpt_param(intf, &req, 3, &rsp, 2, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_PORT;
        ret->data.oem_intel_get_tpt_cfg.cfg.port =
            rsp->data[1] | ((unsigned short)rsp->data[2] << 8);
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE)
    && (-1 != get_tpt_param(intf, &req, 4, &rsp, 1, error))) {
        // TODO: this differs from spec (which seems to be wrong!)
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE;
        ret->data.oem_intel_get_tpt_cfg.cfg.fml_multi_push_mode = rsp->data[1] & 0x80 ? 1 : 0;
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_IN_SIZE)
    && (-1 != get_tpt_param(intf, &req, 5, &rsp, 2, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_IN_SIZE;
        ret->data.oem_intel_get_tpt_cfg.cfg.fml_max_in_size =
            rsp->data[1] | ((unsigned short)rsp->data[2] << 8);
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_OUT_SIZE)
    && (-1 != get_tpt_param(intf, &req, 6, &rsp, 2, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_FML_MAX_OUT_SIZE;
        ret->data.oem_intel_get_tpt_cfg.cfg.fml_max_out_size =
            rsp->data[1] | ((unsigned short)rsp->data[2] << 8);
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_OUTPUT_LIMIT)
    && (-1 != get_tpt_param(intf, &req, 7, &rsp, 2, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_FML_OUTPUT_LIMIT;
        ret->data.oem_intel_get_tpt_cfg.cfg.fml_output_limit =
            rsp->data[1] | ((unsigned short)rsp->data[2] << 8);
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN) 
    && (-1 != get_tpt_param(intf, &req, 8, &rsp, 2, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN;
        ret->data.oem_intel_get_tpt_cfg.cfg.def_ssl = rsp->data[1] & 0x01;
        ret->data.oem_intel_get_tpt_cfg.cfg.def_linger = (rsp->data[2] >> 1) & 0x01;
        ret->data.oem_intel_get_tpt_cfg.cfg.def_coalescing = rsp->data[2] & 0x01;
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_TIMEOUTS) 
    && (-1 != get_tpt_param(intf, &req, 9, &rsp, 1, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_TIMEOUTS;
        ret->data.oem_intel_get_tpt_cfg.cfg.xmit_timeout = rsp->data[1] & 0x0f;
        ret->data.oem_intel_get_tpt_cfg.cfg.recv_timeout = (rsp->data[1] >> 4) & 0x0f;
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_SUPP) 
    && (-1 != get_tpt_param(intf, &req, 10, &rsp, 1, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_SUPP;
        ret->data.oem_intel_get_tpt_cfg.cfg.ssl_supported = rsp->data[1];
    }

    if ((params->data.oem_intel_get_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_DEF) 
    && (-1 != get_tpt_param(intf, &req, 11, &rsp, 1, error))) {
        ret->data.oem_intel_get_tpt_cfg.mask |= PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_DEF;
        ret->data.oem_intel_get_tpt_cfg.cfg.def_ssl_proto = rsp->data[1] & 0x0f;
    }

    return 0;
}

static int set_tpt_param(struct ipmi_intf *intf,
                         struct ipmi_rq *set_tpt_cfg_req,
                         unsigned char idx,
                         unsigned char data_len,
                         int *error)
{
    const char *cmd_name = "[OEM Intel] Set TPT Config Params";
    struct ipmi_rs *rsp;

    set_tpt_cfg_req->msg.data_len = 1 + 1 + data_len; // chan + cmd + data
    set_tpt_cfg_req->msg.data[1] = idx;

//dump_ipmi_msg("#### set_tpt_param", set_tpt_cfg_req);
    rsp = intf->sendrecv(intf, set_tpt_cfg_req, error);
    if (!rsp) {
        ipmi_printf("'%s (%d)' failed\n", cmd_name, idx);
        return -1;
    }
    if (rsp->ccode) {
        ipmi_printf("'%s (%d)' failed: %s\n", cmd_name,
                    idx, val2str(rsp->ccode, completion_code_vals));
        return -1;
    }
    return 0;
}

static int ipmi_oem_intel_set_tpt_cfg(struct ipmi_intf * intf,
                                      pp_ipmi_parameter_t *params,
                                      pp_ipmi_return_t *ret UNUSED,
                                      int * error)
{
    struct ipmi_rq req;

    // prepare common parts of message

    unsigned char b[16];
    memset(b, 0, sizeof(b));
    b[0] = params->data.oem_intel_set_tpt_cfg.chan;

    memset(&req, 0, sizeof(req));
    req.msg.netfn = IPMI_NETFN_OEM_INTEL_TPT;
    req.msg.cmd = BMC_OEM_INTEL_SET_TPT_CFG;
    req.msg.data = b;

    // get each param

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_ENABLES) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.enabled ? 0x01 : 0x00;
        b[2] |= params->data.oem_intel_set_tpt_cfg.cfg.in_standby ? 0x00 : 0x80;
        set_tpt_param(intf, &req, 1, 1, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_ADDIN_FML_SADDR) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.addin_fml_saddr << 1;
        set_tpt_param(intf, &req, 2, 1, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_PORT) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.port & 0xff;
        b[3] = (params->data.oem_intel_set_tpt_cfg.cfg.port >> 8) & 0xff;
        set_tpt_param(intf, &req, 3, 2, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_OP_MODE) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.fml_multi_push_mode ? 0x80 : 0x00;
        set_tpt_param(intf, &req, 4, 1, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_FML_OUTPUT_LIMIT) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.fml_output_limit & 0xff;
        b[3] = (params->data.oem_intel_set_tpt_cfg.cfg.fml_output_limit >> 8) & 0xff;
        set_tpt_param(intf, &req, 7, 2, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_DEF_CONN) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.def_ssl ? 0x01 : 0x00;
        b[3] = params->data.oem_intel_set_tpt_cfg.cfg.def_coalescing ? 0x01 : 0x00;
        b[3] |= params->data.oem_intel_set_tpt_cfg.cfg.def_linger ? 0x02 : 0x00;
        set_tpt_param(intf, &req, 8, 2, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_TIMEOUTS) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.xmit_timeout & 0x0f;
        b[2] |= params->data.oem_intel_set_tpt_cfg.cfg.recv_timeout << 4;
        set_tpt_param(intf, &req, 9, 1, error);
    }

    if (params->data.oem_intel_set_tpt_cfg.mask & PP_IPMI_OEM_INTEL_TPT_CFG_SSL_PROTO_DEF) {
        b[2] = params->data.oem_intel_set_tpt_cfg.cfg.def_ssl_proto & 0x0f;
        set_tpt_param(intf, &req, 11, 1, error);
    }

    return 0;
}

int ipmi_oem_intel_usr_cfg_main(struct ipmi_intf * intf,
                                int subcmd,
                                pp_ipmi_parameter_t *params,
                                pp_ipmi_return_t *ret,
                                int * error)
{
    pp_ipmi_cleanup_ret(ret);
    switch ((pp_ipmi_kvm_subcommand_t) subcmd) {
        case PP_IPMI_OEM_INTEL_GET_USR_CFG:
            return ipmi_oem_intel_get_usr_cfg(intf, params, ret, error);
        case PP_IPMI_OEM_INTEL_SET_USR_CFG:
            return ipmi_oem_intel_set_usr_cfg(intf, params, ret, error);
        default:
            ipmi_printf("Invalid OEM Intel Usr Cfg command: %d\n", subcmd);
                ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
            return -1;
    }
}

int ipmi_oem_intel_tpt_main(struct ipmi_intf * intf,
                            int subcmd,
                            pp_ipmi_parameter_t *params,
                            pp_ipmi_return_t *ret,
                            int * error)
{
    pp_ipmi_cleanup_ret(ret);
    switch ((pp_ipmi_kvm_subcommand_t) subcmd) {
        case PP_IPMI_OEM_INTEL_GET_TPT_STATUS:
            return ipmi_oem_intel_get_tpt_status(intf, params, ret, error);
        case PP_IPMI_OEM_INTEL_TPT_LISTEN:
            return ipmi_oem_intel_tpt_listen(intf, params, ret, error);
        case PP_IPMI_OEM_INTEL_GET_TPT_CFG:
            return ipmi_oem_intel_get_tpt_cfg(intf, params, ret, error);
        case PP_IPMI_OEM_INTEL_SET_TPT_CFG:
            return ipmi_oem_intel_set_tpt_cfg(intf, params, ret, error);
        default:
            ipmi_printf("Invalid OEM Intel TPT command: %d\n", subcmd);
                        ipmi_set_error(error, IPMI_ERROR_INVALID_COMMAND);
            return -1;
    }
}
